set rcsId {$Id: notebook.tcl,v 1.24 1997/07/13 12:32:30 jfontain Exp $}

class notebookPage {}

proc notebookPage::notebookPage {this parentPath args} composite {
    [new frame $parentPath -background $widget::(default,ButtonBackgroundColor) -relief raised -borderwidth 1] $args
} {
    composite::manage $this [new button $parentPath -borderwidth 1 -highlightthickness 1] tag [new frame $parentPath] patch
    composite::complete $this
    raise $this
    place $composite::($this,patch,path) -height 2                                                               ;# set patch height
}

proc notebookPage::~notebookPage {this} {}

proc notebookPage::options {this} {
    return [list\
        [list -command command Command {} {}]\
        [list -font font Font $widget::(default,ButtonFont) $widget::(default,ButtonFont)]\
        [list -tagoffset tagOffset TagOffset 0 0]\
        [list -text text Text {} {}]\
        [list -underline underline Underline -1 -1]\
        [list -x x X 0]\
        [list -y y Y 0]\
    ]
}

foreach option {-command -font -text -underline} {
    proc notebookPage::set$option {this value} "widget::configure \$composite::(\$this,tag) $option \$value"
}

proc notebookPage::set-text {this value} {
    set tag $composite::($this,tag)
    widget::configure $tag -text $value
    place $composite::($this,patch,path) -width [expr {[winfo reqwidth $composite::($this,tag,path)]-4}]             ;# resize patch
}

proc notebookPage::set-tagoffset {this value} {
    place $composite::($this,tag,path) -x [expr {$composite::($this,-x)+$value}]
    place $composite::($this,patch,path) -x [expr {$composite::($this,-x)+$value+2}]
}

proc notebookPage::set-x {this value} {
    place $widget::($this,path) -x $value
    set-tagoffset $this $composite::($this,-tagoffset)                                                             ;# reposition tag
}

proc notebookPage::set-y {this value} {
    place $composite::($this,tag,path) -y $value
    incr value [expr {[winfo reqheight $composite::($this,tag,path)]-2}]
    place $widget::($this,path) -y $value
    place $composite::($this,patch,path) -y [expr {$value-1}]
}

proc notebookPage::raise {this args} {
    eval ::raise $composite::($this,tag,path) $args
    ::raise $widget::($this,path) $composite::($this,tag,path)
    ::raise $composite::($this,patch,path) $widget::($this,path)
}

proc notebookPage::lower {this args} {
    eval ::lower $composite::($this,patch,path) $args
    ::lower $widget::($this,path) $composite::($this,patch,path)
    ::lower $composite::($this,tag,path) $widget::($this,path)
}

proc notebookPage::cover {this other} {
    raise $this $composite::($other,patch,path)                                                       ;# use other page highest path
}

proc notebookPage::uncover {this other} {
    lower $this $composite::($other,tag,path)                                                          ;# use other page lowest path
}

proc notebookPage::tagWidth {this} {
     return [winfo reqwidth $composite::($this,tag,path)]
}

proc notebookPage::tagHeight {this} {
     return [expr {[winfo reqheight $composite::($this,tag,path)]-2}]
}


class notebook {}

set notebook::(list) {}
# these should really be added to existing bindings instead of replacing them but it seems that recursive calls to focus do not work
bind all <Tab> {focus [notebook::focus [tk_focusNext %W] tk_focusNext]}
bind all <Shift-Tab> {focus [notebook::focus [tk_focusPrev %W] tk_focusPrev]}

proc notebook::notebook {this parentPath args} composite {[new frame $parentPath] $args} {
    register $this
    set notebook::($this,parent) $parentPath
    set notebook::($this,columns) 5
    set notebook::($this,nextOffset) 0
    set notebook::($this,framesWidth) 0
    set notebook::($this,framesHeight) 0
    set notebook::($this,tagHeight) 0
    set notebook::($this,tagsHeight) 0
    set notebook::($this,columnOffsets) 0
    set notebook::($this,lastRow) 0
    set notebook::($this,pages) {}
    set notebook::($this,frontRow) 0
    set notebook::($this,active) 0
    composite::complete $this
}

proc notebook::~notebook {this} {
    eval delete $notebook::($this,pages)
    unregister $this
}

proc notebook::register {this} {
    lappend notebook::(list) $this
}

proc notebook::unregister {this} {
    set index [lsearch -exact $notebook::(list) $this]
    set notebook::(list) [lreplace $notebook::(list) $index $index]
}

proc notebook::options {this} {                      ;# force initialization on columns offset so that pixel conversion always occur
    return [list\
        [list -columns columns Columns 5 5]\
        [list -columnoffset columnOffset ColumnOffset 1c]\
        [list -font font Font $widget::(default,ButtonFont) $widget::(default,ButtonFont)]\
        [list -height height Height 0 0]\
        [list -width width Width 0 0]\
    ]
}

proc notebook::set-height {this value} {
    if {$value<=0} {
        updateHeight $this
    } else {
        $widget::($this,path) configure -height $value
    }
}

proc notebook::set-width {this value} {
    if {$value<=0} {
        updateWidth $this
    } else {
        $widget::($this,path) configure -width $value
    }
}

proc notebook::set-columns {this value} {
    if {$composite::($this,complete)} {
        error {option -columns cannot be set dynamically}
    }
}

proc notebook::set-columnoffset {this value} {
    if {$composite::($this,complete)} {
        error {option -columnoffset cannot be set dynamically}
    }
    # always convert to pixels for easier use and error checking
    set composite::($this,-columnoffset) [winfo pixels $widget::($this,path) $value]
}

proc notebook::set-font {this value} {
    if {$composite::($this,complete)} {
        error {option -font cannot be set dynamically}
    }
    foreach page $notebook::($this,pages) {
        widget::configure $page -font $value
    }
}

proc notebook::newPage {this labelText} {
    if {[info exists notebook::($this,selected)]} {
        error "new page cannot be added dynamically"
    }
    set page [new notebookPage $widget::($this,path) -text $labelText]

    # use a frame for clipping so that page border is not obscured when notebook is resized below child user widget requested size
    set path $widget::([new frame $widget::($page,path)],path)
    place $path -relwidth 1 -relheight 1 -width -1 -height -1
    set notebook::($this,pageFrame,$page) $path

    # place sub-widgets centered in a frame with their upper left corner always visible, thanks to the grid manager behavior
    set path $widget::([new frame $path],path)
    grid $path
    bind $path <Configure> "notebook::update $this $path"

    if {[llength $notebook::($this,pages)]==0} {                                   ;# initialize a few things at first page creation
        # assume tags height is constant
        set notebook::($this,tagsHeight) [set notebook::($this,tagHeight) [notebookPage::tagHeight $page]]
        # highlight first page as it is above and therefore should look selected
        widget::configure $page tag -background [widget::cget $page tag -activebackground]
        set notebook::($this,active) $page
    } else {                                                         ;# always place new page below the others and make it invisible
        notebookPage::uncover $page [lindex $notebook::($this,pages) end]
        place $notebook::($this,pageFrame,$page) -anchor se
    }
    widget::configure $page -command "notebook::select $this $page"
    lappend notebook::($this,pages) $page
    placeNew $this $page
    return $path
}

proc notebook::placeNew {this page} {
    set row [expr {([llength $notebook::($this,pages)]-1)/$composite::($this,-columns)}]
    if {$row>$notebook::($this,lastRow)} {
        set notebook::($this,lastRow) $row
        incr notebook::($this,columnOffsets) $composite::($this,-columnoffset)
        incr notebook::($this,tagsHeight) $notebook::($this,tagHeight) 
        for {set index 0} {$index<$row} {incr index} {                                                   ;# reposition existing rows
            positionRow $this $index $index
        }
        set notebook::($this,nextOffset) 0
        # make existing page frames follow notebook widget configuration according to number of rows and column offset
        for {set index 0} {$index<$row} {incr index} {
            foreach other $notebook::($this,rowPages,$index) {
                place $widget::($other,path)\
                    -relwidth 1 -relheight 1 -width -$notebook::($this,columnOffsets) -height -$notebook::($this,tagsHeight)
            }
        }
        updateWidth $this                                       ;# make sure sizes are correct in case we get no configuration event
        updateHeight $this
    }
    place $widget::($page,path) -relwidth 1 -relheight 1\
        -width -$notebook::($this,columnOffsets) -height -$notebook::($this,tagsHeight)
    set notebook::($this,rowPosition,$row) $row
    lappend notebook::($this,rowPages,$row) $page                               ;# store row pages in separate lists for fast access
    set notebook::($this,pagePosition,$page) $row
    set notebook::($this,row,$page) $row
    widget::configure $page -x [expr {$row*$composite::($this,-columnoffset)}] -tagoffset $notebook::($this,nextOffset)
    incr notebook::($this,nextOffset) [notebookPage::tagWidth $page]
    if {$notebook::($this,nextOffset)>$notebook::($this,framesWidth)} {                            ;# make sure all tags are visible
        set notebook::($this,framesWidth) $notebook::($this,nextOffset)
        updateWidth $this
    }
}

proc notebook::updateWidth {this} {
    $widget::($this,path) configure -width [expr {$notebook::($this,framesWidth)+$notebook::($this,columnOffsets)}]
}

proc notebook::updateHeight {this} {
    $widget::($this,path) configure -height [expr {$notebook::($this,framesHeight)+$notebook::($this,tagsHeight)}]
}

proc notebook::update {this path} {
    # take the page border size into account, add 1 pixel to sizes because placer fails to correctly center widgets with odd sizes
    set width [expr {[winfo reqwidth $path]+3}]
    if {$width>$notebook::($this,framesWidth)} {
        set notebook::($this,framesWidth) $width
        updateWidth $this
    }
    set height [expr {[winfo reqheight $path]+3}]
    if {$height>$notebook::($this,framesHeight)} {
        set notebook::($this,framesHeight) $height
        updateHeight $this
    }
}

proc notebook::select {this page} {
    set notebook::($this,selected) {}
    if {$notebook::($this,pagePosition,$page)!=0} {
        set row $notebook::($this,frontRow)                                 ;# if not already in front, swap this row with front row
        positionRow $this $notebook::($this,frontRow) $notebook::($this,pagePosition,$page)
        positionRow $this $notebook::($this,row,$page) 0
        raiseRow $this $notebook::($this,row,$page)                                             ;# make sure selected row is visible
        lowerRow $this $row
    }
    notebookPage::raise $page
    if {$notebook::($this,active)!=0} {                 ;# use non active page tag background to unhighlight previously selected tag
        widget::configure $notebook::($this,active) tag -background [widget::cget $page tag -background]
        place $notebook::($this,pageFrame,$notebook::($this,active)) -anchor se                           ;# and make page invisible
    }
    widget::configure $page tag -background [widget::cget $page tag -activebackground]     ;# highlight tag to better show selection
    place $notebook::($this,pageFrame,$page) -anchor nw                                                         ;# make page visible
    set notebook::($this,active) $page
}

proc notebook::positionRow {this index position} {
    if {$position==0} {
        set notebook::($this,frontRow) $index
    }
    set notebook::($this,rowPosition,$index) $position
    set y [expr {($notebook::($this,lastRow)-$position)*$notebook::($this,tagHeight)}]
    set x [expr {$position*$composite::($this,-columnoffset)}]
    foreach page $notebook::($this,rowPages,$index) {
        set notebook::($this,pagePosition,$page) $position
        widget::configure $page -x $x -y $y
    }
}

proc notebook::raiseRow {this index} {
    foreach page [lsort -integer -decreasing $notebook::($this,rowPages,$index)] {
        notebookPage::raise $page
    }
}

proc notebook::lowerRow {this index} {
    set row $index
    # find last page of row in front of this one, which is guaranteed to be the lowest among its row pages
    set position [expr {$notebook::($this,rowPosition,$index)-1}]
    for {set index 0} {$notebook::($this,rowPosition,$index)!=$position} {incr index} {}
    set last [lindex $notebook::($this,rowPages,$index) end]
    foreach page [lsort -integer -decreasing $notebook::($this,rowPages,$row)] {
        notebookPage::uncover $page $last
    }
}

proc notebook::geometryManager {window} {
    set manager [winfo manager $window]
    if {[string length $manager]==0} {
        return {}
    }
    ### at this time canvas and text managers are not supported, due to the high complexity of finding the manager widget name ###
    switch $manager {
        grid -
        pack {
            array set data [$manager info $window]
            return $data(-in)
        }
        place {
            array set data [place info $window]
            if {[info exists data(-in)]} {
                return $data(-in)
            } else {
                return [winfo parent $window]
            }
        }
    }
}

# find whether window or one of his ancestors is managed geometry-wise by a notebook page
proc notebook::managingPage {this window} {
    set toplevel [winfo toplevel $window]
    while {[string compare $window $toplevel]!=0} {
        set manager [geometryManager $window]
        if {[string length $manager]==0} {
            return {}
        }
        foreach page $notebook::($this,pages) {
            if {[string compare $manager $widget::($page,path)]==0} {
                return $page
            }
        }
        set window $manager                                                                                    ;# up to next manager
    }
    return {}
}

proc notebook::focus {window next} {
    if {[string length $window]==0} {
        return {}
    }
    foreach book $notebook::(list) {
        if {![winfo exists $widget::($book,path)]} {                                                 ;# book may have been destroyed
            unregister $book
            continue
        }
        set page [managingPage $book $window]
        if {[string length $page]>0} {                                                        ;# found page that manages this widget
            if {$page==$notebook::($book,active)} {
                return $window                                                                ;# page in front, widget can get focus
            } else {
                return [focus [$next $window] $next]                               ;# not in front, see if next widget can get focus
            }
        }
    }
    return $window                                          ;# widget is not managed by any notebook, therefore it can get the focus
}
