set rcsId {$Id: stooop.tcl,v 3.22 1997/08/19 21:07:11 jfontain Exp $}

package provide stooop 3.0

catch {rename proc _proc}            ;# rename proc before it is overloaded, ignore error in case of multiple inclusion of this file

namespace eval ::stooop {
    namespace export class virtual                       ;# class and virtual commands are always needed even with dynamic extension
}

if {[llength [info commands ::stooop::new]]==0} {                                              ;# if dynamic extension not available

    namespace eval ::stooop {
        namespace export new delete classof           ;# export public commands otherwise created in the dynamic extension if loaded
        if {![info exists newId]} {
            variable newId 0                    ;# initialize object id counter only once even if this file is sourced several times
        }
    }

    _proc ::stooop::new {classOrId args} {                         ;# create an object of specified class or copy an existing object
        variable newId
        variable class

        # use local variable for identifier because new can be invoked recursively
        if {[scan $classOrId %d dummy]==0} {                                                            ;# first argument is a class
            # we could detect here whether class was ever declared but that would prevent stooop packages to load properly, because
            # constructor would not be invoked and thus class source file never sourced
            set class([set id [incr newId]]) $classOrId
            # invoke the constructor for the class with optional arguments in caller's variable context so that object creation is
            # transparent and that array names as constructor parameters work with a simple upvar
            uplevel ::${classOrId}::$classOrId $id $args
        } else {                      ;# first argument is an object identifier, copy source object to new object of identical class
            if {[catch {set class([set id [incr newId]]) $class($classOrId)}]} {
                error "invalid object identifier $classOrId"
            }
            # invoke the copy constructor for the class in caller's variable context so that object copy is transparent (see above)
            uplevel ::$class($classOrId)::_copy $id $classOrId
        }
        return $id                                                                              ;# return a unique object identifier
    }

    _proc ::stooop::delete {args} {                                                                    ;# delete one or more objects
        variable class

        foreach id $args {
            _delete $class($id) $id
            unset class($id)
        }
    }

    # delete object data starting at specified class layer and going up the base class hierarchy if any
    # invoke the destructor for the object class and unset all the object data members for the class
    # the destructor will in turn delete the base classes layers
    _proc ::stooop::_delete {class id} {
        # invoke the destructor for the class in caller's variable context so that object deletion is transparent
        uplevel 2 ::${class}::~$class $id
        # delete all this object data members if any (assume that they were stored as ${class}::($id,memberName))
        foreach name [array names ::${class}:: $id,*] {
            unset ::${class}::($name)
        }
        # data member arrays deletion is left to the user
    }

    _proc ::stooop::classof {id} {
        variable class

        return $class($id)                                                                                ;# return class of object
    }

    _proc ::stooop::copy {class from to} {                                    ;# copy object data members from one object to another
        set index [string length $from]
        foreach name [array names ::${class}:: $from,*] {                                               ;# copy regular data members
            set ::${class}::($to[string range $name $index end]) [set ::${class}::($name)]
        }
        # if any, array data members copy is left to the class programmer through the then mandatory copy constructor
    }

}                                                                         ;# end of tcl code used if dynamic extension not available

_proc ::stooop::class {args} {
    variable declaring
    variable declared

    set declaring [lindex $args 0]                                                                     ;# trace class being declared
    if {[string match *::* $declaring]} {
        error "class $declaring name must not be qualified"
    }
    # make the class namespace a direct child of the global namespace and create the empty name array used to hold all class objects
    # so that static members can be directly initialized within the class declaration but outside member procedures
    uplevel namespace eval ::$declaring [list "::variable {}\n[lindex $args end]"]
    set declared($declaring) {}      ;# class is now successfully declared, otherwise a tcl error would prevent us from getting here
    unset declaring
}

# virtual operator, to be placed before proc
# virtualize a member procedure, determine whether it is a pure virtual, check for procedures that cannot be virtualized
_proc ::stooop::virtual {keyword name arguments args} {
    variable declaring

    if {![regexp {^(::)?proc$} $keyword]} {
        error "virtual operator works only on proc, not $keyword"
    }
    if {[catch {set declaring} class]} {                                                                  ;# not in class definition
        if {![regexp {^(.+)::(.+)$} $name dummy class procedure]} {
            error "$name is not a valid member procedure name"
        }
    } else {                                                                         ;# procedure definition within class definition
        set procedure $name
    }
    if {[string compare $class $procedure]==0} {
        error "cannot make class $class constructor virtual"
    }
    if {[string compare ~$class $procedure]==0} {
        error "cannot make class $class destructor virtual"
    }
    if {[string compare [lindex $arguments 0] this]!=0} {
        error "cannot make static procedure $procedure of class $class virtual"
    }
    variable pureVirtual     ;# set a flag so that proc knows it is acting upon a virtual procedure, also serves as a pure indicator
    set pureVirtual [expr {[llength $args]==0}]                                              ;# no procedure body means pure virtual
    # process procedure declaration, body being empty for pure virtual procedure
    proc $name $arguments [lindex $args 0]
    unset pureVirtual
}

_proc proc {procedure arguments args} {
    if {
        [catch {set ::stooop::declaring} class]&&
        (![regexp {^([^:]+)::(.+)$} $procedure dummy class procedure]||![info exists ::stooop::declared($class)])
    } {   ;# we are not within a class definition nor in a member procedure definition outside a previously defined class definition
        # uplevel is required instead of eval here otherwise tcl seems to forget the procedure namespace if it exists
        uplevel _proc [list $procedure $arguments] $args
        return                                            ;# not a class member procedure, fall back to normal procedure declaration
    }

    if {[llength $args]==0} {                                                                   ;# check for procedure body presence
        error "missing body for ${class}::$procedure"
    }
    if {[string match *::* $procedure]} {                                       ;# check that member procedure name is not qualified
        error "${class}::$procedure procedure member part $procedure must not be qualified"
    }
    if {[string compare $class $procedure]==0} {                                                     ;# class constructor definition
        if {[string compare [lindex $arguments 0] this]!=0} {
            error "class $class constructor first argument must be this"
        }
        if {[string compare [lindex $arguments 1] copy]==0} {                            ;# user defined copy constructor definition
            if {[llength $arguments]!=2} {
                error "class $class copy constructor must have 2 arguments exactly"
            }
            if {[catch {info body ::${class}::$class}]} {                                   ;# make sure of proper declaration order
                error "class $class copy constructor defined before constructor"
            }
            eval ::stooop::constructorDeclaration $class 1 \{$arguments\} $args
        } else {                                                                                                 ;# main constructor
            eval ::stooop::constructorDeclaration $class 0 \{$arguments\} $args
            ::stooop::generateDefaultCopyConstructor $class
        }
    } elseif {[string compare ~$class $procedure]==0} {                                              ;# class destructor declaration
        if {[llength $arguments]!=1} {
            error "class $class destructor must have 1 argument exactly"
        }
        if {[string compare [lindex $arguments 0] this]!=0} {
            error "class $class destructor argument must be this"
        }
        if {[catch {info body ::${class}::$class}]} {                          ;# use fastest method for testing procedure existence
            error "class $class destructor defined before constructor"                      ;# make sure of proper declaration order
        }
        ::stooop::destructorDeclaration $class $arguments [lindex $args 0]
    } else {                                           ;# regular member procedure, may be static if there is no this first argument
        if {[catch {info body ::${class}::$class}]} {                                       ;# make sure of proper declaration order
            error "class $class member procedure $procedure defined before constructor"
        }
        ::stooop::memberProcedureDeclaration $class $procedure $arguments [lindex $args 0]
    }
}

_proc ::stooop::constructorDeclaration {class copy arguments args} {           ;# copy flag is set for user defined copy constructor
    variable bases
    variable variable

    set number [llength $args]
    if {($number%2)==0} {                                                    ;# check that each base class constructor has arguments
        error "bad $class constructor declaration, a base class, contructor arguments or body may be missing"
    }

    if {[string compare [lindex $arguments end] args]==0} {
        set variable($class) {}                        ;# remember that there is a variable number of arguments in class constructor
    }
    if {!$copy} {
        # do not initialize (or reinitialize in case of multiple class file source statements) base classes for copy constructor
        set bases($class) {}
    }
    foreach {base baseArguments} [lrange $args 0 [expr {$number-2}]] {         ;# check base classes and their constructor arguments
        if {!$copy} {                                     ;# check and save base classes only for main constructor that defines them
            if {[lsearch -exact $bases($class) $base]>=0} {
                error "class $class directly inherits from class $base more than once"
            }
            lappend bases($class) $base
            # in case base class is defined in a file that is part of a package, make sure that file is sourced through the tcl
            # package auto-loading mechanism by directly invoking the base class constructor while ignoring the resulting error
            catch {::${base}::$base}
        }
        # remove new lines in base arguments part in case user has formatted long declarations with new lines
        regsub -all \n $baseArguments {} constructorArguments($base)
    }

    # setup access to class data (an empty named array)
    # fully qualify tcl variable command for it may have been redefined within the class namespace
    set constructorBody \
"::variable {}
"
    if {[llength $bases($class)]>0} {                                                         ;# base class(es) derivation specified
        foreach base $bases($class) {
            if {[string compare $class $base]==0} {
                error "class $class cannot be derived from itself"
            }
            if {[catch {info body ::${base}::$base}]} {                                     ;# make sure of proper declaration order
                error "class $class constructor defined before base class $base constructor"
            }
        }
        # invoke base class constructors before evaluating constructor body
        # then set base part hidden derived member so that virtual procedures are invoked at base class level as in C++
        if {[info exists variable($class)]} {                           ;# variable number of arguments in derived class constructor
            foreach base $bases($class) {
                if {![info exists constructorArguments($base)]} {
                    error "missing base class $base constructor arguments from class $class constructor"
                }
                if {[info exists variable($base)]&&([string first {$args} $constructorArguments($base)]>=0)} {
                    # variable number of arguments in base class constructor and in derived class base class constructor arguments
                    # use eval so that base class constructor sees arguments instead of a list
                    # only the last argument of the base class constructor arguments is considered as a variable list
                    # (it usually is $args but could be a procedure invocation, such as [filter $args])
                    # fully qualify tcl commands such as set, for they may have been redefined within the class namespace
                    append constructorBody \
"::set _list \[::list $constructorArguments($base)\]
::eval ::${base}::$base \$this \[::lrange \$_list 0 \[::expr {\[::llength \$_list\]-2}\]\] \[::lindex \$_list end\]
::unset _list
::set ::${base}::(\$this,_derived) $class
"
                } else {
                    # no special processing needed
                    # variable number of arguments in base class constructor or
                    # variable arguments list passed as is to base class constructor
                    append constructorBody \
"::${base}::$base \$this $constructorArguments($base)
::set ::${base}::(\$this,_derived) $class
"
                }
            }
        } else {                                                                                     ;# constant number of arguments
            foreach base $bases($class) {
                if {![info exists constructorArguments($base)]} {
                    error "missing base class $base constructor arguments from class $class constructor"
                }
                append constructorBody \
"::${base}::$base \$this $constructorArguments($base)
::set ::${base}::(\$this,_derived) $class
"
            }
        }
    }                                                                                     ;# else no base class derivation specified
    if {$copy} {                                        ;# for user defined copy constructor, copy derived class member if it exists
        append constructorBody \
"::catch {::set ::${class}::(\$this,_derived) \[::set ::${class}::(\$[::lindex $arguments 1],_derived)\]}
"
    }
    append constructorBody [lindex $args end]                                          ;# finally append user defined procedure body
    # fully qualify member procedure names as we know that class namespace is a direct child of the global namespace
    if {$copy} {
        _proc ::${class}::_copy $arguments $constructorBody
    } else {
        _proc ::${class}::$class $arguments $constructorBody
    }
}

_proc ::stooop::destructorDeclaration {class arguments body} {
    variable bases

    # setup access to class data
    set body \
"::variable {}
$body
"
    # if there are any, delete base classes parts in reverse order of construction
    for {set index [expr {[llength $bases($class)]-1}]} {$index>=0} {incr index -1} {
        set base [lindex $bases($class) $index]
        append body \
"::stooop::_delete $base \$this
"
    }
    _proc ::${class}::~$class $arguments $body
}

_proc ::stooop::memberProcedureDeclaration {class procedure arguments body} {
    variable pureVirtual

    if {[info exists pureVirtual]} {                                                                          ;# virtual declaration
        if {$pureVirtual} {                                                                              ;# pure virtual declaration
            # setup access to class data
            # evaluate derived procedure which must exists. derived procedure return value is automatically returned
            _proc ::${class}::$procedure $arguments \
"::variable {}
::eval \$::${class}::(\$this,_derived)::$procedure \[::lrange \[::info level 0\] 1 end\]
"
        } else {                                                                                      ;# regular virtual declaration
            # setup access to class data
            # evaluate derived procedure and return if it exists
            # else evaluate the base class procedure which can be invoked from derived class procedure by prepending _
            _proc ::${class}::_$procedure $arguments \
"::variable {}
$body
"
            _proc ::${class}::$procedure $arguments \
"::variable {}
if {!\[::catch {::info body \$::${class}::(\$this,_derived)::$procedure}\]} {
::return \[::eval \$::${class}::(\$this,_derived)::$procedure \[::lrange \[::info level 0\] 1 end\]\]
}
::eval ::${class}::_$procedure \[::lrange \[::info level 0\] 1 end\]
"
        }
    } else {                                                                                              ;# non virtual declaration
        # setup access to class data
        _proc ::${class}::$procedure $arguments \
"::variable {}
$body
"
    }
}

# generate default copy procedure which may be overriden by the user for any class layer
_proc ::stooop::generateDefaultCopyConstructor {class} {
    variable bases

    foreach base $bases($class) {               ;# generate code for cloning base classes layers if there is at least one base class
        append body \
"::${base}::_copy \$this \$sibling
"
    }
    append body \
"::stooop::copy $class \$sibling \$this
"
    _proc ::${class}::_copy {this sibling} $body
}
