/* $Id: stooop.c,v 1.60 1996/10/27 17:34:10 jfontain Exp $ */

/*
    # how to create a dynamically loadable library on different platforms (please send me commands for your platform if not listed):
    # TCLSRC is the Tcl 7.5 source directory path name, TCLSRC=/usr/sources/tcl7.5 for example

    # SunOS
    cc -O2 -pic -I$TCLSRC/generic -c stooop.c                                                                  # with stock compiler
    gcc -O2 -fPIC -Wall -I$TCLSRC/generic -c stooop.c                                                                     # with gcc
    ld -assert pure-text -o libstooop2.3.so stooop.o

    # AT&T SVR4.0
    cc -O -KPIC -Xa -v -I$TCLSRC/generic -c stooop.c                                                           # with stock compiler
    gcc -O2 -fPIC -Wall -I$TCLSRC/generic -c stooop.c                                                                     # with gcc
    ld -G -o libstooop2.3.so -h libstooop2.3.so stooop.o

    # linux
    cc -shared -o libstooop2.3.so -O2 -fPIC -Wall -I$TCLSRC/generic stooop.c

    # AIX 4.1
    gcc -O2 -fPIC -Wall -I$TCLSRC/generic -c stooop.c                                                                     # with gcc
    # use ldAix command from Tcl 7.5 distribution
    $TCLSRC/unix/ldAix /bin/ld -bhalt:4 -bM:SRE -bE:lib.exp -H512 -T512 -o libstooop2.3.so stooop.o -L/usr/local/lib -ltcl7.5 -lc

    # (the Tcl generic source code directory is needed for tclInt.h, its location may be different on your machine)

    # to use it in your Tcl code, just load in stooop.tcl, as in:
        load libstooop2.3.so
*/

#include <stdlib.h>
#ifdef VARARGS
#include <stdarg.h>
#endif /* VARARGS */
#include <tcl.h>
#include <tclInt.h>

static struct Tcl_HashTable identifiers;
static struct Tcl_HashTable classes;
static int newId;

#ifdef VARARGS
static int invokeCommand(char *name, int (*command)(), ClientData clientData, Tcl_Interp *interpreter, ...)
{
    va_list list;
    int i;
    unsigned number;
    char **arguments;

    va_start(list, interpreter);
    for(number = 0; va_arg(list, char *) != 0; number++);                                    /* arguments must be zero terminated */
    va_end(list);
    number++;                                                                                /* count command name first argument */

    arguments = malloc(number * sizeof(char *));
    va_start(list, interpreter);
    i = 0;
    arguments[i++] = name;
    for(; i < number; i++)
        arguments[i] = va_arg(list, char *);
    va_end(list);
    i = command(clientData, interpreter, number, arguments);
    free(arguments);
    return i;
}
#endif /* VARARGS */

static unsigned addIdentifier(char *class, int identifier)          /* returns a boolean, true if identifier was added to classes */
{                                            /* associative array, which should always be the case since the identifier is unique */
    struct Tcl_HashEntry *entry;
    char *pc;
    int created;

    entry = Tcl_CreateHashEntry(&classes, class, &created);                                    /* first eventually register class */
    if(created){                                                                                                     /* new class */
        pc = malloc(strlen(class) + 1);            /* class entries are never freed for there should not be a huge number of them */
        strcpy(pc, class);
        Tcl_SetHashValue(entry, pc);
    } else
        pc = (char *)Tcl_GetHashValue(entry);

    Tcl_SetHashValue(Tcl_CreateHashEntry(&identifiers, (ClientData)identifier, &created), pc);         /* now register identifier */
    return created;
}

static int copyObject(Tcl_Interp *interpreter, int object, char *name)                     /* object identifier and its text form */
{
    struct Tcl_HashEntry *entry;
    struct Tcl_DString string;
    char *class;
    int identifier;                                   /* use local variable for identifier because new can be invoked recursively */

    entry = Tcl_FindHashEntry(&identifiers, (ClientData)object);                                             /* find object class */
    if(entry == 0){
        sprintf(interpreter->result, "invalid object identifier \"%s\"", name);
        return TCL_ERROR;
    }
    class = (char *)Tcl_GetHashValue(entry);
    identifier = ++newId;                                                                                   /* set new identifier */
    Tcl_DStringInit(&string);
    Tcl_DStringAppend(&string, class, -1);                                         /* build class copy constructor procedure call */
    Tcl_DStringAppend(&string, "::_copy ", 8);
    sprintf(interpreter->result, "%d", identifier);                                                        /* with new identifier */
    Tcl_DStringAppend(&string, interpreter->result, -1);
    Tcl_DStringAppend(&string, " ", 1);
    Tcl_DStringAppend(&string, name, -1);                                                               /* and sibling identifier */
    if(Tcl_Eval(interpreter, Tcl_DStringValue(&string)) != TCL_OK)
        return TCL_ERROR;
    Tcl_DStringFree(&string);
    if(!addIdentifier(class, identifier)){
        sprintf(interpreter->result, "fatal error: could not register identifier for new object of class %s", class);
        return TCL_ERROR;
    }
    Tcl_ResetResult(interpreter);                              /* reset to safe static result buffer for eval probably changed it */
    sprintf(interpreter->result, "%d", identifier);                                                      /* return new identifier */
    return TCL_OK;
}

static int new(ClientData clientData, Tcl_Interp *interpreter, int numberOfArguments, char *arguments[])
{
    struct Tcl_DString string;
    int identifier;                                   /* use local variable for identifier because new can be invoked recursively */
    unsigned u;
    char *class;

    if(numberOfArguments < 2){
        sprintf(interpreter->result, "wrong number of arguments, should be: \"%s class ?arg arg ...?\"", arguments[0]);
        return TCL_ERROR;
    }

    if((identifier = atoi(arguments[1])) != 0)
        return copyObject(interpreter, identifier, arguments[1]);          /* first argument is an object identifier: copy object */

    class = arguments[1];
    identifier = ++newId;                                                     /* set new identifier (arrange for 0 to be invalid) */
    Tcl_DStringInit(&string);
    Tcl_DStringAppend(&string, class, -1);                                                              /* call class constructor */
    Tcl_DStringAppend(&string, "::", 2);
    Tcl_DStringAppend(&string, class, -1);
    Tcl_DStringAppend(&string, " ", 1);
    sprintf(interpreter->result, "%d", identifier);                                                        /* with new identifier */
    Tcl_DStringAppend(&string, interpreter->result, -1);
    for(u = 2; u < numberOfArguments; u++)                                                           /* and constructor arguments */
        Tcl_DStringAppendElement(&string, arguments[u]);
    if(Tcl_Eval(interpreter, Tcl_DStringValue(&string)) != TCL_OK)
        return TCL_ERROR;
    Tcl_DStringFree(&string);
    if(!addIdentifier(class, identifier)){
        sprintf(interpreter->result, "fatal error: could not register identifier for new object of class %s", class);
        return TCL_ERROR;
    }
    Tcl_ResetResult(interpreter);                              /* reset to safe static result buffer for eval probably changed it */
    sprintf(interpreter->result, "%d", identifier);                                                      /* return new identifier */
    return TCL_OK;
}

static int classOf(ClientData clientData, Tcl_Interp *interpreter, int numberOfArguments, char *arguments[])
{
    struct Tcl_HashEntry *entry;

    if(numberOfArguments != 2){
        sprintf(interpreter->result, "wrong number of arguments, should be: \"%s identifier\"", arguments[0]);
        return TCL_ERROR;
    }
    entry = Tcl_FindHashEntry(&identifiers, (ClientData)atoi(arguments[1]));
                          /* works because atoi returns 0 if string is not a valid integer and 0 is not a valid object identifier */
    if(entry == 0){
        sprintf(interpreter->result, "invalid object identifier \"%s\"", arguments[1]);
        return TCL_ERROR;
    }
    interpreter->result = (char *)Tcl_GetHashValue(entry);
    return TCL_OK;
}

static int deleteObject(ClientData clientData, Tcl_Interp *interpreter, int numberOfArguments, char *arguments[])
{                                                                        /* warning: for internal use only, no arguments checking */
    struct Tcl_DString string;
    char *class;
    char *nIdentifier;
    int i;
    Tcl_HashEntry *entry;
    Tcl_HashSearch search;
    Var *array;
    char *n;

    class = arguments[1];
    nIdentifier = arguments[2];

    Tcl_DStringInit(&string);
    Tcl_DStringAppend(&string, class, -1);                                                               /* call class destructor */
    Tcl_DStringAppend(&string, "::~", 3);
    Tcl_DStringAppend(&string, class, -1);
    Tcl_DStringAppend(&string, " ", 1);
    Tcl_DStringAppend(&string, nIdentifier, -1);                                                               /* with identifier */
    i = Tcl_Eval(interpreter, Tcl_DStringValue(&string));
    Tcl_DStringFree(&string);
    if(i != TCL_OK)
        return TCL_ERROR;

    entry = Tcl_FindHashEntry(&((Interp *)interpreter)->globalTable, class);                /* class array is at the global level */
    if(entry != 0){                                                                                            /* if array exists */
        Tcl_DStringAppend(&string, nIdentifier, -1);                                  /* build array index prefix with identifier */
        Tcl_DStringAppend(&string, ",", 1);                                                                          /* and comma */
        i = strlen(Tcl_DStringValue(&string));
        array = (Var *)Tcl_GetHashValue(entry);
        for(entry = Tcl_FirstHashEntry(array->value.tablePtr, &search); entry != 0; entry = Tcl_NextHashEntry(&search)){
            n = Tcl_GetHashKey(array->value.tablePtr, entry);
            if(strncmp(n, Tcl_DStringValue(&string), i) == 0)                                    /* unset all object data members */
                Tcl_UnsetVar2(interpreter, class, n, TCL_GLOBAL_ONLY);
        }
        Tcl_DStringFree(&string);
    }

    Tcl_ResetResult(interpreter);                                                                               /* return nothing */
    return TCL_OK;
}

static int delete(ClientData clientData, Tcl_Interp *interpreter, int numberOfArguments, char *arguments[])
{
    unsigned u;
    struct Tcl_HashEntry *entry;
#ifndef VARARGS
    char *an[3];
#endif /* VARARGS */

#ifndef VARARGS
    an[0] = "_delete";                                                                  /* invoke the internal delete Tcl command */
#endif /* VARARGS */
    for(u = 1; u < numberOfArguments; u++){
        entry = Tcl_FindHashEntry(&identifiers, (ClientData)atoi(arguments[u]));
        if(entry == 0){
            sprintf(interpreter->result, "invalid object identifier \"%s\"", arguments[u]);
            return TCL_ERROR;
        }
#ifndef VARARGS
        an[1] = (char *)Tcl_GetHashValue(entry);
        an[2] = arguments[u];
        if(deleteObject(clientData, interpreter, 3, an) != TCL_OK)
#else
        if(
            invokeCommand("_delete", deleteObject, clientData, interpreter, (char *)Tcl_GetHashValue(entry), arguments[u], 0) !=
            TCL_OK
        )
#endif /* VARARGS */
            return TCL_ERROR;
        Tcl_DeleteHashEntry(entry);                                             /* remove identifier from class associative array */
    }
    return TCL_OK;
}

static int copyArrayMembers(Tcl_Interp *interpreter, char *class, char *from, char *to)
{
    struct Tcl_DString fromString;
    struct Tcl_DString toString;
    Tcl_HashEntry *entry;
    Tcl_HashSearch search;
    Var *array;
    char *arrayName;
    char *n;
    int fromLength;

    Tcl_DStringInit(&fromString);
    Tcl_DStringInit(&toString);

    Tcl_DStringAppend(&fromString, class, -1);                      /* build global array name prefix by concatenating class name */
    Tcl_DStringAppend(&fromString, from, -1);                                                            /* and object identifier */
    fromLength = strlen(Tcl_DStringValue(&fromString));

    for(
        entry = Tcl_FirstHashEntry(&((Interp *)interpreter)->globalTable, &search); entry != 0;
        entry = Tcl_NextHashEntry(&search)
    ){
        arrayName = Tcl_GetHashKey(&((Interp *)interpreter)->globalTable, entry);
        if(strncmp(arrayName, Tcl_DStringValue(&fromString), fromLength) != 0)
            continue;                                           /* filter out variables that are not array members for this class */
        Tcl_DStringAppend(&toString, class, -1);                                  /* build destination array name with class name */
        Tcl_DStringAppend(&toString, to, -1);                                                    /* destination object identifier */
        Tcl_DStringAppend(&toString, arrayName + fromLength, -1);                                        /* and member array name */
        array = (Var *)Tcl_GetHashValue(entry);
        for(entry = Tcl_FirstHashEntry(array->value.tablePtr, &search); entry != 0; entry = Tcl_NextHashEntry(&search)){
            n = Tcl_GetHashKey(array->value.tablePtr, entry);
            Tcl_SetVar2(                                                                        /* arrays are at the global level */
                interpreter, Tcl_DStringValue(&toString), n, Tcl_GetVar2(interpreter, arrayName, n, TCL_GLOBAL_ONLY),
                TCL_GLOBAL_ONLY
            );
        }
        Tcl_DStringFree(&toString);
    }

    Tcl_DStringFree(&fromString);

    return TCL_OK;
}

static int copy(ClientData clientData, Tcl_Interp *interpreter, int numberOfArguments, char *arguments[])
{                                                                        /* warning: for internal use only, no arguments checking */
    struct Tcl_DString string;
    struct Tcl_DString fromString;
    struct Tcl_DString toString;
    char *class;
    char *from;
    char *to;
    int fromLength;
    int toLength;
    Tcl_HashEntry *entry;
    Tcl_HashSearch search;
    Var *array;
    char *n;

    class = arguments[1];
    from = arguments[2];
    to = arguments[3];

    Tcl_DStringInit(&string);
    Tcl_DStringInit(&fromString);
    Tcl_DStringInit(&toString);

    entry = Tcl_FindHashEntry(&((Interp *)interpreter)->globalTable, class);                /* class array is at the global level */
    if(entry != 0){                                                                                            /* if array exists */
        Tcl_DStringAppend(&fromString, from, -1);                              /* build array index prefix with source identifier */
        Tcl_DStringAppend(&fromString, ",", 1);                                                                      /* and comma */
        fromLength = strlen(Tcl_DStringValue(&fromString));
        Tcl_DStringAppend(&toString, to, -1);                                                   /* build target identifier prefix */
        Tcl_DStringAppend(&toString, ",", 1);
        toLength = strlen(Tcl_DStringValue(&toString));
        array = (Var *)Tcl_GetHashValue(entry);
        for(entry = Tcl_FirstHashEntry(array->value.tablePtr, &search); entry != 0; entry = Tcl_NextHashEntry(&search)){
            n = Tcl_GetHashKey(array->value.tablePtr, entry);
            if(strncmp(n, Tcl_DStringValue(&fromString), fromLength) != 0)
                continue;
            Tcl_DStringAppend(&string, Tcl_DStringValue(&toString), toLength);             /* copy all normal object data members */
            Tcl_DStringAppend(&string, n + fromLength, -1);                                                 /* append member name */
            Tcl_SetVar2(
                interpreter, class, Tcl_DStringValue(&string), Tcl_GetVar2(interpreter, class, n, TCL_GLOBAL_ONLY), TCL_GLOBAL_ONLY
            );
            Tcl_DStringFree(&string);
        }
        Tcl_DStringFree(&fromString);
        Tcl_DStringFree(&toString);
    }

    if(copyArrayMembers(interpreter, class, from, to) != TCL_OK)
        return TCL_ERROR;

    Tcl_ResetResult(interpreter);                                                                               /* return nothing */
    return TCL_OK;
}

int Stooop_Init(interpreter)
    Tcl_Interp *interpreter;
{
    Tcl_InitHashTable(&identifiers, TCL_ONE_WORD_KEYS);
    Tcl_InitHashTable(&classes, TCL_STRING_KEYS);
    newId = 0;

    Tcl_CreateCommand(interpreter, "new", new, 0, 0);
    Tcl_CreateCommand(interpreter, "classof", classOf, 0, 0);
    Tcl_CreateCommand(interpreter, "delete", delete, 0, 0);
    Tcl_CreateCommand(interpreter, "_delete", deleteObject, 0, 0);
    Tcl_CreateCommand(interpreter, "_copy", copy, 0, 0);

    return TCL_OK;
}

int Stooop_SafeInit(interpreter)
    Tcl_Interp *interpreter;
{
    return Stooop_Init(interpreter);                                                                  /* stooop is safe by nature */
}
