set rcsId {$Id: stooop.tcl,v 1.50 1996/10/09 20:28:41 jfontain Exp $}

# uncomment the following line if you want to use the dynamically loadable extension
# load ./libstooop2.3.so

# rename proc{} before it is overloaded, ignore error in case of multiple inclusion of this file
catch {rename proc ::proc}

if {[llength [info commands new]]==0} {
# if dynamic extension not available

# initialize object id counter only once even if this file is sourced several times
if {![info exists _newId]} {
    set _newId 0
}

::proc new {classOrId args} {
    # create an object of specified class or copy an existing object

    global _newId _class

    # use local variable for identifier because new can be invoked recursively
    if {[catch {expr $classOrId}]} {
        # first argument is a class
        set _class([set id [incr _newId]]) $classOrId
        # invoke the constructor for the class with optional arguments
        eval $classOrId::$classOrId $id $args
    } {
        # first argument is an object identifier
        # copy source object to new object of identical class
        [set _class([set id [incr _newId]]) $_class($classOrId)]::_copy $id $classOrId
    }
    # return a unique object identifier
    return $id
}

::proc delete {args} {
    # delete one or more objects
    global _class

    foreach id $args {
        _delete $_class($id) $id
        unset _class($id)
    }
}

::proc _delete {class id} {
    # delete object data starting at specified class layer and going up the base class hierarchy if any
    # invokes the destructor for the object class and unset all the object data members for the class
    # the destructor will in turn delete the base classes layers

    $class::~$class $id
    # delete all this object array members if any (assume that they were stored as $class($id,memberName))
    global $class
    foreach name [array names $class $id,*] {
        unset ${class}($name)
    }
}

::proc classof {id} {
    # return class of object
    global _class

    return $_class($id)
}

::proc _copy {class from to} {
    # copy object data members from one object to another

    global $class

    # copy regular data members
    set index [string length $from]
    foreach name [array names $class $from,*] {
        set ${class}($to[string range $name $index end]) [set ${class}($name)]
    }
    # copy array data members
    set index [string length $class$from]
    foreach name [info globals $class$from*] {
        global [set target $class$to[string range $name $index end]] $name
        array set $target [array get $name]
    }
}

# end of Tcl code used if dynamic extension not available
}

::proc virtual {keyword name arguments args} {
    # virtual operator, to be placed before proc
    # virtualize a member procedure, determine whether it is a pure virtual
    # check for procedures that cannot be virtualized

    if {[string compare $keyword proc]!=0} {
        error "virtual operator works only on proc, not $keyword"
    }
    if {![regexp {^(.+)::(.+)$} $name dummy class procedure]} {
        error "$name is not a valid member procedure name"
    }
    if {[string compare $class $procedure]==0} {
        error "cannot make class $class constructor virtual"
    }
    if {[string compare ~$class $procedure]==0} {
        error "cannot make class $class destructor virtual"
    }
    if {[string compare [lindex $arguments 0] this]!=0} {
        error "cannot make static procedure $procedure of class $class virtual"
    }
    # set a global flag so that proc knows it is acting upon a virtual procedure, also serves as a pure indicator
    global _pureVirtual
    # no procedure body means pure virtual
    set _pureVirtual [expr [llength $args]==0]
    # process procedure declaration, body being empty for pure virtual procedure
    proc $name $arguments [lindex $args 0]
    unset _pureVirtual
}


::proc proc {name arguments args} {
    # preprocess class member procedure definitions by adding specific object oriented code
    # regular procedures are not affected
    # it is assumed that both constructor and destructor are defined for every class

# use the following when using the tix package, that creates some invalid (for stooop) member procedures through the proc command
#   if {(![regexp {^(.+)::(.+)$} $name dummy class procedure])||[string match tix* $class]}

    if {![regexp {^(.+)::(.+)$} $name dummy class procedure]} {
        # not a class member procedure, fall back to normal procedure declaration
        ::proc $name $arguments [lindex $args 0]
        return
    }
    # check for procedure body presence
    if {[llength $args]==0} {
        error "missing body for $name"
    }
    if {[string compare $class $procedure]==0} {
        # class constructor definition
        if {[string compare [lindex $arguments 0] this]!=0} {
            error "class $class constructor first argument must be this"
        }
        if {[string compare [lindex $arguments 1] copy]==0} {
            # user defined copy constructor definition
            if {[llength $arguments]!=2} {
                error "class $class copy constructor must have 2 arguments exactly"
            }
            if {[catch {info body $class::$class}]} {
                # make sure of proper declaration order
                error "class $class copy constructor defined before constructor"
            }
            eval _constructorDeclaration $class 1 \{$arguments\} $args
        } {
            # main constructor
            eval _constructorDeclaration $class 0 \{$arguments\} $args
            _generateDefaultCopyConstructor $class
        }
    } elseif {[string compare ~$class $procedure]==0} {
        # class destructor declaration
        if {[llength $arguments]!=1} {
            error "class $class destructor must have 1 argument exactly"
        }
        if {[string compare [lindex $arguments 0] this]!=0} {
            error "class $class destructor argument must be this"
        }
        # use fastest method for testing procedure existence
        if {[catch {info body $class::$class}]} {
            # make sure of proper declaration order
            error "class $class destructor defined before constructor"
        }
        _destructorDeclaration $class $arguments [lindex $args 0]
    } {
        # regular member procedure, may be static if there is no this first argument
        if {[catch {info body $class::$class}]} {
            # make sure of proper declaration order
            error "class $class member procedure $procedure defined before constructor"
        }
        _memberProcedureDeclaration $class $procedure $arguments [lindex $args 0]
    }
}

::proc _ancestors {class} {
    global _bases _ancestors

    if {[info exists _ancestors($class)]} {
        return $_ancestors($class)
    }
    foreach base $_bases($class) {
        set ancestors($base) {}
        foreach ancestor [_ancestors $base] {
            set ancestors($ancestor) {}
        }
    }
    return [set _ancestors($class) [array names ancestors]]
}

::proc _constructorDeclaration {class copy arguments args} {
    # copy flag is set for user defined copy constructor
    global _bases _variable

    # check that each base class constructor has arguments
    set number [llength $args]
    if {($number%2)==0} {
        error "bad $class constructor declaration, a base class, contructor arguments or body may be missing"
    }

    if {[string compare [lindex $arguments end] args]==0} {
        # remember that there is a variable number of arguments in class constructor
        set _variable($class) {}
    }
    if {!$copy} {
        # do not initialize (or reinitialize in case of multiple class file source statements) base classes for copy constructor
        set _bases($class) {}
    }
    # check base classes and their constructor arguments
    foreach {base baseArguments} [lrange $args 0 [expr $number-2]] {
        if {!$copy} {
            # check and save base classes only for main constructor that defines them
            if {[lsearch -exact $_bases($class) $base]>=0} {
                error "class $class directly inherits from class $base more than once"
            }
            lappend _bases($class) $base
        }
        # remove new lines in base arguments part in case user has formatted long declarations with new lines
        regsub -all \n $baseArguments {} constructorArguments($base)
    }
    set body [lindex $args end]

    if {[llength $_bases($class)]>0} {
        # base class(es) derivation specified
        foreach base $_bases($class) {
            if {[string compare $class $base]==0} {
                error "class $class cannot be derived from itself"
            }
            if {[catch {info body $base::$base}]} {
                # make sure of proper declaration order
                error "class $class constructor defined before base class $base constructor"
            }
        }
        # setup access to this class and base classes data
        set constructorBody \
"
global [_ancestors $class] $class
"
        # invoke base class constructors before evaluating constructor body
        # then set base part hidden derived member so that virtual procedures are invoked at base class level as in C++
        if {[info exists _variable($class)]} {
            # variable number of arguments in derived class constructor
            foreach base $_bases($class) {
                if {![info exists constructorArguments($base)]} {
                    error "missing base class $base constructor arguments from class $class constructor"
                }
                if {[info exists _variable($base)]&&([string first {$args} $constructorArguments($base)]>=0)} {
                    # variable number of arguments in base class constructor and in derived class base class constructor arguments
                    # use eval so that base class constructor sees arguments instead of a list
                    # only the last argument of the base class constructor arguments is considered as a variable list
                    # (it usually is $args but could be a procedure invocation, such as [filter $args])
                    append constructorBody \
"set _list \[list $constructorArguments($base)\]
eval $base::$base \$this \[lrange \$_list 0 \[expr \[llength \$_list\]-2\]\] \[lindex \$_list end\]
unset _list
set ${base}(\$this,_derived) $class
"
                } {
                    # no special processing needed
                    # variable number of arguments in base class constructor or
                    # variable arguments list passed as is to base class constructor
                    append constructorBody \
"$base::$base \$this $constructorArguments($base)
set ${base}(\$this,_derived) $class
"
                }
            }
        } {
            # constant number of arguments
            foreach base $_bases($class) {
                if {![info exists constructorArguments($base)]} {
                    error "missing base class $base constructor arguments from class $class constructor"
                }
                append constructorBody \
"$base::$base \$this $constructorArguments($base)
set ${base}(\$this,_derived) $class
"
            }
        }
    } {
        # no base class derivation specified
        # setup access to this class data
        set constructorBody \
"
global $class
"
    }
    if {$copy} {
        # for user defined copy constructor, copy derived class member if it exists
        append constructorBody \
"catch {set ${class}(\$this,_derived) \[set ${class}(\$[lindex $arguments 1],_derived)\]}
"
    }
    append constructorBody $body
    if {$copy} {
        ::proc $class::_copy $arguments $constructorBody
    } {
        ::proc $class::$class $arguments $constructorBody
    }
}

::proc _destructorDeclaration {class arguments body} {
    global _bases

    # setup access to this class and base classes data
    # evaluate this class destructor body
    set body \
"
global [_ancestors $class] $class
$body
"
    # if there are any, delete base classes parts in reverse order of construction
    for {set index [expr [llength $_bases($class)]-1]} {$index>=0} {incr index -1} {
        set base [lindex $_bases($class) $index]
        append body \
"_delete $base \$this
"
    }
    ::proc $class::~$class $arguments $body
}

::proc _memberProcedureDeclaration {class name arguments body} {
    global _pureVirtual

    if {[info exists _pureVirtual]} {
        # virtual declaration
        if {$_pureVirtual} {
            # pure virtual declaration
            # setup access to this class and base classes data
            # evaluate derived procedure which must exists. derived procedure return value is automatically returned
            ::proc $class::$name $arguments \
"
global [_ancestors $class] $class
eval \$${class}(\$this,_derived)::$name \[lrange \[info level 0\] 1 end\]
"
        } {
            # regular virtual declaration
            # setup access to this class and base classes data
            # evaluate derived procedure and return if it exists
            # else evaluate the base class procedure which can be invoked from derived class procedure by prepending ::
            ::proc ::$class::$name $arguments \
"
global [_ancestors $class] $class
$body
"
            ::proc $class::$name $arguments \
"
global [_ancestors $class] $class
if {!\[catch {info body \$${class}(\$this,_derived)::$name}\]} {
return \[eval \$${class}(\$this,_derived)::$name \[lrange \[info level 0\] 1 end\]\]
}
eval ::\[info level 0\]
"
        }
    } {
        # non virtual declaration
        # setup access to this class and base classes data
        ::proc $class::$name $arguments \
"
global [_ancestors $class] $class
$body
"
    }
}

::proc _generateDefaultCopyConstructor {class} {
    # generate default copy procedure which may be overriden by the user for any class layer

    global _bases

    # generate code for cloning base classes layers if there is at least one base class
    foreach base $_bases($class) {
        append body \
"$base::_copy \$this \$sibling
"
    }
    append body \
"_copy $class \$sibling \$this
"
    ::proc $class::_copy {this sibling} $body
}
