
/***********************************************
 * PDE Solver
 * Sine-Gordon
 *
 * Dave Beazley
 * Math 658,  Spring 1993
 *
 * May 25, 1993
 ***********************************************/

 #include <stdio.h>
 #include <math.h>
 #include <stdlib.h>
#include <bstring.h>

#include "sg.h"

#define  PI 3.14159265359
#define  SIZE 1000
#define  L  80
#define  V  0.9
#define  X0 5.0

// --------------------------------------------------------
// Create a new simulation
// --------------------------------------------------------

SineGordon::SineGordon(int npts, double dt, double _xmin, double _xmax) {
  int i;
  npoints = npts;
  Dt = dt;
  u = (double *) malloc((npts+1)*sizeof(double));
  v = (double *) malloc((npts+1)*sizeof(double));
  a  = (double *) malloc((npts+1)*sizeof(double));
  b  = (double *) malloc((npts+1)*sizeof(double));
  c  = (double *) malloc((npts+1)*sizeof(double));
  d  = (double *) malloc((npts+1)*sizeof(double));

  for (i = 0; i <= npts; i++) {
    u[i] = v[i] = a[i] = b[i] = c[i] = d[i] = 0.0;
  }

  temp = (double *) malloc((npts+1)*sizeof(double));
  totalsteps = 0;
  xmin = _xmin;
  xmax = _xmax;

}

// -------------------------------------------------------
// Make a snapshot of current simulation
// -------------------------------------------------------

SineGordon::SineGordon(SineGordon *s) {
  npoints = s->npoints;
  xmin = s->xmin;
  xmax = s->xmax;
  totalsteps = s->totalsteps;
  Dt = s->Dt;
  u = (double *) malloc((npoints+1)*sizeof(double));
  v = (double *) malloc((npoints+1)*sizeof(double));
  a  = (double *) malloc((npoints+1)*sizeof(double));
  b  = (double *) malloc((npoints+1)*sizeof(double));
  c  = (double *) malloc((npoints+1)*sizeof(double));
  d  = (double *) malloc((npoints+1)*sizeof(double));
  temp = (double *) malloc((npoints+1)*sizeof(double));

  bcopy(s->u,u,(npoints+1)*sizeof(double));
  bcopy(s->v,v,(npoints+1)*sizeof(double));
  bcopy(s->a,a,(npoints+1)*sizeof(double));
  bcopy(s->b,b,(npoints+1)*sizeof(double));
  bcopy(s->c,c,(npoints+1)*sizeof(double));
  bcopy(s->d,d,(npoints+1)*sizeof(double));
  bcopy(s->temp,temp,(npoints+1)*sizeof(double));

}

// -------------------------------------------------------
// Delete current simulations
// -------------------------------------------------------

SineGordon::~SineGordon() {
  free(u);
  free(v);
  free(a);
  free(b);
  free(c);
  free(d);
  free(temp);
}

/************************************************
 Solves the Sine-Gordon Equation
        Utt = Uxx - Sin U
	Ux(0) = 0
	Ux(L) = 0
	U(x,0) = f(x)
        Ut(x,0)= g(x) 

 ************************************************/

/*/ Solve Sine-Gordon Equation for n steps */

void SineGordon::sg(int nsteps) {

  double  h,k,r,t,p,x,y,h2,k2;
  void   tridiagonal(double *, double *, double *, double *, double *, int);
  int    i,j;

  h = (xmax - xmin)/(double) npoints;
  k = Dt;

  /* Set up initial conditions */
     
  h2 = h*h;
  k2 = k*k;

  /* Calculate tridiagonal Matrix coefficients */

  if (!totalsteps) {
    a[0] = 0;
    c[0] = -2*k2;
    c[npoints] = 0;
    a[npoints] = -2*k2;
    for (i = 1; i < npoints; i++) {
      a[i] = -k2;
      c[i] = -k2;
    }
    for (i = 0; i <= npoints; i++) b[i] = 2*(2*h2 + k2);
    
    
    /* Calculate RHS */
    
    d[0] = 2*k2*u[1] + (4*h2-2*k2)*u[0] + 4*h2*k*v[0] - 2*h2*k2*sin(u[0]);
    d[npoints] = 2*k2*u[npoints-1] + (4*h2-2*k2)*u[npoints] + 4*h2*k*v[npoints] - 2*h2*k2*sin(u[npoints]);
    for (i = 1; i < npoints; i++) {
      d[i] = k2*(u[i+1]+u[i-1])+(4*h2-2*k2)*u[i] +4*h2*k*v[i] - 2*h2*k2*sin(u[i]);
    }
    
    tridiagonal(a,b,c,d,v, npoints+1);

  /* Adjust crank-nicholson coefficients */

  for (i = 0; i <= npoints; i++) b[i] = 2*(h2 + k2);

  }     

  for (i = 1; i <= nsteps; i++) {
    /* Set up RHS */
    d[0] = 2*k2*v[1] + (4*h2-2*k2)*v[0] -2*h2*u[0] - 2*h2*k2*sin(v[0]);
    for (j = 1; j < npoints; j++) {
      d[j] = k2*(v[j+1]+v[j-1])+(4*h2-2*k2)*v[j] - 2*h2*u[j] - 2*h2*k2*sin(v[j]);
    }
    d[npoints] = 2*k2*v[npoints-1] + (4*h2-2*k2)*v[npoints] - 2*h2*u[npoints] - 2*h2*k2*sin(v[npoints]);

    tridiagonal(a,b,c,d,temp, npoints+1);	  

    /* Copy solutions */

    for (j = 0; j <= npoints; j++) {
      u[j] = v[j];
      v[j] = temp[j];
    }

    Time += Dt;
    totalsteps++;
  }
}


/* Equation for a kink with velocity V and center x0 */

double eval_kink(double x, double v, double x0, double k) {

       double y;
       y = 4*atan(exp((x-x0)/sqrt(1-v*v))) + 2*PI*k;
       return(y);
     }
double eval_dkink(double x, double v, double x0) {
       double y;
       y = -4*v*exp((x-x0)/sqrt(1-v*v))/(sqrt(1-v*v)*(1+exp((x-x0)/sqrt(1-v*v))*exp((x-x0)/sqrt(1-v*v))));
       return(y);
     }
double eval_antikink(double x, double v, double x0, double k) {
       double y;
       y = 4*atan(exp(-(x-x0)/sqrt(1-v*v))) + 2*PI*k;
       return(y);
     }
double eval_dantikink(double x, double v, double x0) {
       double y;
       y = 4*v*exp(-(x-x0)/sqrt(1-v*v))/(sqrt(1-v*v)*(1+exp(-(x-x0)/sqrt(1-v*v))*exp(-(x-x0)/sqrt(1-v*v))));
       return(y);
     }

/* Kink equations */

void SineGordon::kink(double vel, double x0, int k) {

  double h,x;
  int i;

  h = (xmax - xmin)/ (double) npoints;
  for (i = 0; i <= npoints; i++) {
    x = xmin+i*h;
    u[i] += eval_kink(x,vel,x0,k);
    v[i] += eval_dkink(x,vel,x0);
  }
  
}

void SineGordon::antikink(double vel, double x0, int k) {

  double h,x;
  int i;

  h = (xmax - xmin)/ (double) npoints;
  for (i = 0; i <= npoints; i++) {
    x = xmin+i*h;
    u[i] += eval_antikink(x,vel,x0,k);
    v[i] += eval_dantikink(x,vel,x0);
  }
}

#define MAXSIZE 32000
     
void tridiagonal(double *a, double *b, double *c, double *d, double *u, int N) {

       int i,j,k;

       double   P[MAXSIZE];
       double   R[MAXSIZE];

       P[0] = b[0];
       R[0] = d[0]/b[0];

       for (i = 1; i < N; i++) {
           P[i] = b[i] - (a[i]*c[i-1])/P[i-1];
           R[i] = (d[i] - a[i]*R[i-1])/P[i];
	 }

       u[N-1] = R[N-1];
       for (i = N-2; i >= 0; i--)
            u[i] = R[i] - (c[i]*u[i+1])/P[i];


     }





