
/**********************************************************************
 * Simplified Wrapper and Interface Generator  (SWIG)
 * 
 * Dave Beazley
 * 
 * Theoretical Division (T-11)           Department of Computer Science
 * Los Alamos National Laboratory        University of Utah
 * Los Alamos, New Mexico  87545         Salt Lake City, Utah  84112
 * beazley@lanl.gov                      beazley@cs.utah.edu
 *
 * Copyright (c) 1995-1996
 * The Regents of the University of California and the University of Utah
 * All Rights Reserved
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that 
 * (1) The above copyright notice and the following two paragraphs
 * appear in all copies of the source code and (2) redistributions
 * including binaries reproduces these notices in the supporting
 * documentation.   Substantial modifications to this software may be
 * copyrighted by their authors and need not follow the licensing terms
 * described here, provided that the new terms are clearly indicated in
 * all files where they apply.
 * 
 * IN NO EVENT SHALL THE AUTHOR, THE UNIVERSITY OF CALIFORNIA, THE 
 * UNIVERSITY OF UTAH OR DISTRIBUTORS OF THIS SOFTWARE BE LIABLE TO ANY
 * PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
 * DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION,
 * EVEN IF THE AUTHORS OR ANY OF THE ABOVE PARTIES HAVE BEEN ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE AUTHOR, THE UNIVERSITY OF CALIFORNIA, AND THE UNIVERSITY OF UTAH
 * SPECIFICALLY DISCLAIM ANY WARRANTIES,INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND 
 * THE AUTHORS AND DISTRIBUTORS HAVE NO OBLIGATION TO PROVIDE MAINTENANCE,
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 * The author requests that all users of this software return any
 * improvements made to beazley@cs.utah.edu and grant the author
 * full redistribution rights.
 *
 **************************************************************************/
/***********************************************************************
 * $Header: /b11/dmb/SWIG/SWIG1.0/Modules/RCS/tcl.cxx,v 1.38 1996/08/25 00:05:25 dmb Exp $
 *
 * tcl.cxx
 *
 * Definitions for creating a simple, stand-alone TCL implementation.
 *
 * -- Revision History
 * $Log: tcl.cxx,v $
 * Revision 1.38  1996/08/25 00:05:25  dmb
 * Added ability to change Tcl result string behavior with %pragma
 *
 * Revision 1.37  1996/08/21 16:51:23  dmb
 * Fixed bugs in constant code.
 *
 * Revision 1.36  1996/08/21 05:50:48  dmb
 * Added support for [incr Tcl] namespaces
 *
 * Revision 1.35  1996/08/16 04:17:34  dmb
 * Added support for Netscape plugin
 *
 * Revision 1.34  1996/08/12 01:51:27  dmb
 * Changes to support new language class
 *
 * Revision 1.33  1996/08/02 02:58:56  dmb
 * Changed to use better parameter list functions
 *
 * Revision 1.32  1996/07/17 14:55:36  dmb
 * Fixed bug in -strict 1 pointer type checking mode.
 *
 * Revision 1.31  1996/06/16 02:22:58  beazley
 * Fixed bug in constant creation.
 *
 * Revision 1.30  1996/06/02  00:14:48  beazley
 * Minor changes
 *
 * Revision 1.29  1996/05/28  23:17:14  beazley
 * Minor fix to initialize function
 *
 * Revision 1.28  1996/05/22  20:20:21  beazley
 * Added banner and cleanup functions to headers() and close() functions.
 * Changed name of class
 *
 * Revision 1.27  1996/05/20  23:36:17  beazley
 * Added a few more constant datatypes.
 *
 * Revision 1.26  1996/05/17  05:53:22  beazley
 * Added return by value support.
 *
 * Revision 1.25  1996/05/14  23:24:04  beazley
 * Minor changes
 *
 * Revision 1.24  1996/05/13  23:45:08  beazley
 * Reworked the module/init procedure
 *
 * Revision 1.23  1996/05/09  23:27:25  beazley
 * Minor changes
 *
 * Revision 1.22  1996/05/06  23:09:58  beazley
 * Some fixes to forward declarations for C++
 *
 * Revision 1.21  1996/05/03  22:29:09  dmb
 * Modified to look for swigtcl.swg file.   Used to include code for
 * UNIX and Windows.
 *
 * Revision 1.20  1996/05/03 05:11:30  dmb
 * Fixed some C++ linking problems.
 *
 * Revision 1.19  1996/05/01 22:42:44  dmb
 * Cleaned up command line option handling.
 *
 * Revision 1.18  1996/04/14 15:24:28  dmb
 * Changed filename to tcl.cc. Fixed some of the headers
 *
 * Revision 1.17  1996/04/09 20:19:10  beazley
 * Minor cleanup.
 *
 * Revision 1.16  1996/04/09  19:29:31  beazley
 * Minor changes to documentation.
 *
 * Revision 1.15  1996/03/28  02:46:38  beazley
 * Minor bug fix to documentation.
 *
 * Revision 1.14  1996/03/22  23:42:16  beazley
 * Added constants.  Cleaned up some stuff.
 *
 * Revision 1.13  1996/03/16  06:29:05  beazley
 * Major overhaul.  Has too many new features to mention
 * here.  Biggest change = run-time type checking.
 *
 * Revision 1.12  1996/03/04  21:29:13  beazley
 * Change usage(), made changes to pointer handling.
 *
 * Revision 1.11  1996/02/19  05:31:35  beazley
 * Changed treatment of pointers to avoid weird problems with
 * scanf("%p").
 *
 * Revision 1.10  1996/02/16  07:06:58  beazley
 * Fixed problems with sprintf()
 *
 * Revision 1.9  1996/02/16  06:38:49  beazley
 * Removed a few unused variables.
 *
 * Revision 1.8  1996/02/15  22:38:31  beazley
 * Fixed several significant bugs.  Changed copyright
 *
 * Revision 1.7  1996/02/12  08:19:34  beazley
 * A few minor changes to the include path.
 *
 * Revision 1.6  1996/02/09  04:54:31  beazley
 * Added "tcl" to wrapper names.
 *
 * Revision 1.5  1996/02/07  05:22:10  beazley
 * A complete rewrite.   Uses new emit functions and now supports
 * almost any datatype (including pointers and complex types).
 *
 * Revision 1.4  1996/01/23  19:41:25  beazley
 * Fixed a few bugs.
 *
 * Revision 1.3  1996/01/13  01:34:26  beazley
 * A few minor changes.
 *
 * Revision 1.2  1996/01/05  22:40:54  dmb
 * *** empty log message ***
 *
 * Revision 1.1  1995/12/30 04:34:14  dmb
 * Initial revision
 *
 *
 ***********************************************************************/

#include "swig.h"
#include "gl_tcl.h"
#include <ctype.h>

static char *Tcl_config="swigtcl.swg";
static char *usage = "\
Tcl Options (available with -tcl)\n\
     -module name    - Set name of module\n\
     -prefix name    - Set a prefix to be appended to all names\n\
     -htcl tcl.h     - Specify the name of the \"tcl.h\" header file\n\
     -htk tk.h       - Specify the name of the \"tk.h\" header file\n\
     -nspace      - Build module into a [incr Tcl] nspace. \n\
     -plugin         - Produce code compatible with Tcl Netscape Plugin\n\n";

static char *ns_name = 0;

// ---------------------------------------------------------------------
// TCL::parse_args(int argc, char *argv[])
//
// Parse tcl specific command line options
// ---------------------------------------------------------------------

void TCL::parse_args(int argc, char *argv[]) {
  
  int i = 1;
  sprintf(LibDir,"%s",tcl_path);

  // Look for certain command line options

  for (i = 1; i < argc; i++) {
      if (argv[i]) {
	  if (strcmp(argv[i],"-htcl") == 0) {
	    if (argv[i+1]) {
	      strcpy(tcl_include,argv[i+1]);
	      mark_arg(i);
	      mark_arg(i+1);
	      i++;
	    } else {
	      arg_error();
	    }
	  } else if (strcmp(argv[i],"-htk") == 0) {
	    if (argv[i+1]) {
	      strcpy(tk_include,argv[i+1]);
	      mark_arg(i);
	      mark_arg(i+1);
	      i++;
	    } else {
	      arg_error();
	    }
	  } else if (strcmp(argv[i],"-prefix") == 0) {
	    if (argv[i+1]) {
	      prefix = new char[strlen(argv[i+1])+2];
	      strcpy(prefix, argv[i+1]);
	      mark_arg(i);
	      mark_arg(i+1);
	      i++;
	    } else {
	      arg_error();
	    }
	  } else if (strcmp(argv[i],"-module") == 0) {
	    if (argv[i+1]) {
	      set_module(argv[i+1]);
	      mark_arg(i);
	      mark_arg(i+1);
	      i++;
	    } else {
	      arg_error();
	    }
	  } else if (strcmp(argv[i],"-plugin") == 0) {
	    Plugin = 1;
	    mark_arg(i);
	  } else if (strcmp(argv[i],"-namespace") == 0) {
	    nspace = 1;
	    mark_arg(i);
	  } else if (strcmp(argv[i],"-help") == 0) {
	    fputs(usage,stderr);
	    exit(0);
	  }
      }
  }

  // If a package has been specified, make sure it ends with a '_'

  if (prefix) {
    ns_name = copy_string(prefix);
    if (prefix[strlen(prefix)] != '_') {
      prefix[strlen(prefix)+1] = 0;
      prefix[strlen(prefix)] = '_';
    }
  } else 
    prefix = "";

}

// ---------------------------------------------------------------------
// void TCL::parse()
//
// Start parsing an interface file for Tcl.
// ---------------------------------------------------------------------

void TCL::parse() {

  fprintf(stderr,"Making wrappers for Tcl\n");
  
  // Print out TCL specific headers
  
  headers();
  
  // Run the parser
  
  yyparse();

}

// ---------------------------------------------------------------------
// TCL::set_module(char *mod_name)
//
// Sets the module name.
// Does nothing if it's already set (so it can be overridden as a command
// line option).
//
//----------------------------------------------------------------------

void TCL::set_module(char *mod_name) {

  char temp[256], *c;
  if (module) return;
  
  module = new char[strlen(mod_name)+1];
  strcpy(module,mod_name);

  // Fix capitalization for Tcl 

  c = module;
  while (*c) {
    *c = (char) tolower(*c);
    c++;
  }

  // Now create an initialization function

  sprintf(temp,"%s_Init", module);
  init_name = new char[strlen(temp) + 1];
  strcpy(init_name, temp);
  *init_name = toupper(*init_name);
  sprintf(temp,"%s_SafeInit",module);
  safe_name = new char[strlen(temp)+1];
  strcpy(safe_name,temp);
  *safe_name = toupper(*safe_name);

  if (!ns_name) ns_name = copy_string(module);

  // If namespaces have been specified, set the prefix to the module name

  if ((nspace) && (strlen(prefix) < 1)) {
    prefix = new char[strlen(module)+2];
    strcpy(prefix,module);
    prefix[strlen(module)] = '_';
    prefix[strlen(module)+1] = 0;
  }
}


// ---------------------------------------------------------------------
// TCL::set_init(char *iname)
//
// Sets the initialization function name.
// Does nothing if it's already set
//
//----------------------------------------------------------------------

void TCL::set_init(char *iname) {

  if (init_name) return;

  init_name = new char[strlen(iname)+1];
  strcpy(init_name, iname);

}

// ---------------------------------------------------------------------
// TCL::headers(void)
//
// Generate the appropriate header files for TCL interface.
// ----------------------------------------------------------------------

void TCL::headers(void)
{

  emit_banner(f_header);
  if (!Plugin) 
    fprintf(f_header,"/* Implementation : TCL */\n\n");
  else
    fprintf(f_header,"/* Implementation : TCL - Netscape plugin */\n\n");
  fprintf(f_header,"#define INCLUDE_TCL    <%s>\n", tcl_include);
  fprintf(f_header,"#define INCLUDE_TK     <%s>\n", tk_include);
  fprintf(f_header,"#include INCLUDE_TCL\n");
  fprintf(f_header,"#include <string.h>\n");
  fprintf(f_header,"#include <stdlib.h>\n");
  
  // Include a Tcl configuration file for Unix,Mac,Wintel.
  
  if (IncludeFile(Tcl_config, f_header) == -1) {
    fprintf(f_header,"#define EXPORT(a,b) a b\n");
  }
  
  // Write out hex conversion functions for pointers
  emit_hex(f_header);

}

// --------------------------------------------------------------------
// TCL::initialize(void)
//
// Produces an initialization function.   Assumes that the init function
// name has already been specified.
// ---------------------------------------------------------------------

void TCL::initialize() 
{

  int i;

  if ((!ns_name) && (nspace)) {
    fprintf(stderr,"Tcl error.   Must specify a namespace.\n");
    exit(1);
  }

  if (!init_name) 
    init_name = "Swig_Init";

  fprintf(f_header,"#define SWIG_init    %s\n\n\n", init_name);

  if (CPlusPlus) {
      fprintf(f_header,"extern \"C\" { EXPORT(int,%s)(Tcl_Interp *);}\n",init_name);
  } else {
    fprintf(f_header,"EXPORT(int,%s)(Tcl_Interp *);\n", init_name);
  }

  if (Plugin) {	
    if (CPlusPlus) {
      fprintf(f_header,"extern \"C\" { EXPORT(int,%s)(Tcl_Interp *);}\n",safe_name);
    } else {
      fprintf(f_header,"EXPORT(int,%s)(Tcl_Interp *);\n", safe_name);
    }
  }



  fprintf(f_init,"EXPORT(int,%s)(Tcl_Interp *%s) {\n", init_name, interp_name);
  if (nspace) {
    fprintf(f_init,"#ifdef ITCL_NAMESPACES\n");
    fprintf(f_init,"\t Itcl_Namespace spaceId;\n");
    fprintf(f_init,"#endif\n");
  }

  // Create a Tcl Hash Table

  fprintf(f_header,"static Tcl_HashTable glTable;\n");
  fprintf(f_init,"\t Tcl_HashEntry *entryPtr;\n");
  fprintf(f_init,"\t int *glconst;\n");
  fprintf(f_init,"\t int n;\n");
  fprintf(f_init,"\t Tcl_InitHashTable(&glTable, TCL_STRING_KEYS);\n");
  
  fprintf(f_init,"\t if (%s == 0) \n", interp_name);
  fprintf(f_init,"\t\t return TCL_ERROR;\n");

  /* Check to see if other initializations need to be performed */

  if (InitNames) {
    i = 0;
    while (InitNames[i]) {
      fprintf(f_init,"\t if (%s(%s) == TCL_ERROR)\n",InitNames[i], interp_name);
      fprintf(f_init,"\t\t return TCL_ERROR;\n");
      i++;
    }
  }

  /* Check to see if we're adding support for [incr Tcl] nspaces */
  if (nspace) {
    fprintf(f_init,"#ifdef ITCL_NAMESPACES\n");
    fprintf(f_init,"\t if (Itcl_CreateNamesp(%s, \"%s\", (ClientData) 0, (Itcl_DeleteProc *) 0, &spaceId) != TCL_OK) {\n", interp_name, ns_name);
    fprintf(f_init,"\t\t return TCL_ERROR;\n");
    fprintf(f_init,"\t}\n");
    fprintf(f_init,"#endif\n");
  }
}

// ---------------------------------------------------------------------
// TCL::close(void)
//
// Wrap things up.  Close initialization function.
// ---------------------------------------------------------------------

void TCL::close(void)
{

  // Dump the pointer equivalency table

  emit_ptr_equivalence("_swig_ptr_tcl", f_wrappers);
  fprintf(f_init,"\t swig_ptr_derived = _swig_ptr_tcl;\n");

  // Close the init file and quit

  fprintf(f_init,"\t return TCL_OK;\n");
  fprintf(f_init,"}\n");

  if (Plugin) {
    fprintf(f_init,"EXPORT(int,%s)(Tcl_Interp *interp) {\n",safe_name);
    fprintf(f_init,"#ifdef SAFE_SWIG\n");
    fprintf(f_init,"   return %s(interp);\n", init_name); 
    fprintf(f_init,"#else\n");
    fprintf(f_init,"   return TCL_ERROR;\n");
    fprintf(f_init,"#endif\n");	
    fprintf(f_init,"}\n");
  }
}

// ----------------------------------------------------------------------
// TCL::get_pointer(int parm, DataType *t)
//
// Emits code to get a pointer from a parameter and do type checking.
// parm is the parameter number.   This function is only used
// in create_function().
// ----------------------------------------------------------------------

void TCL::get_pointer(char *iname, int parm, DataType *t) {

  // Pointers are read as hex-strings with encoded type information

  fprintf(f_wrappers,"\t if (_swig_get_hex(argv[%d], (void **) &_arg%d,",parm+1,parm);
  if (t->type == T_VOID) fprintf(f_wrappers,"(char *) 0)) {\n");
  else 
    fprintf(f_wrappers,"\"%s\")) {\n", t->print_mangle());


  // Now emit code according to the level of strictness desired

  switch(TypeStrict) {
  case 0: // No type checking
    fprintf(f_wrappers,"\t}\n");
    break;
  case 1: // Warning message only
    fprintf(f_wrappers,
      "\t fprintf(stderr,\"Warning : type mismatch in argument %d of %s. Expected %s, received %%s\\n\", argv[%d]);\n", parm+1,iname, t->print_mangle(), parm+1);
    fprintf(f_wrappers,"\t }\n");
    break;
  case 2: // Super strict mode.
    fprintf(f_wrappers,"\t\t Tcl_SetResult(interp, \"Type error in argument %d of %s. Expected %s, received \", TCL_STATIC);\n", parm+1,iname,t->print_mangle());
    fprintf(f_wrappers,"\t\t Tcl_AppendResult(interp, argv[%d], (char *) NULL);\n", parm+1);
    fprintf(f_wrappers,"\t\t return TCL_ERROR;\n");
    fprintf(f_wrappers,"\t}\n");
    break;

  default :
    fprintf(stderr,"Unknown strictness level\n");
    break;
  }

}

// ----------------------------------------------------------------------
// TCL::create_function(char *name, char *iname, DataType *d,
//                             ParmList *l, int ex)
//
// Create a function declaration and register it with the interpreter.
// ----------------------------------------------------------------------

void TCL::create_function(char *name, char *iname, DataType *d, ParmList *l)
{
  Parm *p;
  int   pcount,i;
  char  wname[256];
  char  *usage = 0;

  // Make a wrapper name for this
  
  strcpy(wname,iname);
  make_wrap_name(wname);

  // Now write the wrapper function itself....this is pretty ugly
  
  fprintf(f_wrappers,"static int _wrap_tcl_%s%s(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {\n",prefix,wname);

  // Print out variables for storing arguments.

  pcount = emit_args(d, l, f_wrappers);


  fprintf(f_wrappers,"\t Tcl_HashEntry *entryPtr;\n");
  fprintf(f_wrappers,"\t int *arg;\n");

  // Check the number of arguments

  usage_func(iname,d,l,&usage);       // Create a usage string
  fprintf(f_wrappers,"\n\t if (argc != %d) {\n",pcount+1);
  fprintf(f_wrappers,"\t\t Tcl_SetResult(interp, \"Wrong # args  %s\",TCL_STATIC);\n",usage);
  fprintf(f_wrappers,"\t\t return TCL_ERROR;\n");
  fprintf(f_wrappers,"\t}\n");

  delete usage;
  // Extract parameters.    This case statement should be used to extract
  // Function parameters.   Add more cases if you want to do more.

  i = 0;
  p = l->get_first();
  while (p != 0) {
    if (!p->t->is_pointer) {

      // Extract a parameter by value.

      switch(p->t->type) {

      // Signed Integers

      case T_INT:
      case T_SINT:
      case T_SHORT:
      case T_SSHORT:
      case T_LONG:
      case T_SLONG:
      case T_SCHAR:
	// Check out hash table to see what's up 

	fprintf(f_wrappers,"\t entryPtr = Tcl_FindHashEntry(&glTable, argv[%d]);\n", i+1);
	fprintf(f_wrappers,"\t if (entryPtr) {\n");
	fprintf(f_wrappers,"\t\t arg = (int *) Tcl_GetHashValue(entryPtr);\n");
	fprintf(f_wrappers,"\t\t _arg%d = %s *arg;\n", i, p->t->print_cast());
	fprintf(f_wrappers,"\t} else \n");
	fprintf(f_wrappers,"\t _arg%d = atoi(argv[%d]);\n",i, i+1);
	break;
	
      // Unsigned Integers

      case T_UINT:
      case T_USHORT:
      case T_ULONG:
      case T_UCHAR:
	// Check out hash table to see what's up 

	fprintf(f_wrappers,"\t entryPtr = Tcl_FindHashEntry(&glTable, argv[%d]);\n", i+1);
	fprintf(f_wrappers,"\t if (entryPtr) {\n");
	fprintf(f_wrappers,"\t\t arg = (int *) Tcl_GetHashValue(entryPtr);\n");
	fprintf(f_wrappers,"\t\t _arg%d = %s *arg;\n", i, p->t->print_cast());
	fprintf(f_wrappers,"\t} else \n");
	fprintf(f_wrappers,"\t _arg%d = strtoul(argv[%d], (char **) NULL, 0);\n",i, i+1);
	break;

      // Floating point

      case T_FLOAT:
      case T_DOUBLE:
	fprintf(f_wrappers,"\t _arg%d = %s atof(argv[%d]);\n",
                i, p->t->print_cast(), i+1);
	break;

      // A single character
	
      case T_CHAR :
	fprintf(f_wrappers,"\t _arg%d =  *argv[%d];\n",i,i+1);
	break;

      // Void.. Do nothing.

      case T_VOID :
	break;

      // User defined.   This is an error.

      case T_USER:

      // Unsupported data type

      default :
	fprintf(stderr,"%s : Line %d: Unable to use type %s as a function argument.\n",
               input_file, line_number, p->t->print_type());
	break;
      }
    } else {

      // Function argument is some sort of pointer
      // Look for a string.   Otherwise, just pull off a pointer.

      if ((p->t->type == T_CHAR) && (p->t->is_pointer == 1)) {
	fprintf(f_wrappers,"\t _arg%d = argv[%d];\n",i,i+1);
      } else {

	// Have a generic pointer type here.    Read it in as
        // a hex-string

	get_pointer(iname, i, p->t);

      }
    }
    p = l->get_next();   // Get next parameter and continue
    i++;
  }

  // Now write code to make the function call

  emit_func_call(name,d,l,f_wrappers);

  // Return value if necessary 

  if ((d->type != T_VOID) || (d->is_pointer)) {
    if (!d->is_pointer) {

      // Function returns a "value"

      switch(d->type) {
	// Is a signed integer
      case T_INT:
      case T_SINT:
      case T_SHORT:
      case T_SSHORT:
      case T_LONG :
      case T_SLONG:
      case T_SCHAR:
	fprintf(f_wrappers,"\t sprintf(interp->result,\"%%ld\", (long) _result);\n");
	break;
	
	// Is an unsigned integer
      case T_UINT:
      case T_USHORT:
      case T_ULONG:
      case T_UCHAR:
	fprintf(f_wrappers,"\t sprintf(interp->result,\"%%lu\", (unsigned long) _result);\n");
	break;

	// Is a single character.  Assume we return it as a string
      case T_CHAR :
	fprintf(f_wrappers,"\t sprintf(interp->result,\"%%c\", _result);\n");
	break;
	
	// Floating point number
      case T_DOUBLE :
      case T_FLOAT :
	fprintf(f_wrappers,"\t sprintf(interp->result,\"%%0.17f\",(double) _result);\n");
	break;

	// User defined type
      case T_USER :
	
	// Okay. We're returning malloced memory at this point.
	// Probably dangerous, but what the hell

	d->is_pointer++;
	fprintf(f_wrappers,"\t _swig_make_hex(interp->result, (void *) _result,\"%s\");\n",
		d->print_mangle());
	d->is_pointer--;
	break;

	// Unknown type
      default :
	fprintf(stderr,"%s : Line %d: Unable to use return type %s in function %s.\n",
               input_file, line_number, d->print_type(), name);
	break;
      }
    } else {

    // Is a pointer return type

      if ((d->type == T_CHAR) && (d->is_pointer == 1)) {
	// Return a character string
	fprintf(f_wrappers,"\t Tcl_SetResult(interp, (char *) _result, %s);\n", char_result);
      } else {
	// Is an ordinary pointer type.
	fprintf(f_wrappers,"\t  _swig_make_hex(interp->result, (void *) _result,\"%s\");\n",
                d->print_mangle());
      }
    }
  }

    // Wrap things up (in a manner of speaking)

  fprintf(f_wrappers,"\t return TCL_OK;\n}\n");

  // Now register the function with Tcl

  if (nspace) {
    fprintf(f_init,"#ifdef ITCL_NAMESPACES\n");
    fprintf(f_init,"\t Tcl_CreateCommand(%s, \"%s::%s\", _wrap_tcl_%s%s, (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);\n",interp_name, ns_name, iname, prefix,wname);
    fprintf(f_init,"#else\n");
  }
  
  fprintf(f_init,"\t Tcl_CreateCommand(%s, \"%s%s\", _wrap_tcl_%s%s, (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);\n",interp_name, prefix, iname, prefix,wname);

  if (nspace) {
    fprintf(f_init,"#endif\n");
  }
}

// -----------------------------------------------------------------------
// TCL::link_variable(char *name, char *iname, DataType *t,
//                           int ex)
//
// Create a TCL link to a variable.
// -----------------------------------------------------------------------

void TCL::link_variable(char *name, char *iname, DataType *t)
{

  String    s;
  
  // Check the datatype.  Must be a valid Tcl type (there aren't many)

  if (((t->type == T_INT) && (!t->is_pointer)) ||
      ((t->type == T_UINT) && (!t->is_pointer)) ||
      ((t->type == T_SINT) && (!t->is_pointer)) ||
      ((t->type == T_DOUBLE) && (!t->is_pointer)) ||
      ((t->type == T_CHAR) && (t->is_pointer == 1))) {

    // This is a valid TCL type. 

    if (t->type == T_UINT)
      fprintf(stderr,"%s : Line %d : ** Warning. Linkage of unsigned type may be unsafe.\n",
              input_file,  line_number);

    // Now add symbol to the TCL interpreter

    switch(t->type) {
    case T_CHAR :
      s << "(char *) &" << name << ", TCL_LINK_STRING";
      break;
    case T_INT :
    case T_UINT:
    case T_SINT:
      s << "(char *) &" << name << ", TCL_LINK_INT";
      break;
    case T_DOUBLE :
      s << "(char *) &" << name << ", TCL_LINK_DOUBLE";
      break;
    default :
      fprintf(f_init,"Fatal error. Internal error (Tcl:link_variable)\n");
      break;
    }

    if (Status & STAT_READONLY)
      s << " | TCL_LINK_READ_ONLY);\n";
    else
      s << ");\n";

    if (nspace) {
      fprintf(f_init,"#ifdef ITCL_NAMESPACES\n");
      fprintf(f_init,"\t Tcl_LinkVar(%s,\"%s::%s\", %s", interp_name, ns_name, iname, s.get());
      fprintf(f_init,"#else\n");
    }
    fprintf(f_init,"\t Tcl_LinkVar(%s, \"%s%s\", %s",interp_name, prefix, iname, s.get());    
    if (nspace) {
      fprintf(f_init,"#endif\n");
    }
  } else {

	// Have some sort of "other" type.
	// We're going to emit some functions to set/get it's value instead
	//
	emit_set_get(name,iname, t);
  }	
}	

// -----------------------------------------------------------------------
// TCL::declare_const(char *name, DataType *type, char *value)
//
// Makes a constant.  Really just creates a variable and links to it.
// Tcl variable linkage allows read-only variables so we'll use that
// instead of just creating a Tcl variable.
// ------------------------------------------------------------------------

void TCL::declare_const(char *name, DataType *type, char *value) {

  if (type->type == T_INT) {
    fprintf(f_init,"\t entryPtr = Tcl_CreateHashEntry(&glTable,\"%s\",&n);\n", name);
    fprintf(f_init,"\t glconst = (int *) malloc(sizeof(int));\n");
    fprintf(f_init,"\t *glconst = %s;\n", value);
    fprintf(f_init,"\t Tcl_SetHashValue(entryPtr, glconst);\n");
  } else {
    fprintf(stderr,"%s : Line %d.  Illegal Constant.\n", input_file, line_number);
  }
}

// ----------------------------------------------------------------------
// TCL::usage_var(char *iname, DataType *t, char **s)
//
// Produces a usage string for a tcl variable.  Stores it in s
// ----------------------------------------------------------------------

void TCL::usage_var(char *iname, DataType *t, char **s) {

  char temp[1024], *c;

  sprintf(temp,"$%s%s : (%s)", prefix, iname, t->print_type());
  c = temp+strlen(temp);

  if (!(((t->type == T_INT) && (!t->is_pointer)) ||
       ((t->type == T_UINT) && (!t->is_pointer)) ||
       ((t->type == T_DOUBLE) && (!t->is_pointer)) ||
       ((t->type == T_CHAR) && (t->is_pointer)))) {
     sprintf(c," - unsupported");
  }

  if (*s == 0) 
    *s = new char[strlen(temp)+1];
  strcpy(*s, temp);

}

// ---------------------------------------------------------------------------
// TCL::usage_func(char *iname, DataType *t, ParmList *l,
//                        char **s)
// 
// Produces a usage string for a function in Tcl and stores it in s
// ---------------------------------------------------------------------------

void TCL::usage_func(char *iname, DataType *t, ParmList *l, char **s) {

  char temp[1024];
  char *c;
  int  i;
  Parm  *p;

  sprintf(temp,"%s : %s%s ", t->print_type(), prefix, iname);
  c = temp + strlen(temp);
  
  /* Now go through and print parameters */

  p = l->get_first();
  while (p != 0) {
    
    /* If parameter has been named, use that.   Otherwise, just print a type  */

    if ((p->t->type != T_VOID) || (p->t->is_pointer)) {
      if (strlen(p->name) > 0) {
	sprintf(c,"%s ",p->name);
	c += strlen(p->name)+1;
      }
      else {
	sprintf(c,"%s",p->t->name);
	c += strlen(p->t->name);
	if (p->t->is_pointer) {
	  for (i = 0; i < (p->t->is_pointer-p->t->implicit_ptr); i++) {
	    sprintf(c,"*");
	    c++;
	  }
	}
	sprintf(c," ");
	c++;
      }
    }
      p = l->get_next();
  }

  if (*s == 0) 
    *s = new char[strlen(temp)+1];
  strcpy(*s, temp);
}

// -----------------------------------------------------------------
// TCL::usage_const(char *name, DataType *type, char *value,
//                         char **s)
//
// Makes a usage string and places it into s;
// -----------------------------------------------------------------

void TCL::usage_const(char *name, DataType *, char *value, char **s) {

  char temp[1024];

  sprintf(temp,"$%s%s = %s", prefix, name, value);
  if (*s == 0) 
    *s = new char[strlen(temp)+1];
  strcpy(*s, temp);

}
    
// -------------------------------------------------------------------
// TCL::add_native(char *name, char *funcname)
//
// This adds an already written Tcl wrapper function to our
// initialization function.
// -------------------------------------------------------------------


void TCL::add_native(char *name, char *funcname) {
  if (nspace) {
    fprintf(f_init,"#ifdef ITCL_NAMESPACES\n");
    fprintf(f_init,"\t Tcl_CreateCommand(%s, \"%s::%s\", %s, (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);\n",interp_name, ns_name, name, funcname);
    fprintf(f_init,"#else\n");
  }
  fprintf(f_init,"\t Tcl_CreateCommand(%s, \"%s%s\", %s, (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);\n",interp_name, prefix, name, funcname);

  if (nspace) {
    fprintf(f_init,"#endif\n");
  }
}

// -------------------------------------------------------------------
// TCL::pragma(char *lname, char *name, char *value)
//
// This is an experimental %pragma handler.  Officially unsupported
// in this release, but requested in e-mail.
// --------------------------------------------------------------------

void TCL::pragma(char *lname, char *name, char *value) {

  if (strcmp(lname,"tcl") == 0) {
    if (strcmp(name,"dynamic") == 0) {
      char_result = "TCL_DYNAMIC";
    } else if (strcmp(name,"static") == 0) {
      char_result = "TCL_STATIC";
    } else if (strcmp(name,"volatile") == 0) {
      char_result = "TCL_VOLATILE";
    }
  }
}

    





