/**********************************************************************
 * Simplified Wrapper and Interface Generator  (SWIG)
 * 
 * Dave Beazley
 * 
 * Theoretical Division (T-11)           Department of Computer Science
 * Los Alamos National Laboratory        University of Utah
 * Los Alamos, New Mexico  87545         Salt Lake City, Utah  84112
 * beazley@lanl.gov                      beazley@cs.utah.edu
 *
 * Copyright (c) 1995-1996 The Regents of the University of California and
 *                         The University of Utah
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA OR THE UNIVERSITY OF
 * UTAH BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL,
 * OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF CALIFORNIA OR UNIVERSITY
 * OF UTAH HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA AND UNIVERSITY OF UTAH SPECIFICALLY DISCLAIM
 * ANY WARRANTIES,INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 * CALIFORNIA OR UNIVERITY OF UTAH HAS NO OBLIGATION TO PROVIDE MAINTENANCE,
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 **************************************************************************/
/***********************************************************************
 * $Header: /b11/dmb/SWIG/SWIG1.0/Modules/RCS/perl5.cxx,v 1.30 1996/09/04 21:08:27 dmb Exp dmb $
 *
 * perl5.c
 *
 * Definitions for adding functions to Perl 5
 *
 * How to extend perl5 (note : this is totally different in Perl 4) :
 *
 * 1.   Variable linkage
 *
 *      Must declare two functions :
 *
 *          _var_set(SV *sv, MAGIC *mg);
 *          _var_get(SV *sv, MAGIC *mg);
 *
 *      These functions must set/get the values of a variable using
 *      Perl5 internals.
 *
 *      To add these to Perl5 (which isn't entirely clear), need to
 *      do the following :
 *
 *            SV  *sv;
 *            MAGIC  *m;
 *            sv = perl_get_sv("varname",TRUE);
 *            sv_magic(sv,sv, 'U', "varname", strlen("varname));
 *            m = mg_find(sv, 'U');
 *            m->mg_virtual = (MGVTBL *) malloc(sizeof(MGVTBL));
 *            m->mg_virtual.svt_get = _var_set;
 *            m->mg_virtual.svt_set = _var_get;
 *            m->mg_virtual.svt_len = 0;
 *            m->mg_virtual.svt_free = 0;
 *            m->mg_virtual.svt_clear = 0;
 *
 *
 * 2.   Function extension
 *
 *      Functions are declared as :
 *             XS(_wrap_func) {
 *                 dXSARGS;
 *                 if (items != parmcount) {
 *                     croak("Usage :");
 *                 }
 *              ... get arguments ...
 *
 *              ... call function ...
 *              ... set return value in ST(0) 
 *                 XSRETURN(1);
 *              }
 *      To extract function arguments, use the following :
 *              _arg = (int) SvIV(ST(0))
 *              _arg = (double) SvNV(ST(0))
 *              _arg = (char *) SvPV(ST(0),na);
 *
 *      For return values, use :
 *              ST(0) = sv_newmortal();
 *              sv_setiv(ST(0), (IV) RETVAL);     // Integers
 *              sv_setnv(ST(0), (double) RETVAL); // Doubles
 *              sv_setpv((SV*) ST(0), RETVAL);    // Strings
 *
 *      New functions are added using 
 *              newXS("name", _wrap_func, file)
 *
 *    
 * 3.   Compilation.
 *
 *      Code should be compiled into an object file for dynamic
 *      loading into Perl.
 *
 * -- Revision History
 * $Log: perl5.cxx,v $
 * Revision 1.30  1996/09/04 21:08:27  dmb
 * Fixed minor bug with packages
 *
 * Revision 1.29  1996/08/27 23:01:51  dmb
 * Minor changes to error handling
 *
 * Revision 1.28  1996/08/21 16:51:14  dmb
 * Minor cleanup to eliminate warnings
 *
 * Revision 1.27  1996/08/21 05:49:45  dmb
 * Some fixes to the new pointer type-checker.
 *
 * Revision 1.26  1996/08/15 05:08:57  dmb
 * Major overhaul.  Changed generation of wrapper functions to
 * eliminate problems with undeclared variables.
 *
 * Also switched over to Perl5 references which seems to work much
 * better---well, at least with other Perl5 extensions.
 *
 * Revision 1.25  1996/08/12 01:49:33  dmb
 * Changes to support new language class.   Also changed implementation
 * of the variable initialization code
 *
 * Revision 1.24  1996/08/02 02:58:01  dmb
 * Changed to use better parameter list functions
 *
 * Revision 1.23  1996/07/17 14:55:15  dmb
 * Fixed bug in -strict 1 pointer type checking mode.
 *
 * Revision 1.22  1996/06/02 00:14:17  beazley
 * Changed something--but I can't remember what.
 *
// Revision 1.21  1996/05/22  20:20:21  beazley
// Add banner and cleanup functions to headers() and close() functions
//
// Revision 1.20  1996/05/20  23:35:56  beazley
// Added a few more constant datatypes.
//
// Revision 1.19  1996/05/17  05:53:12  beazley
// Added return by value support.
//
// Revision 1.18  1996/05/13  23:45:28  beazley
// Reworked the module/init procedure
//
// Revision 1.17  1996/05/01  22:41:30  dmb
// Cleaned up command line option handling.
//
 * Revision 1.16  1996/04/16 17:13:01  dmb
 * Fixed bug when linking to pointer variables.
 *
 * Revision 1.15  1996/04/09 20:19:01  beazley
 * Minor cleanup
 *
// Revision 1.14  1996/04/08  22:09:28  beazley
// Minor cleanup
//
// Revision 1.13  1996/04/03  22:48:44  beazley
// Minor changes to module naming.
//
// Revision 1.12  1996/03/28  02:46:56  beazley
// Minor bug fix to documentation.
//
// Revision 1.11  1996/03/24  22:14:51  beazley
// Cleaned up wrapper file construction. Took out "system" calls.
//
// Revision 1.10  1996/03/22  23:41:31  beazley
// Fixed to work with new class structure.  Fixed variable linkage
// problem.   Added support for constants.
//
// Revision 1.9  1996/03/04  21:28:54  beazley
// Changed usage(), made changes to pointer handling.
//
// Revision 1.8  1996/02/20  04:16:25  beazley
// Took out streams.
//
// Revision 1.7  1996/02/19  08:34:28  beazley
// Fixed a bug with pointer return values.
//
// Revision 1.6  1996/02/19  05:30:58  beazley
// Changed treatment of pointers to hexadecimal convert functions.
// Fixed quite a few minor bugs in the handling of several datatypes.
//
// Revision 1.5  1996/02/17  22:55:19  beazley
// Fixed documentation and a few things with package names.
//
// Revision 1.4  1996/02/16  07:20:17  beazley
// Fixed problems with sprintf().  Added package name to documentation.
//
// Revision 1.3  1996/02/16  06:38:49  beazley
// Removed a few unused variables.
//
// Revision 1.2  1996/02/16  05:20:22  beazley
// Changed variable linkage procedure to add "var_init" function.
// Fixed bugs related to fixes in other modules.
//
// Revision 1.1  1996/02/15  22:39:31  beazley
// Initial revision
//
 *
 ***********************************************************************/

#include "swig.h"
#include "perl5.h"

static char *usage = "\
Perl5 Options (available with -perl5)\n\
     -module name    - Set module name\n\
     -package name   - Set package prefix\n\
     -static         - Omit code related to dynamic loading.\n\
     -exportall      - Export all symbols (not generally recommended)\n\n";  

// ---------------------------------------------------------------------
// PERL5::parse_args(int argc, char *argv[])
//
// Parse command line options.
// ---------------------------------------------------------------------

void
PERL5::parse_args(int argc, char *argv[]) {

  int i = 1;

  export_all = 0;
  sprintf(LibDir,"%s",perl_path);

  // Look for certain command line options

  // Get options
  for (i = 1; i < argc; i++) {
      if (argv[i]) {
	  if(strcmp(argv[i],"-package") == 0) {
	    if (argv[i+1]) {
	      package = new char[strlen(argv[i+1])+1];
	      strcpy(package, argv[i+1]);
	      mark_arg(i);
	      mark_arg(i+1);
	      i++;
	    } else {
	      arg_error();
	    }
	  } else if (strcmp(argv[i],"-module") == 0) {
	    if (argv[i+1]) {
	      module = new char[strlen(argv[i+1])+1];
	      strcpy(module, argv[i+1]);
	      mark_arg(i);
	      mark_arg(i+1);
	      i++;
	    } else {
	      arg_error();
	    }
	  } else if (strcmp(argv[i],"-exportall") == 0) {
	      export_all = 1;
	      mark_arg(i);
	  } else if (strcmp(argv[i],"-static") == 0) {
	      is_static = 1;
	      mark_arg(i);
	  } else if (strcmp(argv[i],"-help") == 0) {
	    fputs(usage,stderr);
	    exit(0);
	  }
      }
  }
}

// ------------------------------------------------------------------
// PERL5::parse()
//
// Parse an interface file
// ------------------------------------------------------------------

void
PERL5::parse() {


  printf("Generating wrappers for Perl 5\n");

  // Print out PERL5 specific headers
  
  headers();
  
  // Run the parser
  
  yyparse();
  fputs(vinit.get(),f_wrappers);
}


// ---------------------------------------------------------------------
// PERL5::set_module(char *mod_name)
//
// Sets the module name.
// Does nothing if it's already set (so it can be overridden as a command
// line option).
//
//----------------------------------------------------------------------

void PERL5::set_module(char *mod_name) {

  if (module) return;
  
  module = new char[strlen(mod_name)+1];
  strcpy(module,mod_name);

}

// ---------------------------------------------------------------------
// PERL5::set_init(char *iname)
//
// Sets the initialization function name.
// Does nothing if it's already set
//
//----------------------------------------------------------------------

void PERL5::set_init(char *iname) {
  set_module(iname);
}

// ---------------------------------------------------------------------
// PERL5::headers(void)
//
// Generate the appropriate header files for PERL5 interface.
// ----------------------------------------------------------------------

void PERL5::headers(void)
{

  emit_banner(f_header);

  fprintf(f_header,"/* Implementation : PERL 5 */\n\n");
  fprintf(f_header,"#ifdef __cplusplus\n");
  fprintf(f_header,"extern \"C\" {\n");
  fprintf(f_header,"#endif\n");
  fprintf(f_header,"#include \"EXTERN.h\"\n");
  fprintf(f_header,"#include \"perl.h\"\n");
  fprintf(f_header,"#include \"XSUB.h\"\n");
  fprintf(f_header,"#include <string.h>\n");
//  fprintf(f_header,"#include <stdlib.h>\n");
  fprintf(f_header,"#ifdef __cplusplus\n");
  fprintf(f_header,"}\n");
  fprintf(f_header,"#endif\n");

  if (IncludeFile("perl5ptr.swg", f_header) == -1) {
    fprintf(stderr,"SWIG : Fatal error.  Unable to locate 'perl5ptr.swg' in SWIG library.\n");
    SWIG_exit(1);
  }
  /* Emit an entirely different pointer handling mechanism for Perl5 */
  // emit_hex(f_header);
}

// --------------------------------------------------------------------
// PERL5::initialize()
//
// Output initialization code that registers functions with the
// interface.
// ---------------------------------------------------------------------

void PERL5::initialize()
{

  char filen[256];

  if (!module)
    module = "swig";

  if (!package) {
    package = new char[strlen(module)+1];
    strcpy(package,module);
  }

  /* Create a .pm file */

  sprintf(filen,"%s.pm", module);
  if ((f_pm = fopen(filen,"w")) == 0) {
    fprintf(stderr,"Unable to open %s\n", filen);
    SWIG_exit(0);
  }
  
  fprintf(f_header,"#define SWIG_init    boot_%s\n\n", module);
  fprintf(f_header,"#define SWIG_name   \"%s::boot_%s\"\n", package, module);
  fprintf(f_header,"#define SWIG_varinit \"%s::var_%s_init();\"\n", package, module);
  fprintf(f_header,"#ifdef __cplusplus\n");
  fprintf(f_header,"extern \"C\"\n");
  fprintf(f_header,"#endif\n");
  fprintf(f_header,"void boot_%s _((CV* cv));\n", module);
  fprintf(f_init,"#ifdef __cplusplus\n");
  fprintf(f_init,"extern \"C\"\n");
  fprintf(f_init,"#endif\n");
  fprintf(f_init,"XS(boot_%s) {\n", module);
  fprintf(f_init,"\t dXSARGS;\n");
  fprintf(f_init,"\t char *file = __FILE__;\n");
  fprintf(f_init,"\t newXS(\"%s::var_%s_init\", _wrap_perl5_%s_var_init, file);\n",package,module, module);
  vinit << "XS(_wrap_perl5_" << module << "_var_init) {\n"
        << tab4 << "SV *sv;\n"
        << tab4 << "MAGIC *mg;\n"
        << tab4 << "dXSARGS;\n";
  
  fprintf(f_pm,"package %s;\n",module);
  fprintf(f_pm,"require Exporter;\n");
  if (!is_static) {
    fprintf(f_pm,"require DynaLoader;\n");
    fprintf(f_pm,"@ISA = qw(Exporter DynaLoader);\n");
  } else {
    fprintf(f_pm,"@ISA = qw(Exporter);\n");
  }    
  fprintf(f_pm,"@EXPORT = qw( ");

  /* Process additional initialization files here */
  // Not currently supported.

}


// ---------------------------------------------------------------------
// PERL5::close(void)
//
// Wrap things up.  Close initialization function.
// ---------------------------------------------------------------------

void PERL5::close(void)
{

  emit_ptr_equivalence("_swig_ptr_perl5",f_wrappers);
  fprintf(f_init,"\t swig_ptr_derived = _swig_ptr_perl5;\n");

  fprintf(f_init,"\t ST(0) = &sv_yes;\n");
  fprintf(f_init,"\t XSRETURN(1);\n");
  fprintf(f_init,"}\n");

  vinit << tab4 << "XSRETURN(1);\n"
        << "}\n";

  fprintf(f_pm,");\n");
  fprintf(f_pm,"package %s;\n", package);	
  if (!is_static) {
    fprintf(f_pm,"bootstrap %s;\n", module);
  } else {
    fprintf(f_pm,"boot_%s();\n", module);
  }
  fprintf(f_pm,"var_%s_init();\n", module);
  fprintf(f_pm,"1;\n");
  fclose(f_pm);
  
}

// ----------------------------------------------------------------------
// char *PERL5::type_mangle(DataType *t)
//
// Mangles a datatype into a Perl5 name compatible with xsubpp type
// T_PTROBJ.
// ----------------------------------------------------------------------

char *
PERL5::type_mangle(DataType *t) {
  static char result[128];
  int   i;
  char *r, *c;

  r = result;
  c = t->name;

  for ( c = t->name; *c; c++,r++) {
      *r = *c;
  }
  for (i = 0; i < (t->is_pointer-t->implicit_ptr); i++, r++) {
    strcpy(r,"Ptr");
    r+=2;
  }
  *r = 0;
  return result;
}

// ----------------------------------------------------------------------
// PERL5::get_pointer(char *iname, char *srcname, char *src, char *target,
//                     DataType *t, WrapperFunction &f, char *ret)
//
// Emits code to get a pointer from a parameter and do type checking.
// parm is the parameter number.   This function is only used
// in create_function().
// ----------------------------------------------------------------------

void PERL5::get_pointer(char *iname, char *srcname, char *src, char *dest,
			DataType *t, String &f, char *ret) {

  // Now get the pointer value from the string and save in dest
  
  f << tab4 << "if (_swig_get_hex(" << src << ",(void **) &" << dest << ",";

  // If we're passing a void pointer, we give the pointer conversion a NULL
  // pointer, otherwise pass in the expected type.
  
  if (t->type == T_VOID) f << "(char *) 0 )) {\n";
  else
    f << "\"" << t->print_mangle() << "\")) {\n";

  // This part handles the type checking according to three different
  // levels.   0 = no checking, 1 = warning message, 2 = strict.

  switch(TypeStrict) {
  case 0: // No type checking
    f << tab4 << "}\n";
    break;

  case 1: // Warning message only

    // Change this part to how you want to handle a type-mismatch warning.
    // By default, it will just print to stderr.

    f << tab8 << "fprintf(stderr,\"Warning : type mismatch in " << srcname
      << " of " << iname << ". Expected " << t->print_mangle()
      << ", received %s\\n\"," << src << ");\n"
      << tab4 << "}\n";

    break;
  case 2: // Super strict mode.

    // Change this part to return an error.

    f << tab8 << "croak(\"Type error in " << srcname
	   << " of " << iname << ". Expected " << t->print_mangle() << ".\");\n"
	   << tab8 << ret << ";\n"
	   << tab4 << "}\n";

    break;
    
  default :
    fprintf(stderr,"SWIG Error. Unknown strictness level\n");
    break;
  }
}

// ----------------------------------------------------------------------
// PERL5::create_function(char *name, char *iname, DataType *d,
//                             ParmList *l)
//
// Create a function declaration and register it with the interpreter.
// ----------------------------------------------------------------------

void PERL5::create_function(char *name, char *iname, DataType *d, ParmList *l)
{
  Parm *p;
  int   pcount,i;
  char  wname[256];
  char *usage = 0;
  WrapperFunction f;
  char  src[32];
  char  temp[256];
  char  target[256];
  
  // Make a wrapper name for this
  strcpy(wname,iname);
  make_wrap_name(wname);
  
  // Now write the wrapper function itself....this is pretty ugly

  f.def << "XS(_wrap_" << wname << ") {\n";
  pcount = emit_args(d, l, f);
  f.code << tab4 << "dXSARGS;\n";

  // Check the number of arguments

  usage_func(iname,d,l,&usage);
  f.code << tab4 << "if (items != " << pcount << ") \n"
	 << tab8 << "croak(\"Usage: " << usage << "\");\n";
  delete usage;

  // Write code to extract parameters.
  // This section should be able to extract virtually any kind 
  // parameter, represented as a string

  i = 0;
  p = l->get_first();
  while (p != 0) {
    if (!p->t->is_pointer) {

      // Extract a parameter by "value"

      switch(p->t->type) {

	// Integers

      case T_INT :
      case T_SHORT :
      case T_LONG :
      case T_SINT :
      case T_SSHORT:
      case T_SLONG:
      case T_SCHAR:
      case T_UINT:
      case T_USHORT:
      case T_ULONG:
      case T_UCHAR:
	f.code << tab4 << "_arg" << i << " = " << p->t->print_cast()
	       << "SvIV(ST(" << i << "));\n";
	  break;
      case T_CHAR :
	f.code << tab4 << "_arg" << i << " = (char) *SvPV(ST(" << i << "),na);\n";
	break;

	// Doubles

      case T_DOUBLE :
      case T_FLOAT :
	f.code << tab4 << "_arg" << i << " = " << p->t->print_cast()
	       << " SvNV(ST(" << i << "));\n";
	break;

	// Void.. Do nothing.

      case T_VOID :
	break;

	// User defined.   This is invalid here.   Note, user-defined types by
	// value are handled in the parser.

      case T_USER:

	// Unsupported data type

      default :
	fprintf(stderr,"%s : Line %d. Unable to use type %s as a function argument.\n",input_file, line_number, p->t->print_type());
	break;
	}
    } else {

      // Argument is a pointer type.   Special case is for char *
      // since that is usually a string.

      if ((p->t->type == T_CHAR) && (p->t->is_pointer == 1)) {
	f.code << tab4 << "_arg" << i << " = (char *) SvPV(ST(" << i << "),na);\n";
      } else {

	// Have a generic pointer type here.    Read it in as a swig
        // typed pointer.

	sprintf(src,"ST(%d)",i);
	sprintf(target,"_arg%d", i);
	sprintf(temp,"argument %d", i+1);
	get_pointer(iname,temp,src,target, p->t, f.code, "XSRETURN(1)");
      }
    }
    p = l->get_next();
    i++;
  }

  // Now write code to make the function call

  emit_func_call(name,d,l,f);

  if ((d->type != T_VOID) || (d->is_pointer)) {
    // Now have return value, figure out what to do with it.

    if (!d->is_pointer) {

      // Function returns a "value"
      f.code << tab4 << "ST(0) = sv_newmortal();\n";
      switch(d->type) {
      case T_INT: case T_SINT: case T_UINT:
      case T_SHORT: case T_SSHORT: case T_USHORT:
      case T_LONG : case T_SLONG : case T_ULONG:
      case T_SCHAR: case T_UCHAR :
	f.code << tab4 << "sv_setiv(ST(0),(IV) _result);\n";
	break;
      case T_DOUBLE :
      case T_FLOAT :
	f.code << tab4 << "sv_setnv(ST(0), (double) _result);\n";
	break;
      case T_CHAR :
	f.add_local("char", "_ctemp[2]");
	f.code << tab4 << "_ctemp[0] = _result;\n"
	       << tab4 << "_ctemp[1] = 0;\n"
	       << tab4 << "sv_setpv((SV*)ST(0),_ctemp);\n";
	break;

	// Return a complex type by value

      case T_USER:
	d->is_pointer++;
	f.code << tab4 << "sv_setref_pv(ST(0),\"" << d->print_mangle()
	       << "\", (void *) _result);\n";
	d->is_pointer--;
	break;

      default :
	fprintf(stderr,"%s: Line %d. Unable to use return type %s in function %s.\n", input_file, line_number, d->print_type(), name);
	break;
      }
    } else {

    // Is a pointer return type
      f.code << tab4 << "ST(0) = sv_newmortal();\n";
      if ((d->type == T_CHAR) && (d->is_pointer == 1)) {
	
	// Return a character string
	f.code << tab4 << "sv_setpv((SV*)ST(0),_result);\n";

      } else {
	// Is an ordinary pointer type.
	f.code << tab4 << "sv_setref_pv(ST(0),\"" << d->print_mangle()
	       << "\", (void *) _result);\n";
      }
    }
  }
  
  // Wrap things up (in a manner of speaking)

  f.code << tab4 << "XSRETURN(1);\n}\n";

  // Dump this function out

  f.print(f_wrappers);

  // Now register the function

  fprintf(f_init,"\t newXS(\"%s::%s\", _wrap_%s, file);\n", package, iname, wname);
  if (export_all) 
    fprintf(f_pm,"%s ", iname);
}


// -----------------------------------------------------------------------
// PERL5::link_variable(char *name, char *iname, DataType *d)
//
// Create a link to a C variable.
// -----------------------------------------------------------------------

void PERL5::link_variable(char *name, char *iname, DataType *t)
{
  char  set_name[256];
  char  val_name[256];
  WrapperFunction  getf, setf;
  

  sprintf(set_name,"_wrap_set_%s",iname);
  sprintf(val_name,"_wrap_val_%s",iname);

  // Create a Perl function for setting the variable value

  setf.def << "int " << set_name << "(SV* sv, MAGIC *mg) {\n";
    if (Status & STAT_READONLY) {
      setf.code << tab4 << "croak(\"Unable to set " << iname << ". Variable is read-only.\");\n";
    } else {
      if (!t->is_pointer) {

	// Set the value to something 

	switch(t->type) {
	case T_INT : case T_SINT : case T_UINT:
	case T_SHORT : case T_SSHORT : case T_USHORT:
	case T_LONG : case T_SLONG : case T_ULONG:
	case T_UCHAR: case T_SCHAR:
	  setf.code << tab4 << name << " = " << t->print_cast() << " SvIV(sv);\n";
	  break;
	case T_DOUBLE :
	case T_FLOAT :
	  setf.code << tab4 << name << " = " << t->print_cast() << " SvNV(sv);\n";
	  break;
	case T_CHAR :
	  setf.code << tab4 << name << " = (char) *SvPV(sv,na);\n";
	  break;

	case T_USER:
	  
	  // Add support for User defined type here
	  // Get as a pointer value

	  t->is_pointer++;
	  setf.add_local("void","*_temp");
	  get_pointer(iname,"value","sv","_temp", t, setf.code, "return(1)");
	  setf.code << tab4 << name << " = *(" << t->print_cast() << ") _temp;\n";
	  t->is_pointer--;
	  break;
	  
	default :
	  fprintf(stderr,"%s : Line %d.  Unable to link with datatype %s (ignored).\n", input_file, line_number, t->print_type());
	  return;
	  break;
	}
      } else {
	// Have some sort of pointer type here, Process it differently
	if ((t->type == T_CHAR) && (t->is_pointer == 1)) {
	  setf.add_local("char","*_a");
	  setf.code << tab4 << "_a = (char *) SvPV(sv,na);\n";

	  if (CPlusPlus)
	    setf.code << tab4 << "if (" << name << ") delete " << name << ";\n"
		   << tab4 << name << " = new char[strlen(_a)+1];\n";
	  else
	    setf.code << tab4 << "if (" << name << ") free(" << name << ");\n"
		   << tab4 << name << " = (char *) malloc(strlen(_a)+1);\n";
	  setf.code << "strcpy(" << name << ",_a);\n";
	} else {
	  // Set the value of a pointer

	  setf.add_local("void","*_temp");
	  get_pointer(iname,"value","sv","_temp", t, setf.code, "return(1)");
	  setf.code << tab4 << name << " = " << t->print_cast() << " _temp;\n";
	}
      }
    }
    setf.code << tab4 << "return 1;\n"
	      << "}\n";

    setf.print(f_wrappers);
    
    // Now write a function to evaluate the variable

    getf.def << "int " << val_name << "(SV *sv, MAGIC *mg) {\n";
    if (!t->is_pointer) {
      switch(t->type) {
      case T_INT : case T_SINT: case T_UINT:
      case T_SHORT : case T_SSHORT: case T_USHORT:
      case T_LONG : case T_SLONG : case T_ULONG:
      case T_UCHAR: case T_SCHAR:
	getf.code << tab4 << "sv_setiv(sv, (IV) " << name << ");\n";
	break;
      case T_DOUBLE :
      case T_FLOAT :
	getf.code << tab4 << "sv_setnv(sv, (double) " << name << ");\n";
	break;
      case T_CHAR :
	getf.add_local("char","_ptemp[2]");
	getf.code << tab4 << "_ptemp[0] = " << name << ";\n"
		  << tab4 << "_ptemp[1] = 0;\n"
		  << tab4 << "sv_setpv((SV*) sv, _ptemp);\n";
	break;
      case T_USER:
	t->is_pointer++;
	getf.code << tab4 << "sv_setref_pv((SV*) sv,\"" << t->print_mangle()
		  << "\", (void *) &" << name << ");\n";
	t->is_pointer--;
	
	break;
      default :
	break;
      }
    } else {
      
      // Have some sort of arbitrary pointer type.  Return it as a string
      
      if ((t->type == T_CHAR) && (t->is_pointer == 1))
	getf.code << tab4 << "sv_setpv((SV*) sv, " << name << ");\n";
      else {
	getf.code << tab4 << "sv_setref_pv((SV*) sv,\"" << t->print_mangle()
	       << "\", (void *) " << name << ");\n";
      }
    }
    getf.code << tab4 << "return 1;\n"
	      << "}\n";

    getf.print(f_wrappers);
    
    // Now add symbol to the PERL interpreter
    vinit << tab4 << "sv = perl_get_sv(\"" << iname << "\", TRUE);\n"
	  << tab4 << "sv_magic(sv, sv, 'U', \"" << iname <<"\", " << strlen(iname) << ");\n"
	  << tab4 << "mg = mg_find(sv, 'U');\n";

    // perl5 uses a common magic virtual table.   We need to create our own 
    // in order for our linkage to work.

    vinit << tab4 << "mg->mg_virtual = (MGVTBL *) malloc(sizeof(MGVTBL));\n"
	  << tab4 << "mg->mg_virtual->svt_get = " << val_name << ";\n"
	  << tab4 << "mg->mg_virtual->svt_set = " << set_name << ";\n"
	  << tab4 << "mg->mg_virtual->svt_len = 0;\n"
	  << tab4 << "mg->mg_virtual->svt_clear = 0;\n"
	  << tab4 << "mg->mg_virtual->svt_free = 0;\n";
}

// -----------------------------------------------------------------------
// PERL5::declare_const(char *name, DataType *type, char *value)
//
// Makes a constant.  Really just creates a variable and creates a read-only
// link to it.
// ------------------------------------------------------------------------

void
PERL5::declare_const(char *name, DataType *type, char *value)
{

  int OldStatus = Status;      // Save old status flags
  char   var_name[256];

  Status = STAT_READONLY;      // Enable readonly mode.

  // Make a static variable;

  sprintf(var_name,"_wrap_const_%s",name);


  if ((type->type == T_USER) && (!type->is_pointer)) {
    fprintf(stderr,"%s : Line %d.  Unsupported constant value.\n", input_file, line_number);
    return;
  }

  // Create variable and assign it a value

  fprintf(f_header,"static %s %s = ", type->print_type(), var_name);
  if ((type->type == T_CHAR) && (type->is_pointer <= 1)) {
    fprintf(f_header,"\"%s\";\n", value);
  } else {
    fprintf(f_header,"%s;\n", value);
  }

  // Now create a variable declaration

  link_variable(var_name, name, type);
  Status = OldStatus;

}

// ----------------------------------------------------------------------
// PERL5::usage_var(char *iname, DataType *t, char **s)
//
// Produces a usage string for a Perl 5 variable.
// ----------------------------------------------------------------------

void PERL5::usage_var(char *iname, DataType *t, char **s) {

  char temp[1024], *c;

  sprintf(temp,"$%s::%s : (%s)", package, iname, t->print_type());
  c = temp + strlen(temp);
  if (!((t->type != T_USER) || (t->is_pointer))) {
    sprintf(c," - unsupported");
  }
  if (*s == 0) 
    *s = new char[strlen(temp)+1];
  strcpy(*s,temp);

}

// ---------------------------------------------------------------------------
// PERL5::usage_func(char *iname, DataType *t, ParmList *l, char **s)
// 
// Produces a usage string for a function in Perl
// ---------------------------------------------------------------------------

void PERL5::usage_func(char *iname, DataType *t, ParmList *l, char **s) {

  char temp[1024];
  char *c;
  int  i;
  Parm  *p;

  sprintf(temp,"%s : %s::%s(",t->print_type(), package, iname);
  c = temp + strlen(temp);
  
  /* Now go through and print parameters */

  p = l->get_first();
  while (p != 0) {
    
    /* If parameter has been named, use that.   Otherwise, just print a type  */

    if ((p->t->type != T_VOID) || (p->t->is_pointer)) {
      if (strlen(p->name) > 0) {
	sprintf(c,"%s ",p->name);
	c += strlen(p->name) + 1;
      } else {
	sprintf(c,"%s",p->t->name);
	c += strlen(p->t->name);
	for (i = 0; i < (p->t->is_pointer-p->t->implicit_ptr); i++) {
	  sprintf(c,"*");
	  c++;
	}
      }
    }
    p = l->get_next();
    if (p != 0) {
      sprintf(c,",");
      c++;
    }
  }
  sprintf(c,");");

  if (*s == 0) 
    *s = new char[strlen(temp)+1];
  strcpy(*s,temp);

}

// ----------------------------------------------------------------------
// PERL5::usage_const(char *iname, DataType *type, char *value, char **s)
//
// Produces a usage string for a Perl 5 constant
// ----------------------------------------------------------------------

void PERL5::usage_const(char *iname, DataType *, char *value, char **s) {

  char temp[1024];

  sprintf(temp,"$%s::%s = %s", package, iname, value);

  if (*s == 0) 
    *s = new char[strlen(temp)+1];
  strcpy(*s,temp);

}


// -----------------------------------------------------------------------
// PERL5::add_native(char *name, char *funcname)
//
// Add a native module name to Perl5.
// -----------------------------------------------------------------------

void PERL5::add_native(char *name, char *funcname) {
  fprintf(f_init,"\t newXS(\"%s::%s\", %s, file);\n", package,name, funcname);
  if (export_all)
    fprintf(f_pm,"%s ", name);
}
