
/**********************************************************************
 * Simplified Wrapper and Interface Generator  (SWIG)
 * 
 * Dave Beazley
 * 
 * Theoretical Division (T-11)           Department of Computer Science
 * Los Alamos National Laboratory        University of Utah
 * Los Alamos, New Mexico  87545         Salt Lake City, Utah  84112
 * beazley@lanl.gov                      beazley@cs.utah.edu
 *
 * Copyright (c) 1995-1996
 * The Regents of the University of California and the University of Utah
 * All Rights Reserved
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that 
 * (1) The above copyright notice and the following two paragraphs
 * appear in all copies of the source code and (2) redistributions
 * including binaries reproduces these notices in the supporting
 * documentation.   Substantial modifications to this software may be
 * copyrighted by their authors and need not follow the licensing terms
 * described here, provided that the new terms are clearly indicated in
 * all files where they apply.
 * 
 * IN NO EVENT SHALL THE AUTHOR, THE UNIVERSITY OF CALIFORNIA, THE 
 * UNIVERSITY OF UTAH OR DISTRIBUTORS OF THIS SOFTWARE BE LIABLE TO ANY
 * PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
 * DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION,
 * EVEN IF THE AUTHORS OR ANY OF THE ABOVE PARTIES HAVE BEEN ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE AUTHOR, THE UNIVERSITY OF CALIFORNIA, AND THE UNIVERSITY OF UTAH
 * SPECIFICALLY DISCLAIM ANY WARRANTIES,INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND 
 * THE AUTHORS AND DISTRIBUTORS HAVE NO OBLIGATION TO PROVIDE MAINTENANCE,
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 * The author requests that all users of this software return any
 * improvements made to beazley@cs.utah.edu and grant the author
 * full redistribution rights.
 *
 **************************************************************************/
/**********************************************************************
 * $Header: /b11/dmb/SWIG/SWIG1.0/Modules/RCS/python.cxx,v 1.25 1996/08/29 05:41:00 dmb Exp $
 *
 * python.cxx
 *
 * All new SWIG Python module.   This module has been completely rewritten
 * using some new functions to appear in SWIG 2.0.
 *
 * $Log: python.cxx,v $
 * Revision 1.25  1996/08/29 05:41:00  dmb
 * Fixed help message
 *
 * Revision 1.24  1996/08/27 23:02:21  dmb
 * Minor changes to error handling
 *
 * Revision 1.23  1996/08/25 00:05:13  dmb
 * More improvements to shadow classes
 *
 * Revision 1.22  1996/08/21 16:51:37  dmb
 * Cleanup to eliminate warning messages
 *
 * Revision 1.21  1996/08/21 05:50:28  dmb
 * More changes to shadow classes.
 *
 * Revision 1.20  1996/08/16 04:17:21  dmb
 * Added SWIG_name declaration
 *
 * Revision 1.19  1996/08/12 01:50:03  dmb
 * Almost a completely new version. Changed variable linking, constants,
 * and added support for new language class structure
 *
 * Revision 1.18  1996/08/02 02:57:41  dmb
 * Modified to use better parameter list functions()
 *
 * Revision 1.17  1996/07/17 15:06:30  dmb
 * Fixed a few more bugs with -strict 1 mode.
 *
 * Revision 1.16  1996/07/17 14:57:42  dmb
 * Fixed bug in -strict 1 pointer type checking mode.
 *
 * Revision 1.15  1996/06/02 00:14:36  beazley
 * Minor fixes
 *
// Revision 1.14  1996/05/22  20:20:21  beazley
// Add banner and cleanup functions to headers() and close() functions
//
// Revision 1.13  1996/05/20  23:36:07  beazley
// Added a few more constant datatypes.
//
// Revision 1.12  1996/05/17  05:53:32  beazley
// Added return by value support
//
// Revision 1.11  1996/05/16  03:58:34  beazley
// Fixed bug in function return type error message.
//
// Revision 1.10  1996/05/13  23:45:37  beazley
// Reworked the module/init procedure
//
// Revision 1.9  1996/05/10  23:38:26  beazley
// Minor changes
//
// Revision 1.8  1996/05/07  03:24:10  beazley
// Fixed bug with linking pointer variable types.
//
// Revision 1.7  1996/05/03  05:11:15  dmb
// Fixed up some C++ linking problems.
//
 * Revision 1.6  1996/05/01 22:42:01  dmb
 * Cleaned up command line option handling.
 *
 * Revision 1.5  1996/04/13 19:02:05  beazley
 * Fixed bug in module system and variable linking.
 *
// Revision 1.4  1996/04/09  20:18:35  beazley
// Minor cleanup.
//
// Revision 1.3  1996/04/08  22:09:36  beazley
// Minor cleanup
//
// Revision 1.2  1996/03/28  02:47:14  beazley
// Some bug fixes, etc...
//
// Revision 1.1  1996/03/24  22:15:14  beazley
// Initial revision
//
 *
 **************************************************************************/


#include "swig.h"
#include "python.h"

static char *usage = "\
Python Options (available with -python)\n\
     -module name    - Set module name\n\
     -globals name   - Set name used to access C global variable ('cvar' by default).\n\
     -shadow         - Generate shadow classes. \n\n";

// ---------------------------------------------------------------------
// PYTHON::parse_args(int argc, char *argv[])
//
// ---------------------------------------------------------------------

void PYTHON::parse_args(int argc, char *argv[]) {

  int i = 1;

  sprintf(LibDir,"%s",path);


  // Look for additional command line options.
  for (i = 1; i < argc; i++) {	
      if (argv[i]) {
	  if(strcmp(argv[i],"-module") == 0) {
	    if (argv[i+1]) {
	      module = new char[strlen(argv[i+1])+2];
	      strcpy(module, argv[i+1]);
	      mark_arg(i);
	      mark_arg(i+1);
	      i+=1;
	    } else {
	      arg_error();
	    } 
	  } else if (strcmp(argv[i],"-globals") == 0) {
	    if (argv[i+1]) {
	      global_name = new char[strlen(argv[i+1])+1];
	      strcpy(global_name, argv[i+1]);
	      mark_arg(i);
	      mark_arg(i+1);
	      i++;
	    } else {
	      arg_error();
	    }
	  } else if (strcmp(argv[i],"-shadow") == 0) {
	    shadow = 1;
	    mark_arg(i);
	  } else if (strcmp(argv[i],"-help") == 0) {
	    fputs(usage,stderr);
	    exit(0);
	  }
      }
  }
}

// ---------------------------------------------------------------------
// PYTHON::parse()
//
// Parse the interface file
// ---------------------------------------------------------------------

void
PYTHON::parse() {
  
  printf("Generating wrappers for Python\n"); 
  headers();
  
  // Run the SWIG parser
  
  yyparse();
}

// ---------------------------------------------------------------------
// PYTHON::set_module(char *mod_name)
//
// Sets the module name.
// Does nothing if it's already set (so it can be overridden as a command
// line option).
//
//----------------------------------------------------------------------

void PYTHON::set_module(char *mod_name) {

  if (module) return;

  module = new char[strlen(mod_name)+1];
  strcpy(module,mod_name);

}

// ---------------------------------------------------------------------
// PYTHON::set_init(char *iname)
//
// Sets the initialization function name.
// Does nothing if it's already set
//
//----------------------------------------------------------------------

void PYTHON::set_init(char *iname) {
  set_module(iname);
}

// ----------------------------------------------------------------------
// PYTHON::add_method(char *name, char *function)
//
// Add some symbols to the methods table
// ----------------------------------------------------------------------

void PYTHON::add_method(char *name, char *function) {

  Method *n;

  n = new Method;
  n->name = new char[strlen(name)+1];
  strcpy(n->name,name);
  n->function = new char[strlen(function)+1];
  strcpy(n->function, function);

  n->next = head;
  head = n;
}

// ---------------------------------------------------------------------
// PYTHON::print_methods()
//
// Prints out the method array.
// ---------------------------------------------------------------------

void PYTHON::print_methods() {

  Method *n;

  fprintf(f_wrappers,"static PyMethodDef %sMethods[] = {\n", module);
  n = head;
  while (n) {
    fprintf(f_wrappers,"\t { \"%s\", %s, 1 },\n", n->name, n->function);
    n = n->next;
  }
  fprintf(f_wrappers,"\t { NULL, NULL }\n");
  fprintf(f_wrappers,"};\n");
}

// ---------------------------------------------------------------------
// PYTHON::emit_ptr_args(ParmList *l)
//
// We need to produce some args for receiving SWIG pointers.
// ---------------------------------------------------------------------

void PYTHON::emit_ptr_args(ParmList *l) {

  int i;
  Parm *p;

  i = 0;
  p = l->get_first();
  while (p != 0) {

    // Check if this parameter is a pointer.  If not, we'll get values

    if (p->t->is_pointer) {
      if ((p->t->type != T_CHAR) || (p->t->is_pointer > 1)) 
	fprintf(f_wrappers,"\t char *_argc%d;\n", i);
    }
    p = l->get_next();
    i++;
  }
}

// ---------------------------------------------------------------------
// PYTHON::headers(void)
//
// ----------------------------------------------------------------------

void PYTHON::headers(void)
{

  emit_banner(f_header);

  fprintf(f_header,"/* Implementation : PYTHON 1.3 */\n\n");

  fprintf(f_header,"#include <string.h>\n");
  fprintf(f_header,"#include <stdlib.h>\n");
  
  // Include configuration file with variable linkage.
  
  if (IncludeFile("python.swg", f_header) == -1) {
    fprintf(stderr,"SWIG : Fatal error. Unable to locate python.swg. (Possible installation problem).\n");
    SWIG_exit(1);
  }
  
  // I may implement SWIG pointers as a new Python type, but for
  // now I'll keep them as strings.

  emit_hex(f_header);

}

// --------------------------------------------------------------------
// PYTHON::initialize(void)
//
// This function outputs the starting code for a function to initialize
// your interface.   It is only called once by the parser.
//
// ---------------------------------------------------------------------

void PYTHON::initialize(void)
{

  int i;
  char  filen[256];
  char  *temp;
  char  *oldmodule;
  if (!module)
    module = "swig";

  // If shadow classing is enabled, we're going to change the module
  // name to "modulec"

  if (shadow) {
    temp = new char[strlen(module)+2];
    sprintf(temp,"%sc",module);
    oldmodule = module;
    module = temp;
  } 
  
  fprintf(f_header,"#define SWIG_init    init%s\n\n", module);
  fprintf(f_header,"#define SWIG_name    \"%s\"\n", module);	

  // Output the start of the init function.  
  // Modify this to use the proper return type and arguments used
  // by the target Language

  fprintf(f_init,"static PyObject *SWIG_globals;\n");

  if (CPlusPlus) 
      fprintf(f_init,"extern \"C\" ");

  fprintf(f_init,"void init%s() {\n",module);
  fprintf(f_init,"\t PyObject *m, *d;\n");

  if (InitNames) {
    i = 0;
    while (InitNames[i]) {
      fprintf(f_init,"\t %s();\n", InitNames[i]);
      i++;
    }
  }
  fprintf(f_init,"\t SWIG_globals = swig_newvarlink();\n");
  fprintf(f_init,"\t m = Py_InitModule(\"%s\", %sMethods);\n", module, module);
  fprintf(f_init,"\t d = PyModule_GetDict(m);\n");

  /* Create a shadow file (if enabled).*/
  if (shadow) {
    sprintf(filen,"%s.py", oldmodule);
    if ((f_shadow = fopen(filen,"w")) == 0) {
      fprintf(stderr,"Unable to open %s\n", filen);
      SWIG_exit(0);
    }
    fprintf(f_shadow,"# This file was created automatically by SWIG.\n");
    fprintf(f_shadow,"import %s\n", module);
  }
}

// ---------------------------------------------------------------------
// PYTHON::close(void)
//
// Called when the end of the interface file is reached.  Closes the
// initialization function and performs cleanup as necessary.
// ---------------------------------------------------------------------

void PYTHON::close(void)
{

  print_methods();

  emit_ptr_equivalence("_swig_ptr_python", f_wrappers);
  fprintf(f_init,"\t swig_ptr_derived = _swig_ptr_python;\n");
  fprintf(f_init,"}\n");

  if (shadow) {
    fprintf(f_shadow,"\n\n#-------------- FUNCTION WRAPPERS ------------------\n\n");
    fprintf(f_shadow,"%s",func.get());
    fprintf(f_shadow,"\n\n#-------------- VARIABLE WRAPPERS ------------------\n\n");
    fprintf(f_shadow,"%s",vars.get());
    fclose(f_shadow);
  }
}

// ----------------------------------------------------------------------
// PYTHON::get_pointer(char *iname, char *srcname, char *src, char *target,
//                     DataType *t, WrapperFunction &f, char *ret)
//
// Emits code to get a pointer and do type checking.
//      iname = name of the function/method  (used for error messages)
//      srcname = Name of source (used for error message reporting).
//      src   = name of variable where source string is located.
//      dest  = name of variable where pointer value is stored.
//      t     = Expected datatype of the parameter
//      f     = Wrapper function object being used to generate code.
//      ret   = return code upon failure.
//
// Note : pointers are stored as strings so you first need to get
// a string and then call _swig_get_hex() to extract a point.
//
// This module is pretty ugly, but type checking is kind of nasty
// anyways.
// ----------------------------------------------------------------------

void
PYTHON::get_pointer(char *iname, char *srcname, char *src, char *dest,
		    DataType *t, String &f, char *ret)
{

  // Now get the pointer value from the string and save in dest
  
  f << tab4 << "if (_swig_get_hex(" << src << ",(void **) &" << dest << ",";

  // If we're passing a void pointer, we give the pointer conversion a NULL
  // pointer, otherwise pass in the expected type.
  
  if (t->type == T_VOID) f << "(char *) 0 )) {\n";
  else
    f << "\"" << t->print_mangle() << "\")) {\n";

  // This part handles the type checking according to three different
  // levels.   0 = no checking, 1 = warning message, 2 = strict.

  switch(TypeStrict) {
  case 0: // No type checking
    f << tab4 << "}\n";
    break;

  case 1: // Warning message only

    // Change this part to how you want to handle a type-mismatch warning.
    // By default, it will just print to stderr.

    f << tab8 << "fprintf(stderr,\"Warning : type mismatch in " << srcname
	   << " of " << iname << ". Expected " << t->print_mangle()
	   << ", received %s\\n\"," << src << ");\n"
	   << tab4 << "}\n";

    break;
  case 2: // Super strict mode.

    // Change this part to return an error.

    f << tab8 << "PyErr_SetString(PyExc_TypeError,\"Type error in " << srcname
	   << " of " << iname << ". Expected " << t->print_mangle() << ".\");\n"
	   << tab8 << "return " << ret << ";\n"
	   << tab4 << "}\n";
    break;
    
  default :
    fprintf(stderr,"SWIG Error. Unknown strictness level\n");
    break;
  }
}

// ----------------------------------------------------------------------
// PYTHON::create_function(char *name, char *iname, DataType *d,
//                             ParmList *l)
//
// This function creates a wrapper function and registers it with the
// interpreter.   
//  
// Inputs :
//     name  = actual name of the function that's being wrapped
//    iname  = name of the function in the interpreter (may be different)
//        d  = Return datatype of the functions.
//        l  = A linked list containing function parameter information.
//
// ----------------------------------------------------------------------

void PYTHON::create_function(char *name, char *iname, DataType *d, ParmList *l)
{
  Parm    *p;
  int     pcount,i;
  char    wname[256];
  char    src[64], target[64], temp[256];
  char    *usage = 0;
  WrapperFunction f;
  String   parse_args;
  String   arglist;
  String   get_pointers;
  
  // Make a valid name for this function.   This removes special symbols
  // that would cause problems in the C compiler.

  strcpy(wname,iname);
  make_wrap_name(wname);
  
  // Now emit the function declaration for the wrapper function.  You
  // should modify this to return the appropriate types and use the
  // appropriate parameters.

  f.def << "static PyObject *_wrap_" << wname
	<< "(PyObject *self, PyObject *args) {";
  
  // Get the function usage string for later use
  
  usage_func(iname,d,l,&usage);   

  // Write code to extract function parameters.
  // This is done in one pass, but we need to construct three independent
  // pieces.
  //      1.    Python format string such as "iis"
  //      2.    The actual arguments to put values into
  //      3.    Pointer conversion code.
  //
  // we can do this in one pass by building up some strings.

  pcount = emit_args(d, l, f);
  parse_args << tab4 << "if(!PyArg_ParseTuple(args,\"";
  i = 0;
  p = l->get_first();
  while (p != 0) {

    // Check if this parameter is a pointer.  If not, we'll get values

    if (!p->t->is_pointer) {
      // Extract a parameter by "value"
      
      switch(p->t->type) {

	// Handle integers here.  Usually this can be done as a single
	// case if you appropriate cast things.   However, if you have
	// special cases, you'll need to add more code.  

      case T_INT : case T_UINT: case T_SINT:
	parse_args << "i";
	break;
      case T_SHORT: case T_USHORT: case T_SSHORT:
	parse_args << "h";
	break;
      case T_LONG : case T_ULONG: case T_SLONG :
	parse_args << "l";
	break;
      case T_SCHAR : case T_UCHAR :
	parse_args << "b";
	break;
      case T_CHAR:
	parse_args << "c";
	break;
      case T_FLOAT :
	parse_args << "f";
	break;
      case T_DOUBLE:
	parse_args << "d";
	break;

	// Void.. Do nothing.

      case T_VOID :
	break;

	// User defined.   This is usually invalid.   No way to pass a
	// complex type by "value".  We'll just pass into the unsupported
	// datatype case.

      case T_USER:

	// Unsupported data type

      default :
	fprintf(stderr,"%s : Line %d. Unable to use type %s as a function argument.\n",input_file, line_number, p->t->print_type());
	break;
      }
      
      // Emit code for parameter list
      
      if (p->t->type != T_VOID)
	arglist << "&_arg" << i;

    } else {

      // Is some other kind of variable.   
      
      parse_args << "s";

      if ((p->t->type == T_CHAR) && (p->t->is_pointer == 1)) {
	arglist << "&_arg" << i;
      } else {

	// Have some sort of pointer variable.  Create a temporary local
	// variable for the string and read the pointer value into it.

	sprintf(src,"_argc%d", i);
	sprintf(target,"_arg%d", i);
	sprintf(temp,"argument %d",i+1);
	
	f.add_local("char *", src);
	arglist << "&" << src;
	get_pointer(iname, temp, src, target, p->t, get_pointers, "NULL");
      }
    }
    p = l->get_next();
    if (p) 
      arglist << ",";
    i++;
  }

  if (pcount > 0)
    parse_args << "\",";    // Have arguments to parse
  else
    parse_args << "\"";     // No additional arguments

  parse_args << arglist << ")) \n"
	     << tab8 << "return NULL;\n";
  
  /* Now slap the whole first part of the wrapper function together */

  f.code << parse_args << get_pointers;

  // This function emits code to call the real function.  Assuming you read
  // the parameters in correctly, this will work.

  emit_func_call(name,d,l,f);

  // Now emit code to return the functions return value (if any).
  // If there was a result, it was saved in _result.
  // If the function is a void type, don't do anything.

  if ((d->type != T_VOID) || (d->is_pointer)) {
    // Now have return value, figure out what to do with it.

    if (!d->is_pointer) {

      // Function returns a "value"

      switch(d->type) {

	// Return an integer type
	
      case T_INT: case T_SINT: case T_UINT:
	f.code << tab4 << "return Py_BuildValue(\"i\",_result);\n";
	break;
      case T_SHORT: case T_SSHORT: case T_USHORT:
	f.code << tab4 << "return Py_BuildValue(\"h\",_result);\n";
	break;
      case T_LONG : case T_SLONG : case T_ULONG:
	f.code << tab4 << "return Py_BuildValue(\"l\",_result);\n";
	break;
      case T_SCHAR: case T_UCHAR :
	f.code << tab4 << "return Py_BuildValue(\"b\",_result);\n";
	break;

	// Return a floating point value

      case T_DOUBLE :
	f.code << tab4 << "return Py_BuildValue(\"d\",_result);\n";
	break;
      case T_FLOAT :
	f.code << tab4 << "return Py_BuildValue(\"f\",_result);\n";
	break;

	// Return a single ASCII value.  Usually we need to convert
	// it to a NULL-terminate string and return that instead.

      case T_CHAR :
	f.code << tab4 << "return Py_BuildValue(\"c\",_result);\n";
	break;

      case T_USER :

	// Return something by value
	// We're living dangerously here, but life is short...play hard

	// Oops.  Need another local variable
	f.add_local("char","_ptemp[128]");
	
	d->is_pointer++;
	f.code << tab4 << "_swig_make_hex(_ptemp, (void *) _result,\""
	  << d->print_mangle() << "\");\n";
	d->is_pointer--;
	// Return a character string containing our pointer.

	f.code << tab4 << "return Py_BuildValue(\"s\",_ptemp);\n";
	break;
      default :
	fprintf(stderr,"%s: Line %d. Unable to use return type %s in function %s.\n", input_file, line_number, d->print_type(), name);
	break;
      }
    } else {

      // Return type is a pointer.   We'll see if it's a char * and return
      // a string. Otherwise, we'll convert it into a SWIG pointer and return
      // that.

      if ((d->type == T_CHAR) && (d->is_pointer == 1)) {
	
	// Return a character string
	f.code << tab4 << "return Py_BuildValue(\"s\", _result);\n";

      } else {

	// Build a SWIG pointer.
	f.add_local("char","_ptemp[128]");

	f.code << tab4 << "_swig_make_hex(_ptemp, (char *) _result,\""
	  << d->print_mangle() << "\");\n";

	// Return a character string containing our pointer.
	f.code << tab4 << "return Py_BuildValue(\"s\",_ptemp);\n";
      }
    }
  } else {
    // no return value
    f.code << tab4 << "Py_INCREF(Py_None);\n";
    f.code << tab4 << "return Py_None;\n";
  }

  f.code << "}\n";

  // Dump this sucker out

  f.print(f_wrappers);

  // Now register the function with the interpreter.  

  sprintf(temp,"_wrap_%s", wname);
  add_method(iname, temp);

  delete usage;

  // ----------------------------------------------
  // Create a shadow for this function (if enabled)
  // ----------------------------------------------

  if (shadow) {

    func << "def " << iname << "(";
    p = l->get_first();
    i = 0;
    while(p) {
      func << "arg" << i;
      p = l->get_next();
      if (p)
	func << ",";
      i++;
    }
    func << "):\n";
    func << tab4;
    if ((d->type != T_VOID) || (d->is_pointer > 0))
      func << "val = ";

    if ((hash.lookup(d->name)) && (d->is_pointer <=1))
      func << (char *) hash.lookup(d->name) << "Ptr(";
    func << module << "." << iname << "(";
    p = l->get_first();
    i = 0;
    while(p) {
      func << "arg" << i;
      if ((hash.lookup(p->t->name)) && (p->t->is_pointer <= 1))
	func << ".this";
      p = l->get_next();
      if (p)
	func << ",";
      i++;
    }
    if ((hash.lookup(d->name)) && (d->is_pointer <=1))
      func << "))\n";
    else
      func << ")\n";
    if ((hash.lookup(d->name)) && (d->is_pointer < 1))
      func << tab4 << "val.thisown = 1\n";
    if ((d->type != T_VOID) || (d->is_pointer > 0))
      func << tab4 << "return val\n\n";
    else
      func << "\n";
  }
}

// -----------------------------------------------------------------------
// PYTHON::link_variable(char *name, char *iname, DataType *d)
//
// Input variables:
//     name = the real name of the variable being linked
//    iname = Name of the variable in the interpreter (may be different)
//        d = Datatype of the variable.
//
// This creates a pair of functions for evaluating/setting the value
// of a variable.   These are then added to the special SWIG global
// variable type.
// -----------------------------------------------------------------------

void PYTHON::link_variable(char *name, char *iname, DataType *t) {

    char   wname[256];
    static int have_globals = 0;

    WrapperFunction getf, setf;

    // If this is our first call, add the globals variable to the
    // Python dictionary.

    if (!have_globals) {
      fprintf(f_init,"\t PyDict_SetItemString(d,\"%s\", SWIG_globals);\n",global_name);
      have_globals=1;
      if (shadow) {
	vars << global_name << " = " << module << "." << global_name << "\n";
      }
    }
    // First make a sanitized version of the function name (in case it's some
    // funky C++ thing).
    
    strcpy(wname, name);
    make_wrap_name(wname);

    // ---------------------------------------------------------------------
    // Create a function for setting the value of the variable
    // ---------------------------------------------------------------------

    setf.def << "static int _wrap_" << wname << "_set(PyObject *val) {";
    if (!(Status & STAT_READONLY)) {
      if ((t->type != T_VOID) || (t->is_pointer)) {
	if (!t->is_pointer) {
	  
	  // Have a real value here 

	  switch(t->type) {
	  case T_INT: case T_SHORT: case T_LONG :
	  case T_UINT: case T_USHORT: case T_ULONG:
	  case T_SINT: case T_SSHORT: case T_SLONG:
	  case T_SCHAR: case T_UCHAR:
	    // Get an integer value
	    setf.add_local(t->print_type(), "tval");
	    setf.code << tab4 << "tval = " << t->print_cast() << "PyInt_AsLong(val);\n"
		      << tab4 << "if (PyErr_Occurred()) {\n"
		      << tab8 << "PyErr_SetString(PyExc_TypeError,\"C variable '"
		      << iname << "'(" << t->print_type() << ")\");\n"
		      << tab8 << "return 1; \n"
		      << tab4 << "}\n"
		      << tab4 << name << " = tval;\n";
	    break;

	  case T_FLOAT: case T_DOUBLE:
	    // Get a floating point value
	    setf.add_local(t->print_type(), "tval");
	    setf.code << tab4 << "tval = " << t->print_cast() << "PyFloat_AsDouble(val);\n"
		      << tab4 << "if (PyErr_Occurred()) {\n"
		      << tab8 << "PyErr_SetString(PyExc_TypeError,\"C variable '"
		      << iname << "'(" << t->print_type() << ")\");\n"
		      << tab8 << "return 1; \n"
		      << tab4 << "}\n"
		      << tab4 << name << " = tval;\n";
	    break;

	    // A single ascii character
	    
	  case T_CHAR:
	    setf.add_local("char *", "tval");
	    setf.code << tab4 << "tval = (char *) PyString_AsString(val);\n"
		      << tab4 << "if (PyErr_Occurred()) {\n"
		      << tab8 << "PyErr_SetString(PyExc_TypeError,\"C variable '"
		      << iname << "'(" << t->print_type() << ")\");\n"
		      << tab8 << "return 1; \n"
		      << tab4 << "}\n"
		      << tab4 << name << " = *tval;\n";
	    break;
	  case T_USER:
	    setf.add_local("char *", "tval");
	    t->is_pointer++;
	    setf.add_local(t->print_type(),"temp");
	    setf.code << tab4 << "tval = (char *) PyString_AsString(val);\n"
		      << tab4 << "if (PyErr_Occurred()) {\n"
		      << tab8 << "PyErr_SetString(PyExc_TypeError,\"C variable '"
		      << iname << "'(" << t->print_type() << ")\");\n"
		      << tab8 << "return 1; \n"
		      << tab4 << "}\n";

	    get_pointer(iname,"value","tval","temp",t,setf.code,"1");
	    setf.code << tab4 << name << " = *temp;\n";
	    t->is_pointer--;
	    break;
	  default:
	    fprintf(stderr,"%s : Line %d. Unable to link with type %s.\n", input_file, line_number, t->print_type());
	  }
	} else {

	  // Parse a pointer value

	  if ((t->type == T_CHAR) && (t->is_pointer == 1)) {
	    setf.add_local("char *", "tval");
	    setf.code << tab4 << "tval = (char *) PyString_AsString(val);\n"
		      << tab4 << "if (PyErr_Occurred()) {\n"
		      << tab8 << "PyErr_SetString(PyExc_TypeError,\"C variable '"
		      << iname << "'(" << t->print_type() << ")\");\n"
		      << tab8 << "return 1; \n"
		      << tab4 << "}\n";

	    if (CPlusPlus) {
	      setf.code << tab4 << "if (" << name << ") delete " << name << ";\n"
			<< tab4 << name << " = new char[strlen(tval)+1];\n"
			<< tab4 << "strcpy(" << name << ",tval);\n";
	    } else {
	      setf.code << tab4 << "if (" << name << ") free(" << name << ");\n"
			<< tab4 << name << " = (char *) malloc(strlen(tval)+1);\n"
			<< tab4 << "strcpy(" << name << ",tval);\n";
	    }
	  } else {
	    
	    // Is a generic pointer value.
	    
	    setf.add_local("char *", "tval");
	    setf.add_local(t->print_type(),"temp");
	    setf.code << tab4 << "tval = (char *) PyString_AsString(val);\n"
		      << tab4 << "if (PyErr_Occurred()) {\n"
		      << tab8 << "PyErr_SetString(PyExc_TypeError,\"C variable '"
		      << iname << "'(" << t->print_type() << ")\");\n"
		      << tab8 << "return 1; \n"
		      << tab4 << "}\n";

	    get_pointer(iname,"value","tval","temp",t,setf.code,"1");
	    setf.code << tab4 << name << " = temp;\n";
	  }
	}
      }
      setf.code << tab4 << "return 0;\n";
    } else {
      // Is a readonly variable.  Issue an error
      setf.code << tab4 << "PyErr_SetString(PyExc_TypeError,\"Variable " << iname
		<< " is read-only.\");\n"
		<< tab4 << "return 1;\n";
    }

    setf.code << "}\n";

    // Dump out function for setting value

    setf.print(f_wrappers);
    
    // ----------------------------------------------------------------
    // Create a function for getting the value of a variable
    // ----------------------------------------------------------------

    getf.def << "static PyObject *_wrap_" << wname << "_get() {";
    if ((t->type != T_VOID) || (t->is_pointer)) {
      if (!t->is_pointer) {

	/* Is a normal datatype */
	switch(t->type) {
	case T_INT: case T_SINT: case T_UINT:
	case T_SHORT: case T_SSHORT: case T_USHORT:
	case T_LONG: case T_SLONG: case T_ULONG:
	case T_SCHAR: case T_UCHAR:
	  getf.code << tab4 << "return PyInt_FromLong((long) " << name << ");\n";
	  break;
	case T_FLOAT: case T_DOUBLE:
	  getf.code << tab4 << "return PyFloat_FromDouble((double) " << name << ");\n";
	  break;
	case T_CHAR:
	  getf.add_local("char","ptemp[2]");
	  getf.code << tab4 << "ptemp[0] = " << name << ";\n"
		    << tab4 << "ptemp[1] = 0;\n"
		    << tab4 << "return PyString_FromString(ptemp);\n";
	  break;
	case T_USER:
	  // Hack this into a pointer
	  getf.add_local("char", "ptemp[128]");
	  t->is_pointer++;
	  getf.code << tab4 << "_swig_make_hex(ptemp,(char *) &" << name
		    << "," << quote << t->print_mangle() << quote << ");\n"
		    << tab4 << "return PyString_FromString(ptemp);\n";
	  t->is_pointer--;
	  break;
	default:
	  fprintf(stderr,"Unable to link with type %s\n", t->print_type());
	  break;
	}
      } else {

	// Is some sort of pointer value
	if ((t->type == T_CHAR) && (t->is_pointer == 1)) {
	  getf.code << tab4 << "if (" << name << ")\n"
		    << tab8 << "return PyString_FromString(" << name << ");\n"
		    << tab4 << "else return PyString_FromString(\"(NULL)\");\n";
	} else {
	  getf.add_local("char","ptemp[128]");
	  getf.code << tab4 << "_swig_make_hex(ptemp, (char *) " << name << ",\""
		    << t->print_mangle() << "\");\n"
		    << tab4 << "return PyString_FromString(ptemp);\n";
	}
      }
    }
    getf.code << "}\n";

    getf.print(f_wrappers);

    // Now add this to the variable linking mechanism

    fprintf(f_init,"\t swig_addvarlink((swig_varlinkobject *) SWIG_globals,\"%s\",_wrap_%s_get, _wrap_%s_set);\n", iname, wname, wname);

    // ----------------------------------------------------------
    // Output a shadow variable.  (If applicable and possible)
    // ----------------------------------------------------------
    if (shadow) {
      if ((hash.lookup(t->name)) && (t->is_pointer < 1)) {
	vars << iname << " = " << (char *) hash.lookup(t->name) << "Ptr(" << module << "." << global_name
	     << "." << iname << ")\n";
      }
    }
}

// -----------------------------------------------------------------------
// PYTHON::declare_const(char *name, DataType *type, char *value)
//
// Makes a constant as defined with #define.  Constants are added to the
// module's dictionary and are **NOT** guaranteed to be read-only,
// sorry.
//
// ------------------------------------------------------------------------

void PYTHON::declare_const(char *name, DataType *type, char *value) {

  char   var_name[256];

  // Make a static python object

  sprintf(var_name,"_wrap_python_const_%s", name);
  fprintf(f_header,"static PyObject *%s;\n",var_name);

  if ((type->type == T_USER) && (!type->is_pointer)) {
    fprintf(stderr,"%s : Line %d.  Unsupported constant value.\n", input_file, line_number);
    return;
  }
  
  if (type->is_pointer == 0) {
    switch(type->type) {
    case T_INT:case T_SINT: case T_UINT:
    case T_SHORT: case T_SSHORT: case T_USHORT:
    case T_LONG: case T_SLONG: case T_ULONG:
    case T_SCHAR: case T_UCHAR:
      fprintf(f_init,"\t %s = PyInt_FromLong(%s);\n", var_name,value);
      break;
    case T_DOUBLE:
      fprintf(f_init,"\t %s = PyFloat_FromDouble(%s);\n", var_name,value);
      break;
    case T_CHAR :
      fprintf(f_init,"\t %s = PyString_FromString(\"%s\");\n", var_name, value);
      break;
    default:
      fprintf(stderr,"%s : Line %d. Unsupported constant value.\n", input_file, line_number);
      break;
    }
  } else {
    if ((type->type == T_CHAR) && (type->is_pointer == 1)) {
      fprintf(f_init,"\t %s = PyString_FromString(\"%s\");\n", var_name, value);
    } else {
      // A funky user-defined type.  We'll really create it as a string pointer value
      fprintf(f_header,"static %s %s_value = %s;\n", type->print_type(), var_name, value);
      fprintf(f_header,"static char %s_char[%d];\n", var_name, strlen(type->print_mangle()) + 20);
      fprintf(f_init,"\t _swig_make_hex(%s_char, (void *) %s_value,\"%s\");\n",
	      var_name, var_name, type->print_mangle());
      fprintf(f_init,"\t %s = PyString_FromString(%s_char);\n", var_name, var_name);
    }
  }
    
  fprintf(f_init,"\t PyDict_SetItemString(d,\"%s\", %s);\n", name, var_name);

  if (shadow) {
    vars << name << " = " << module << "." << name << "\n";
  }    

}

// ----------------------------------------------------------------------
// PYTHON::usage_var(char *iname, DataType *t, char **s)
//
// This function produces a string indicating how to use a variable.
// It is called by the documentation system to produce syntactically
// correct documentation entires.
//
// s is a pointer to a character pointer.   You should create
// a string and set this pointer to point to it.
// ----------------------------------------------------------------------

void PYTHON::usage_var(char *iname, DataType *t, char **s) {

  char temp[1024], *c;

  sprintf(temp,"%s.%s_get(), %s.%s_set(%s)", module, iname, module,iname,t->print_type());
  c = temp + strlen(temp);

  // This appends a message to unsupported datatype.

  if (!((t->type != T_USER) || (t->is_pointer))) {
    sprintf(c," - unsupported");
  }

  // Create result.  Don't modify this

  if (*s == 0) 
    *s = new char[strlen(temp)+1];
  strcpy(*s,temp);

}

// ---------------------------------------------------------------------------
// PYTHON::usage_func(char *iname, DataType *t, ParmList *l, char **s)
// 
// Produces a string indicating how to call a function in the target
// PYTHONuage.  Called by the documentation system.
//
// ---------------------------------------------------------------------------

void PYTHON::usage_func(char *iname, DataType *t, ParmList *l, char **s) {

  char temp[1024];
  char *c;
  int  i;
  Parm  *p;

  sprintf(temp,"%s : %s.%s(",t->print_type(), module, iname);
  c = temp + strlen(temp);
  
  // Now go through and print parameters 
  // You probably don't need to change this

  p = l->get_first();
  while (p != 0) {
    
    /* If parameter has been named, use that.   Otherwise, just print a type  */

    if ((p->t->type != T_VOID) || (p->t->is_pointer)) {
      if (strlen(p->name) > 0) {
	sprintf(c,"%s ",p->name);
	c += strlen(p->name) + 1;
      } else {
	sprintf(c,"%s",p->t->name);
	c += strlen(p->t->name);
	for (i = 0; i < (p->t->is_pointer-p->t->implicit_ptr); i++) {
	  sprintf(c,"*");
	  c++;
	}
      }
    }
    p = l->get_next();
    if (p != 0) {
      sprintf(c,",");
      c++;
    }
  }
  sprintf(c,");");

  // Create result. Don't change this

  if (*s == 0) 
    *s = new char[strlen(temp)+1];
  strcpy(*s,temp);

}


// ----------------------------------------------------------------------
// PYTHON::usage_const(char *iname, DataType *type, char *value, char **s)
//
// Produces a string for a constant.   Really about the same as
// usage_var() except we'll indicate the value of the constant.
// ----------------------------------------------------------------------

void PYTHON::usage_const(char *iname, DataType *, char *value, char **s) {

  char temp[1024];

  sprintf(temp,"%s.%s = %s", module, iname, value);

  if (*s == 0) 
    *s = new char[strlen(temp)+1];
  strcpy(*s,temp);

}

// -----------------------------------------------------------------------
// PYTHON::add_native(char *name, char *funcname)
//
// Add a native module name to the methods list.
// -----------------------------------------------------------------------

void PYTHON::add_native(char *name, char *funcname) {
  add_method(name, funcname);
}
