
/**********************************************************************
 * Simplified Wrapper and Interface Generator  (SWIG)
 * 
 * Dave Beazley
 * 
 * Theoretical Division (T-11)           Department of Computer Science
 * Los Alamos National Laboratory        University of Utah
 * Los Alamos, New Mexico  87545         Salt Lake City, Utah  84112
 * beazley@lanl.gov                      beazley@cs.utah.edu
 *
 * Copyright (c) 1995-1996
 * The Regents of the University of California and the University of Utah
 * All Rights Reserved
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that 
 * (1) The above copyright notice and the following two paragraphs
 * appear in all copies of the source code and (2) redistributions
 * including binaries reproduces these notices in the supporting
 * documentation.   Substantial modifications to this software may be
 * copyrighted by their authors and need not follow the licensing terms
 * described here, provided that the new terms are clearly indicated in
 * all files where they apply.
 * 
 * IN NO EVENT SHALL THE AUTHOR, THE UNIVERSITY OF CALIFORNIA, THE 
 * UNIVERSITY OF UTAH OR DISTRIBUTORS OF THIS SOFTWARE BE LIABLE TO ANY
 * PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
 * DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION,
 * EVEN IF THE AUTHORS OR ANY OF THE ABOVE PARTIES HAVE BEEN ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE AUTHOR, THE UNIVERSITY OF CALIFORNIA, AND THE UNIVERSITY OF UTAH
 * SPECIFICALLY DISCLAIM ANY WARRANTIES,INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND 
 * THE AUTHORS AND DISTRIBUTORS HAVE NO OBLIGATION TO PROVIDE MAINTENANCE,
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 * The author requests that all users of this software return any
 * improvements made to beazley@cs.utah.edu and grant the author
 * full redistribution rights.
 *
 **************************************************************************/

#include "internal.h"

// -----------------------------------------------------------------------
// $Header: /b11/dmb/SWIG/SWIG1.0/SWIG/RCS/cplus.cxx,v 1.23 1996/08/27 22:58:26 dmb Exp $
// Super Experimental C++ module
//
// This is the super-experimental C++ module.  All functions here are
// undocumented and under development.   Eventually this might turn
// into a very simple C++ class parser.   Of course, this might not ever show
// up if it becomes too complicated.
//
// General comments :
//
// 1.  The words "simple" and "C++" are rarely used in the same
//     sentence.   Therefore this module is going to be some sort
//     of compromise.
//
// 2.  I'm using the "Annotated C++ Reference Manual" (ARM) as my
//     reference for handling certain cases.     Of course, there
//     is a high probability that I have misinterpreted or overlooked
//     certain cases.
//
// 3.  It is not my intent to write a full C++ compiler.
//
//     My goals are simple :
//           -  Support simple ANSI C-like class member functions and data.
//           -  Support constructors and destructors.
//           -  static member functions.
//           -  basic inheritance.
//           -  virtual functions.
//           -  References
//
//     I do not plan to support the following anytime in the near future 
//           -  Operator overloading
//           -  Polymorphism
//           -  templates
//
// $Log: cplus.cxx,v $
// Revision 1.23  1996/08/27 22:58:26  dmb
// Slight change to addmethods handling
//
// Revision 1.22  1996/08/26 23:34:56  dmb
// Added support for user-definable methods
//
// Revision 1.21  1996/08/25 00:03:06  dmb
// Added parameter to cpp_inherit() to allow language modules to explicitly
// disable the inclusion of functions in base classes
//
// Revision 1.20  1996/08/23 18:49:19  dmb
// Fixed bug in handling of object attributes of type "void *"
//
// Revision 1.19  1996/08/21 16:48:40  dmb
// Minor cleanup to eliminate compiler warnings.
//
// Revision 1.18  1996/08/21 05:46:51  dmb
// A major overhaul of some internals.    It's not perfect, but solves
// a bunch of problems related to C structures and processing
// unnamed structures.
//
// Revision 1.17  1996/08/20 19:34:21  dmb
// Fixes to the handling of C++ references.
//
// Revision 1.16  1996/08/12 01:47:15  dmb
// Some cleanup.   Fixes to support for C++ references.
//
// Revision 1.15  1996/06/18 16:44:26  dmb
// Fixed declaration bugs
//
// Revision 1.14  1996/06/15 23:41:06  beazley
// Fixed bug in verbose mode.
//
// Revision 1.13  1996/06/02  00:13:53  beazley
// Fixed a few bugs with references
//
// Revision 1.12  1996/05/22  20:20:21  beazley
// Modified to use new type-equivalence tables in types.cxx
//
// Revision 1.11  1996/05/20  23:34:52  beazley
// Split up functions into parser functions and default code generation methods
//
// Revision 1.10  1996/05/17  19:54:41  beazley
// Minor changes
//
// Revision 1.9  1996/05/17  05:52:30  beazley
// Fixed up the assignment of the derived pointer table.
//
// Revision 1.8  1996/05/14  23:22:48  beazley
// Got inheritance support added.  Of course, who knows how
// "risky" it is.....
//
// Revision 1.7  1996/05/13  23:43:10  beazley
// A few more minor fixes.  Still need to add documentation
// generation.
//
// Revision 1.6  1996/05/10  23:38:37  beazley
// Added enum support.
//
// Revision 1.5  1996/05/09  23:27:54  beazley
// Fixed bug in variable linking.
//
// Revision 1.3  1996/05/07  14:41:43  beazley
// Added support for references (maybe)
//
// Revision 1.2  1996/05/06  23:09:38  beazley
// Added support for member data (variables) and static functions.
//
// Revision 1.1  1996/05/03  22:28:07  dmb
// Initial revision
//
// -----------------------------------------------------------------------

static char  *ClassName;        // Current Class Name 
static char  *ClassRename;      // Name of class in target
static char  *ClassType;        // Type of class

static int    Have_Constructor = 0;
static int    Have_Destructor = 0;
static int    Have_Virtual = 0;
static int    Inherit_mode = 0;
static int    NewMethods = 0;

static char new_name[256];
static char new_iname[256];

extern int  Add_Symbol(char *, DataType *, char *);
extern int  Update_Symbol(char *, DataType *, char *);

void set_classname(char *name, char *rname);
void cplus_member_func(char *, char *, DataType *, ParmList *, int);
void cplus_constructor(char *, char *, ParmList *);
void cplus_destructor(char *, char *);
void cplus_variable(char *, char *, DataType *);
void cplus_static_func(char *, char *, DataType *, ParmList *);
void cplus_static_var(char *, char *, DataType *);
void cplus_declare_const(char *, char *, DataType *, char *);

// -----------------------------------------------------------------------
// class ClassDef
//
// This class maintains a table of all C++ classes wrapped so far.
// This is used so that we can properly handle public inheritance.
//
// For each class, we are going to save the following :
//
//       1.    Non-virtual member functions.
//       2.    Static member functions.
//       3.    Member data.
//       4.    Enumerations.
//
// Constructors, Destructors, and virtual functions are not saved.
// 
// For public inheritance, we'll look up a class in our table
// and create functions for all of the above.
// -------------------------------------------------------------------------

// -------------------------------------------------------------------------
// Struct for holding a function prototype
// -------------------------------------------------------------------------

struct CPP_function {
  int  is_static;
  char *name;
  char *iname;
  DataType *r_type;
  ParmList *parms;
  int   new_method;
  CPP_function(char *n, char *i, DataType *t, ParmList *l,  int s) {
    name = copy_string(n);
    iname = copy_string(i);
    r_type = new DataType(t);
    parms = new ParmList(l);
    is_static = s;
    new_method = NewMethods;
  }
  void inherit() {
    if (!new_method) {
      if (is_static) {
	cplus_static_func(name, iname, r_type, parms);
      } else {
	cplus_member_func(name, iname, r_type, parms,0);
      }
    }
  }
  void emit() {
    NewMethods = new_method;
    if (is_static) {
      lang->cpp_static_func(name, iname, r_type, parms);
    } else {
      lang->cpp_member_func(name, iname, r_type, parms);
    }
  }
  void print() {
    fprintf(stderr,"   Function : ");
    emit_extern_func(name, r_type, parms, 0, stderr);
  }
};

// --------------------------------------------------------------------------
// Struct for holding a constructor definition
// --------------------------------------------------------------------------

struct CPP_constructor {
  char *name;
  char *iname;
  ParmList *parms;
  int  unnamed;
  int  new_method;
  CPP_constructor(char *n, char *i, ParmList *l, int u) {
    name = copy_string(n);
    iname = copy_string(i);
    parms = new ParmList(l);
    unnamed = u;
    new_method = NewMethods;
  }
  void emit() {

    // If it was unnamed, make sure it matches up with the classname now
    
    if (unnamed) {
      if (strcmp(name,ClassName) != 0) {
	fprintf(stderr,"%s : Line %d.  Constructor %s doesn't match class name of %s.\n", input_file, line_number, name, ClassName);
	return;
      }
    }
    NewMethods = new_method;
    lang->cpp_constructor(ClassName,iname,parms);
  }
  void inherit() {
  }
  void print() {
    fprintf(stderr,"   Constructor : %s\n", name);
  }
  
};


// --------------------------------------------------------------------------
// Struct for holding a destructor definition
// --------------------------------------------------------------------------

struct CPP_destructor {
  char *name;
  char *iname;
  int   unnamed;
  int   new_method;
  CPP_destructor(char *n, char *i, int u) {
    name = copy_string(n);
    iname = copy_string(i);
    unnamed = u;
    new_method = NewMethods;
  }
  void emit() {
    if ((unnamed) && (strcmp(ClassName,name) != 0)) {
      fprintf(stderr,"%s : Line %d. Destructor %s doesn't match class name of %s\n", input_file, line_number, name, ClassName);
      return;
    }
    NewMethods = new_method;
    lang->cpp_destructor(ClassName, (char *) 0 );
  }
  void inherit() {
  }
  void print() {
    fprintf(stderr,"   Destructor : %s\n", name);
  }
};


// -------------------------------------------------------------------------
// Struct for holding a variable declaration
// -------------------------------------------------------------------------

struct CPP_variable {
  int  is_static;
  char *name;
  char *iname;
  DataType *type;
  CPP_variable(char *n, char *i, DataType *t, int s) {
    name = copy_string(n);
    iname = copy_string(i);
    type = new DataType(t);
    is_static = s;
  }
  void emit() {
    if (!is_static)
      lang->cpp_variable(name,iname,type);
    else
      lang->cpp_static_var(name,iname,type);
  }
  void inherit() {
    if (!is_static)
      cplus_variable(name,iname,type);
    else
      cplus_static_var(name,iname,type);
  }
  void print() {
    fprintf(stderr,"   Variable : ");
    emit_extern_var(name, type, 0,stderr);
  }

};

// -------------------------------------------------------------------------
// Struct for holding a constant/enum value
// -------------------------------------------------------------------------

struct CPP_constant {
  char  *name;
  char  *iname;
  char  *value;
  DataType *type;
  CPP_constant(char *n, char *i, DataType *t, char *v) {
    name = copy_string(n);
    iname = copy_string(i);
    type = new DataType(t);
    value = copy_string(v);
  }
  void emit() {
    lang->cpp_declare_const(name,iname,type,value);
  }
  void inherit() {
    cplus_declare_const(name, iname, type, value);
  }
  void print() {
    if (value) 
      fprintf(stderr,"   Constant : %s = %s\n", name, value);
    else
      fprintf(stderr,"   Constant : %s\n", name);
  }
};

#define  CPP_func      1
#define  CPP_var       2
#define  CPP_enum      3
#define  CPP_new       4
#define  CPP_delete    5

// -------------------------------------------------------------------------
// We'll use this to hold one of the above.  Kind of a kludge
// -------------------------------------------------------------------------

struct CPP_declaration {
  int decl_type;
  CPP_function *cpp_func;
  CPP_variable *cpp_var;
  CPP_constant *cpp_const;
  CPP_constructor *cpp_construct;
  CPP_destructor  *cpp_destruct;
}
;

// -------------------------------------------------------------------------
// Class for holding functions, variables, and constants in each
// class.
// -------------------------------------------------------------------------

class CPP_class {
private:
  struct decl_node {
    CPP_declaration *cpp_decl;
    decl_node *next;
  };
  decl_node  *head;
  decl_node  *z;
  struct type_node {
    char      *name;
    type_node *next;
  };
public:
  char      *classname;
  char      *classtype;	
  CPP_class *baseclass;
  type_node  *derived_list;
  CPP_class(char *name, char *ctype) {
    head = new decl_node;
    z = new decl_node;
    head->next = z;
    z->next = z;
    classname = copy_string(name);
    classtype = copy_string(ctype);
    baseclass = (CPP_class *) 0;
    derived_list = (type_node *) 0;
  };

  void add(CPP_declaration *d) {
    decl_node *n;
    n = new decl_node;
    n->cpp_decl = d;
    n->next = head->next;
    head->next = n;
  };
  void add_derived(char *cname) {
    type_node *t;
    t = new type_node;
    t->name = copy_string(cname);
    t->next = derived_list;
    derived_list = t;
  };

  // Print out a comma separated list of derived classes (in quotes)

  void print_derived(FILE *f) {
    type_node *t;
    t = derived_list;
    while (t) {
      fprintf(f,"\"%s\",", t->name);
      t = t->next;
    }
    fprintf(f,"0");
  };

  // Search for a member with the given name. Returns a 1 on success, 0 on failure

  int search(char *name) {
    decl_node *n;
    CPP_declaration *d;
    char *c;
    n = head->next;
    while (n != z) {
      d = n->cpp_decl;
      switch(d->decl_type) {
      case CPP_func:
	c = d->cpp_func->iname ? d->cpp_func->iname : d->cpp_func->name;
	break;
      case CPP_var:
	c = d->cpp_var->iname ? d->cpp_var->iname : d->cpp_var->name;
	break;
      case CPP_enum:
	c = d->cpp_const->iname ? d->cpp_const->iname : d->cpp_const->name;
	break;
      case CPP_new:
	c = d->cpp_construct->iname ? d->cpp_construct->iname : d->cpp_construct->name;
	break;
      case CPP_delete:
	c = d->cpp_destruct->iname ? d->cpp_destruct->iname : d->cpp_destruct->name;
	break;
      default:
	c = "";
	break;
      }
      if (strcmp(c,name) == 0) return 1;
      n = n->next;
    }
    return 0;
  }

  // Inherit.  Put all the declarations associate with this class into the current
  // context.

  void inherit_decls() {
    decl_node *n;
    CPP_declaration *d;
    n = head->next;
    while (n != z) {
      d = n->cpp_decl;
      switch(d->decl_type) {
      case CPP_func:
	d->cpp_func->inherit();
	break;
      case CPP_new:
	d->cpp_construct->inherit();
	break;
      case CPP_delete:
	d->cpp_destruct->inherit();
	break;
      case CPP_var:
	d->cpp_var->inherit();
	break;
      case CPP_enum:
	d->cpp_const->inherit();
	break;
      default:
	break;
      }
      n = n->next;
    }
  }
  // Emit all of the declarations associated with this class into the current
  // class (ie. inherit)

  void emit_decls() {
    decl_node *n;
    CPP_declaration *d;
    n = head->next;
    while (n != z) {
      d = n->cpp_decl;
      switch(d->decl_type) {
      case CPP_func:
	d->cpp_func->emit();
	break;
      case CPP_new:
	d->cpp_construct->emit();
	break;
      case CPP_delete:
	d->cpp_destruct->emit();
	break;
      case CPP_var:
	d->cpp_var->emit();
	break;
      case CPP_enum:
	d->cpp_const->emit();
	break;
      default:
	break;
      }
      n = n->next;
    }
  }
  
  void print() {
    decl_node *n;
    CPP_declaration *d;
    n = head->next;
    while (n != z) {
      d = n->cpp_decl;
      switch(d->decl_type) {
      case CPP_func:
	d->cpp_func->print();
	break;
      case CPP_new:
	d->cpp_construct->print();
	break;
      case CPP_delete:
	d->cpp_destruct->print();
	break;
      case CPP_var:
	d->cpp_var->print();
	break;
      case CPP_enum:
	d->cpp_const->print();
	break;
      default:
	break;
      }
      n = n->next;
    }
  }
};

// -------------------------------------------------------------------------
// Class for maintaining lists of all C++ classes seen so far.
// -------------------------------------------------------------------------

class CPP_classlist {
private:
  struct class_node {
    CPP_class *cpp_class;
    class_node *next;
  };
  class_node *head, *z;
public:
  CPP_classlist() {
    head = new class_node;
    z = new class_node;
    head->next = z;
    z->next = z;
  };
  void add(CPP_class *c) {
    class_node *n;
    n = new class_node;
    n->cpp_class = c;
    n->next = head->next;
    head->next = n;
  };
  CPP_class *search(char *classname) {
    class_node *n;
    n = head->next;
    while (n != z) {
      if (strcmp(n->cpp_class->classname, classname) == 0) return n->cpp_class;
      n = n->next;
    }
    return (CPP_class *) 0;
  };
  void print() {
    class_node *n;
    n = head->next;
    while (n != z) {
      n->cpp_class->print();
      n = n->next;
    }
  };
  
};

// Whew!   Hope you got all that above.....
// So here's the deal.
//
// CPP_classlist   --- holds a linked list of all classes we've
//                     created.
// CPP_class       --- An instance of each class.  This in turn
//                     has a linked list of all member data added.
// CPP_declaration --- A struct holding information about a particular declaration
//
// In order to handle inheritance, we'll need to search the CPP_classlist
// for a baseclass, and add all of it's methods to the current class.
//
// Need to make sure we don't do something stupid like add virtual
// classes however.


static CPP_classlist *classlist;
static CPP_class     *current_class;
static CPP_class     *base_class;

// ----------------------------------------------------------------------
// void set_classname(char *name, char *rname, char *ctype)
//
// Sets the current class or structure name
// If rname is provided, then we'll set up a class renaming scheme
// ----------------------------------------------------------------------

void set_classname(char *name, char *rname, char *ctype) {
  extern void TypeEq_Mangle(char *, char *);
  char temp[256];

  ClassName = name;
  if (rname) {
    ClassRename = copy_string(rname);
  } else {
    ClassRename = (char *) 0;
  }

  if (!CPlusPlus) {
    ClassType = new char[strlen(ctype)+2];
    sprintf(ClassType,"%s ", ctype);
  } else {
    ClassType = "";
  }
  
  /* Build up a new class */
  
  if (!classlist)
    classlist = new CPP_classlist;
  current_class = new CPP_class(name,ctype);

  // We only add our class to the list if it is "named".
  
  if (name)
    classlist->add(current_class);
  
  // Now call language specific function
  
  Have_Constructor = 0;
  Have_Destructor = 0;
  Have_Virtual = 0;
  NewMethods = 0;
  lang->cpp_open_class(name, ClassRename, ctype);
  
  // Make a typedef for both long and short versions of this datatype

  if (ClassName) {
    sprintf(temp,"%s %s", ctype, ClassName);   
    TypeEq_Mangle(temp,ClassName);
    TypeEq_Mangle(ClassName,temp);
  }
}

// -------------------------------------------------------------------------
// void cplus_class_close(char *name)
//
// Close a C++ class definition.   Up to this point, we've only been collecting
// member definitions.     This will dump them all out.
//
// If name is non-null, it means that the name of the class has actually been
// set after all of the definitions.  For example :
//
//    typedef struct {
//             double x,y,z
//    } Vector;
//
// If this is the case, we'll use that as our classname and datatype.
// Otherwise, we'll just go with the classname and classtype set earlier.
// ---------------------------------------------------------------------------

void cplus_class_close(char *name) {


  if (Verbose) {
    printf("closing class %s\n", ClassName);
    current_class->print();
  }
  
  lang->cpp_close_start(name); 

  if (Verbose) {
    if (name)
      printf("class renamed to %s\n", name);
  }
  if (name) {
    // Okay.  We have a late-named class.   Fix everything up //
    ClassName = copy_string(name);
    ClassType = "";                   /* Since this is a typedef, we'll just use that */
  }

  /* Now create all of the class wrappers */

  Inherit_mode = 0;
  current_class->emit_decls();
  lang->cpp_close_class();
  NewMethods = 0;

}

void cplus_cleanup(void) {

  //  if (classlist)
  //    classlist->equivalence(f_wrappers);
  // emit_ptr_equivalence("_swig_ptr_table", f_wrappers);

  lang->cpp_cleanup();
}

// ---------------------------------------------------------------------------
// void cplus_inherit_default(char *baseclass)
//
// Inherit all of the functions in the given baseclass.  If class doesn't
// exist, issue a warning.   Probably pretty important to call this function
// before closing the derived class.
// ---------------------------------------------------------------------------\

void cpp_inherit_default(char *baseclass, int inherit) {
  CPP_class *c;
  extern void TypeEq_Mangle(char *, char *);
  char   temp1[128];
  char   temp2[128];

  c = classlist->search(baseclass);
  if (c) {

    // Found a base class.  Emit a bunch of functions
    
    Inherit_mode = 1;
    current_class->baseclass = c;;
    base_class = c;
    if (inherit) {
      c->inherit_decls();
    }
    Inherit_mode = 0;
    // Add the current class name to the list of derived types for the base class
    while (c) {
      c->add_derived(ClassName);
      sprintf(temp1,"%s %s", current_class->classtype, ClassName);
      sprintf(temp2,"%s %s", c->classtype, c->classname);

      // Add various equivalences to the pointer table
      TypeEq_Mangle(c->classname, ClassName);
      TypeEq_Mangle(temp2, ClassName);
      TypeEq_Mangle(temp2, temp1);
      TypeEq_Mangle(c->classname, temp1);
      c = c->baseclass;
    }

  } else {
    fprintf(stderr,"%s : Line %d. Base class %s not defined (ignored).\n",
	           input_file, line_number, baseclass);
  }
}

//
// Entry point for the parser
//

void cplus_inherit(char *baseclass) {
  lang->cpp_inherit(baseclass);
}

// ---------------------------------------------------------------------------
// ParmList *cplus_emit_member_func(char *name, char *iname, DataType *type, ParmList *l)
//
// Create a helper function to wrap a C++ member function.
// This function only emits the helper function.  It does not 
// provide anything else such as inheritance, naming, documentation or
// even wrapping the resulting function.   
//
// This function returns a modified parameter list that can be used to
// produce a wrapper function.
// 
// Works as follows:
//     class Foo {
//       int bar(args)
//     }
//
//     becomes ....
//     Foo_bar(Foo *obj, args) {
//         obj->bar(args);
//     }
//
// Then we wrap Foo_bar().
//
// References are a little tricky.   A function 
//       int bar(double &a)
//
// Is going to get wrapped as 
//       int Foo_bar(Foo *obj, double *a) {
//            return obj->bar(*a);
// }
//
// A return by reference is even more tricky.   This is indicated by
// The "ref" flag above.
//
//       int& bar() 
//
// Gets turned into the following if ref = 1:
//	
//	int *Foo_bar(Foo *obj) {
//            int *ret1;
//            int& ret = obj->bar();
//            ret1 = &ret;
//            return ret1;
//      }
//
// If C++ mode is disabled, this plays a trick : It creates a wrapper
// for a "C" function with the name ClassName_func  (for shadow
// classing, this allows one to write object oriented C).
//
// -------------------------------------------------------------------------

ParmList *cplus_emit_member_func(char *name, char *iname, DataType *type, ParmList *l) {

    Parm *p;
    ParmList *newparms;
    int i;


    if (!NewMethods) {
      fprintf(f_header,"static %s %s(%s%s *obj", type->print_full(), iname, ClassType, ClassName);

      // Walk down the parameter list and Spit out arguments
 
      i = 0;
      p = l->get_first();
      while (p != 0) {
	if ((p->t->type != T_VOID) || (p->t->is_pointer)) {
	  fprintf(f_header,",");
	  fprintf(f_header,"%s _arg%d", p->t->print_full(), i);
	  i++;
	}
	p = l->get_next();
      }

      fprintf(f_header,") {\n");

      // Emit the function call.  Need to do something special for
      // References however.

      if (type->type != T_VOID) {
	// Declare the return value

	fprintf(f_header,"\t %s _result;\n", type->print_full());

	if (type->is_reference) {
	  type->is_pointer--;
	  fprintf(f_header,"\t %s& _result_ref = ", type->print_full());
	  type->is_pointer++;
	} else {
	  fprintf(f_header,"\t _result = ");
	}
      } else {
	fprintf(f_header,"\t ");
      }
      
      fprintf(f_header," obj->%s(",name);
      
      i = 0;
      p = l->get_first();
      while(p != 0) {
	if ((p->t->type != T_VOID) || (p->t->is_pointer)) {
	  if (p->t->is_reference) {
	    fprintf(f_header,"*");
	  }
	  fprintf(f_header,"_arg%d",i);
	  i++;
	}
	p = l->get_next();
	if (p != 0)
	  fprintf(f_header,",");
      }
      fprintf(f_header,");\n");
      
      if (type->type != T_VOID) {
	if (type->is_reference) {
	  fprintf(f_header,"\t _result = &_result_ref;\n");
	}
	fprintf(f_header,"\t return _result;\n");
      }
      
      fprintf(f_header,"}\n");
    }

    // Add a parameter to the beginning of the function

    newparms = new ParmList(l);
    p = new Parm;
    p->t = new DataType;
    p->t->type = T_USER;
    p->t->is_pointer = 1;
    p->call_type = 0;

    sprintf(p->t->name,"%s%s", ClassType, ClassName);
    p->name = "";
    newparms->add_param(p);

    // If we had any C++ references, get rid of them now.

    if (!NewMethods) {
      p = newparms->get_first();
      while (p) {
	p->t->is_reference = 0;
	p = newparms->get_next();
      }
    }
    return newparms;
}

// ---------------------------------------------------------------------------
// void cpp_member_func_default(char *name, char *iname, DataType *type, ParmList *,
//                              int inherit)
//
// Wrap a C++ member function (default method)
//
// This function takes a member function definition, produces a helper
// function and wraps that.  Does not process inheritance.
// 
// inherit flag indicates whether or not this function can be inherited or not.
// ----------------------------------------------------------------------------

void cpp_member_func_default(char *name, char *iname, DataType *type, ParmList *l) {

    ParmList *newparms;

    // Create name and iname for this sucker...

    if (ClassRename) {
      if (strlen(ClassRename)) 
	sprintf(new_iname,"%s_%s", ClassRename, iname);
      else
	sprintf(new_iname,"%s",iname);
    } else {
      sprintf(new_iname,"%s_%s", ClassName,iname);
    }

    sprintf(new_name,"CPP:%s", new_iname);
    if (Add_Symbol(new_name,(DataType *) 0,(char *) 0)) {
      fprintf(stderr,"%s : Line %d. Function %s multiply defined (2nd definition ignored).\n", input_file, line_number, iname);
      return;
    }

    // Create a function name for this function

    if (!NewMethods)
      sprintf(new_name,"%s_%s", ClassName, iname);
    else
      sprintf(new_name,"%s_%s", ClassName, name);

    // Emit a help function

    newparms = cplus_emit_member_func(name, new_name, type, l);
    
    if (type->is_reference) {
      type->is_reference = 0;
      lang->create_function(new_name,new_iname, type, newparms);
      type->is_reference = 1;
    } else {
      lang->create_function(new_name,new_iname, type, newparms);
    }

    // Create a new documentation entry for this one
      
    doc_entry = new DocEntry(category,new_iname, type, newparms);
    doc->add_doc(doc_entry);
}

// ---------------------------------------------------------------------------
// void clus_member_func(char *name, char *iname, DataType *type, ParmList *)
//
// Parser entry point to creating a C++ member function.   This function
// does a number of things :
//     1.   Build up names for the function (iname).
//     2.   Add function to inheritance tables.
//     3.   Check if function is being inherited or not.
//     4.   Call out to language c++ member function method.
// -------------------------------------------------------------------------

void cplus_member_func(char *name, char *iname, DataType *type, ParmList *l,
		       int is_virtual) {

  char *temp_iname;
  CPP_declaration *cd;

  // First, we build up the interpreter name of this function

  if (!iname) 
    temp_iname = name;
  else 
    temp_iname = iname;

  // If we're in inherit mode, we need to check for duplicates.
  // Issue a warning.
    
  if (Inherit_mode) {
    if (current_class->search(temp_iname)) {
      // Have a duplication
      if (Verbose) {
	fprintf(stderr,"Warning : function %s from baseclass %s not inherited into class %s\n", iname, base_class->classname, current_class->classname);
	fprintf(stderr,"(function name is duplicated in derived class)\n");
      }
      return;
    }
  }
  
  // Add it to our C++ class list

  cd = new CPP_declaration;
  cd->cpp_func = new CPP_function(name, temp_iname, type, l, 0);
  cd->decl_type = CPP_func;
  current_class->add(cd);

  if (is_virtual) Have_Virtual = 1;

}

// ----------------------------------------------------------------------
// void cpp_constructor_default(char *name, char *iname, ParmList *)
//
// Creates a constructor function.
//
// name is the name of the class.  
// iname is the name to be used for the construct (if renamed).
// ParmList is a list of parameters associated with the constructor (if any).
//
// This function will create wrapper functions called
//       new_ClassRename   (the default)
//    or new_<iname>       (if renamed)
//
// -------------------------------------------------------------------------

void cpp_constructor_default(char *name, char *iname, ParmList *l) {

    Parm *p;
    int i;
    DataType *type;
    char   *temp_iname;
    // Figure out how we're going to rename this class.
    
    if (iname) 
      temp_iname = iname;
    else
      temp_iname = name;

    // Check if we've already added this function 

    sprintf(new_name,"CPP:%s_%s_construct", ClassName, temp_iname);
    if (Add_Symbol(new_name,(DataType *) 0,(char *) 0)) {
      fprintf(stderr,"%s : Line %d. Constructor %s::%s multiply defined (2nd definition ignored).\n", input_file, line_number, ClassName, temp_iname);
      return;
    }

    type = new DataType;
    type->type = T_USER;
    sprintf(type->name,"%s%s", ClassType, name);
    type->is_pointer = 1;

    // Got to figure out how many parms for correct C handling

    i = 0;
    p = l->get_first();
    while (p) {
      p = l->get_next();
      i++;
    }
    
    if (!NewMethods) {
      
      fprintf(f_header,"static %s%s *new_%s(", ClassType, name, temp_iname);

      // Walk down the parameter list and Spit out arguments
      
      i = 0;
      p = l->get_first();
      while (p != 0) {
	if ((p->t->type != T_VOID) || (p->t->is_pointer)) {
	  fprintf(f_header,"%s _arg%d", p->t->print_full(), i);
	  i++;
	}
	p = l->get_next();
	if (p) {
	  fprintf(f_header,",");
	}
      }
      
      fprintf(f_header,") {\n");

      // Emit the real function call

      if (CPlusPlus) {
	fprintf(f_header,"\t return new %s(",name);
	
	i = 0;
	p = l->get_first();
	while(p != 0) {
	  if ((p->t->type != T_VOID) || (p->t->is_pointer)) {
	    fprintf(f_header,"_arg%d",i);
	    i++;
	  }
	  p = l->get_next();
	  if (p != 0)
	    fprintf(f_header,",");
	}
	fprintf(f_header,");\n");
      } else {
	fprintf(f_header,"\t return %s malloc(sizeof(%s%s));\n",type->print_cast(), ClassType, name);
      }
    
      fprintf(f_header,"}\n");

    }

    if (NewMethods) {
      sprintf(new_name,"new_%s", name);
    } else {
      sprintf(new_name,"new_%s", temp_iname);
    }
    
    if (!iname) {
      if (ClassRename) {
	if (strlen(ClassRename))
	  sprintf(new_iname,"new_%s", ClassRename);
	else 
	  sprintf(new_iname,"new");
      }
      else
	sprintf(new_iname,"new_%s", ClassName);
    } else 
      sprintf(new_iname,"new_%s", iname);

    lang->create_function(new_name, new_iname, type, l);
    // Create a new documentation entry for this one

    doc_entry = new DocEntry(category,new_iname, type, l);
    doc->add_doc(doc_entry);
    delete type;
}

// ----------------------------------------------------------------------
// void cplus_constructor(char *name, char *iname, ParmList *,l)
//
// Creates a constructor function.
//
// name is the name of the class.
// iname is a renaming of the constructor.
//
// Checks to make sure name is the same as the current class.  If not,
// Then we'll set the type to an int and call cplus_member_func().
// -------------------------------------------------------------------------

void cplus_constructor(char *name, char *iname, ParmList *l) {

    DataType *type;
    CPP_declaration *cd;

    if ((strlen(ClassName) > 0) && (strcmp(name, ClassName) != 0)) {
      type = new DataType;
      type->type = T_INT;
      sprintf(type->name,"int");
      type->is_pointer = 0;
      cplus_member_func(name, iname, type, l,0);
      delete type;
      return;
    }

    // Call language dependent constructor function

    Have_Constructor = 1;
    cd = new CPP_declaration;
    if (strlen(ClassName) > 0)
      cd->cpp_construct = new CPP_constructor(name,iname,l,0);
    else
      cd->cpp_construct = new CPP_constructor(name,iname,l,1);
    cd->decl_type = CPP_new;
    current_class->add(cd);

}

// ----------------------------------------------------------------------
// void cpp_destructor_default(char *name, char *iname)
//
// Default method for creating a C++ destructor.   name is
// the classname.    iname is the renamed function (if
// applicable).
//
// Works as follows:
//     class Foo {
//       ...
//     ~Foo();
//     }
//
//     becomes ....
//     void delete_Foo(Foo *f) {
//         delete f;
//     }
//
// Then we wrap delete_Foo().
//
// -------------------------------------------------------------------------

void cpp_destructor_default(char *name, char *iname) {

    Parm *p;
    DataType *type;
    ParmList *l;

    sprintf(new_name,"CPP:%s_%s_destroy", ClassName, name);
    if (Add_Symbol(new_name,(DataType *) 0,(char *) 0)) {
      fprintf(stderr,"%s : Line %d. Destructor %s::~%s multiply defined (2nd definition ignored).\n", input_file, line_number, ClassName, name);
      return;
    }

    if (!NewMethods) {
      // Spit out a helper function for this member function

      fprintf(f_header,"static void delete_%s(%s%s *obj) {\n", name, ClassType, name);
      if (CPlusPlus) 
	fprintf(f_header,"\t delete obj;\n");
      else
	fprintf(f_header,"\t free((char *) obj);\n");
      
      fprintf(f_header,"}\n");
    }
    
    // Add a parameter to the beginning of the function
    
    l = new ParmList;
    p = new Parm;
    p->t = new DataType;
    p->t->type = T_USER;
    p->t->is_pointer = 1;
    p->call_type = 0;
    sprintf(p->t->name,"%s%s", ClassType, ClassName);
    p->name = "";
    l->add_param(p);
    
    type = new DataType;
    type->type = T_VOID;
    sprintf(type->name,"void");
    type->is_pointer = 0;

    sprintf(new_name,"delete_%s", name);

    // We ignore the name directive if supplied.

    if (iname) 
      fprintf(stderr,"%s : Line %d.  %%name() directive ignored.\n", input_file,
	      line_number);

    if (ClassRename) {
      if (strlen(ClassRename)) 
	sprintf(new_iname,"delete_%s", ClassRename);
      else
	sprintf(new_iname,"delete");
    } else
      sprintf(new_iname,"delete_%s", ClassName);

    lang->create_function(new_name, new_iname, type, l);

    // Add a documentation entry for it

    doc_entry = new DocEntry(category,new_iname, type, l);
    doc->add_doc(doc_entry);
    delete type;
}

// ----------------------------------------------------------------------
// void cplus_destructor(char *name, char *iname)
//
// Create a wrapper function for a C++ destructor
//
// Works as follows:
//     class Foo {
//       ...
//     ~Foo();
//     }
//
//     becomes ....
//     void delete_Foo(Foo *f) {
//         delete f;
//     }
//
// Then we wrap delete_Foo().
//
// -------------------------------------------------------------------------

void cplus_destructor(char *name, char *iname) {

  CPP_declaration *cd;

  if ((strlen(ClassName) > 0) && (strcmp(name,ClassName) != 0)) {
    fprintf(stderr,"%s : Line %d : Error. %s not a valid destructor.\n", input_file, line_number, name);
    return;
  }

  // If class is unnamed, we'll let it pass through unscathed.

  cd = new CPP_declaration;
  if (strlen(ClassName) > 0) 
    cd->cpp_destruct = new CPP_destructor(name,iname,0);
  else
    cd->cpp_destruct = new CPP_destructor(name,iname,1);
  cd->decl_type = CPP_delete;
  current_class->add(cd);
}

// -------------------------------------------------------------------
// cpp_variable_default(char *name, char *iname, DataType *t)
//
// Create functions to access member variables of a C++ class.
// 
// class Foo {
//      double x;
// }
//
// Gets translated into the following :
//
// double set_Foo_x(Foo *obj, double x) {
//      return obj->x = x;
// }
//
// double get_Foo_x(Foo *obj) {
//      return obj->x;
// }
//
// Need to handle special cases for char * and for user
// defined types. 
//
// 1.  char *
//
//     Will free previous contents (if any) and allocate
//     new storage.   Could be risky, but it's a reasonably
//     natural thing to do.
//
// 2.  User_Defined
//     Will assign value from a pointer. 
//     Will return a pointer to current value.
//
// --------------------------------------------------------------------

void cpp_variable_default(char *name, char *iname, DataType *t) {

    Parm *p;
    ParmList *l;
    char  *temp_name;


    sprintf(new_name,"CPP:%s_%s", ClassName, name);
    if (Add_Symbol(new_name,(DataType *) 0,(char *) 0)) {
      fprintf(stderr,"%s : Line %d. Member data %s::%s multiply defined (2nd definition ignored).\n", input_file, line_number, ClassName, name);
      return;
    }

    // First write a function to set the variable of the
    // Variable

    if (!(Status & STAT_READONLY)) {
      if ((t->type == T_USER) && (!t->is_pointer)) {
	t->is_pointer++;
	fprintf(f_header,"static %s %s_%s_set(%s%s *obj, %s val) {\n",
		t->print_type(), ClassName, name, ClassType, ClassName, t->print_type());
	t->is_pointer--;
      } else {
	fprintf(f_header,"static %s %s_%s_set(%s%s *obj, %s val) {\n",
		t->print_type(), ClassName, name, ClassType, ClassName, t->print_type());
      }

      if ((t->type != T_VOID) || (t->is_pointer)){
	if (!t->is_pointer) {

	  // Have a real value here 
	  // If it's a user defined type, we'll do something special.
	  // Otherwise, just assign it.

	  if (t->type != T_USER) {
	    fprintf(f_header,"\t return (obj->%s = val);\n", name);
	  } else {
	    fprintf(f_header,"\t obj->%s = *(val);\n", name);
	    fprintf(f_header,"\t return &obj->%s;\n", name);
	  }
	} else {
      
	  // Is a pointer type here.  If string, we do something
	  // special.  Otherwise. No problem.

	  if ((t->type == T_CHAR) && (t->is_pointer == 1)) {
	    if (CPlusPlus) {
	      fprintf(f_header,"\t if (obj->%s) delete obj->%s;\n", name,name);
	      fprintf(f_header,"\t obj->%s = new char[strlen(val)+1];\n",name);
	      fprintf(f_header,"\t strcpy(obj->%s,val);\n", name);
	      fprintf(f_header,"\t return obj->%s;\n", name);
	    } else {
	      fprintf(f_header,"\t if (obj->%s) free(obj->%s);\n", name,name);
	      fprintf(f_header,"\t obj->%s = (char *) malloc(strlen(val)+1);\n",name);
	      fprintf(f_header,"\t strcpy(obj->%s,val);\n", name);
	      fprintf(f_header,"\t return obj->%s;\n", name);
	    }
	  } else {
	    fprintf(f_header,"\t return obj->%s = val;\n", name);
	  }
	}
      }

      fprintf(f_header,"}\n");

      // Now wrap it.

      l = new ParmList;
      p = new Parm;
      p->t = new DataType;
      p->t->type = t->type;
      p->t->is_pointer = t->is_pointer;
      p->call_type = 0;	
      if ((t->type == T_USER) && (!t->is_pointer)) p->t->is_pointer++;
      p->t->implicit_ptr = t->implicit_ptr;
      strcpy(p->t->name, t->name);
      if (t->qualifier) {
	p->t->qualifier = new char[strlen(t->qualifier)+1];
	strcpy(p->t->qualifier,t->qualifier);
      }
      p->name = "";
      l->add_param(p);
      p = new Parm;
      p->t =  new DataType;
      p->t->type = T_USER;
      p->call_type = 0;	
      p->t->is_pointer = 1;
      sprintf(p->t->name,"%s%s", ClassType, ClassName);
      p->name = "";
      l->add_param(p);

      sprintf(new_name,"%s_%s_set", ClassName,name);

      if (!iname) temp_name = name;
      else temp_name = iname;
      if (ClassRename) {
	if (strlen(ClassRename))
	  sprintf(new_iname,"%s_%s_set", ClassRename,temp_name);
	else
	  sprintf(new_iname,"%s_set",temp_name);
      } else
	sprintf(new_iname,"%s_%s_set", ClassName, temp_name);

      if ((t->type == T_USER) && (!t->is_pointer)) {
	t->is_pointer++;
	lang->create_function(new_name, new_iname, t, l);
	t->is_pointer--;
      } else {
	lang->create_function(new_name, new_iname, t, l);
      }

      // Add documentation

      doc_entry = new DocEntry(category,new_iname,t,l);
      doc->add_doc(doc_entry);
      delete l;

    } // if READONLY

    // Now write a function to get the value of the variable

    if ((t->type == T_USER) && (!t->is_pointer)) {
      t->is_pointer++;
      fprintf(f_header,"static %s %s_%s_get(%s%s *obj) { \n",
	      t->print_type(), ClassName, name, ClassType, ClassName);
      fprintf(f_header,"\t return &obj->%s;\n", name);
      t->is_pointer--;
    } else {
      fprintf(f_header,"static %s %s_%s_get(%s%s *obj) { \n",
	      t->print_type(), ClassName, name, ClassType, ClassName);
      fprintf(f_header,"\t return obj->%s;\n", name);
    }

    fprintf(f_header,"}\n");
    
    // Wrap this function

    l = new ParmList;
    p = new Parm;
    p->t =  new DataType;
    p->t->type = T_USER;
    p->t->is_pointer = 1;
    p->call_type = 0;	
    p->name = "";
    sprintf(p->t->name,"%s%s", ClassType, ClassName);
    l->add_param(p);

    sprintf(new_name,"%s_%s_get", ClassName,name);

    if (!iname) temp_name = name;
    else temp_name = iname;
    if (ClassRename) {
      if (strlen(ClassRename))
	sprintf(new_iname,"%s_%s_get", ClassRename,temp_name);
      else
	sprintf(new_iname,"%s_get",temp_name);
    } else
      sprintf(new_iname,"%s_%s_get", ClassName, temp_name);
    
    if ((t->type == T_USER) && (!t->is_pointer)) {
      t->is_pointer++;
      lang->create_function(new_name, new_iname, t, l);
      t->is_pointer--;
    } else {
      lang->create_function(new_name, new_iname, t, l);
    }

    // Add documentation

    doc_entry = new DocEntry(category,new_iname,t,l);
    doc->add_doc(doc_entry);
    delete l;

}

// -------------------------------------------------------------------
// cplus_variable(char *name, char *iname, DataType *t)
//
// Parser entry point for creating a variable.  We'll check
// for inheritance, name duplication, etc...
// -------------------------------------------------------------------

void cplus_variable(char *name, char *iname, DataType *t) {

    CPP_declaration *cd;

    // If we're in inherit mode, we need to check for duplicates.
    
    if (Inherit_mode) {
      if (current_class->search(name)) {
	// Have a duplication
	if (Verbose) {
	  fprintf(stderr,"Warning : variable %s from baseclass %s not inherited into class %s\n", name, base_class->classname, current_class->classname);
	  fprintf(stderr,"(is already declared in derived class)\n");
	}
	return;
      }
    }

    cd = new CPP_declaration;
    cd->cpp_var = new CPP_variable(name, iname, t, 0);
    cd->decl_type = CPP_var;
    current_class->add(cd);
}

// -------------------------------------------------------------------
// cpp_static_func_default(char *name, char *iname, DataType *type, ParmList *l)
//
// A static member function.   We'll just hack up a new name for it
// and create a wrapper function.
// -------------------------------------------------------------------

void cpp_static_func_default(char *name, char *iname, DataType *type, ParmList *l) {

    // Check if already defined
  sprintf(new_name,"CPP:%s_%s", ClassName, name);
  if (Add_Symbol(new_name,(DataType *) 0,(char *) 0)) {
    fprintf(stderr,"%s : Line %d. Static member function %s::%s multiply defined (2nd definition ignored).\n", input_file, line_number, ClassName, name);
    return;
  }

  sprintf(new_name,"%s::%s",ClassName, name);
  if (!iname) {
    if (ClassRename) {
      if (strlen(ClassRename)) 
	sprintf(new_iname,"%s_%s", ClassRename,name);
      else 
	sprintf(new_iname,"%s",name);
    } else
      sprintf(new_iname,"%s_%s",ClassName,name);
  } else
    sprintf(new_iname,"%s", iname);

  lang->create_function(new_name, new_iname, type, l);
  // Create a new documentation entry for this one
  doc_entry = new DocEntry(category,new_iname, type, l);
  doc->add_doc(doc_entry);
}

// -------------------------------------------------------------------
// cplus_static_func(char *name, char *iname, DataType *type, ParmList *l)
//
// A static member function.   We'll just hack up a new name for it
// and create a wrapper function.
// -------------------------------------------------------------------

void cplus_static_func(char *name, char *iname, DataType *type, ParmList *l) {

    // If we're in inherit mode, we need to check for duplicates.
    // Possibly issue a warning or perhaps a remapping
    
    if (Inherit_mode) {
      if (current_class->search(name)) {
	// Have a duplication
	if (Verbose) {
	  fprintf(stderr,"Warning : static function %s from baseclass %s not inherited into class %s\n", name, base_class->classname, current_class->classname);
	  fprintf(stderr,"(is duplicated in derived class)\n");
	}
	return;
      }
    }


  CPP_declaration *cd = new CPP_declaration;
  cd->cpp_func = new CPP_function(name, iname, type, l, 1);
  cd->decl_type = CPP_func;
  current_class->add(cd);
}


// ---------------------------------------------------------------------
// cpp_declare_const_default(char *name, char *iname, DataType *type, char *value)
//
// Creates a C++ enum value.
//
// Will be created as ClassName_name or the value of iname if specified.
// ---------------------------------------------------------------------

void cpp_declare_const_default(char *name, char *iname, DataType *type, char *value) {

  char new_value[512];

  // Check if already defined

  sprintf(new_name,"CPP:%s_%s", ClassName, name);
  if (Add_Symbol(new_name,type,value)) {
    fprintf(stderr,"%s : Line %d. Constant value %s::%s multiply defined (2nd definition ignored).\n", input_file, line_number, ClassName, name);
    return;
  }

  if (!iname) {
    if (ClassRename) {
      if (strlen(ClassRename))
	sprintf(new_name,"%s_%s", ClassRename, name);
      else
	sprintf(new_name,"%s",name);
    } else
      sprintf(new_name,"%s_%s",ClassName, name);
  }  else 
    sprintf(new_name,"%s", iname);

  if (!value) 
    sprintf(new_value,"%s::%s",ClassName,name);
  else
    sprintf(new_value,"%s",value);

  lang->declare_const(new_name,type, new_value);

  // Add documentation

  doc_entry = new DocEntry(category,new_iname, type->type,new_value);
  doc_entry->usage = 0;
  lang->usage_const(new_name,type,new_value,&doc_entry->usage);
  doc->add_doc(doc_entry);

}

// ---------------------------------------------------------------------
// cplus_declare_const(char *name, char *iname, DataType *type, char *value)
//
// Creates a C++ enum value.
//
// This is the parser entry point.
// ---------------------------------------------------------------------

void cplus_declare_const(char *name, char *iname, DataType *type, char *value) {

  // If we're in inherit mode, we need to check for duplicates.
  // Possibly issue a warning or perhaps a remapping
    
  if (Inherit_mode) {
    if (current_class->search(name)) {
      // Have a duplication
      if (Verbose) {
	fprintf(stderr,"Warning : enum %s from baseclass %s not inherited into class %s\n", name, base_class->classname, current_class->classname);
	fprintf(stderr,"(symbol is duplicated in derived class)\n");
      }
      return;
    }
  }


  CPP_declaration *cd = new CPP_declaration;
  cd->cpp_const = new CPP_constant(name, iname, type, value);
  cd->decl_type = CPP_enum;
  current_class->add(cd);

  // Add this symbol

  Update_Symbol(name, type, value);
}


// -------------------------------------------------------------------
// cpp_static_var_default(char *name, char *iname, DataType *type)
//
// A static member data item.  We'll just hack it's name and set it into
// C functions.
// -------------------------------------------------------------------

void cpp_static_var_default(char *name, char *iname, DataType *type) {

    // Check if already defined
  sprintf(new_name,"CPP:%s_%s", ClassName, name);
  if (Add_Symbol(new_name,(DataType *) 0,(char *) 0)) {
    fprintf(stderr,"%s : Line %d. Static member variable %s::%s multiply defined (2nd definition ignored).\n", input_file, line_number, ClassName, name);
    return;
  }

  sprintf(new_name,"%s::%s",ClassName, name);
  if (!iname) {
    if (ClassRename) {
      if (strlen(ClassRename)) 
	sprintf(new_iname,"%s_%s", ClassRename,name);
      else 
	sprintf(new_iname,"%s",name);
    } else
      sprintf(new_iname,"%s_%s",ClassName,name);
  } else
    sprintf(new_iname,"%s", iname);

  lang->link_variable(new_name, new_iname, type);
  // Create a new documentation entry for this one
  doc_entry = new DocEntry(category,new_iname, type);
  doc->add_doc(doc_entry);
}

// -------------------------------------------------------------------
// cplus_static_var(char *name, char *iname, DataType *type)
//
// -------------------------------------------------------------------

void cplus_static_var(char *name, char *iname, DataType *type) {

    // If we're in inherit mode, we need to check for duplicates.
    // Possibly issue a warning or perhaps a remapping
    
    if (Inherit_mode) {
      if (current_class->search(name)) {
	// Have a duplication
	if (Verbose) {
	  fprintf(stderr,"Warning : static variable %s from baseclass %s not inherited into class %s\n", name, base_class->classname, current_class->classname);
	  fprintf(stderr,"(is duplicated in derived class)\n");
	}
	return;
      }
    }

  CPP_declaration *cd = new CPP_declaration;
  cd->cpp_var = new CPP_variable(name, iname, type, 1);
  cd->decl_type = CPP_var;
  current_class->add(cd);

}

// -------------------------------------------------------------------
// cplus_newmethod(int i)
//
// Set the status of the C++ new method flag
// -------------------------------------------------------------------

void cplus_newmethod(int i) {

  NewMethods = i;

}












