
/**********************************************************************
 * Simplified Wrapper and Interface Generator  (SWIG)
 * Version 2.0 (pre-pre-pre-alpha)
 * 
 * Dave Beazley
 * 
 * Theoretical Division (T-11)           Department of Computer Science
 * Los Alamos National Laboratory        University of Utah
 * Los Alamos, New Mexico  87545         Salt Lake City, Utah  84112
 * beazley@lanl.gov                      beazley@cs.utah.edu
 *
 * Copyright (c) 1995-1996
 * The Regents of the University of California and the University of Utah
 * All Rights Reserved
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that 
 * (1) The above copyright notice and the following two paragraphs
 * appear in all copies of the source code and (2) redistributions
 * including binaries reproduces these notices in the supporting
 * documentation.   Substantial modifications to this software may be
 * copyrighted by their authors and need not follow the licensing terms
 * described here, provided that the new terms are clearly indicated in
 * all files where they apply.
 * 
 * IN NO EVENT SHALL THE AUTHOR, THE UNIVERSITY OF CALIFORNIA, THE 
 * UNIVERSITY OF UTAH OR DISTRIBUTORS OF THIS SOFTWARE BE LIABLE TO ANY
 * PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
 * DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION,
 * EVEN IF THE AUTHORS OR ANY OF THE ABOVE PARTIES HAVE BEEN ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE AUTHOR, THE UNIVERSITY OF CALIFORNIA, AND THE UNIVERSITY OF UTAH
 * SPECIFICALLY DISCLAIM ANY WARRANTIES,INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND 
 * THE AUTHORS AND DISTRIBUTORS HAVE NO OBLIGATION TO PROVIDE MAINTENANCE,
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 **************************************************************************/

// ----------------------------------------------------------------
// hash.cxx
//
// This file defines a very simple hash table class.  Eventually,
// this will be part of SWIG 2.0
//
// Simple class for building hash tables.  Uses strings as key
// values, but can store references to anything.
//
// ----------------------------------------------------------------

#include "internal.h"

#define INIT_SIZE   67
// ----------------------------------------------------------------
// Hash()
//
// Construct a new hash table.
// ----------------------------------------------------------------

Hash::Hash() {
  int i;
  hashsize = INIT_SIZE;
  hashtable = (Node **) malloc(hashsize*sizeof(Node *));
  for (i = 0; i < hashsize; i++) {
    hashtable[i] = 0;
  }
}

// ----------------------------------------------------------------
// ~Hash()
//
// Delete this hash table.
// ----------------------------------------------------------------

Hash::~Hash() {
  int   i;
  Node *n,*next;

  for (i = 0; i < hashsize; i++) {
    if (hashtable[i]) {
      n = hashtable[i];
      while (n) {
	next = n->next;
	delete n;
	n = next;
      }
    }
  }
  free(hashtable);
}

// ----------------------------------------------------------------
// int h1(const char *key)
//
// Hashing function
// ----------------------------------------------------------------

int Hash::h1(const char *key) {
  int h = 0;
  const char *c;

  c = key;
  while (*c) {
    h = (128*h + *c) % hashsize;
    c++;
  }
  return h;
}

// ----------------------------------------------------------------
// int Hash::add(const char *k, void *obj)
//
// Add a new entry to the hash table.   Returns 0 on success, -1
// if the item is already in the hash table.
// ----------------------------------------------------------------

int Hash::add(const char *k, void *obj) {

  int  hv;
  Node *n,*prev;

  hv = h1(k);                                  // Get hash value
  n = hashtable[hv];
  prev = n;
  while (n) {
    if (strcmp(n->key,k) == 0) return -1;      // Already in hash table
    prev = n;
    n = n->next;
  }

  // Safe to add this to the table

  n = new Node(k,obj,0);
  if (prev) prev->next = n;
  else hashtable[hv] = n;
  return 0;
}


// ----------------------------------------------------------------
// int Hash::add(const char *k, void *obj, void (*d)(void *))
//
// Add a new entry to the hash table.   Returns 0 on success, -1
// if the item is already in the hash table.
//
// Allows additional specification of a deletion callback for the
// object.
// ----------------------------------------------------------------

int Hash::add(const char *k, void *obj, void (*d)(void *)) {

  int  hv;
  Node *n,*prev;

  hv = h1(k);                                  // Get hash value
  n = hashtable[hv];
  prev = n;
  while (n) {
    if (strcmp(n->key,k) == 0) return -1;      // Already in hash table
    prev = n;
    n = n->next;
  }

  // Safe to add this to the table

  n = new Node(k,obj,d);
  if (prev) prev->next = n;
  else hashtable[hv] = n;
  return 0;
}

// -----------------------------------------------------------------
// void *lookup(const char *k)
//
// Lookup a value in the hash table.   Returns a pointer to the
// object pointer if found.  Otherwise, returns a NULL pointer.
// -----------------------------------------------------------------

void *Hash::lookup(const char *k) {
  int hv;
  Node *n;

  hv = h1(k);                                // Get hash value
  n = hashtable[hv];

  while (n) {
    if (strcmp(n->key,k) == 0) return n->object;
    n = n->next;
  }

  return 0;
}


