
/**********************************************************************
 * Simplified Wrapper and Interface Generator  (SWIG)
 * 
 * Dave Beazley
 * 
 * Theoretical Division (T-11)           Department of Computer Science
 * Los Alamos National Laboratory        University of Utah
 * Los Alamos, New Mexico  87545         Salt Lake City, Utah  84112
 * beazley@lanl.gov                      beazley@cs.utah.edu
 *
 * Copyright (c) 1995-1996
 * The Regents of the University of California and the University of Utah
 * All Rights Reserved
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that 
 * (1) The above copyright notice and the following two paragraphs
 * appear in all copies of the source code and (2) redistributions
 * including binaries reproduces these notices in the supporting
 * documentation.   Substantial modifications to this software may be
 * copyrighted by their authors and need not follow the licensing terms
 * described here, provided that the new terms are clearly indicated in
 * all files where they apply.
 * 
 * IN NO EVENT SHALL THE AUTHOR, THE UNIVERSITY OF CALIFORNIA, THE 
 * UNIVERSITY OF UTAH OR DISTRIBUTORS OF THIS SOFTWARE BE LIABLE TO ANY
 * PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
 * DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION,
 * EVEN IF THE AUTHORS OR ANY OF THE ABOVE PARTIES HAVE BEEN ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE AUTHOR, THE UNIVERSITY OF CALIFORNIA, AND THE UNIVERSITY OF UTAH
 * SPECIFICALLY DISCLAIM ANY WARRANTIES,INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND 
 * THE AUTHORS AND DISTRIBUTORS HAVE NO OBLIGATION TO PROVIDE MAINTENANCE,
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 * The author requests that all users of this software return any
 * improvements made to beazley@cs.utah.edu and grant the author
 * full redistribution rights.
 *
 **************************************************************************/
/************************************************
 * $Header: /b11/dmb/SWIG/SWIG1.0/SWIG/RCS/scanner.cxx,v 1.32 1996/08/28 05:37:08 dmb Exp $
 * scanner.c
 *
 * Dave Beazley
 * January 1996
 *
 * Input scanner.   This scanner finds and returns tokens
 * for the wrapper generator.   Since using lex/flex from
 * C++ is so F'ed up, I've written this function to replace
 * them entirely.     It's not as fast, but hopefully it will
 * eliminate alot of compilation problems.
 *
 * $Log: scanner.cxx,v $
 * Revision 1.32  1996/08/28 05:37:08  dmb
 * Took out weird error message.
 *
 * Revision 1.31  1996/08/27 22:59:27  dmb
 * Minor change to error handling
 *
 * Revision 1.30  1996/08/26 23:35:31  dmb
 * Added %addmethods directive
 *
 * Revision 1.29  1996/08/25 00:03:46  dmb
 * Added %pragma keyword
 *
 * Revision 1.28  1996/08/21 05:48:26  dmb
 * Added a little better error handling
 *
 * Revision 1.27  1996/08/15 05:05:05  dmb
 * Changed typedef/%typedef handling
 *
 * Revision 1.26  1996/08/12 01:50:29  dmb
 * Added some new keywords--primarily for C++.
 *
 * Revision 1.25  1996/07/23 16:51:57  dmb
 * Fixed minor bug in skip_define()
 *
 * Revision 1.24  1996/06/18 15:33:43  dmb
 * Declared yylex as an extern "C"
 *
 * Revision 1.23  1996/05/28 23:17:01  beazley
 * Added brackets
 *
 * Revision 1.22  1996/05/22  20:20:21  beazley
 * Cleaned up some error handling.
 *
 * Revision 1.21  1996/05/20  23:37:22  beazley
 * Fixed constant rules to allow suffixes of 'u' and 'l'
 *
 * Revision 1.20  1996/05/17  05:53:48  beazley
 * Fixed support for multi-character 'char' constants.
 *
 * Revision 1.19  1996/05/13  23:45:53  beazley
 * Added %module directive
 *
 * Revision 1.18  1996/05/10  23:38:53  beazley
 * Added %alpha, %raw, %text directives
 *
 * Revision 1.17  1996/05/07  14:42:04  beazley
 * Took out redundant '&' symbol.
 *
 * Revision 1.16  1996/05/06  23:10:30  beazley
 * Added some more C++ keywords.
 *
 * Revision 1.15  1996/05/03  22:28:59  dmb
 * Preliminary C++ support.
 *
 * Revision 1.14  1996/05/01 22:42:14  dmb
 * Changed to use "parser.h" for parser include file.
 *
 * Revision 1.13  1996/04/16 17:07:25  dmb
 * Fixed minor bug with free() command.
 *
 * Revision 1.12  1996/04/14 15:24:11  dmb
 * Fixed headers.
 *
 * Revision 1.11  1996/04/08 22:09:43  beazley
 * Minor cleanup
 *
 * Revision 1.10  1996/04/08  19:44:24  beazley
 * Minor fixes.
 *
 * Revision 1.9  1996/04/07  20:31:42  beazley
 * Fixed a few things. Added enums.
 *
 * Revision 1.8  1996/04/03  22:48:56  beazley
 * Added option for ignoring comments.
 *
 * Revision 1.7  1996/03/22  23:41:53  beazley
 * Took out some obsolete directives.
 *
 * Revision 1.6  1996/03/16  06:28:01  beazley
 * Added new tokens.   Cleaned it up.
 *
 * Revision 1.5  1996/02/20  04:15:43  beazley
 * Fixed up buffers.  Made object file much much smaller.
 *
 * Revision 1.4  1996/02/16  06:38:49  beazley
 * Removed a few unused variables.
 *
 * Revision 1.3  1996/02/15  22:38:20  beazley
 * Changed copyright. Added %typedef token
 *
 * Revision 1.2  1996/02/07  05:23:55  beazley
 * A lot of changes to support documentation and new parser features.
 *
 * Revision 1.1  1996/01/16  00:56:15  beazley
 * Initial revision
 *
 *
 ************************************************/


#include "internal.h"
#include "parser.h"
#include <string.h>
#include <ctype.h>

#define  MAXHLEN  256000
#define  YYBSIZE  8192

struct InFile {
  FILE   *f;
  int    line_number;
  char  *in_file;
  struct InFile *prev;
};

InFile  *in_head;

FILE    *LEX_in = NULL;
static char    *header;
static char    *yybuffer;
static int     lex_pos = 0;
static int     lex_len = 0;
char    yytext[YYBSIZE];
static int     yylen = 0;
int     line_number = 1;
int     start_line = 1;
char   *input_file;
static int     scan_init  = 0;
static  int    num_brace = 0;
static  int    last_brace = 0;
static  int    in_define = 0;
extern int Error;


/**************************************************************
 * scanner_init()
 *
 * Initialize buffers
 **************************************************************/

void scanner_init() {

  header = (char *) malloc(MAXHLEN);
  yybuffer = (char *) malloc(YYBSIZE);
  scan_init = 1;
}

/**************************************************************
 * scanner_file(FILE *f)
 *
 * Start reading from new file 
 **************************************************************/
void scanner_file(FILE *f) {
  InFile *in;

  in = new InFile;
  in->f = f;
  in->in_file = input_file;
  if (in_head) in_head->line_number = line_number+1;
  if (!in_head) in->prev = 0;
  else in->prev = in_head;
  in_head = in;
  LEX_in = f;
  line_number = 1;
}

/**************************************************************
 * scanner_close()
 *
 * Close current input file and go to next 
 **************************************************************/

void scanner_close() {

  InFile *p;

  fclose(LEX_in);
  if (!in_head) return;
  p = in_head->prev;
  if (p != 0) {
    LEX_in = p->f;
    line_number = p->line_number;
    input_file = p->in_file;
  } else {
    LEX_in = NULL;
  }
  free((char *) in_head);
  in_head = p;
  ForceExtern = 0;             // Turn off any special processing
}

/**************************************************************
 * char nextchar()
 *
 * gets next character from input
 **************************************************************/

char nextchar() {

    char c;

    if (lex_pos >= lex_len) {
	if (!LEX_in) {
	  SWIG_exit(1);
	}
	while(fgets(yybuffer, YYBSIZE, LEX_in) == NULL) {
	  scanner_close();        // Close current input file
	  if (!LEX_in) return 0;  // No more, we're outta here
	}
	lex_len = strlen(yybuffer);
	lex_pos = 0;
      }

    lex_pos++;
    c = yybuffer[lex_pos-1];
    yytext[yylen] = c;
    yylen++;
    return(c);

}

void retract(int n) {

    lex_pos = lex_pos - n;
    yylen = yylen - n;

}

/**************************************************************
 * yycomment(char *)
 *
 * Inserts a comment into a documentation entry.    If no doc
 * entry exists, this ignores a comment.   Otherwise, it saves
 * the comment and clears the documentation entry.
 **************************************************************/

void yycomment(char *s) {

  if (doc_entry == 0) return;   // Do nothing

  if (IgnoreComments) return;   // Throw out comments

  doc_entry->descrip = copy_string(s);
  doc_entry = 0;                // Clear entry (disabling further comments)
}

/**************************************************************
 * void skip_brace(void)
 *
 * Found a {.
 * Skip all characters until we find a matching closed }.
 *
 * This function is used to skip over inlined C code and other
 * garbage found in interface files.
 ***************************************************************/

void skip_brace(void) {

  char c;
  while (num_brace > last_brace) {
    if ((c = nextchar()) == 0) {
      fprintf(stderr,"%s : Line %d.  Missing '}'. Reached end of input.\n",
	      input_file, line_number);
      return;
    }
    if (c == '{') num_brace++;
    if (c == '}') num_brace--;
    if (c == '\n') line_number++;
    yylen = 0;
  }
}

/**************************************************************
 * void skip_decl(void)
 *
 * This tries to skip over an entire declaration.   For example
 *
 *  friend ostream& operator<<(ostream&, const char *s);
 * 
 * or 
 *  friend ostream& operator<<(ostream&, const char *s) { };
 *
 **************************************************************/

void skip_decl(void) {
  char c;
  int  done = 0;
  while (!done) {
    if ((c = nextchar()) == 0) {
      fprintf(stderr,"%s : Line %d.  Missing semicolon. Reached end of input.\n",
	      input_file, line_number);
      return;
    }
    if (c == '\n') line_number++;
    if (c == '{') {
      last_brace = num_brace;
      num_brace++;
      break;
    }
    yylen = 0;
    if (c == ';') done = 1;
  }
  if (!done) {
    while (num_brace > last_brace) {
      if ((c = nextchar()) == 0) {
	fprintf(stderr,"%s : Line %d.  Missing '}'. Reached end of input.\n",
		input_file, line_number);
	return;
      }
      if (c == '{') num_brace++;
      if (c == '}') num_brace--;
      if (c == '\n') line_number++;
      yylen = 0;
    }
  }
}

/**************************************************************
 * void skip_define(void)
 *
 * Skips to the end of a #define statement.
 *
 **************************************************************/

void skip_define(void) {
  char c;
  while (in_define) {
    if ((c = nextchar()) == 0) return;
    if (c == '\\') in_define = 2;
    if (c == '\n') {
      if (in_define == 2) {
	line_number++;
	in_define = 1;
      } else if (in_define == 1) {
	line_number++;
	in_define = 0;
      }
    }
    yylen = 0;
  }
}
      
/**************************************************************
 * int yylook()
 *
 * Lexical scanner.
 * See Aho,Sethi, and Ullman,  pg. 106
 **************************************************************/

int yylook(void) {

    int      state;
    char     c;
    int      hi = 0;

    state = 0;
    yylen = 0;
    start_line = line_number;
    while(1) {

/*	printf("State = %d\n", state);   */
	switch(state) {

	case 0 :
	  if((c = nextchar()) == 0) return(0);

	  /* Process delimeters */

	  if ((c == ' ') || (c == '\t')) {
	    state = 0;   
	    yylen = 0;
	  } else if (c == '\n') {
	    state = 0;
	    yylen = 0;
	    line_number++;
	    if (in_define == 1) {
	      in_define = 0;
	      return(ENDDEF);
	    } else if (in_define == 2) {
	      in_define = 1;
	    }
	  }
	      
	  /* Look for single character symbols */
	      
	  else if (c == '(') return (LPAREN);
	  else if (c == ')') return (RPAREN);
	  else if (c == ';') return (SEMI);
	  else if (c == ',') return (COMMA);
	  else if (c == '*') return (STAR);
	  else if (c == '}') {
	    num_brace--;
	    if (num_brace < 0) {
	      fprintf(stderr,"%s : Line %d. Error. Extraneous '}' (Ignored)\n",
		      input_file, line_number);
	      state = 0;
	      num_brace = 0;
	    } else {
	      return (RBRACE);
	    }
	  }
	  else if (c == '{') {
	    last_brace = num_brace;
	    num_brace++;
	    return (LBRACE);
	  }
	  else if (c == '=') return (EQUAL);
	  else if (c == '+') return (PLUS);
          else if (c == '-') return (MINUS);
	  else if (c == '&') return (AND);
	  else if (c == '|') return (OR);
	  else if (c == '^') return (XOR);
          else if (c == '<') state = 60;
	  else if (c == '>') state = 61;
	  else if (c == '~') return (NOT);
	  else if (c == '\\') {
	    if (in_define == 1) {
	      in_define = 2;
	      state = 0;
	    } else 
	      state = 99;
	  }
  	  else if (c == '[') return (LBRACKET);
	  else if (c == ']') return (RBRACKET);

	  /* Look for multi-character sequences */
	  
	  else if (c == '/') state = 1;    // Comment (maybe) 
	  else if (c == '\"') state = 2;   // Possibly a string
	  else if (c == '#') state = 3;    // CPP
	  else if (c == '%') state = 4;    // Directive
	  else if (c == ':') state = 5;    // maybe double colon
	  else if (c == '0') state = 83;   // An octal or hex value
	  else if (c == '\'') state = 9;   // A character constant
	  else if (isdigit(c)) state = 8;  // A numerical value
	  else if ((isalpha(c)) || (c == '_')) state = 7;
	  else state = 99;
	  break;
	case 1:  /*  Comment block */
	  if ((c = nextchar()) == 0) return(0);
	  if (c == '/') state = 10;        // C++ style comment
	  else if (c == '*') state = 11;   // C style comment
	  else {
	    retract(1);
	    return(SLASH);
	  }
	  break;
	case 10:  /* C++ style comment */
	  if ((c = nextchar()) == 0) {
	    fprintf(stderr,"%s : EOF. Unterminated comment detected.\n",input_file);
	    return 0;
	  }
	  if (c == '\n') {
	    retract(1);
	    // Add the comment to documentation
	    yytext[yylen] = 0;
	    yycomment(yytext+2);
	    state = 0;
	  } else {
	    state = 10;
	  }
	  break;
	case 11: /* C style comment block */
	  if ((c = nextchar()) == 0) {
	    fprintf(stderr,"%s : EOF. Unterminated comment detected.\n", input_file);
	    return 0;
	  }
	  if (c == '\n') line_number++;
	  if (c == '*') 
	    state = 12;
	  else {
	    state = 11;
	  }
	  break;
	case 12: /* Still in C style comment */
	  if ((c = nextchar()) == 0) {
	    fprintf(stderr,"%s : EOF. Unterminated comment detected.\n", input_file);
	    return 0;
	  }
	  
	  if (c == '\n') line_number++;
	  if (c == '*')
	    state = 12;
	  else if (c == '/') {
	    yytext[yylen-2] = 0;
	    yycomment(yytext+2);
	    state = 0;
	  }
	  else 
	    state = 11;
	  break;

	case 2: /* Processing a string */
	  if ((c = nextchar()) == 0) {
	    fprintf(stderr,"%s : EOF. Unterminated string detected.\n", input_file);
	    return 0;
	  }
	  if (c == '\"') {
	    yytext[yylen-1] = 0;
	    yylval.id = copy_string(yytext+1);
	    return(STRING);
	  } else if (c == '\\') {
	    state = 21;             /* Possibly an escape sequence. */
	    break;
	  } else state = 2;
	  break;
	case 21: /* An escape sequence. get next character, then go
		    back to processing strings */

	  if ((c = nextchar()) == 0) return 0;
	  state = 2;
	  break;

	case 3: /* a CPP directive */

	  if (( c= nextchar()) == 0) return 0;
	  if (c == '\n') {
	    retract(1);
	    yytext[yylen] = 0;
	    yylval.id = yytext;
	    return(POUND);
	  } else {
	    yytext[yylen] = 0;
	    if (strcmp(yytext,"#define") == 0) {
	      in_define = 1;
	      return(DEFINE);
	    } else if (strcmp(yytext,"#ifdef") == 0) {
	      return(IFDEF);
	    } else if (strcmp(yytext,"#ifndef") == 0) {
	      return(IFNDEF);
	    } else if (strcmp(yytext,"#else") == 0) {
	      return(ELSE);
	    } else if (strcmp(yytext,"#endif") == 0) {
	      return(ENDIF);
	    }
	  }
	  state = 3;
	  break;
	case 4: /* A wrapper generator directive (maybe) */
	  if (( c= nextchar()) == 0) return 0;
	  if (c == '{') {
	    start_line++;
	    state = 40;   /* Include block */
	    hi = 0;
	  }
	  else if ((isalpha(c)) || (c == '_')) state = 7;
	  else {
	    retract(1);
	    state = 99;
	  }
	  break;
	  
	case 40: /* Process an include block */
	  if ((c = nextchar()) == 0) {
	    fprintf(stderr,"%s : EOF. Unterminated include block detected.\n", input_file);
	    return 0;
	  }
	  yylen = 0;
	  if (c == '\n') line_number++;
	  if (c == '%') state = 41;
	  else {
	    header[hi] = c;
	    hi++;
	    state = 40;
	  }
	  break;
	case 41: /* Still processing include block */
	  if ((c = nextchar()) == 0) {
	    fprintf(stderr,"%s : EOF. Unterminated include block detected.\n", input_file);
	    return 0;
	  }
	  if (c == '\n') line_number++;
	  if (c == '}') {
	    header[hi] = 0;
	    yylval.id = header;
	    return(HBLOCK);
	  } else {
	    header[hi] = '%';
	    header[hi+1] = c;
	    hi+=2;
	    state = 40;
	  }
	  break;

	case 5: /* Maybe a double colon */

	  if (( c= nextchar()) == 0) return 0;
	  if ( c == ':') return DCOLON;
	  else {
	    retract(1);
	    return COLON;
	  }
	  break;

	case 60: /* shift operators */
	  if ((c = nextchar()) == 0) return (0);
	  if (c == '<') return LSHIFT;
	  else {
	    retract(1);
	    state = 99;
	  }
	  break;
	case 61: 
	  if ((c = nextchar()) == 0) return (0);
	  if (c == '>') return RSHIFT;
	  else {
	    retract(1);
	    state = 99;
	  }
	  break;
	case 7: /* Identifier */
	  if ((c = nextchar()) == 0) return(0);
	  if (isalnum(c) || (c == '_') || (c == '>') 
              || (c == '.') || (c == '-'))
	    state = 7;
	  else {retract(1);
	  return(ID);
	  }
	  break;
	case 8: /* A numerical digit */
	  if ((c = nextchar()) == 0) return(0);
	  if (c == '.') {state = 81;}
	  else if ((c == 'e') || (c == 'E')) {state = 86;}
	  else if (isdigit(c)) { state = 8;}
	  else if ((c == 'l') || (c == 'L')) {
	    state = 87;
	  } else if ((c == 'u') || (c == 'U')) {
	    state = 88;
	  } else {
	      retract(1);
	      yytext[yylen] = 0;
	      yylval.id = copy_string(yytext);
	      return(NUM_INT);
	    }
	  break;
	case 81: /* A floating pointer number of some sort */
	  if ((c = nextchar()) == 0) return(0);
	  if (isdigit(c)) state = 81;
	  else if ((c == 'e') || (c == 'E')) state = 82;
	  else {
	    retract(1);
	    yytext[yylen] = 0;
	    yylval.id = copy_string(yytext);
	    return(NUM_FLOAT);
	  }
	  break;
	case 82:
	  if ((c = nextchar()) == 0) return(0);
	  if ((isdigit(c)) || (c == '-') || (c == '+')) state = 86;
	  else {
	    retract(2);
	    yytext[yylen-1] = 0;
	    yylval.id = copy_string(yytext);
	    return(NUM_INT);
	  }
	  break;
	case 83:
	  /* Might be a hexidecimal or octal number */
	  if ((c = nextchar()) == 0) return(0);
	  if (isdigit(c)) state = 84;
	  else if ((c == 'x') || (c == 'X')) state = 85;
	  else if (c == '.') state = 81;
	  else {
	    retract(1);
	    yytext[yylen] = 0;
	    yylval.id = copy_string(yytext);
	    return(NUM_INT);
	  }
	  break;
	case 84:
	  /* This is an octal number */
	  if ((c = nextchar()) == 0) return (0);
	  if (isdigit(c)) state = 84;
	  else if ((c == 'l') || (c == 'L')) {
	    state = 87;
	  } else if ((c == 'u') || (c == 'U')) {
	    state = 88;
	  } else {
	    retract(1);
	    yytext[yylen] = 0;
	    yylval.id = copy_string(yytext);
	    return(NUM_INT);
	  }
	  break;
	case 85:
	  /* This is an hex number */
	  if ((c = nextchar()) == 0) return (0);
	  if ((isdigit(c)) || (c=='a') || (c=='b') || (c=='c') ||
	      (c=='d') || (c=='e') || (c=='f') || (c=='A') ||
	      (c=='B') || (c=='C') || (c=='D') || (c=='E') ||
	      (c=='F'))
	    state = 85;
	  else if ((c == 'l') || (c == 'L')) {
	    state = 87;
	  } else if ((c == 'u') || (c == 'U')) {
	    state = 88;
	  } else {
	    retract(1);
	    yytext[yylen] = 0;
	    yylval.id = copy_string(yytext);
	    return(NUM_INT);
	  }
	  break;

	case 86:
	  /* Rest of floating point number */

	  if ((c = nextchar()) == 0) return (0);
	  if (isdigit(c)) state = 86;
	  else {
	    retract(1);
	    yytext[yylen] = 0;
	    yylval.id = copy_string(yytext);
	    return(NUM_FLOAT);
	  }
	  /* Parse a character constant. ie. 'a' */
	  break;

	case 87 :
	  /* A long integer of some sort */
	  if ((c = nextchar()) == 0) return (0);
	  if ((c == 'u') || (c == 'U')) {
	    yytext[yylen] = 0;
	    yylval.id = copy_string(yytext);
	    return(NUM_ULONG);
	  } else {
	    retract(1);
	    yytext[yylen] = 0;
	    yylval.id = copy_string(yytext);
	    return(NUM_LONG);
	  } 
	  break;
	case 88:
	  /* An unsigned integer of some sort */
	  if ((c = nextchar()) == 0) return (0);
	  if ((c == 'l') || (c == 'L')) {
	    yytext[yylen] = 0;
	    yylval.id = copy_string(yytext);
	    return(NUM_ULONG);
	  } else {
	    retract(1);
	    yytext[yylen] = 0;
	    yylval.id = copy_string(yytext);
	    return(NUM_UNSIGNED);
	  } 
	  break;
	case 9:
	  if ((c = nextchar()) == 0) return (0);
	  if (c == '\'') {
	    yytext[yylen-1] = 0;
	    yylval.id = copy_string(yytext+1);
	    return(CHARCONST);
	  }
	  break;

	default:
	  if (!Error) 
	    fprintf(stderr,"%s : Line %d ::Illegal character.\n",input_file, line_number);
	  state = 0;	
  	  Error = 1;	
	  return(ILLEGAL);
	}
    }
}


/**************************************************************
 * int yylex()
 *
 * Gets the lexene and returns tokens.
 *************************************************************/

extern "C" int yylex(void) {
    
    int   l;

    if (!scan_init) {
      scanner_init();
      if (LEX_in == NULL) LEX_in = stdin;
      scanner_file(LEX_in);
    }
    
    l = yylook();

    /* Copy the lexene */

    yytext[yylen] = 0;
    switch(l) {

      case ID:

	/* Look for keywords now */

	if (strcmp(yytext,"int") == 0) {
	  yylval.type = new DataType;
	  yylval.type->type = T_INT;
	  strcpy(yylval.type->name,yytext);
	  return(TYPE_INT);
	}
	if (strcmp(yytext,"double") == 0) {
	  yylval.type = new DataType;
	  yylval.type->type = T_DOUBLE;
	  strcpy(yylval.type->name,yytext);
	  return(TYPE_DOUBLE);
	}
	if (strcmp(yytext,"void") == 0) {
	  yylval.type = new DataType;
	  yylval.type->type = T_VOID;
	  strcpy(yylval.type->name,yytext);
	  return(TYPE_VOID);
	}
	if (strcmp(yytext,"char") == 0) {
	  yylval.type = new DataType;
	  yylval.type->type = T_CHAR;
	  strcpy(yylval.type->name,yytext);
	  return(TYPE_CHAR);
	}
	if (strcmp(yytext,"short") == 0) {
	  yylval.type = new DataType;
	  yylval.type->type = T_SHORT;
	  strcpy(yylval.type->name,yytext);
	  return(TYPE_SHORT);
	}
	if (strcmp(yytext,"long") == 0) {
	  yylval.type = new DataType;
	  yylval.type->type = T_LONG;
	  strcpy(yylval.type->name,yytext);
	  return(TYPE_LONG);
	}
	if (strcmp(yytext,"float") == 0) {
	  yylval.type = new DataType;
	  yylval.type->type = T_FLOAT;
	  strcpy(yylval.type->name,yytext);
	  return(TYPE_FLOAT);
	}
	if (strcmp(yytext,"signed") == 0) {
	  yylval.type = new DataType;
	  yylval.type->type = T_SINT;
	  strcpy(yylval.type->name,yytext);
	  return(TYPE_SIGNED);
	}
	if (strcmp(yytext,"unsigned") == 0) {
	  yylval.type = new DataType;
	  yylval.type->type = T_UINT;
	  strcpy(yylval.type->name,yytext);
	  return(TYPE_UNSIGNED);
	}
	
	if (strcmp(yytext,"extern") == 0) return(EXTERN);
	if (strcmp(yytext,"signed") == 0) return(SIGNED);
	if (strcmp(yytext,"unsigned") == 0) return(UNSIGNED);
	if (strcmp(yytext,"const") == 0) return(CONST);
	if (strcmp(yytext,"struct") == 0) return(STRUCT);
	if (strcmp(yytext,"union") == 0) return(UNION);
	if (strcmp(yytext,"enum") == 0) return(ENUM);
	if (strcmp(yytext,"sizeof") == 0) return(SIZEOF);
	if (strcmp(yytext,"%module") == 0) return(MODULE);
	if (strcmp(yytext,"%init") == 0)  return(INIT);
	if (strcmp(yytext,"%readonly") == 0) return(READONLY);
	if (strcmp(yytext,"%readwrite") == 0) return(READWRITE);
	if (strcmp(yytext,"%name") == 0) return(NAME);
	if (strcmp(yytext,"%include") == 0) return(INCLUDE);
	if (strcmp(yytext,"%extern") == 0) return(WEXTERN);
	if (strcmp(yytext,"%val") == 0) return(CVALUE);
	if (strcmp(yytext,"%section") == 0) return(CATEGORY);
	if (strcmp(yytext,"%title") == 0) return(TITLE);
	if (strcmp(yytext,"%typedef") == 0) {
	  yylval.ivalue = 1;
	  return(TYPEDEF);
	}
	if (strcmp(yytext,"typedef") == 0) {
	  yylval.ivalue = 0;
	  return(TYPEDEF);
	}
	if (strcmp(yytext,"%alpha") == 0) return(ALPHA_MODE);
	if (strcmp(yytext,"%raw") == 0) return(RAW_MODE);
	if (strcmp(yytext,"%text") == 0) return(TEXT);
	if (strcmp(yytext,"%native") == 0) return(NATIVE);
	if (strcmp(yytext,"%disabledoc") == 0) return(DOC_DISABLE);
	if (strcmp(yytext,"%enabledoc") == 0) return(DOC_ENABLE);
	if (strcmp(yytext,"%ifdef") == 0) return(IFDEF);
	if (strcmp(yytext,"%else") == 0) return(ELSE);
	if (strcmp(yytext,"%ifndef") == 0) return(ELIF);
	if (strcmp(yytext,"%endif") == 0) return(ENDIF);
	if (strcmp(yytext,"%pragma") == 0) return(PRAGMA);
	if (strcmp(yytext,"%addmethods") == 0) return(ADDMETHODS);

	// C++ keywords
	
	if (strcmp(yytext,"class") == 0) return(CLASS);
	if (strcmp(yytext,"private") == 0) return(PRIVATE);
	if (strcmp(yytext,"public") == 0) return(PUBLIC);
	if (strcmp(yytext,"protected") == 0) return(PROTECTED);
        if (strcmp(yytext,"friend") == 0) return(FRIEND);
	if (strcmp(yytext,"static") == 0) return(STATIC);
	if (strcmp(yytext,"virtual") == 0) return(VIRTUAL);
	if (strcmp(yytext,"operator") == 0) return(OPERATOR);

	// Ignored keywords 

	if (strcmp(yytext,"inline") == 0) return(yylex());

	yylval.id = copy_string(yytext);
	return(ID);
      default:
	return(l);
    }
}




