/*
 * Copyright 1994 The University of Newcastle upon Tyne
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation for any purpose other than its commercial exploitation
 * is hereby granted without fee, provided that the above copyright
 * notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that
 * the name of The University of Newcastle upon Tyne not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission. The University of
 * Newcastle upon Tyne makes no representations about the suitability of
 * this software for any purpose. It is provided "as is" without express
 * or implied warranty.
 * 
 * THE UNIVERSITY OF NEWCASTLE UPON TYNE DISCLAIMS ALL WARRANTIES WITH
 * REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF
 * NEWCASTLE UPON TYNE BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * If you wish to use this software commercially please contact the author
 * for information as to how this can be arranged.
 * 
 * Author:   Ian Campbell
 *          (Ian.Campbell@newcastle.ac.uk)
 *
 *          Department of Computing Science
 *          University of Newcastle upon Tyne, UK
 */

/* --------------------------------------------------------------------------
@NAME       : wrapper.c
@INPUT      : 
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: Functions and variables used in the interface between
              user applications and the Berkely MPEG decoder.  This
              file essentially comprises the tkMovie front end to the
	      Berkeley decoder; it has access to the decoder's global
	      variables (via globals.h), and also has a few of its own;
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : This file is losely based on mpeg_lib by greg.
@MODIFIED   : 
---------------------------------------------------------------------------- */

#include <stdio.h>
#include <errno.h>
#include "video.h"
#include "proto.h"
#include "util.h"
#include "dither.h"
#include "globals.h"

#define BUF_LENGTH 1000  /*80000*/


/* Global variables shared between this file and globals.c 
 * (but nowhere else!) 
 */
Boolean	    FrameDone;
char       *CurrentImage;
ImageDesc   ImageInfo;
int         totNumFrames = 0;

/* Global variables used in this file only (but passed to mpegVidRsrc()): */

static VidStream *theStream;


/* Prototypes for functions local to this file: */

void GetMPEGInfo (VidStream *vid_stream, ImageDesc *Info);
void openMPEGDither (Display *display, Window window, DitherEnum dither) ;


/* -------------------------------------------------------------------------
@NAME       : OpenMPEG
@INPUT      : 
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: Creates and initializes the variables needed to start 
              reading/decoding an MPEG stream.  

@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : Ian Campbell 1/3/95 adapted from 94/6/16, Greg Ward (adapted 
	      from main() in the original Berkeley source)
@MODIFIED   : 
---------------------------------------------------------------------------- */
Boolean OpenMPEG (Display *display, Window window, 
                  FILE *MPEGfile, ImageDesc *ImgInfo)
{
   /* 
    * Create the video stream and read the first chunk to get movie
    * stats -- width and height in particular.
    */
    extern Boolean EOF_flag;
    ImgInfo->input = MPEGfile;
    ImgInfo->theStream = NULL ;
    ditherType = ImgInfo->ditherType ;

    RewindMPEG(ImgInfo) ;
   /* Allocate/initialize tables used for dithering (?) */

   lum_values = (int *) malloc(LUM_RANGE*sizeof(int));
   cr_values = (int *) malloc(CR_RANGE*sizeof(int));
   cb_values = (int *) malloc(CB_RANGE*sizeof(int));

   init_tables();	

   openMPEGDither(display, window, ImgInfo->ditherType) ;
     
}     /* OpenMPEG () */


/* ----------------------------- MNI Header -----------------------------------
@NAME       : CloseMPEG
@INPUT      : (none)
@OUTPUT     : (none)
@RETURNS    : (void)
@DESCRIPTION: Frees up some of the memory allocated by OpenMPEG() (some
              is not freed because the brain-damaged Berkeley code
	      doesn't take into account the fact that somebody might 
	      want to, say, free up the memory it allocates... someday,
	      I'll probably have to hack into it to fix that, but not
	      today thanks.)
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : Ian Campbell 1/3/95
@MODIFIED   : 
---------------------------------------------------------------------------- */
void CloseMPEG ( ImageDesc *ImgInfo)
{
   DestroyVidStream (ImgInfo->theStream);
/*   free (lum_values);
     free (cr_values);
     free (cb_values); */
}



/* -------------------------------------------------------------------------
@NAME       : RewindMPEG
@INPUT      : 
@OUTPUT     : (none)
@RETURNS    : (void)
@DESCRIPTION: Resets things so that the caller can start reading the MPEG
              stream from the start again.  Note that the caller does NOT
	      need to call OpenMPEG() again -- after a call to RewindMPEG(),
	      the next call to GetMPEGFrame() will return the first frame
	      of the MPEG.
@METHOD     : 
@GLOBALS    : EOF_flag, curBits, bitOffset, bufLength, bitBuffer, totNumFrames
@CALLS      : 
@CREATED    : Ian Campbell 1/3/95
@MODIFIED   : 
@COMMENTS   : 
---------------------------------------------------------------------------- */
void RewindMPEG (ImageDesc *Image)
{
   extern Boolean EOF_flag;
 	  
   if(Image->theStream != NULL) {
	DestroyVidStream (Image->theStream);
	rewind (Image->input);
	} 

   ditherType = Image->ditherType ; 
   theStream = NewVidStream(BUF_LENGTH);
   input = Image->input ;
   EOF_flag = FALSE;
   totNumFrames = 0 ;   
   mpegVidRsrc(0, theStream) ;

   GetMPEGInfo (theStream, Image);
   Image->theStream = theStream ;
   Image->input = input ;
   Image->EOF_flag = EOF_flag ;
   Image->totNumFrames = totNumFrames ;
   	
}



/* ----------------------------- MNI Header -----------------------------------
@NAME       : GetMPEGInfo
@INPUT      : vid_stream - a video stream that is open and has had at 
                           least one call to mpegVidRsrc() performed on it
              Info       - a pointer to an ImageDesc struct in the caller's
	                   space (i.e., the argument to OpenMPEG()) where
			   the image information will be copied
@OUTPUT     : 
@RETURNS    : (void)
@DESCRIPTION: From the video stream, determines the width, height, pixel
              size and depth (in bits) and total image size (in bytes)
	      for an MPEG stream.  Sets the global variable ImageInfo
	      (part of the interface between wrapper.c and globals.c),
	      and then copies that struct to the user application's 
	      space via the Info pointer.
@METHOD     : 
@GLOBALS    : ImageInfo
@CALLS      : 
@CREATED    : Ian Campbell 1/3/95 based on 94/6/17, Greg Ward: based on code
	      from ExecuteDisplay() in the original Berkeley source
@MODIFIED   : 
---------------------------------------------------------------------------- */
void GetMPEGInfo (VidStream *vid_stream, ImageDesc *Info)
{
   switch (Info->ditherType)
   {
      case Twox2_DITHER:
      {
	 ImageInfo.Height = vid_stream->mb_height * 32;
	 ImageInfo.Width = vid_stream->mb_width * 32; 
	 ImageInfo.Depth = 8;
	 ImageInfo.PixelSize = 8;
	 ImageInfo.BitmapPad = 8;
	 break;
      } 
      case FULL_COLOR_DITHER:
      {
	 ImageInfo.Height = vid_stream->mb_height * 16;
	 ImageInfo.Width = vid_stream->mb_width * 16; 
	 ImageInfo.Depth = 24;	/* ???? */
	 ImageInfo.PixelSize = 32;
	 ImageInfo.BitmapPad = 32;
	 break;
      } 
      case MONO_DITHER:
      case MONO_THRESHOLD:
      {
	 ImageInfo.Height = vid_stream->mb_height * 16;
	 ImageInfo.Width = vid_stream->mb_width * 16; 
	 ImageInfo.Depth = 1;
	 ImageInfo.PixelSize = 1;
	 ImageInfo.BitmapPad = 8;
	 break;
      } 
      default:
      {
	 ImageInfo.Height = vid_stream->mb_height * 16;
	 ImageInfo.Width = vid_stream->mb_width * 16; 
	 ImageInfo.Depth = 8;
	 ImageInfo.PixelSize = 8;
	 ImageInfo.BitmapPad = 8;
	 break;
      }
   }     /* switch on ditherType */

   ImageInfo.Size = ImageInfo.Height * ImageInfo.Width * 
      (ImageInfo.PixelSize/8);
   ImageInfo.ditherType = Info->ditherType ;  
   ImageInfo.theStream = Info->theStream ;
   ImageInfo.input = Info->input ;
   ImageInfo.EOF_flag = Info->EOF_flag ;
   ImageInfo.totNumFrames = Info->totNumFrames ;  

   memcpy (Info, &ImageInfo, sizeof (ImageDesc));

}     /* GetMPEGInfo () */



/* --------------------------------------------------------------------------
@NAME       : SetMPEGDither
@INPUT      : 
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: Set dither
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : Ian Campbell 1/3/95.
@MODIFIED   : 
@SIDE EFFECTS :	The stream is rewound to the start.
---------------------------------------------------------------------------- */
void SetMPEGDither (Display *display, Window window,
		    ImageDesc *Image, DitherEnum Value)
{    
	openMPEGDither(display, window, Value) ;
	RewindMPEG(Image) ;

}     /* SetMPEGDither () */



/* ----------------------------- MNI Header -----------------------------------
@NAME       : GetMPEGFrame
@INPUT      : 
@OUTPUT     : Frame - the image data, converted to RGB space, is
              copied to the area pointed to by Frame.  There must be
              enough room for the entire image; the ImageDesc
              structure returned by OpenMPEG() will tell you just how
              much memory this is.  The format of the data depends on
              the dithering type; for full colour dithering, there are
              four bytes per pixel: red, green, blue, and unused.
              (Perfect for passing to lrectwrite() or XPutImage().)
@RETURNS    : TRUE if there are still frames left to decode
              FALSE if we have just decoded the last frame
@DESCRIPTION: Part of the tkMovie front end to the Berkeley MPEG decoder.  
              Essentially reads, decodes, and converts to RGB space the
	      next frame in the MPEG stream opened with OpenMPEG().
@METHOD     : 
@GLOBALS    : 
@CALLS      : mpegVidRsrc ()
@CREATED    : Ian Campbell 1/3/95 adapted from 94/6/16, Greg Ward
@MODIFIED   : 
---------------------------------------------------------------------------- */
Boolean GetMPEGFrame ( ImageDesc *ImgInfo, char *Frame)
{
   extern Boolean EOF_flag ;
   Boolean   MovieDone = FALSE;

   FrameDone = FALSE;
   input = ImgInfo->input ;
   theStream = ImgInfo->theStream ;
   EOF_flag = ImgInfo->EOF_flag;
   totNumFrames = ImgInfo->totNumFrames ;
   ditherType = ImgInfo->ditherType ;
   dprintf ("GetMPEGFrame: starting or continuing movie\n");
 	
   while (!MovieDone && !FrameDone) /* FrameDone is set by ExecuteDisplay() */
   {
      MovieDone = (mpegVidRsrc (0, theStream) == NULL);
   }

   dprintf ("\nGetMPEGFrame: just received a finished frame: "
	    "copying from %08X to %08X\n", CurrentImage, Frame);
   memcpy (Frame, CurrentImage, ImgInfo->Size);
   ImgInfo->theStream = theStream ;
   ImgInfo->input = input ;
   ImgInfo->EOF_flag = EOF_flag ;
   ImgInfo->totNumFrames = totNumFrames ;

   return (!MovieDone);

}     /* GetMPEGFrame () */


/* ----------------------------- MNI Header -----------------------------------
@NAME       : openMPEGDither
@INPUT      : dither - a DitherEnum to open
	      display - pointer to a display on which to create the dither
	      window - a window so that color maps can be altered if neccessary
	      There will be a close dither when I get round to it, to free up
	      some of the holes in the memory when dithers are changed.
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: Creates and initializes the variables needed to start 
              using a dither.  

@METHOD     : 
@GLOBALS    : 
@CALLS      : InitDisplay(), InitColour() etc procs in gdith.c
@CREATED    : 1/3/95, Ian Campbell.ncl.ac.uk
@MODIFIED   : 
---------------------------------------------------------------------------- */
void openMPEGDither (Display *display, Window window, DitherEnum dither)
{

   switch (dither) {
    
  case HYBRID_DITHER:
    
    InitColor();
    InitHybridDither();
    InitDisplay(display, window);
    break;
    
    case HYBRID2_DITHER:
    InitColor();
    InitHybridErrorDither();
    InitDisplay(display, window);
    break;
    
  case FS4_DITHER:
    InitColor();
    InitFS4Dither();
      InitDisplay(display, window);
    break;
    
  case FS2_DITHER:
    InitColor();
    InitFS2Dither();
    InitDisplay(display, window);
    break;
    
  case FS2FAST_DITHER:
    InitColor();
    InitFS2FastDither();
    InitDisplay(display, window);
    break;
    
  case Twox2_DITHER:
    InitColor();
    Init2x2Dither();
    InitDisplay(display, window);
    PostInit2x2Dither();
    break;

  case GRAY_DITHER:
    InitGrayDisplay(display, window);
    break;

  case FULL_COLOR_DITHER:
    InitColorDither();
    InitColorDisplay(display);
    break;

  case NO_DITHER:
/*    shmemFlag = 0; */
    break;

  case ORDERED_DITHER:
    InitColor();
    InitOrderedDither();
    InitDisplay(display, window);
    break;

  case MONO_DITHER:
  case MONO_THRESHOLD:
    InitMonoDisplay(display);
    break;

  case ORDERED2_DITHER:
    InitColor();
    InitDisplay(display, window);
    InitOrdered2Dither();
    break;

  case MBORDERED_DITHER:
    InitColor();
    InitDisplay(display, window);
    InitMBOrderedDither();
    break;

  }
}
