/**************************************************************************
 *  CTAXX.C : a Computer Player for Ataxx
 *  Copyright (c) 1995 John Williams <williams@cs.weber.edu>
 *  Distributed with ABSOLUTELY NO WARRANTY (GNU General Public Licence)
 **************************************************************************/

/* This is pretty generic C code.  It should compile pretty easily on just
   about any ansi compiler.  If you experience problems, please notify the
   author at the email address above. */
/* Successfully compiled on:
     Linux  : gcc -O2 -fno-strength-reduce -o ctaxx ctaxx.c
     HP-UX  : gcc -O2 -o ctaxx ctaxx.c
              cc -Aa -O -o ctaxx ctaxx.c
     SCO    : cc -O -o ctaxx ctaxx.c
     ULTRIX : gcc -O2 -o ctaxx ctaxx.c
              cc -O -o ctaxx ctaxx.c
     SunOS  : gcc -O2 -o ctaxx ctaxx.c
              cc -O -o ctaxx ctaxx.c
     AIX    : cc -O -o ctaxx ctaxx.c
     dgux   : cc -O -o ctaxx ctaxx.c
     OSF1   : cc -O -o ctaxx ctaxx.c
*/

#include <stdlib.h>
#include <unistd.h>

#define MAXDEPTH	10
#define MAXSIZE		9
int	BOARDSIZE = 7;

#define MAXVAL		30000
#define MINVAL		-30000
#define FIRSTMOVE	0x0000

#define PLAYER1		'1'
#define PLAYER2		'2'
#define BLANK		'.'
#define BLOCK		'#'

#define	abs(a)		( (a)<0 ? -(a) : (a) )
#define min(a,b)	( (a)<(b) ? (a) : (b) )
#define max(a,b)	( (a)<(b) ? (b) : (a) )
#define minmax(m,x,y)	( (m)^((x)>(y)) ? (y) : (x) )

int	Skill = 1;
int	Verbose = 0;
int	Hexxagon = 0;
char*	Board = NULL;
int	movbuf[MAXDEPTH+1];
char	brdbuf[MAXDEPTH+1][MAXSIZE*MAXSIZE+1];

/* Boards are input as a single string, by concatenating each row */
/* An extra character is added to the end to indicate whose move it is */
/* The Hexxagon board is tweaked to look square by adding BLOCKs */
char*	initial_board = "1.....2...................................2.....11";
char*	Hexxagon_board = "1...2####......###.......##........#2.......1#........##.......###......####1...21";
/* Ataxx Board:
      A B C D E F G 
   1  1 . . . . . 2  1
   2  . . . . . . .  2
   3  . . . . . . .  3
   4  . . . . . . .  4
   5  . . . . . . .  5
   6  . . . . . . .  6
   7  2 . . . . . 1  7
      A B C D E F G 	*/
/* Hexxagon Board:
             1 2 3 4 5
            / / / / / 6
        A- 1 . . . 2 / 7     
       B- . . . . . . / 8    
      C- . . . . . . . / 9   
     D- . . . . . . . . /     
    E- 2 . . . . . . . 1     
     F- . . . . . . . .      
      G- . . . . . . .       
       H- . . . . . .        
        I- 1 . . . 2         */


 

int	oneMove[MAXSIZE*MAXSIZE][9];
int	twoMove[MAXSIZE*MAXSIZE][17];


#define LEGAL(x,y)	( ( (x)>=0 ) && ( (y)>=0 ) && \
			  ( (x)<BOARDSIZE ) && ( (y)<BOARDSIZE ) && \
			  ( board[(y)*BOARDSIZE+(x)] != BLOCK ) )

void compute_moves(char *board)
{
int	x, y;
int	p, n;
	for (x=0, y=0; y<BOARDSIZE ; x=(x+1)%BOARDSIZE, y+=!x)
	{
		p = y*BOARDSIZE+x;
	/* from space legal? */
		if (board[p] == BLOCK)
		{
			oneMove[p][0] = -1;
			twoMove[p][0] = -1;
			continue;
		}
	/* do all one space moves */
		n = 0;
		if (LEGAL(x-1,y-1))			oneMove[p][n++] = (y-1)*BOARDSIZE+(x-1);
		if (LEGAL(x  ,y-1))			oneMove[p][n++] = (y-1)*BOARDSIZE+(x  );
		if (LEGAL(x+1,y-1) && !Hexxagon)	oneMove[p][n++] = (y-1)*BOARDSIZE+(x+1);
		if (LEGAL(x-1,y  ))			oneMove[p][n++] = (y  )*BOARDSIZE+(x-1);
		if (LEGAL(x+1,y  ))			oneMove[p][n++] = (y  )*BOARDSIZE+(x+1);
		if (LEGAL(x-1,y+1) && !Hexxagon)	oneMove[p][n++] = (y+1)*BOARDSIZE+(x-1);
		if (LEGAL(x  ,y+1))			oneMove[p][n++] = (y+1)*BOARDSIZE+(x  );
		if (LEGAL(x+1,y+1))			oneMove[p][n++] = (y+1)*BOARDSIZE+(x+1);
		oneMove[p][n] = -1;
	/* do all two space moves */
		n = 0;
		if (LEGAL(x-2,y-2))			twoMove[p][n++] = (y-2)*BOARDSIZE+(x-2);
		if (LEGAL(x-1,y-2))			twoMove[p][n++] = (y-2)*BOARDSIZE+(x-1);
		if (LEGAL(x  ,y-2))			twoMove[p][n++] = (y-2)*BOARDSIZE+(x  );
		if (LEGAL(x+1,y-2) && !Hexxagon)	twoMove[p][n++] = (y-2)*BOARDSIZE+(x+1);
		if (LEGAL(x+2,y-2) && !Hexxagon)	twoMove[p][n++] = (y-2)*BOARDSIZE+(x+2);
		if (LEGAL(x-2,y-1))			twoMove[p][n++] = (y-1)*BOARDSIZE+(x-2);
		if (LEGAL(x+2,y-1) && !Hexxagon)	twoMove[p][n++] = (y-1)*BOARDSIZE+(x+2);
		if (LEGAL(x+1,y-1) &&  Hexxagon)	twoMove[p][n++] = (y-1)*BOARDSIZE+(x+1);
		if (LEGAL(x-2,y  ))			twoMove[p][n++] = (y  )*BOARDSIZE+(x-2);
		if (LEGAL(x+2,y  ))			twoMove[p][n++] = (y  )*BOARDSIZE+(x+2);
		if (LEGAL(x-2,y+1) && !Hexxagon)	twoMove[p][n++] = (y+1)*BOARDSIZE+(x-2);
		if (LEGAL(x-1,y+1) &&  Hexxagon)	twoMove[p][n++] = (y+1)*BOARDSIZE+(x-1);
		if (LEGAL(x+2,y+1))			twoMove[p][n++] = (y+1)*BOARDSIZE+(x+2);
		if (LEGAL(x-2,y+2) && !Hexxagon)	twoMove[p][n++] = (y+2)*BOARDSIZE+(x-2);
		if (LEGAL(x-1,y+2) && !Hexxagon)	twoMove[p][n++] = (y+2)*BOARDSIZE+(x-1);
		if (LEGAL(x  ,y+2))			twoMove[p][n++] = (y+2)*BOARDSIZE+(x  );
		if (LEGAL(x+1,y+2))			twoMove[p][n++] = (y+2)*BOARDSIZE+(x+1);
		if (LEGAL(x+2,y+2))			twoMove[p][n++] = (y+2)*BOARDSIZE+(x+2);
		twoMove[p][n] = -1;
	}

	if (Verbose > 2)
	{
		for (p=0 ; p<BOARDSIZE*BOARDSIZE ; p++)
		{
			for (n=0 ; oneMove[p][n]!= -1 ; n++)
				printf("%d - %d\n",p,oneMove[p][n]);
			for (n=0 ; twoMove[p][n]!= -1 ; n++)
				printf("%d - %d\n",p,twoMove[p][n]);
		}
	}	
}


char* tweak_hex_board(char* shortboard)
{
char*	newboard;
char*	hp;
char*	np;
	newboard = (char*)malloc(BOARDSIZE*BOARDSIZE+2);
	np = newboard;
	hp = Hexxagon_board;
	while(*hp)
	{
		while(*hp==BLOCK)
			*np++ = *hp++;
		*np++ = *shortboard++;
		hp++;
	}
	*np = '\0';
/*	printf("tweaked: %s\n",newboard);  */
	return(newboard);
}


void print_move(int move)
{
int	fr, to, fx, fy, tx, ty;
	if (move == 0)
		printf("pass\n");
	else
	{
		fr = (move & 0x00ff);
		tx = (move & 0x0f00) >> 8;
		ty = (move & 0x1000) != 0;
		if (ty)
		{
			to = twoMove[fr][tx];
		}
		else
		{
			to = fr;
			fr = oneMove[fr][tx];
		}
		fx = fr % BOARDSIZE;
		fy = (fr-fx) / BOARDSIZE;
		tx = to % BOARDSIZE;
		ty = (to-tx) / BOARDSIZE;
		if (Hexxagon)
			printf("%c%c-%c%c\n", fy+'a', fx+'1', ty+'a', tx+'1');
		else
			printf("%c%c-%c%c\n", fx+'a', fy+'1', tx+'a', ty+'1');
	}
}


void Move(char* board, int fr, int to, int onetwo, char who)
{
int	n = 0;
char	notwho = ( who==PLAYER1 ? PLAYER2 : PLAYER1 );
int	sp;

	if (onetwo)
		board[fr] = BLANK;
	sp = oneMove[to][n];
	while (sp != -1)
	{
		if (board[sp] == notwho)
			board[sp] = who;
		n++;
		sp = oneMove[to][n];
	}
	board[to] = who;
	board[BOARDSIZE*BOARDSIZE] = notwho;
}


char* generate_move( char* old, int depth, int* move)
{
char*	who = old+(BOARDSIZE*BOARDSIZE);
char*	new = brdbuf[depth];
int	fr, to, onetwo;

	strncpy(new,old,BOARDSIZE*BOARDSIZE+1);

	if (*move==0)
	{
		fr = 0;
		to = -1;
		onetwo = 0;
	}
	else
	{
		fr = (*move & 0x00ff);
		to = (*move & 0x0f00) >> 8;
		onetwo = (*move & 0x1000) != 0;
	}
    while(1)
    {	
	to++;
	if (onetwo)
	{	/* two space moves */
		if (( twoMove[fr][to] == -1 ) ||	/* end of list, or */
		    ( old[fr] != *who ) )		/* this space no good */
		{
			fr++;
			to = -1;
			if (fr >= BOARDSIZE*BOARDSIZE) return(NULL);
			continue;
		}
		if ( old[twoMove[fr][to]] == BLANK )
		{
			Move(new, fr, twoMove[fr][to], onetwo, *who);
			*move = 0x4000 | (onetwo<<12) | (to<<8) | fr;
			return(new);
		}
	}
	else
	{	/* one space moves */
		if (( oneMove[fr][to] == -1 ) ||		/* end of list, or */
		    ( old[fr] != BLANK ) ||			/* this space no good, or */
		    ( to > 0 && old[oneMove[fr][to-1]] == *who) )	/* valid move last time */
		{
			fr++;
			to = -1;
			if (fr >= BOARDSIZE*BOARDSIZE)
			{
				fr = 0;
				onetwo = 1;
			}
			continue;
		}
		if (old[oneMove[fr][to]] == *who)
		{
			Move(new, oneMove[fr][to], fr, onetwo, *who);
			*move = 0x4000 | (onetwo<<12) | (to<<8) | fr;
			return(new);
		}
	}
    }
}


int evaluate(char* board)
{
char*	c;
int	i;
int	n1,n2,nb;

	n1 = n2 = nb = 0;
	c = board;
	for( i=BOARDSIZE*BOARDSIZE; i ; i-- )
	{
		switch(*c)
		{
		case PLAYER1:
			n1++; break;
		case PLAYER2:
			n2++; break;
		case BLANK:
			nb++; break;
		}
		c++;
	}
	if (nb==0 && n1>n2)	return(MAXVAL);
	if (nb==0 && n1<n2)	return(MINVAL);
	if (n1==0)		return(MINVAL);
	if (n2==0)		return(MAXVAL);
	return(n1-n2);
}



int search(char *board, int depth, int clip)
{
char*	new;
int	mm;
int	bestval;
int	value;
int	move;
int	newbest;
int	bestmove;
int	ties;

	if (depth>=Skill)
	{
		return(evaluate(board));
	}

	mm = (board[BOARDSIZE*BOARDSIZE]==PLAYER1);
	bestval = minmax(!mm,MINVAL,MAXVAL);
	move = FIRSTMOVE;
	ties = 0;

	while( new = generate_move(board,depth,&move) )
	{
		if (Verbose) {
			printf("%s   ",new);
			print_move(move);
		}
		value = search( new, depth+1, bestval );
/*if (depth>=0) {printf("%d  %d  ",depth,value); print_move(move); gets(brdbuf+200);}*/
		if ( value!=clip && minmax(!mm,value,clip)==clip)
			return(value);

		newbest = minmax(mm,value,bestval);
		if (value==newbest)
		{	/* a lot of code to choose randomly between ties */
			if (newbest!=bestval)	ties=0;
			ties++;
			if ( (rand()%ties)==0 )
				bestmove = move;
			if (depth==0)
				movbuf[0] = bestmove;
		}
		bestval = newbest;

		if ( bestval == minmax(mm,MINVAL,MAXVAL) )
			return(bestval);
	}
	return(bestval);
}


void
usage()
{
	printf("Usage: ataxx [-dmvH] [board]\n");
	printf("       -d depth   set maximum search depth\n");
	printf("       -H         Hexxagon variant\n");
}


int
process_opts(int argc, char* argv[])
{
int	opt;
extern char	*optarg;
extern int	optind, opterr;

	do
	{
		opt = getopt(argc, argv, "d:vH");
		switch(opt)
		{
		case 'd':
			Skill = atoi(optarg);
			if (Skill<1 || Skill>MAXDEPTH)
			{
				printf("Please use a depth value between 1 and %d\n",MAXDEPTH);
				usage();
				return(1);
			}
			break;
		case 'v':
			Verbose++;
			break;
		case 'H':
			Hexxagon = 1;
			BOARDSIZE = 9;
			initial_board = Hexxagon_board;
			break;
		case -1:
			if (optind<argc && strlen(argv[optind])==(BOARDSIZE*BOARDSIZE+1))
			{	
				Board = argv[optind];
			}	
			else
			{
				if (optind < argc)
				{
					if (Hexxagon && strlen(argv[optind])==62)
					{
						Board = tweak_hex_board(argv[optind]);
						break;
					}
					printf("Bad board.  Length=%d, should be %d\n",
						strlen(argv[optind]),(Hexxagon?62:50));
				}
				usage();
				return(1);
			}
			break;
		case '?':
		case ':':
		default:
			usage();
			return(1);
		}
	}
	while(opt != -1);
	return(0);
}


int
main(int argc, char *argv[])
{
int	clip;

	srand(time(0));
	if (process_opts(argc,argv)) return(1);	
	if (!Board) return(1);
	compute_moves(initial_board);
	movbuf[0] = 0;

	clip = minmax(Board[BOARDSIZE*BOARDSIZE]==PLAYER1,MINVAL,MAXVAL);
	search( Board, 0, clip );

	print_move(movbuf[0]);
		
	return(0);
}

