# man2html2.tcl --
#
# This file defines procedures that are used during the second pass of the
# man page to html conversion process. It is sourced by man2html.tcl.
#
# Copyright (c) 1997 by Tom L. Davis.



# Global variables used by these scripts:
#
# manFile -	name of the current man page file, e.g. 'array.n'.
#
# file -	Where to output the generated HTML. (file pointer)
#
# package -	package directory name. Man pages are assumed to be in
#		$package/doc directory.
#
# THinfo -	Array containing information about the current man page
#		found in the TH macro. For example:
#		    THinfo(fname)   - Tcl_UpVar
#		    THinfo(page)    - 3
#		    THinfo(vers)    - 7.4
#		    THinfo(lib)	    - Tcl
#		    THinfo(pname)   - {Tcl Library Procedures}
#
# NAME_file -	array indexed by NAME and containing file names used
#	     	for hyperlinks. Read only during pass 2.
#
# state -	state variable defining behavior of 'text' proc during
#		pass 2.
#
# inDT -	set by 'TPmacro', cleared by 'newline'. Used to terminate
#		a dictionary term <DT> and start the dictionary definition
#	     	<DD> while in a dictionary list <DL>.
#
# inVS -	In change bar section. See the VS and VE macros.
#
# noFill -	When set, newlines cause a linebreak <BR>.
#
# inPRE -	When set, text is pre-formated <PRE>. 
#
# paraStk -	stack oriented list containing currently active
#	     	HTML tags (UL, OL, DL). Local to 'para' proc.
#
# curFont -	Name of special font that is currently in
#	     	use.  Null means the default paragraph font
#		is being used.
#
# charCnt -	Number of characters on current line. Used by tab proc
#
# tabString - 	String defining the current tab settings. Set by setTabs
#		and used by the tab proc.
#
# footer -	Info to insert at end of each page.
#
# vertSpace -	Flag used to ignore first .PP of each section. Set by
#		SHmacro; cleared by newline, tested and set by putPP & .sp.


# initGlobals --
#
# This procedure is invoked to set the initial values of all of the
# global variables local to pass 2, before processing a man page.
#
# Arguments:
#   None.

proc initGlobals {} {
    global state curFont inPRE inVS inDT noFill 

    para init
    set inDT 0
    set inPRE 0
    set inVS 0
    set noFill 0
    set state 0
    set curFont ""
    setTabs 0.5i
}



# putPP --
#
# Prints <P> to the HTML output file while avoiding writting extraneous
# <P>'s.
#
# Arguments:
#   None.

proc putPP {} {
    global file vertSpace
    if !$vertSpace {
	set vertSpace 1
	puts $file "<P>"
    }
}





# perror --
#
# Prints error strings to stderr and the HTML output file. Use for
# debugging only.
#
# Arguments:
#   string -- the error message to write.

proc perror string {
    global file
    set name "main"
    set level [info level]
    if {$level > 1} {
	set name "[lindex [info level [expr $level-1]] 0]"
    }
    puts stderr "$name: $string"
    puts -nonewline $file "<!$name: $string >"
}





# font --
#
# This procedure is invoked to handle font changes in the text
# being output.
#
# Arguments:
#   type -    Type of font: R, I, B, or S.

proc font type {
    global state
    switch $type {
	C -
	B {
	    beginFont B
	    if {$state == "INSERT"} {
		set state REF
	    }
	}
	I {
	    beginFont I
	    if {$state == "REF"} {
		set state INSERT
	    }
	}
	P -
	R {
	    endFont
	    if {$state == "REF"} {
		set state INSERT
	    }
	}
	S {
	}
	default {
	    perror "Unknown: $type"
	}
    }
}



# beginFont --
#
# Arranges for future text to use a special font, rather than
# the default paragraph font.
#
# Arguments:
#   font -    Name of new font to use.

proc beginFont font {
    global curFont file

    if {$curFont == $font} {
	return
    }
    endFont
    puts -nonewline $file <$font>
    set curFont $font
}



# endFont --
#
# Reverts to the default font for the paragraph type.
#
# Arguments:
#   None.

proc endFont {} {
    global curFont file 

    if {$curFont != ""} {
	puts -nonewline $file </$curFont>
	set curFont ""
    }
}



# head --
#
# This procedure outputs header information for the current 
# paragraph. 
#
# Arguments:
#   None.

proc head {} {
}



# text --
#
# This procedure adds text to the current paragraph.  
#
# Arguments:
#   string -	Text to output in the paragraph.

proc text string {
    global file state inDT inPRE charCnt vertSpace

    set vertSpace 0
    incr charCnt [string length $string]
    regsub -all {&} $string {\&amp;}  string
    regsub -all {<} $string {\&lt;}   string
    regsub -all {>} $string {\&gt;}   string
    regsub -all {"} $string {\&quot;} string
    switch $state {
	REF { 
	    if {!$inDT && !$inPRE} {
		set string [insertRef $string]
	    }
	}
	SEE {
	    foreach i [split $string ,] {
		lappend str [insertRef [string trim $i]]
	    }
	    set string [join $str ", "]
	}
    }
    puts -nonewline $file "$string"
}



# insertRef --
#
#
# Arguments:
#   string -	    Text to output in the paragraph.

proc insertRef string {
    global NAME_file manFile package

    # does the string start with a word that matches a name in NAME_file?
    #
    regexp -nocase {[a-z_0-9]+} $string name
    if ![catch {set ref $NAME_file($name)} ] {
	if {$ref != "$package/$manFile"} {
	    set ref [split $ref /]
	    if {[lindex $ref 0] == $package} {
		set ref [lindex $ref 1]
	    } else {
		set ref "../[join $ref /]"
	    }
	    set string "<A HREF=\"$ref.html\">$string</A>"    
	}
    }
    return $string
}



# macro --
#
# This procedure is invoked to process macro invocations that start
# with "." (instead of ').
#
# Arguments:
#   name -   The name of the macro (without the ".").
#   args -   Any additional arguments to the macro.

proc macro {name args} {
#   perror "$name $args"
    global inDT
    set inDT 0
    switch $name {
	ad {}
	AP {
	    if {[llength $args] != 3} {
		perror ".AP: .$name [join $args " "]"
	    }
	    setTabs {1.25i 2.5i 3.75i}
	    TPmacro {}
	    font B
	    text "[lindex $args 0]  "
	    font I
	    text "[lindex $args 1]"
	    font R
	    text " ([lindex $args 2])"
	    newline
	}
	AS {}
	B {
	    global file
	    puts $file <B>$args</B>
	}
	BE {
	    global file inPRE
	    if $inPRE {			 ;# end of SYNOPSIS section
		puts $file </PRE>
		set inPRE 0
	    }
   	    para reset
	    puts $file "<hr>"
	}
	br {
	    global file inPRE charCnt
	    if !$inPRE {
		puts $file <BR>
	    } else {
		puts $file {}
	    }
	    set charCnt 0
	}
	BS {
	    global file
	    puts $file "<hr>"
	}
	
	CE -
	DE {
	    global file inPRE noFill
	    puts $file </PRE></BLOCKQUOTE>
	    set inPRE 0
	    set noFill 0
	}
	CS -
	DS {				    ;# code section
	    global file inPRE charCnt
	    puts -nonewline $file <BLOCKQUOTE><PRE>
	    set inPRE 1
	    set charCnt 0
	}
	fi {				    ;# fill
	    global noFill
	    set noFill 0
	}
	ft {				    ;#change font (for tclX)
	    switch $args {
		CW {
		    global file inPRE
		    puts -nonewline $file <PRE>
		    incr inPRE
		}
		R {
		    global file inPRE
		    puts $file </PRE>
		    if {$inPRE>0} {incr inPRE -1}
		}
		default {
		    perror "unknown .ft option: $args"
		}
	    }
	}
	HS {
	    HSmacro $args
	}
	if  {
	    if {[llength $args]>0} {
		perror ".if [join $args " "]"
	    }
	}
	IP {
	    IPmacro $args
	}
	LP -
	PP {
	    global file paraStk
	    if {[lindex $paraStk end] == "NEW"} {
		putPP
	    } else {
		para item "DL COMPACT" DT
	    }
	}
	ne {}
	nf {				    ;#no fill
	    global noFill charCnt
	    set noFill 1
	    set charCnt 0
	}
	OP {
	    OPmacro $args
	}
	RE {
	    para decr   
	}
	RS {
	    para incr
	}
	SE {
	    global state inPRE file

	    font R
	    puts -nonewline $file </PRE>
	    set inPRE 0
	    para reset
	    putPP
	    text "See the "
	    font B
	    set temp $state
	    set state REF
	    text options
	    set state $temp
	    font R
	    text " manual entry for detailed descriptions of the above options."
	    puts $file ""
	}
	SH {
	    SHmacro $args
	}
	SO {
	    global inPRE file charCnt

	    SHmacro "STANDARD OPTIONS"
	    setTabs {4c 8c 12c}
	    puts -nonewline $file <PRE>
	    set inPRE 1
	    font B
	}
	so {
	    if {$args != "man.macros"} {
		perror "so: .$name [join $args " "]"
	    }
	}
	sp {
	    global file inPRE vertSpace charCnt
	    if {$inPRE && !$vertSpace} {
		set vertSpace 1
		puts $file {}
#		if $charCnt {puts $file {}}
	    } else {
		putPP
	    }
            set charCnt 0
	}
	SS {				    ;# for tclX
	    global file
	    para reset
	    puts -nonewline $file <H4>
	    text $args
	    puts $file </H4>
	}
	ta {
	    setTabs $args
	}
	TH {
	    THmacro $args
	}
	TP {
	    TPmacro $args
	}
	UL {				    ;# underline
	    global file
	    puts -nonewline $file "<U><B>"
	    text [lindex $args 0]
	    puts -nonewline $file "</B></U>"
	    if {[llength $args] == 2} {
		set x [lindex $args 1]
		if {$x == "\t"} {
		    font B
		    tab
		    font R
		} else {
		    perror "UL macro: found 2nd arg not a tab"
		    text [lindex $args 1]
		}
	    }
	}
	VE {
	    global file inVS
	    while {$inVS>0} {
		puts -nonewline $file "</FONT>"
		incr inVS -1
	    }

	}
	VS {
	    global file inVS
	    if {[llength $args] > 0} {
		puts -nonewline $file "<BR>"
	    }
	    puts -nonewline $file "<FONT COLOR=\"007700\">"
	    incr inVS
	}
	default {
	    perror "Unknown: .$name [join $args " "]"
	}
    }
}


# formattedText --
#
# Insert a text string that may also have \fB-style font changes
# and a few other backslash sequences in it.
#
# Arguments:
#   text -    Text to insert.

proc formattedText text {
    while {$text != ""} {
	set index [string first \\ $text]
	if {$index < 0} {
	    text $text
	    return
	}
	text [string range $text 0 [expr $index-1]]
	set c [string index $text [expr $index+1]]
	switch -- $c {
	    f {
		font [string index $text [expr $index+2]]
		set text [string range $text [expr $index+3] end]
	    }
	    e {
		text \\
		set text [string range $text [expr $index+2] end]
	    }
	    - {
		dash
		set text [string range $text [expr $index+2] end]
	    }
	    | {
		set text [string range $text [expr $index+2] end]
	    }
	    default {
		perror "Unknown sequence: \\$c"
		set text [string range $text [expr $index+2] end]
	    }
	}
    }
}



# dash --
#
# This procedure is invoked to handle dash characters ("\-" in
# troff).  It outputs a special dash character.
#
# Arguments:
#   None.

proc dash {} {
    global state charCnt
    if {$state == "NAME"} {
	set state 0
    } elseif {$state == "REF"} {
	set state INSERT
    }
    incr charCnt
    text "-"
}



# tab --
#
# This procedure is invoked to handle tabs in the troff input.
#
# Arguments:
#   None.

proc tab {} {
    global file inPRE charCnt tabString
    if {$inPRE == 1} {
	set pos [expr $charCnt % [string length $tabString] ]
	set spaces [string first "1" [string range $tabString $pos end] ]
	text [format "%*s" [incr spaces] " "]
    } else {
#	perror "found tab outside of <PRE> block"
	set pos [expr $charCnt % [string length $tabString] ]
	set spaces [string first "1" [string range $tabString $pos end] ]
	for {set i 0} {$i <= $spaces} {incr i} {
	    puts -nonewline $file "&nbsp "
	}
    }
}



# setTabs --
#
# This procedure handles the ".ta" macro, which sets tab stops.
#
# Arguments:
#   tabList -   List of tab stops, each consisting of a number
#	       followed by "i" (inch) or "c" (cm).

proc setTabs {tabList} {
    global file tabString

    set last 0
    set tabString {}
    set charsPerInch 14.0
    set numTabs [llength $tabList]
    foreach arg $tabList {
	if {[scan $arg "%f%s" distance units] != 2} {
	    perror "bad distance units \"$arg\""
	    return 0
	}
	switch -- $units {
	    c {
#		perror "using $distance c"
		set distance [expr $distance * $charsPerInch / 2.54 ]
	    }
	    i {
		set distance [expr $distance * $charsPerInch]
	    }
	    default {
		perror "bad units in distance \"$arg\""
		continue
	    }
	}
	lappend tabString [format "%*s1" [expr round($distance-$last-1)] " "]
	set last $distance
    }
    set tabString [join $tabString {}]
}



# newline --
#
# This procedure is invoked to handle newlines in the troff input.
# It outputs either a space character or a newline character, depending
# on fill mode.
#
# Arguments:
#   None.

proc newline {} {
    global file inDT charCnt noFill inPRE

    if {$inDT} {		    ;# terminate dictionary term
	puts $file "\n<DD>"	    ;# start dictionary definition
	set inDT 0
    } elseif {$noFill && !$inPRE} {
	puts $file "<BR>"
    } else {
	puts $file {}
    }
    set charCnt 0
}



# char --
#
# This procedure is called to handle a special character.
#
# Arguments:
#   name -    Special character named in troff \x or \(xx construct.

proc char name {
    global file charCnt

    incr charCnt
    switch $name {
	{\ } -
	{\0} {					    ;#  \0
	    puts -nonewline $file " "
	}
	{\\} {					    ;#  \\
	    puts -nonewline $file "\\"
	}
	{\(+-} {				    ;#  +/-
	    puts -nonewline $file "&#177;"
	}
	{\|} {					    ;#  \|
	}
	{\%} {					    ;#  \%
	}
	{\&} {					    ;#  \&
	}
	default {
	    perror "Unknown character: $name"
	}
    }
}



# macro2 --
#
# This procedure handles macros that are invoked with a leading "'"
# character instead of space.  Right now it just generates an
# error diagnostic.
#
# Arguments:
#   name -    The name of the macro (without the ".").
#   args -    Any additional arguments to the macro.

proc macro2 {name args} {
    perror "Unknown macro: '$name [join $args " "]"
}



# HSmacro --

proc HSmacro {argList} {
    set arg0 [lindex $argList 0]
    set arg1 [lindex $argList 1]
    if {[llength $argList] < 3} {
	lappend argList { }	
    }
    set arg2 [lindex $argList 2]
#	iwid	{THmacro "$arg0 1 $arg2 Tk  {\[incr\ Widgets\]}"}

    switch $arg1 {
	tcl	{THmacro "$arg0 n $arg2 Tcl Tcl\ Built-In\ Commands"}
	tk	{THmacro "$arg0 n $arg2 Tk  Tk\ Commands"}
	tclc	{THmacro "$arg0 3 $arg2 Tcl Tcl\ Library\ Procedures"}
	tkc	{THmacro "$arg0 3 $arg2 Tk  Tk\ Library\ Procedures"}
	tclcmds	{THmacro "$arg0 1 $arg2 Tk  Tcl\ Applications"}
	tkcmds	{THmacro "$arg0 1 $arg2 Tk  Tk\ Applications"}
	iwid	{THmacro [list $arg0 1 $arg2 iwidgets {[incr Widgets]}]}
	cmds -
	lib  -
	ncmds {
	    THmacro $arg0 1 $arg2 [lindex $argList 3] [lindex $argList 4]
	}
    }
}




# IPmacro --
#
# This procedure is invoked to handle ".IP" macros, which may take any
# of the following forms:
#
# IPmacro	   Translate to a bulleted list (UL).
# IPmacro x	 Translate to a numbered list (OL). x is ignored.
# IPmacro text n    Translate to a dictionary list (DL). n is ignored.
#
# Arguments:
#   argList -	   List of arguments to the I macro.

proc IPmacro argList {
    global file

    setTabs 0.5i
    set length [llength $argList]
    if {$length == 0} {
#	para item UL LI
	para item "DL COMPACT" DT
	puts $file "\n<DD>"
	return
    }
    set string [string trim [lindex $argList 0]]
    if {[regexp {^\[([0-9])+\]$} $string]} {
	para item OL LI
	return
    } else {
	para item "DL COMPACT" DT
	formattedText [lindex $argList 0]
	puts $file "\n<DD>"
	return
    }
    perror "[join $argList " "]"
}



# OPmacro --
#
# This procedure is invoked to handle ".OP" macros. 
#
# Arguments:
#   argList -	List of arguments to the macro. (Command-Line
#		Name, Database Name & Database Class)

proc OPmacro argList {
    global inDT file inPRE
    if {[llength $argList] != 3} {
	perror ".OP: .$name [join $argList " "]"
    }

    regsub -all {\\-} [lindex $argList 0] - x

    para item "DL COMPACT" DT

    puts $file "<TABLE CELLSPACING=0 CELLPADDING=0 border=0>"
    puts $file "<TR>"
    puts $file "<TD width=144>Command-Line Name:</TD>"
    puts $file "<TD><B>$x</B></TD>"
    puts $file "</TR>"

    puts $file "<TR>"
    puts $file "<TD>Database Name:</TD>"
    puts $file "<TD><B>[lindex $argList 1]</B></TD>"
    puts $file "</TR>"

    puts $file "<TR>"
    puts $file "<TD>Database Class:</TD>"
    puts $file "<TD><B>[lindex $argList 2]</B></TD>"
    puts $file "</TR>"
    puts $file "</TABLE>"

    set inDT 1			;# next newline writes <DD> 
    newline
}



# SHmacro --
#
# Subsection head; handles the .SH macro.
#
# Arguments:
#   name -  Section name.

proc SHmacro argList {
    global file state noFill inPRE vertSpace charCnt

    set args [join $argList " "]
    if {[llength $argList] < 1} {
	perror ".SH: .$name $args"
    }

    if $inPRE {			 ;# end of SYNOPSIS section
	puts $file </PRE>
	set inPRE 0
    }

    para reset
    set state 0
    set noFill 0

    puts -nonewline $file "<H3>"
    text $args
    puts $file "</H3>"


    # control what the text proc does with text
    #
    switch $args {
	NAME {set state NAME}
	SYNOPSIS {
	    set state 0
	    puts -nonewline $file <PRE>
	    set inPRE 1		 ;# cleared at next section
	}
	"SEE ALSO" {set state SEE}
	default {set state INSERT}
    }
    set vertSpace 1
    set charCnt 0
}


# THmacro --
#
# This procedure handles the .TH macro.  For man pages, it generates 
# the non-scrolling header section for a given man page. For the html
# pages, it sets a five element array corresponding to its five arguments.
# The THmacro also writes the first few lines of html for the page.
# The .TH macro has the following form:
#
# THmacro fname page vers lib pname
#
# Arguments:
#   argList -	List of 5 arguments to the .TH macro.

proc THmacro {argList} {
    global file THinfo

    if {[llength $argList] != 5} {
	set args [join $argList " "]
	perror "$args"
    }
    set THinfo(fname) [lindex $argList 0]      ;# Tcl_UpVar
    set THinfo(page)  [lindex $argList 1]      ;# 3
    set THinfo(vers)  [lindex $argList 2]      ;# 7.4
    set THinfo(lib)   [lindex $argList 3]      ;# Tcl
    set THinfo(pname) [lindex $argList 4]      ;# {Tcl Library Procedures}

    puts -nonewline $file "<HTML><HEAD><TITLE>"
    text "$THinfo(lib) - $THinfo(fname) ($THinfo(page))"
    puts $file "</TITLE></HEAD><BODY>\n"

    puts -nonewline $file "<H2><CENTER>"
    text $THinfo(pname)
    puts $file "</CENTER></H2>\n"
}



# TPmacro --
#
# This procedure is invoked to handle ".TP" macros, which may take any
# of the following forms:
#
# TPmacro x  Translate to a dictionary list (DL). n is ignored.
#
# TPmacro      Translate to a dictionary list (DL).
#
# Arguments:
#   argList -	List of arguments to the .TP macro (ignored).

proc TPmacro {argList} {
    global inDT
    para item "DL COMPACT" DT
    set inDT 1		  ;# next newline writes <DD>
    setTabs 0.5i
}



# para --
#
# This procedure takes care of inserting the tags associated with the
# IP, TP, RS, RE, LP and PP macros. Only 'para item' takes arguments.
#
# Arguments:
#   op -	    operation: item, incr, decr, reset, init
#   listStart -	begin list tag: OL, UL, DL.
#   listItem -	item tag:	LI, LI, DT.

proc para {op {listStart "NEW"} {listItem {} } } {
    global file paraStk charCnt
	
    switch $op {
	item {
            putPP
            set top [lindex $paraStk end]
            if {$top == "NEW" } {
                set paraStk [lreplace $paraStk end end $listStart]
                puts $file "<$listStart>"
             } elseif {$top != $listStart} {
                para decr
                lappend paraStk $listStart
                puts $file "<$listStart>"
            }
            puts $file "<$listItem>"
	}
	decr {
	    if {[llength $paraStk] == 0} {
		perror "decr: stack is empty"
		set paraStk NEW
	    }
	    set tag [lindex $paraStk end]
	    set paraStk [lreplace $paraStk end end]
	    if {$tag != "NEW"} {
		puts $file "</$tag>"
	    } else {
		puts $file "<DT>"
	    }
	}
	incr {			  ;# paired with & follows a decr
	    para debug RS
 	    if {[lindex $paraStk end] == "NEW" } {
# 		perror "incr with NEW at top of stack"
 		set paraStk [lreplace $paraStk end end "DL COMPACT"]
 		puts $file "<DL COMPACT>"
 	    }
	    puts $file "<DD>"
	   lappend paraStk NEW
	}
	reset {
	    while {[llength $paraStk] > 0} {
		para decr
	    }
	    set paraStk NEW
	}
	init {
	    set paraStk NEW
	}
	debug {
	    puts $file ""
	    puts  -nonewline $file "<!$listStart"
	    foreach i $paraStk {
		puts -nonewline $file " $i"
	    }
	    puts $file ">"
	}
    }
    set charCnt 0
}



## para --
#
# This procedure takes care of inserting the tags associated with the
# IP, TP, RS, RE, LP and PP macros. Only 'para item' takes arguments.
#
# Arguments:
#   op -	    operation: item, incr, decr, reset, init
#   listStart -	begin list tag: OL, UL, DL.
#   listItem -	item tag:	LI, LI, DT.

proc para_old {op {listStart "NEW"} {listItem {} } } {
    global file paraStk charCnt
	
    switch $op {
	item {
	    putPP
	    set top [lindex $paraStk end]
	    if {$top == "NEW" } {
		set paraStk [lreplace $paraStk end end $listStart]
		puts $file "<$listStart>"
	     } elseif {$top != $listStart} {
		para decr
		lappend paraStk $listStart
		puts $file "<$listStart>"
	    }
	    puts $file "<$listItem>"
	}
	decr {
	    if {[llength $paraStk] == 0} {
		perror "decr: stack is empty"
		set paraStk NEW
	    }
	    set tag [lindex $paraStk end]
	    if {$tag != "NEW"} {
		puts $file "</$tag>"
	    }
	    catch {set paraStk [lreplace $paraStk end end]}
	}
	incr {			  ;# paired with & follows a decr
 	    if {[lindex $paraStk end] == "NEW" } {
 		perror "incr with NEW at top of stack"
 		set paraStk [lreplace $paraStk end end "DL COMPACT"]
 		puts $file "<DL COMPACT>\n<DD>"
 		return
 	    }
	   lappend paraStk NEW
	}
	reset {
	    while {[llength $paraStk] > 0} {
		para decr
	    }
	    set paraStk NEW
	}
	init {
	    set paraStk NEW
	}
	debug {
	    puts  -nonewline $file "<!"
	    foreach i $paraStk {
		puts -nonewline $file " $i"
	    }
	    puts -nonewline $file ">"
	}
    }
    set charCnt 0
}



# do --
#
# This is the toplevel procedure that generates an HTML page.
# It sources $fileName.tcl (which was generated during pass1) 
# to generate the HTML file.
#
# Arguments:
#   fileName -	Name of the file to translate.
#   html_dir -	HTML pages are stored in this directory.

proc do {fileName html_dir} {
    global file manFile package footer THinfo

    set manFile "[file tail $fileName]"
    set file [open "$html_dir/$package/$manFile.html" w]
    puts "  Pass 2 -- $fileName"
    update
    initGlobals
    if [catch {
	eval [man2tcl $fileName]
#      source $fileName.tcl
#      file delete $fileName.tcl
    } msg] {
	global errorInfo
	puts stderr $msg
	puts "in"
	puts stderr $errorInfo
	close $file
	return
    }
    para reset
    macro VE
    putPP
    puts $file "<center>Last change: $THinfo(vers)</center>"
    puts $file $footer
    puts $file "</BODY></HTML>"
    close $file
}



# doPass2 --
#
# This is the toplevel procedure for pass 2 processing of a package. 
# Given a directory as argument, generate all the man pages in that 
# directory.
#
# Arguments:
#   dir -	   Name of the directory containing man pages.
#   html_dir -	  Name of the directory to save HTML pages

proc doPass2 {dir html_dir} {
    foreach f [lsort [glob [file join $dir/*.\[13n\]]]] {
	do $f $html_dir
	update
    }
}


