package provide mmucl 0.99.7

##########################################################################
# mmucl.tcl - The core mmucl library.
#
# Copyright (C) 1997 Mark Patton
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
##########################################################################

# Contents of global array Mmucl:
#  connection - whether or not connected to mud
#  history    - list containing record of input
#  histloc    - location of input in history
#  input      - string parsed by parse_input
#  host       - location of current mud
#  port       - port of current mud
#  char       - current character loaded by LoadChar if any
#  sockid     - socket identifier for connection to mud
#  rc_dir     - location of users charlist and character init files.
#  lib_dir    - location of libraries, help files etc.

# Some elements need to be set.
# rc_dir and lib_dir should be set by program using the library.
# The other elements will be set by various procedures.

array set Mmucl {
    connection 0 
    histloc    0     
    history    {}
}

# Internal procs that should not be called from the command line.
#################################################################

# echo and display are callbacks that need to be defined.
#  display displays mud output and may do special formatting.
#  echo just displays the given string.

# get output from mud, check it for actions and display it.
# if an action does match do substitution for @n
proc read_mud {} {
    global action_array
    upvar \#0 Mmucl(sockid) sockid

    if {[catch {set mudout [read $sockid]} error]!=0} {
	echo "** $error **"
	disconnect
	return
    }    
    if {[eof $sockid]} {
        disconnect
        return
    }  
    
    display $mudout

    foreach pattern [array names action_array] {
	if {[regexp -nocase $pattern $mudout 0 1 2 3 4 5 6 7 8 9]} {
	    regsub -all {[\$]} [join $action_array($pattern)] {\\&} cmds
	    regsub -all {@([0-9])} $cmds {$\1} cmds
	    if {[catch {uplevel \#0 [subst -nocommands $cmds]} error]!=0} {
		echo "** Error: $error **"
	    }   
	}
    }
}
    
# Check input to see what should be done with it.
#  \ means send it to the mud
#  # means input is a tcl command.
# input is considered to consist of commands seperated
# by ;. Each command is checked to see if it matches an
# alias and if so substitution of n arg for %n is done.

# Have to do a special case for "" (return) because
# the split eats it up.
# Maybe the history stuff should be changed to use
# the built in history command.

proc parse_input {} {
    global alias_array
    upvar \#0 Mmucl(input) input Mmucl(history) history Mmucl(histloc) loc

    set fchar [string index $input 0]
    if {$fchar=="\\"} {
	mudputs [string trimleft $input \\]
    } elseif {$fchar=="#"} {
	if {[catch {uplevel \#0 [string trimleft $input \#]} error]!=0} {
	    echo "** Error: $error **"
	}
    } elseif {$input==""} {
 	mudputs ""
    } else {
	foreach cmd [split $input \;] {
	    set first [string range $cmd 0 [expr {[string wordend $cmd 0]-1}]]
	    if {[info exists alias_array($first)]} {
		regsub -all %0 [join $alias_array($first)] [string range \
		  $cmd [expr {[string wordend $cmd 0]+1}] end] alias
		regsub -all {[\[]} $alias {\\&} alias
		regsub -all {%([1-9])} $alias \
		  "\[lindex [list $cmd] \\1]" alias
		if {[catch {uplevel \#0 [subst -nov $alias]} error]!=0} {
		    echo "** Error: $error **"
		}	  
	    } else {mudputs $cmd}
	}
    }

    lappend history $input
    set input ""
    set history [lreplace $history 0 [expr {$loc-30}]]
    set loc [llength $history]
}

# These are procs, some of which are used internally,
# which can be called by the user.
#######################################################

# connect to a mud. if given the string login is written
# to the mud on a successful connect.
proc connect {host port {login ""}} {
    global Mmucl

    if {$Mmucl(connection)} {
	echo "** Already connected. **"
	return
    }
    
    echo "** Attempting to connect... **"
    
    # Do the close and after cancel to make sure
    # we're only attempting one connection at a time.
    
    catch {close $Mmucl(sockid)}
    after cancel "if {!\$Mmucl(connection)} then \
	    {echo \"** Timed out. **\";disconnect}"
    
    # Set up a timeout.
    set afterid [after 20000 "if {!\$Mmucl(connection)} then \
	    {echo \"** Timed out. **\";disconnect}"]
    
    if {[catch {set Mmucl(sockid) [socket -async $host $port]} error]!=0} {
	after cancel $afterid
	echo "** Error: $error **"
    } else {
	fconfigure $Mmucl(sockid) -buffering line -blocking 0
	
	# Fileevent only get called once a connection.
	# It sets a variable the after commands check to see if we
	# connect in a specific period of time and time out if we don't..
	
	fileevent $Mmucl(sockid) readable "
	      after cancel $afterid
	      set Mmucl(connection) 1
              set Mmucl(host) $host
              set Mmucl(port) $port
              if {[string length $login]} {writemud $login}
              fileevent $Mmucl(sockid) readable read_mud"
    }
}

# disconnect from a mud
proc disconnect {} {
    upvar \#0 Mmucl(sockid) sockid Mmucl(connection) connect

    catch {close $sockid}
    set connect 0
    echo "** Connection closed. **"
}

# break up str into commands seperated by ; and
# write them to the mud.
proc writemud {str} {
    upvar \#0 Mmucl(sockid) sockid Mmucl(connection) connect
    
    if {!$connect} {
	echo "** Not connected. **"
    } else {
	foreach cmd [split $str \;] {
	    puts $sockid $cmd
	}
	flush $sockid
    }
}

# write str to mud
proc mudputs {str} {
    upvar \#0 Mmucl(sockid) sockid Mmucl(connection) connect
    
    if {!$connect} {
	echo "** Not connected. **"
    } else {
	puts $sockid $str
	flush $sockid
    }
}

# add or display aliases
proc alias {{name *} {str ""}} {
    global alias_array
    
    if {$str==""} {
	set result [array names alias_array $name]
	if {$result !=""} {
	    foreach match [lsort $result] {
		echo "** Alias $match: $alias_array($match) **"
	    }
	} else {
	    echo "** No alias matching: $name **"
	}
    } else {
	set alias_array($name) [list $str]
	echo "** Alias \"$name\" set to \{$str\} **"
    }
}

# delete an alias
proc unalias {name} {
    global alias_array
    
    set result [array names alias_array $name]
    if {$result !=""} {
	foreach match $result {
	    unset alias_array($match)
	    echo "** \"$match\" is no longer an alias. **"
	}
    } else {
	echo "** No alias matching: $name. **"
    }
}

# add or display actions
proc action {{pattern *} {cmds ""}} {
    global action_array
    
    if {$cmds==""} {
	set result [array names action_array $pattern]
	if {$result !=""} {
	    foreach match [lsort $result] {
		echo "** Action \{$match\}: $action_array($match) **"
	    }
	} else {
	    echo "** No action matching: $pattern **"
	}
    } else {
	# Check to see if the pattern is a valid regexp.
	if {[catch {regexp $pattern ""} error] !=0} {
	    echo "** Warning: $error"
	}
	set action_array($pattern) [list $cmds]
	echo "** Action \{$pattern\} triggers \{$cmds\} **"
    }
}

# delete an action
proc unaction {pattern} {
    global action_array

    # make sure pattern doesn't blow up in array names
    regsub -all {[[\]} $pattern {\\&} pattern
    
    set result [array names action_array $pattern]
    if {$result !=""} {
	foreach match $result {
	    unset action_array($match)
	    echo "** \{$match\} is no longer an action. **"
	}
    } else {
	echo "** No action matching: $pattern. **"
    }
}

# exec an alias and do the substitutions of args for %n
# args should be a valid list
proc exec_alias {name {args ""}} {
    global alias_array
    
    if {![info exists alias_array($name)]} {
	echo "** No alias $name defined. **"
	return
    }
   
    regsub -all %0 [join $alias_array($name)] [join $args] cmds
    regsub -all {%([1-9])} $cmds "\[lindex $args \[expr {\\1 -1}]]" cmds  
    
    if {[catch {uplevel \#0 $cmds} error]!=0} {
	echo "** Error: $error **"
    }	  
}       

# Add a character to the file charlist in rc_dir
proc charadd {character mudname location port {login ""}} {
    global character_array
    upvar \#0 Mmucl(rc_dir) rc_dir

    if {[file exists [file join $rc_dir charlist]]} {
	source [file join $rc_dir charlist]
    }

    set character_array($character) [list $mudname $location $port $login]
    set fileid [open [file join $rc_dir charlist] w]
    puts $fileid "array set character_array \"[array get character_array]\"" 
    flush $fileid
    close $fileid
    
    if {[catch {set fileid [open [file join $rc_dir \
			 $character.$mudname] {WRONLY CREAT}]} error]!=0} {
	echo "** Error: $error **"
	return
    }
    close $fileid

    echo "** Created character $character. **"
    echo "** Init file: [file join $rc_dir $character.$mudname] **"   
}

# Connect to defined character's mud and load his init file.
proc charload {name} {
     upvar \#0 Mmucl(rc_dir) rc_dir Mmucl(char) char
    
    if {![file exists [file join $rc_dir charlist]]} {
	echo "** No characters defined. **"
	return
    }
    source [file join $rc_dir charlist]
    if {![info exists character_array($name)]} {
	echo "** No character \"$name\" defined. **"
	return
    }
    set charlist $character_array($name)
    
    if {[file exists [file join $rc_dir $name.[lindex $charlist 0]]]} {
	source [file join $rc_dir $name.[lindex $charlist 0]]
    }
    connect [lindex $charlist 1] [lindex $charlist 2] [lindex $charlist 3]
    set char $name
}

# delete a character
proc chardelete {name} {
    global character_array
    upvar \#0 Mmucl(rc_dir) rc_dir

    if {![file exists [file join $rc_dir charlist]]} {
	echo "** No characters defined in charlist. **"
    }

    if {![info exists character_array($name)]} {
	echo "** No defined character $name. **"
    }

    source [file join $rc_dir charlist]
    set charlist $character_array($name)
    unset character_array($name)
    set fileid [open [file join $rc_dir charlist] w]
    puts $fileid "array set character_array \"[array get character_array]\"" 
    flush $fileid
    close $fileid
    catch {file delete [file join $rc_dir $name.[lindex $charlist 0]]}
    echo "** Character $name deleted. **"
}

# pop up a local html browser.
proc help {{subject contents}} {
    upvar \#0 Mmucl(lib_dir) lib_dir
    package require html_browse

    set file [file join $lib_dir html $subject.html]
    
    if {[file exists $file]} {
	HM_browse $file
    } else {
	HM_browse contents.html
    }
}

# save currently defined aliases and actions to a file.
proc save {filename} {
    global alias_array action_array

    if {[catch {set fileid [open $filename w]} error]!=0} {
	echo "** Error: $error **"
	return
    }
    foreach alias [array names alias_array] {
	puts $fileid "alias \{$alias\} $alias_array($alias)"
    }    
    foreach action [array names action_array] {
	puts $fileid "action \{$action\} $action_array($action)"
    }
    flush $fileid
    close $fileid
    echo "** All defined aliases and actions saved in $filename. **"
}

# read a file directly into the mud.
proc textin {filename} {
    upvar \#0 Mmucl(sockid) sockid

    if {[catch {set fileid [open $filename r]} error]!=0} {
	echo "** $error **"
    } else {
	puts $sockid [read $fileid]
    }
}
