# copyright (C) 1997-98 Jean-Luc Fontaine (mailto:jfontain@mygale.org)
# this program is free software: please refer to the BSD type license enclosed in this package

set rcsId {$Id: datatab.tcl,v 1.30 1998/04/29 22:52:34 jfontain Exp $}

class dataTable {                       ;# module data view in the form of a table, that can be sorted using a column as a reference

    set dataTable::(scrollbarBorderWidth) [expr {$widget::(default,ScrollbarBorderWidth)==0?0:1}]
    set dataTable::(scrollbarWidth) [expr {2*$widget::(default,ScrollbarWidth)/3}]

    proc dataTable {this parentPath args} composite {[new scroll table $parentPath] $args} {

        widget::configure $composite::($this,base) horizontal\
            -width $dataTable::(scrollbarWidth) -borderwidth $dataTable::(scrollbarBorderWidth)
        widget::configure $composite::($this,base) vertical\
            -width $dataTable::(scrollbarWidth) -borderwidth $dataTable::(scrollbarBorderWidth)

        set path $composite::($composite::($this,base),scrolled,path)
        # only allow interactive colun resizing
        # use arrow cursor instead of default insertion cursor, meaningless since cell editing is disabled
        $path configure -font $font::(mediumNormal) -rows 2 -state disabled -titlerows 1 -roworigin -1 -colstretchmode unset\
            -variable dataTable::${this}data -resizeborders col -cursor {} -bordercursor sb_h_double_arrow
        $path tag configure select -background white

        bindtags $path "$path . all"           ;# remove all class bindings for we do not use any and they would cause interferences

        # allow dragging from data array
        set dataTable::($this,drag) [new dragSite -path $path -validcommand "dataTable::validateDrag $this"]
        dragSite::provide $dataTable::($this,drag) DATACELLS "dataTable::dragData $this"

        set dataTable::($this,selector) [new tableSelector -selectcommand "dataTable::setCellsState $this"]
        bind $path <ButtonRelease-1> "dataTable::buttonRelease $this %x %y"
        bind $path <Control-ButtonRelease-1> "dataTable::toggleSelection $this %x %y"
        bind $path <Shift-ButtonRelease-1> "dataTable::extendSelection $this %x %y"

        # allow border resizing with first button. does not interfere with drag bindings since command does nothing unless mouse
        # click occured in a column border, which cannot be the case when dragging (see drag validation procedure in this class)
        bind $path <ButtonPress-1> "dataTable::buttonPress $this %x %y"
        bind $path <Button1-Motion> {%W border dragto %x %y}

        set dataTable::($this,sortOrder) increasing
        set dataTable::($this,tablePath) $path
        composite::complete $this
    }

    proc ~dataTable {this} {
        variable ${this}data

        catch {unset ${this}data}                                                                      ;# eventually free table data
        catch {eval delete $dataTable::($this,arrow) $dataTable::($this,tips)}
        delete $dataTable::($this,drag) $dataTable::($this,selector)
    }

    proc options {this} {
        return [list\
            [list -columns columns Columns 0 0]\
            [list -titlefont titleFont TitleFont $font::(mediumBold) $font::(mediumBold)]\
            [list -data data Data {} {}]\
        ]
    }

    proc set-columns {this value} {
        $dataTable::($this,tablePath) configure -cols $value
    }

    proc set-titlefont {this value} {
        if {[string length $composite::($this,-data)]==0} return                                         ;# nothing to do if no data
        upvar $composite::($this,-data) data

        set path $dataTable::($this,tablePath)
        if {![winfo exists $path.0]} {
            createTitles $this
        }
        for {set column 0} {$column<$data(columns)} {incr column} {
            $path.$column.label configure -font $value
        }
    }

    proc set-data {this value} {                                                ;# value must be a fully qualified module data array
        upvar $value data

        if {![array exists $value]} {
            error "\"$value\" argument is not an existing array"
        }
        if {![winfo exists $dataTable::($this,tablePath).0]} {
            createTitles $this
        }
        set dataTable::($this,sortColumn) [lindex $data(sort) 0]
        set dataTable::($this,sortOrder) [lindex $data(sort) 1]
        trace variable ${value}(updates) w "dataTable::update $this"                                    ;# track module data updates
    }

    proc buttonRelease {this x y} {
        if {!$dataTable::($this,columnBorderHit)} {                        ;# if column was resized, do not interfere with selection
            set number [expr {[$dataTable::($this,tablePath) cget -rows]-1}]                        ;# calculate number of data rows
            if {$number==0} return
            scan [$dataTable::($this,tablePath) index @$x,$y] %d,%d row column
            if {$row<0} return                                                                           ;# title row, nothing to do
            selector::select $dataTable::($this,selector) $row,$column
        }
        unset dataTable::($this,columnBorderHit)
    }

    proc columnSort {this column} {                        ;# sort table rows using the column that the user selected as a reference
        if {$column==$dataTable::($this,sortColumn)} {                                                       ;# sort the same column
            if {[string compare $dataTable::($this,sortOrder) increasing]==0} {                                ;# but toggle sorting
                set dataTable::($this,sortOrder) decreasing
            } else {
                set dataTable::($this,sortOrder) increasing
            }
        } else {                                                                ;# sort for the first time or for a different column
            set dataTable::($this,sortColumn) $column
            set dataTable::($this,sortOrder) increasing
        }
        selector::clear $dataTable::($this,selector)       ;# deselect all cells since reordering rows renders selection meaningless
        update $this                                                                                     ;# update table immediately
    }

    proc update {this args} {                                   ;# update display using module data. ignore eventual trace arguments
        if {[string length $composite::($this,-data)]==0} return                                         ;# nothing to do if no data

        variable ${this}data
        upvar $composite::($this,-data) data

        set path $dataTable::($this,tablePath)

        set cursor [$path cget -cursor]                                                                               ;# save cursor
        $path configure -cursor watch                                                                  ;# show user that we are busy
        ::update idletasks

        set sortColumn $dataTable::($this,sortColumn)
        set lists {}
        foreach name [array names data *,0] {
            scan $name %u dataRow
            lappend lists [list $dataRow $data($dataRow,$sortColumn)]
        }
        # sort data rows according to sort column (column numbering is identical for table data and source data)
        set lists [lsort -$dataTable::($this,sortOrder) -$data($sortColumn,type) -index 1 $lists]

        set selector $dataTable::($this,selector)
        set columns $data(columns)
        set changed 0                                                        ;# keep track of whether any rows were added or removed
        set row 0
        set rows {}
        foreach pair $lists {
            set dataRow [lindex $pair 0]
            set exists [info exists ${this}data($row,-1)]
            if {![info exists ${this}data($row,-1)]} {                                                            ;# gather new rows
                lappend rows $row
            }
            set ${this}data($row,-1) $dataRow                                             ;# keep track of table / data rows mapping
            for {set column 0} {$column<$columns} {incr column} {
                set ${this}data($row,$column) $data($dataRow,$column)
            }
            incr row
        }
        $path configure -rows [expr {$row+1}]                                           ;# fit to data (take into account title row)

        if {[llength $rows]>0} {                                                                      ;# one or more rows were added
            set changed 1
            set cells {}
            foreach new $rows {
                for {set column 0} {$column<$columns} {incr column} {
                    lappend cells $new,$column
                }
            }
            selector::add $selector $cells                                               ;# make selector aware of new cells at once
        }

        set rows {}
        while {[info exists ${this}data($row,-1)]} {                                                          ;# gather removed rows
            lappend rows $row
            incr row
        }
        if {[llength $rows]>0} {                                                                    ;# one or more rows were removed
            set changed 1
            set cells {}
            foreach old $rows {
                unset ${this}data($old,-1)
                for {set column 0} {$column<$columns} {incr column} {
                    lappend cells $old,$column
                    unset ${this}data($old,$column)
                }
            }
            selector::remove $selector $cells                                        ;# make selector aware of removed cells at once
        }

        if {$changed} {
           selector::clear $selector                   ;# deselect all cells since new or deleted rows renders selection meaningless
        }

        updatedSortingArrow $this

        $path configure -cursor $cursor                                                                            ;# restore cursor
        ::update idletasks
    }

    proc dragData {this format} {
        variable ${this}data

        set data $composite::($this,-data)
        set list {}
        foreach cell [selector::selected $dataTable::($this,selector)] {
            scan $cell %d,%d row column
            lappend list ${data}([set ${this}data($row,-1)],$column)                        ;# data cell format is array(row,column)
        }
        return $list
    }

    proc validateDrag {this x y} {
        if {$dataTable::($this,columnBorderHit)} {
            return 0                                                                              ;# resizing a column: prevent drag
        }
        if {[llength [selector::selected $dataTable::($this,selector)]]>0} {
            return 1                                                                                   ;# at least one selected cell
        } else {
            return 0                                                                                              ;# nothing to drag
        }
    }

    proc setCellsState {this cells select} {
        if {$select} {
            eval $dataTable::($this,tablePath) tag cell select $cells
        } else {
            eval $dataTable::($this,tablePath) tag cell {{}} $cells
        }
    }

    proc toggleSelection {this x y} {
        set cell [$dataTable::($this,tablePath) index @$x,$y]
        scan $cell %d row
        if {$row<0} return                                                                         ;# prevent selection on title row
        selector::toggle $dataTable::($this,selector) $cell
    }

    proc extendSelection {this x y} {
        set cell [$dataTable::($this,tablePath) index @$x,$y]
        scan $cell %d row
        if {$row<0} return                                                                         ;# prevent selection on title row
        selector::extend $dataTable::($this,selector) $cell
    }

    proc updatedSortingArrow {this} {
        set path $widget::($dataTable::($this,arrow),path)
        set column $dataTable::($this,sortColumn)

        set label $dataTable::($this,tablePath).$column.label      ;# copy title label bindings for contextual help and mouse action
        foreach event {<Enter> <Leave> <ButtonRelease-1>} {
            bind $path $event [bind $label $event]
        }
        if {[string compare $dataTable::($this,sortOrder) increasing]==0} {
            widget::configure $dataTable::($this,arrow) -direction down
        } else {
            widget::configure $dataTable::($this,arrow) -direction up
        }
        # place arrow in sorted column title frame on the right side of label
        place $path -in $dataTable::($this,tablePath).$column -anchor e -relx 1 -rely 0.5 -relheight 1
    }

    proc createTitles {this} {
        upvar $composite::($this,-data) data

        set path $dataTable::($this,tablePath)
        set font $composite::($this,-titlefont)
        for {set column 0} {$column<$data(columns)} {incr column} {                 ;# create table title labels in separate windows
            set frame [frame $path.$column]                       ;# use a frame as a container for label and eventual sorting arrow
            set label [label $path.$column.label -font $font -text $data($column,label)]
            place $label -relwidth 1 -relheight 1         ;# use placer so that sorting arrow can eventually be displayed over label
            $path window configure -1,$column -window $frame -padx 2 -pady 2 -sticky nsew
            bind $label <ButtonRelease-1> "dataTable::columnSort $this $column"
            # setup context sensitive help on titles using help strings from module data
            bind $label <Enter> "lifoLabel::push $::messenger [list $data($column,message)]"
            bind $label <Leave> "lifoLabel::pop $::messenger"
            lappend dataTable::($this,tips) [new widgetTip -path $label -text {click to toggle sort}]
        }
        $path configure -cols $column                                                              ;# fit table to number of columns
        if {![info exists dataTable::($this,arrow)]} {                                    ;# use 1 sorting arrow indicator per table
            set arrow [new arrowButton $path -state disabled -borderwidth 0 -highlightthickness 0 -width 12]
            widget::configure $arrow -disabledforeground [widget::cget $arrow -foreground]               ;# make arrow fully visible
            lappend dataTable::($this,tips) [new widgetTip -path $widget::($arrow,path) -text {click to toggle sort}]
            set dataTable::($this,arrow) $arrow
        }
    }

    proc buttonPress {this x y} {
        foreach {row column} [$dataTable::($this,tablePath) border mark $x $y] {}
        set dataTable::($this,columnBorderHit) [expr {[info exists column]&&([string length $column]>0)}]
    }

}
