#!/bin/sh
# the next line restarts using the interpreter \
exec wish "$0" "$@"

# copyright (C) 1997-98 Jean-Luc Fontaine (mailto:jfontain@mygale.org)
# this program is free software: please refer to the BSD type license enclosed in this package

set rcsId {$Id: moodss.tcl,v 1.76 1998/04/30 20:58:42 jfontain Exp $}


source getopt.tcl

proc printUsage {exitCode} {
    puts stderr "usage: $::argv0 \[-update seconds\] module \[module\ ...]"
    exit $exitCode
}

set arguments(-update) {}                                                                                    ;# set default as empty
if {[catch {set argv [parseCommandLineArguments {-update 1} $argv arguments]} message]} {
    puts stderr $message
    printUsage 1
}

if {[llength $argv]<1} {
    printUsage 1
}


proc loadModules {modules} {
    foreach module $modules {
        if {[info exists loaded($module)]} {
            puts stderr "$::argv0: module \"$module\" was specified more than once"
            exit 1
        }
        if {[catch {package require $module}]} {
            puts stderr "$::argv0: could not load package \"$module\""
            exit 1
        }
        set loaded($module) {}
        if {![info exists ${module}::data(indexColumns)]} {
            set ${module}::data(indexColumns) 0                                ;# use first column as single index column by default
        }
    }
}

proc setPollTimes {modules {commandLineTime {}}} {
    global pollTimes pollTime

    set default 0
    set minimum 0
    foreach module $modules {
        set times [set ${module}::data(pollTimes)]
        if {[llength $times]==0} {
            error "module $module poll times list is empty"
        }
        set time [lindex $times 0]
        if {$time>$default} {                                                                  ;# default value is the first in list
            set default $time                                                        ;# keep the greater default time of all modules
        }
        set times [lsort -integer $times]                                                                         ;# sort poll times
        set time [lindex $times 0]
        if {$time>$minimum} {
            set minimum $time                                                        ;# keep the greater minimum time of all modules
            set minimumModule $module
        }
        foreach time $times {                                        ;# poll times list is the combination of all modules poll times
            set data($time) {}
        }
    }
    set pollTimes [lsort -integer [array names data]]         ;# sort and restrict poll times above maximum module minimum poll time
    set pollTimes [lrange $pollTimes [lsearch -exact $pollTimes $minimum] end]
    set pollTime $default
    if {[string length $commandLineTime]>0} {                                           ;# eventually validate command line override
        if {$commandLineTime<$minimum} {
            puts stderr "$::argv0: minimum poll time is $minimum seconds for module $minimumModule"
            exit 1
        }
        set pollTime $commandLineTime
    }
}

proc modulesString {} {
    global modules

    for {set index 0} {$index<([llength $modules]-1)} {incr index} {
        append string "[lindex $modules $index], "
    }
    append string [lindex $modules $index]
    return $string
}

lappend auto_path .                                                   ;# search in current directory sub-directories for development
set modules $argv
loadModules $modules
setPollTimes $modules $arguments(-update)
set modulesString [modulesString]                                                      ;# store often used concatenated module names

if {[catch {package require stooop 3.5.1}]} {
    source stooop.tcl                                                                     ;# in case stooop package is not installed
}
namespace import stooop::*
if {[catch {package require switched 1.4}]} {                                           ;# in case switched package is not installed
    source switched.tcl
}

if {[catch {package require scwoop 2.2}]} {
    source scwoop.tcl                                                                     ;# in case scwoop package is not installed
    source bindings.tcl
    source widgetip.tcl
    source arrowbut.tcl
    source spinent.tcl
    source panner.tcl
}

if {[catch {package require Tktable 2.0}]} {                                             ;# in case tkTable package is not installed
    switch $tcl_platform(platform) {
        unix {
            load ./Tktable.so.2.00
        }
        windows {
            load ./tktable.dll
        }
    }
}

if {[catch {package require BLT}]} {                                                         ;# in case BLT package is not installed
    switch $tcl_platform(platform) {
        unix {                                                              ;# try version 2.3 patched for Tcl 8.1 or unofficial 8.0
            if {[catch {load ./libBLT2.3.so} message23]&&[catch {load ./libBLT8.0.so} message80]} {
                puts $message23
                puts $message80
                exit 1
            }
        }
        windows {
            load ./blt80.dll                                                  ;# so far, only unofficial 8.0 is supported on Windows
        }
    }
}
set officialBLT [expr {$blt_version==2.3}]                                               ;# remember which BLT version is being used

if {[catch {package require tkpiechart 4.3}]} {                                       ;# in case tkpiechart package is not installed
    source pielabel.tcl
    source boxlabel.tcl
    source canlabel.tcl
    source labarray.tcl
    source perilabel.tcl
    source slice.tcl
    source selector.tcl
    source objselec.tcl
    source pie.tcl
}

source utility.tcl
source config.tcl
source font.tcl
source scroll.tcl
source lifolbl.tcl
source keyslink.tcl
source dialog.tcl
source viewer.tcl
source blt2d.tcl
source tablesel.tcl
source datatab.tcl
source datagraf.tcl
source databar.tcl
source datapie.tcl
source htmllib.tcl                                                ;# Tcl HTML library from Sun, used for viewing HTML help documents
source htmldata.tcl
source html.tcl                                             ;# must be source after HTML library since some peocedures are redefined
source help.tcl
source drag.tcl
source drop.tcl
source canvhand.tcl


proc updateTitle {} {
    global modulesString pollTime

    wm title . "moodss: $modulesString data (every $pollTime seconds)"                 ;# show modules and poll time in window title
}

proc createMenuWidget {parentPath} {
    global modules

    set menu [menu $parentPath.menu -tearoff 0]

    $menu add cascade -label File -menu [menu $menu.file -tearoff 0] -underline 0
    $menu.file add command -label Exit -command exit -underline 1 -accelerator Alt+X
    bind $parentPath <Alt-x> exit

    $menu add cascade -label Options -menu [menu $menu.options -tearoff 0] -underline 0
    $menu.options add command -label {Poll time...} -command inquirePollTime -underline 0

    $menu add cascade -label Help -menu [menu $menu.help -tearoff 0] -underline 0
    $menu.help add command -label Global... -underline 0 -accelerator F1 -command helpWindow
    bind $parentPath <F1> helpWindow
    $menu.help add command -label Modules... -underline 0 -command "modulesHelpDialogBox $modules"
    $menu.help add command -label About... -underline 0 -command aboutDialogBox

    $parentPath configure -menu $menu
}

proc createMessageWidget {parentPath} {
    global messenger

    set messenger [new lifoLabel $parentPath -headerfont $font::(mediumBold) -font $font::(mediumNormal)]
    return $widget::($messenger,path)                                                                     ;# return actual tk widget
}

proc createDragAndDropZone {parentPath} {
    global canvas pollTime

    set frame [frame $parentPath.drops]

    set label [label $frame.graph -image [image create photo -data [dataGraph::iconData]]]
    pack $label -pady 1 -side left
    new dropSite -path $label -formats DATACELLS\
        -command "viewer::view \[new dataGraph $canvas -interval $pollTime\] \$dragSite::data(DATACELLS)"
    new widgetTip -path $label -text {graph drop site}

if {$::officialBLT} {
    set label [label $frame.overlapBarChart -image [image create photo -data [dataOverlapBarChart::iconData]]]
    pack $label -pady 1 -side left
    new dropSite\
        -path $label -formats DATACELLS -command "viewer::view \[new dataOverlapBarChart $canvas\] \$dragSite::data(DATACELLS)"
    new widgetTip -path $label -text {overlap bar chart drop site}
}

    set label [label $frame.sideBarChart -image [image create photo -data [dataSideBarChart::iconData]]]
    pack $label -pady 1 -side left
    new dropSite\
        -path $label -formats DATACELLS -command "viewer::view \[new dataSideBarChart $canvas\] \$dragSite::data(DATACELLS)"
    new widgetTip -path $label -text {side bar chart drop site}

    set label [label $frame.stackedBarChart -image [image create photo -data [dataStackedBarChart::iconData]]]
    pack $label -pady 1 -side left
    new dropSite -path $label -formats DATACELLS\
        -command "viewer::view \[new dataStackedBarChart $canvas\] \$dragSite::data(DATACELLS)"
    new widgetTip -path $label -text {stacked bar chart drop site}

    set label [label $frame.2DPieChart -image [image create photo -data [data2DPieChart::iconData]]]
    pack $label -pady 1 -side left
    new dropSite -path $label -formats DATACELLS\
        -command "viewer::view \[new data2DPieChart $canvas\] \$dragSite::data(DATACELLS)"
    new widgetTip -path $label -text {2D pie chart drop site}

    set label [label $frame.3DPieChart -image [image create photo -data [data3DPieChart::iconData]]]
    pack $label -pady 1 -side left
    new dropSite -path $label -formats DATACELLS\
        -command "viewer::view \[new data3DPieChart $canvas\] \$dragSite::data(DATACELLS)"
    new widgetTip -path $label -text {3D pie chart drop site}

    set trashData {
        R0lGODlhKAAoAKUAAP8A/8/Lz+/r71FRUZ6anpaSlv/7/wgICL66vkFBQYaChtfT17LA3LaytjAw
        MFlZWZaant/b3/fz96aipnFxcWlpaWFhYTAoKCAgICgoKOfj576yvklJSRAQEBAQCI6Slq6yrp6S
        nhgYGAAAAI6KjoaKhnl5ecfDx66ipiAoIOfr5ygwMJ6Slnlxce/j5zg4OL7Dx6aint/T38fLz4Z5
        hvf7/8/Tz66qtqairv/z939/f/fz79fLz0lRUXmChllRUSH5BAEAAAAALAAAAAAoACgAAAb+QIBw
        SBwGisikckkUDAhMYiEaNQwO06ggS00iDIXEQcFcLA5dpUDAaCgch8GDDIVEDJJJIp2URBQUFQoW
        CRcYGRkGihoBCnt8RQYbDxwZhx0eBwcYHyAMIRMDDpBEEwYPGBgiHSOtrh0PJCVhGaRDCw0JHq68
        vAkDJh0cfBIEHAoNGb3LvR0ODlBDJwsTSHcnBRaszNytAxMaa4oGJyRCAn4LBggMBAfd8BwQCxL1
        AiiPAOrjDRMU8PBSUDghIJyKFcOEBDAQIMKJCQqUAeQW54QEDREKjOACQIOGAhQ4iJgY0KIAFgco
        FFlIgoKDXSS7UZAggIKIJAJCtFjxLib+NwcuNLwYkARGDAovevpchkFGhBFLEJhIupTbDA2jlBho
        8K/qsg400CzJ80Cp11Ydhi6JEIBEgm1nW2HgkBDJFwQVHGCI6wqRWGs1ClRIMJLviAQmoCbBY4LD
        i1R8LUSwYSJJAAE3OhzogDBuBgISYNRCouFEhgHmAFCAWbUCugkvcKLCMYREA7MxNRjQMCFrkRMY
        og0xUCGFzwMcDORYMImYBhYJJHbTgUPGDow8eORKI2AHAkeqzB54QYG6ABkCFgSYESDACOFM0GGr
        8MCCBQo9FJgowAAHjjUCGODCAjxE0EAHFnSh2wkfkOADDSGwoIA/CTgwgH8MKKCAAC5S8MBAASLk
        w8QJBsgwAQEk0PBDYX4dgAMCrfg2AUqjUaGCAQsgQMIKRCHxT4K2EEEAOjZYIAJHRKQWJBEgCJCD
        kUtCgoJRHkQJCQkWbGTllrYEAQA7
    }
    set label [label $frame.trash -image [image create photo -data $trashData]]
    pack $label -pady 1 -side right
    new dropSite -path $label -formats OBJECTS -command "eval delete \$dragSite::data(OBJECTS)"
    new widgetTip -path $label -text {objects trash drop site}
    return $frame
}

proc inquirePollTime {} {
    global pollTimes pollTime

    set dialog\
        [new dialogBox . -buttons oc -default o -title {moodss: Poll time} -die 0 -x [winfo pointerx .] -y [winfo pointery .]]
    set frame [frame $widget::($dialog,path).frame]
    set minimum [lindex $pollTimes 0]
    set message [message $frame.message\
        -width [winfo screenwidth .] -font $font::(mediumNormal) -justify center\
        -text "Please enter new poll time\n(greater than $minimum):"\
    ]
    pack $message
    set entry [new spinEntry $frame -width 4 -list $pollTimes -side right]
    spinEntry::set $entry $pollTime
    pack $widget::($entry,path) -anchor e -side left -expand 1 -padx 2       ;# evenly pack entry and label together near the center
    pack [label $frame.label -text seconds] -anchor w -side right -expand 1 -padx 2
    dialogBox::display $dialog $frame
    widget::configure $dialog -command "
        set time \[spinEntry::get $entry\]
        if {\$time<$minimum} {                                                                    ;# check against minimum poll time
            bell
            $message configure -text {Please enter new poll time\n(must be greater than $minimum):}
        } else {                                                                                                ;# new time is valid
            if {\$time!=\$pollTime} {                                             ;# but check that it actually differs from current
                set pollTime \$time
                viewer::updateInterval \$time
                updateTitle
                refresh                                      ;# update immediately in case poll time was set to a much greater value
            }
            delete $dialog                                                                                     ;# destroy dialog box
        }
    "
    bind $frame <Destroy> "delete $entry"                                            ;# delete objects not managed by the dialog box
}

proc refresh {} {
    global updateEvent modulesString modules messenger pollTime

    catch {after cancel $updateEvent}                                                             ;# eventually cancel current event

    lifoLabel::push $messenger "launching $modulesString data update..."        ;# we do not know when data will actually be updated
    update idletasks
    foreach module $modules {
        ${module}::update                                                                   ;# ask module to update its dynamic data
    }
    lifoLabel::pop $messenger
    update idletasks                                                                      ;# restore original cursor at the earliest
    set updateEvent [after [expr {1000*$pollTime}] refresh]                                               ;# convert to milliseconds
}

createMenuWidget .
pack [createMessageWidget .] -side bottom -fill x
updateTitle

set view [new scroll canvas .]
set canvas $composite::($view,scrolled,path)
set width [winfo screenwidth .]
set height [winfo screenheight .]
$canvas configure -background white -width $width -height $height -scrollregion [list 0 0 $width $height]
wm geometry . 400x300

pack [createDragAndDropZone .] -fill x
pack $widget::($view,path) -fill both -expand 1

set x 0
set y 0
foreach module $modules {
    set table [new dataTable $canvas -data ${module}::data]
    new canvasHandles $canvas -path $widget::($table,path) -x $x -y $y
    incr x $configuration::(xWindowManagerInitialOffset)
    incr y $configuration::(yWindowManagerInitialOffset)
}

refresh                                                                                                ;# initialize refresh process
