/* Time open hash table. */

#include <stdio.h>
#include <string.h>
#include "Array.h"
#include "hashfuncs.h"
#include "ohashmap.h"
#include "ohashset.h"


/*
 * Measure resource usage.
 */

extern "C" {
#include <sys/time.h>
#include <sys/resource.h>
}

class Rusage {
  public:
    /* Start collecting usage */
    Rusage();

    /* Reset collection */
    void Reset();

    /* Show usage */
    double UserTime();
    double SystemTime();
  private:
    struct rusage start;
};

inline Rusage::Rusage() {
    getrusage(RUSAGE_SELF, &start);
}

inline void Rusage::Reset() {
    getrusage(RUSAGE_SELF, &start);
}

inline double Rusage::UserTime() {
    struct rusage u;

    getrusage(RUSAGE_SELF, &u);

    struct timeval result;
    result.tv_sec  = u.ru_utime.tv_sec  - start.ru_utime.tv_sec;
    result.tv_usec = u.ru_utime.tv_usec - start.ru_utime.tv_usec;

    return double(result.tv_sec) + double(result.tv_usec) / 1000000.0;
}

inline double Rusage::SystemTime() {
    struct rusage u;

    getrusage(RUSAGE_SELF, &u);

    struct timeval result;
    result.tv_sec  = u.ru_stime.tv_sec  - start.ru_stime.tv_sec;
    result.tv_usec = u.ru_stime.tv_usec - start.ru_stime.tv_usec;

    return double(result.tv_sec) + double(result.tv_usec) / 1000000.0;
}

declareOpenHashMap(IntMap,int,char const*,hash_int,cmp_int)
implementOpenHashMap(IntMap,int,char const*,hash_int,cmp_int)

declareOpenHashSet(IntSet,int,hash_int,cmp_int)
implementOpenHashSet(IntSet,int,hash_int,cmp_int)

declareOpenHashSet(StringSet,char const*,hash_string,cmp_string)
implementOpenHashSet(StringSet,char const *,hash_string,cmp_string)

declareArray(boolArray,bool)
implementArray(boolArray,bool)

static void time_fixed_store();
static void time_fixed_fetch();

static void time_array_grow();
static void time_array_grow_predicted();
static void time_array_replace();
static void time_array_fetch();
static void time_array_remove();

static void time_omap_grow();
static void time_omap_grow_predicted();
static void time_omap_replace();
static void time_omap_fetch();
static void time_omap_remove();

static void time_set_grow();
static void time_set_grow_predicted();
static void time_set_replace();
static void time_set_fetch();
static void time_set_remove();

static void time_sset_grow();
static void time_sset_grow_predicted();
static void time_sset_replace();
static void time_sset_fetch();
static void time_sset_remove();

static void str_init();

static const int count = 10000;

static void report(char const* title, double t) {
    printf("%-20s %8.2f usec\n",
	   title,
	   ((double) (t * 1000000.0 / count)));
}

static char const** strings;

int
main() {
    str_init();

    time_fixed_store();
    time_fixed_fetch();

    time_array_grow();
    time_array_grow_predicted();
    time_array_replace();
    time_array_fetch();
    time_array_remove();

    time_omap_grow();
    time_omap_grow_predicted();
    time_omap_replace();
    time_omap_fetch();
    time_omap_remove();

    time_set_grow();
    time_set_grow_predicted();
    time_set_replace();
    time_set_fetch();
    time_set_remove();

//    time_sset_grow();
//    time_sset_grow_predicted();
//    time_sset_replace();
//    time_sset_fetch();
//    time_sset_remove();

    return 0;
}

static void str_init() {
    /* Initialize strings */
    strings = new const char*[count];

    char* buffer = new char[count * 12];
    for (int i = 0; i < count; i++) {
	sprintf(buffer, "str%d", i);
	strings[i] = buffer;
	buffer += 12;
    }
}

static void time_omap_grow() {
    IntMap map;
    Rusage t;

    t.Reset();
    for (int i = 0; i < count; i++) {
	map.store(i, 0);
    }
    double ut = t.UserTime();

    report("omap_grow", ut);
}

static void time_omap_grow_predicted() {
    IntMap map(count);
    Rusage t;

    t.Reset();
    for (int i = 0; i < count; i++) {
	map.store(i, 0);
    }
    double ut = t.UserTime();

    report("omap_predict/grow", ut);
}

static void time_omap_replace() {
    IntMap map(count);
    Rusage t;

    for (int i = 0; i < count; i++) {
	map.store(i, 0);
    }

    t.Reset();
    for (i = 0; i < count; i++) {
	map.store(i, 0);
    }
    double ut = t.UserTime();

    report("omap_replace", ut);
}

static void time_omap_fetch() {
    IntMap map(count);
    Rusage t;

    for (int i = 0; i < count; i++) {
	map.store(i, 0);
    }

    t.Reset();
    for (i = 0; i < count; i++) {
	char const* val;
	map.fetch(i, val);
    }
    double ut = t.UserTime();

    report("omap_fetch", ut);
}

static void time_omap_remove() {
    IntMap map(count);
    Rusage t;

    for (int i = 0; i < count; i++) {
	map.store(i, 0);
    }

    t.Reset();
    for (i = 0; i < count; i++) {
	map.remove(i);
    }
    double ut = t.UserTime();

    report("omap_remove", ut);
}

static void time_set_grow() {
    IntSet set;
    Rusage t;

    t.Reset();
    for (int i = 0; i < count; i++) {
	set.insert(i);
    }
    double ut = t.UserTime();

    report("set_grow", ut);
}

static void time_set_grow_predicted() {
    IntSet set(count);
    Rusage t;

    t.Reset();
    for (int i = 0; i < count; i++) {
	set.insert(i);
    }
    double ut = t.UserTime();

    report("set_predict/grow", ut);
}

static void time_set_replace() {
    IntSet set(count);
    Rusage t;

    for (int i = 0; i < count; i++) {
	set.insert(i);
    }

    t.Reset();
    for (i = 0; i < count; i++) {
	set.insert(i);
    }
    double ut = t.UserTime();

    report("set_replace", ut);
}

static void time_set_fetch() {
    IntSet set(count);
    Rusage t;
    bool r;

    for (int i = 0; i < count; i++) {
	set.insert(i);
    }

    r = 1;
    t.Reset();
    for (i = 0; i < count; i++) {
	r ^= set.contains(i);
    }
    double ut = t.UserTime();

    report("set_fetch", ut);
}

static void time_set_remove() {
    IntSet set(count);
    Rusage t;

    for (int i = 0; i < count; i++) {
	set.insert(i);
    }

    t.Reset();
    for (i = 0; i < count; i++) {
	set.remove(i);
    }
    double ut = t.UserTime();

    report("set_remove", ut);
}

static void time_sset_grow() {
    StringSet set;
    Rusage t;

    t.Reset();
    for (int i = 0; i < count; i++) {
	set.insert(strings[i]);
    }
    double ut = t.UserTime();

    report("sset_grow", ut);
}

static void time_sset_grow_predicted() {
    StringSet set(count);
    Rusage t;

    t.Reset();
    for (int i = 0; i < count; i++) {
	set.insert(strings[i]);
    }
    double ut = t.UserTime();

    report("sset_predict/grow", ut);
}

static void time_sset_replace() {
    StringSet set(count);
    Rusage t;

    for (int i = 0; i < count; i++) {
	set.insert(strings[i]);
    }

    t.Reset();
    for (i = 0; i < count; i++) {
	set.insert(strings[i]);
    }
    double ut = t.UserTime();

    report("sset_replace", ut);
}

static void time_sset_fetch() {
    StringSet set(count);
    Rusage t;
    bool r;

    for (int i = 0; i < count; i++) {
	set.insert(strings[i]);
    }

    r = 1;
    t.Reset();
    for (i = 0; i < count; i++) {
	r ^= set.contains(strings[i]);
    }
    double ut = t.UserTime();

    report("sset_fetch", ut);
}

static void time_sset_remove() {
    StringSet set(count);
    Rusage t;

    for (int i = 0; i < count; i++) {
	set.insert(strings[i]);
    }

    t.Reset();
    for (i = 0; i < count; i++) {
	set.remove(strings[i]);
    }
    double ut = t.UserTime();

    report("sset_remove", ut);
}

static void time_fixed_store() {
    bool* map = new bool[count];
    Rusage t;

    t.Reset();
    for (int i = 0; i < count; i++) {
	map[i] = 0;
    }
    double ut = t.UserTime();

    report("fixed_store", ut);

    delete [] map;
}

static void time_fixed_fetch() {
    bool* map = new bool[count];
    Rusage t;
    bool r;

    for (int i = 0; i < count; i++) {
	map[i] = 0;
    }

    r = 1;
    t.Reset();
    for (i = 0; i < count; i++) {
	r ^= map[i];
    }
    double ut = t.UserTime();

    report("fixed_fetch", ut);

    delete [] map;
}

static void time_array_grow() {
    boolArray array;
    Rusage t;

    t.Reset();
    for (int i = 0; i < count; i++) {
	array.append(0);
    }
    double ut = t.UserTime();

    report("array_grow", ut);
}

static void time_array_grow_predicted() {
    boolArray array(count);
    Rusage t;

    t.Reset();
    for (int i = 0; i < count; i++) {
	array.append(0);
    }
    double ut = t.UserTime();

    report("array_predict/grow", ut);
}

static void time_array_replace() {
    boolArray array(count);
    Rusage t;

    for (int i = 0; i < count; i++) {
	array.append(0);
    }

    t.Reset();
    for (i = 0; i < count; i++) {
	array[i] = 1;
    }
    double ut = t.UserTime();

    report("array_replace", ut);
}

static void time_array_fetch() {
    boolArray array(count);
    Rusage t;
    bool r;

    for (int i = 0; i < count; i++) {
	array.append(0);
    }

    r = 1;
    t.Reset();
    for (i = 0; i < count; i++) {
	r ^= array[i];
    }
    double ut = t.UserTime();

    report("array_fetch", ut);
}

static void time_array_remove() {
    boolArray array(count);
    Rusage t;

    for (int i = 0; i < count; i++) {
	array.append(0);
    }

    t.Reset();
    for (i = 0; i < count; i++) {
	array.remove();
    }
    double ut = t.UserTime();

    report("array_remove", ut);
}

static void time_array_store() {
    boolArray array(count);
    Rusage t;

    t.Reset();
    for (int i = 0; i < count; i++) {
	array.append(0);
    }
    double ut = t.UserTime();

    report("store/array", ut);
}
