/* tkgButton.c
 * Modified version of tkButton.c (copyright below) 
 */

/* 
 * Copyright (c) 1990-1994 The Regents of the University of California.
 * Copyright (c) 1994-1995 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * SCCS: @(#) tkButton.c 1.128 96/03/01 17:34:49
 */

#include "tkgButton.h"
#include "tkPort.h"
#include "tkInt.h"

/*
 * A data structure of the following type is kept for each
 * widget managed by this file:
 */

typedef struct {
  Tk_Window tkwin;		/* Window that embodies the button.  NULL
				 * means that the window has been destroyed. */
  Display *display;		/* Display containing widget.  Needed to
				 * free up resources after tkwin is gone. */
  Tcl_Interp *interp;		/* Interpreter associated with button. */
  Tcl_Command widgetCmd;	/* Token for button's widget command. */
  int type;			/* Type of widget:  restricts operations
				 * that may be performed on widget.  See
				 * below for possible values. */

  /*
   * Information about what's in the button.
   */

  char *text;			/* Text to display in button (malloc'ed)
				 * or NULL. */
  int numChars;			/* # of characters in text. */
  int underline;		/* Index of character to underline.  < 0 means
				 * don't underline anything. */
  char *textVarName;		/* Name of variable (malloc'ed) or NULL.
				 * If non-NULL, button displays the contents
				 * of this variable. */
  Pixmap bitmap;		/* Bitmap to display or None.*/
  char *imageString;		/* Name of image to display (malloc'ed), or
				 * NULL. */
  Tk_Image image;		/* Image to display in window, or NULL if
				 * none. */
  char *iconside; 		/* Side of button occupied by bitmap 
				 * (left, right, top, bottom, background). (malloc'ed) */

  char *menuName;		/* Name of menu associated with widget.
				 * Malloc-ed. */

 
  /*
   * Information used when displaying widget:
   */

  Tk_Uid state;			/* State of button for display purposes:
				 * normal, active, or disabled. */
  Tk_3DBorder normalBorder;	/* Structure used to draw 3-D
				 * border and background when window
				 * isn't active.  NULL means no such
				 * border exists. */
  Tk_3DBorder activeBorder;	/* Structure used to draw 3-D
				 * border and background when window
				 * is active.  NULL means no such
				 * border exists. */
  int borderWidth;		/* Width of border. */
  int relief;			/* 3-d effect: TK_RELIEF_RAISED, etc. */
  int highlightWidth;		/* Width in pixels of highlight to draw
				 * around widget when it has the focus.
				 * <= 0 means don't draw a highlight. */
  XColor *highlightBgColorPtr;	/* Color for drawing traversal highlight
				 * area when highlight is off. */
  XColor *highlightColorPtr;	/* Color for drawing traversal highlight. */
  int inset;			/* Total width of all borders, including
				 * traversal highlight and 3-D border.
				 * Indicates how much interior stuff must
				 * be offset from outside edges to leave
				 * room for borders. */
  XFontStruct *fontPtr;		/* Information about text font, or NULL. */
  XColor *normalFg;		/* Foreground color in normal mode. */
  XColor *activeFg;		/* Foreground color in active mode.  NULL
				 * means use normalFg instead. */
  XColor *disabledFg;		/* Foreground color when disabled.  NULL
				 * means use normalFg with a 50% stipple
				 * instead. */
  GC normalTextGC;		/* GC for drawing text in normal mode.  Also
				 * used to copy from off-screen pixmap onto
				 * screen. */
  GC activeTextGC;		/* GC for drawing text in active mode (NULL
				 * means use normalTextGC). */
  Pixmap gray;			/* Pixmap for displaying disabled text if
				 * disabledFg is NULL. */
  GC disabledGC;		/* Used to produce disabled effect.  If
				 * disabledFg isn't NULL, this GC is used to
				 * draw button text or icon.  Otherwise
				 * text or icon is drawn with normalGC and
				 * this GC is used to stipple background
				 * across it.  For labels this is None. */
  GC copyGC;			/* Used for copying information from an
				 * off-screen pixmap to the screen. */
  char *widthString;		/* Value of -width option.  Malloc'ed. */
  char *heightString;		/* Value of -height option.  Malloc'ed. */
  int width, height;		/* If > 0, these specify dimensions to request
				 * for window in pixels. */
  int wrapLength;		/* Line length (in pixels) at which to wrap
				 * onto next line.  <= 0 means don't wrap
				 * except at newlines. */
  int padX, padY;		/* Extra space around contents of button. */
  int sep;	    		/* Extra space between text and image. */
  Tk_Anchor textanchor;		/* Where text should be displayed
				 * inside button region. */
  Tk_Anchor imageanchor;	/* Where bitmap should be displayed
				 * inside button region. */
  Tk_Justify justify;		/* Justification to use for multi-line text. */
  int textWidth;		/* Width needed to display text as requested,
				 * in pixels. */
  int textHeight;		/* Height needed to display text as requested,
				 * in pixels. */
  /*
   * Miscellaneous information:
   */

  Tk_Cursor cursor;		/* Current cursor for window, or None. */
  char *takeFocus;		/* Value of -takefocus option;  not used in
				 * the C code, but used by keyboard traversal
				 * scripts.  Malloc'ed, but may be NULL. */
  char *command;		/* Command to execute when button is
				 * invoked; valid for buttons only.
				 * If not NULL, it's malloc-ed. */
  char *exec;			/* Unix command to execute when button is
				 * invoked */
  int staydown;               	/* Should the button stay down while unix command 
				 is running? */
  char *windowname;           	/* What windowname is produced by unix command? */
  int flags;			/* Various flags;  see below for
				 * definitions. */
} Tkgbutton;

/*
 * Possible "type" values for buttons.  These are the kinds of
 * widgets supported by this file.  The ordering of the type
 * numbers is significant:  greater means more features and is
 * used in the code.
 */

#define TYPE_TKGBUTTON		0
#define TYPE_TKGMENUBUTTON	1

/*
 * Class names for buttons, indexed by one of the type values above.
 */

static char *classNames[] = {"Tkgbutton", "Tkgmenubutton"};

/*
 * Flag bits for buttons:
 *
 * REDRAW_PENDING:		Non-zero means a DoWhenIdle handler
 *				has already been queued to redraw
 *				this window.
 * GOT_FOCUS:			Non-zero means this button currently
 *				has the input focus.
 * POSTED:			Non-zero means that the menu associated
 *				with this button has been posted (typically
 *				because of an active button press).
 */

#define REDRAW_PENDING		1
#define GOT_FOCUS		2
#define POSTED			4

/*
 * Mask values used to selectively enable entries in the
 * configuration specs:
 */

#define TKGBUTTON_MASK	        TK_CONFIG_USER_BIT
#define TKGMENUBUTTON_MASK	TK_CONFIG_USER_BIT << 1
#define ALL_MASK		(TKGBUTTON_MASK|TKGMENUBUTTON_MASK)

static int configFlags[] = {TKGBUTTON_MASK,TKGMENUBUTTON_MASK};

/*
 * Information used for parsing configuration specs:
 */

static Tk_ConfigSpec configSpecs[] = {
  {TK_CONFIG_BORDER, "-activebackground", "activeBackground", "Foreground",
   DEF_TKGBUTTON_ACTIVE_BG_COLOR, Tk_Offset(Tkgbutton, activeBorder),
   TKGBUTTON_MASK|TKGMENUBUTTON_MASK
   |TK_CONFIG_COLOR_ONLY},
  {TK_CONFIG_BORDER, "-activebackground", "activeBackground", "Foreground",
   DEF_TKGBUTTON_ACTIVE_BG_MONO, Tk_Offset(Tkgbutton, activeBorder),
   TKGBUTTON_MASK|TKGMENUBUTTON_MASK|TK_CONFIG_MONO_ONLY},
  {TK_CONFIG_COLOR, "-activeforeground", "activeForeground", "Background",
   DEF_TKGBUTTON_ACTIVE_FG_COLOR, Tk_Offset(Tkgbutton, activeFg), 
   TKGBUTTON_MASK|TKGMENUBUTTON_MASK|TK_CONFIG_COLOR_ONLY},
  {TK_CONFIG_COLOR, "-activeforeground", "activeForeground", "Background",
   DEF_TKGBUTTON_ACTIVE_FG_MONO, Tk_Offset(Tkgbutton, activeFg), 
   TKGBUTTON_MASK|TKGMENUBUTTON_MASK
   |TK_CONFIG_MONO_ONLY},
  {TK_CONFIG_ANCHOR, "-textanchor", "textanchor", "textAnchor",
   DEF_TKGBUTTON_ANCHOR, Tk_Offset(Tkgbutton, textanchor), ALL_MASK},
  {TK_CONFIG_ANCHOR, "-imageanchor", "imageanchor", "imageAnchor",
   DEF_TKGBUTTON_ANCHOR, Tk_Offset(Tkgbutton, imageanchor), ALL_MASK},
  {TK_CONFIG_BORDER, "-background", "background", "Background",
   DEF_TKGBUTTON_BG_COLOR, Tk_Offset(Tkgbutton, normalBorder),
   ALL_MASK | TK_CONFIG_COLOR_ONLY},
  {TK_CONFIG_BORDER, "-background", "background", "Background",
   DEF_TKGBUTTON_BG_MONO, Tk_Offset(Tkgbutton, normalBorder),
   ALL_MASK | TK_CONFIG_MONO_ONLY},
  {TK_CONFIG_SYNONYM, "-bd", "borderWidth", (char *) NULL,
   (char *) NULL, 0, ALL_MASK},
  {TK_CONFIG_SYNONYM, "-bg", "background", (char *) NULL,
   (char *) NULL, 0, ALL_MASK},
  {TK_CONFIG_BITMAP, "-bitmap", "bitmap", "Bitmap",
   DEF_TKGBUTTON_BITMAP, Tk_Offset(Tkgbutton, bitmap),
   ALL_MASK|TK_CONFIG_NULL_OK},
  {TK_CONFIG_PIXELS, "-borderwidth", "borderWidth", "BorderWidth",
   DEF_TKGBUTTON_BORDER_WIDTH, Tk_Offset(Tkgbutton, borderWidth), ALL_MASK},
  {TK_CONFIG_STRING, "-command", "command", "Command",
   DEF_TKGBUTTON_COMMAND, Tk_Offset(Tkgbutton, command),
   TKGBUTTON_MASK|TK_CONFIG_NULL_OK},
  {TK_CONFIG_ACTIVE_CURSOR, "-cursor", "cursor", "Cursor",
   DEF_TKGBUTTON_CURSOR, Tk_Offset(Tkgbutton, cursor),
   ALL_MASK|TK_CONFIG_NULL_OK},
  {TK_CONFIG_COLOR, "-disabledforeground", "disabledForeground",
   "DisabledForeground", DEF_TKGBUTTON_DISABLED_FG_COLOR,
   Tk_Offset(Tkgbutton, disabledFg), 
   TKGBUTTON_MASK|TKGMENUBUTTON_MASK|TK_CONFIG_COLOR_ONLY|TK_CONFIG_NULL_OK},
  {TK_CONFIG_COLOR, "-disabledforeground", "disabledForeground",
   "DisabledForeground", DEF_TKGBUTTON_DISABLED_FG_MONO,
   Tk_Offset(Tkgbutton, disabledFg), TKGBUTTON_MASK|TKGMENUBUTTON_MASK
   |TK_CONFIG_MONO_ONLY|TK_CONFIG_NULL_OK},
  {TK_CONFIG_SYNONYM, "-fg", "foreground", (char *) NULL,
   (char *) NULL, 0, ALL_MASK},
  {TK_CONFIG_FONT, "-font", "font", "Font",
   DEF_TKGBUTTON_FONT, Tk_Offset(Tkgbutton, fontPtr),
   ALL_MASK},
  {TK_CONFIG_COLOR, "-foreground", "foreground", "Foreground",
   DEF_TKGBUTTON_FG, Tk_Offset(Tkgbutton, normalFg), ALL_MASK},
  {TK_CONFIG_STRING, "-height", "height", "Height",
   DEF_TKGBUTTON_HEIGHT, Tk_Offset(Tkgbutton, heightString), ALL_MASK},
  {TK_CONFIG_COLOR, "-highlightbackground", "highlightBackground",
   "HighlightBackground", DEF_TKGBUTTON_HIGHLIGHT_BG,
   Tk_Offset(Tkgbutton, highlightBgColorPtr), ALL_MASK},
  {TK_CONFIG_COLOR, "-highlightcolor", "highlightColor", "HighlightColor",
   DEF_TKGBUTTON_HIGHLIGHT, Tk_Offset(Tkgbutton, highlightColorPtr),
   ALL_MASK},
  {TK_CONFIG_PIXELS, "-highlightthickness", "highlightThickness",
   "HighlightThickness",
   DEF_TKGBUTTON_HIGHLIGHT_WIDTH, Tk_Offset(Tkgbutton, highlightWidth),
   TKGBUTTON_MASK|TKGMENUBUTTON_MASK},
  {TK_CONFIG_STRING, "-image", "image", "Image",
   DEF_TKGBUTTON_IMAGE, Tk_Offset(Tkgbutton, imageString),
   ALL_MASK|TK_CONFIG_NULL_OK},
  {TK_CONFIG_STRING, "-iconside", "iconside", "Iconside",
   DEF_TKGBUTTON_ICONSIDE, Tk_Offset(Tkgbutton, iconside),
   TKGBUTTON_MASK|TKGMENUBUTTON_MASK},
  {TK_CONFIG_JUSTIFY, "-justify", "justify", "Justify",
   DEF_TKGBUTTON_JUSTIFY, Tk_Offset(Tkgbutton, justify), ALL_MASK},
  {TK_CONFIG_STRING, "-menu", "menu", "Menu",
   DEF_TKGMENUBUTTON_MENU, Tk_Offset(Tkgbutton, menuName),
   TKGMENUBUTTON_MASK|TK_CONFIG_NULL_OK},
  {TK_CONFIG_PIXELS, "-padx", "padX", "Pad",
   DEF_TKGBUTTON_PADX, Tk_Offset(Tkgbutton, padX), TKGBUTTON_MASK|TKGMENUBUTTON_MASK},
  {TK_CONFIG_PIXELS, "-pady", "padY", "Pad",
   DEF_TKGBUTTON_PADY, Tk_Offset(Tkgbutton, padY), TKGBUTTON_MASK|TKGMENUBUTTON_MASK},
  {TK_CONFIG_RELIEF, "-relief", "relief", "Relief",
   DEF_TKGBUTTON_RELIEF, Tk_Offset(Tkgbutton, relief),
   TKGBUTTON_MASK|TKGMENUBUTTON_MASK},
  {TK_CONFIG_PIXELS, "-sep", "sep", "Sep",
   DEF_TKGBUTTON_SEP, Tk_Offset(Tkgbutton, sep), TKGBUTTON_MASK|TKGMENUBUTTON_MASK},
  {TK_CONFIG_UID, "-state", "state", "State",
   DEF_TKGBUTTON_STATE, Tk_Offset(Tkgbutton, state),
   TKGBUTTON_MASK|TKGMENUBUTTON_MASK},
  {TK_CONFIG_BOOLEAN, "-staydown", "staydown", "Staydown",
   DEF_TKGBUTTON_STAYDOWN, Tk_Offset(Tkgbutton, staydown),
   TKGBUTTON_MASK},
  {TK_CONFIG_STRING, "-takefocus", "takeFocus", "TakeFocus",
   DEF_TKGBUTTON_TAKE_FOCUS, Tk_Offset(Tkgbutton, takeFocus),
   TKGBUTTON_MASK|TKGMENUBUTTON_MASK|TK_CONFIG_NULL_OK},
  {TK_CONFIG_STRING, "-text", "text", "Text",
   DEF_TKGBUTTON_TEXT, Tk_Offset(Tkgbutton, text), ALL_MASK},
  {TK_CONFIG_STRING, "-textvariable", "textVariable", "Variable",
   DEF_TKGBUTTON_TEXT_VARIABLE, Tk_Offset(Tkgbutton, textVarName),
   ALL_MASK|TK_CONFIG_NULL_OK},
  {TK_CONFIG_INT, "-underline", "underline", "Underline",
   DEF_TKGBUTTON_UNDERLINE, Tk_Offset(Tkgbutton, underline), ALL_MASK},
  {TK_CONFIG_STRING, "-exec", "exec", "Exec",
   DEF_TKGBUTTON_EXEC, Tk_Offset(Tkgbutton, exec),
   TKGBUTTON_MASK|TK_CONFIG_NULL_OK},
  {TK_CONFIG_STRING, "-width", "width", "Width",
   DEF_TKGBUTTON_WIDTH, Tk_Offset(Tkgbutton, widthString), ALL_MASK},
  {TK_CONFIG_STRING, "-windowname", "windowname", "Windowname",
   DEF_TKGBUTTON_WINDOWNAME, Tk_Offset(Tkgbutton, windowname),
   TKGBUTTON_MASK|TK_CONFIG_NULL_OK},
  {TK_CONFIG_PIXELS, "-wraplength", "wrapLength", "WrapLength",
   DEF_TKGBUTTON_WRAP_LENGTH, Tk_Offset(Tkgbutton, wrapLength), ALL_MASK},
  {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
   (char *) NULL, 0, 0}
};

/*
 * String to print out in error messages, identifying options for
 * widget commands for different types of labels or buttons:
 */

static char *optionStrings[] = {
  "cget, configure, flash, or invoke",
  "cget or configure"
};

/*
 * Forward declarations for procedures defined later in this file:
 */

static void TkgbuttonCmdDeletedProc _ANSI_ARGS_((ClientData clientData));
static int TkgbuttonCreate _ANSI_ARGS_((ClientData clientData,
					Tcl_Interp *interp, int argc, char **argv, int type));
static void TkgbuttonEventProc _ANSI_ARGS_((ClientData clientData,
					    XEvent *eventPtr));
static void TkgbuttonImageProc _ANSI_ARGS_((ClientData clientData,
					    int x, int y, int width, int height,
					    int imgWidth, int imgHeight));
static char * TkgbuttonTextVarProc _ANSI_ARGS_((ClientData clientData,
						Tcl_Interp *interp, char *name1, char *name2,
						int flags));
static int TkgbuttonWidgetCmd _ANSI_ARGS_((ClientData clientData,
					   Tcl_Interp *interp, int argc, char **argv));
static void ComputeTkgbuttonGeometry _ANSI_ARGS_((Tkgbutton *butPtr));
static int ConfigureTkgbutton _ANSI_ARGS_((Tcl_Interp *interp,
					   Tkgbutton *butPtr, int argc, char **argv,
					   int flags));
static void DestroyTkgbutton _ANSI_ARGS_((Tkgbutton *butPtr));
static void DisplayTkgbutton _ANSI_ARGS_((ClientData clientData));
static int InvokeTkgbutton  _ANSI_ARGS_((Tkgbutton *butPtr));


int
TkgbuttonCmd(clientData, interp, argc, argv)
ClientData clientData;		/* Main window associated with
				 * interpreter. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
  return TkgbuttonCreate(clientData, interp, argc, argv, TYPE_TKGBUTTON);
}

int
TkgmenubuttonCmd(clientData, interp, argc, argv)
ClientData clientData;	/* Main window associated with
			 * interpreter. */
Tcl_Interp *interp;	/* Current interpreter. */
int argc;		/* Number of arguments. */
char **argv;		/* Argument strings. */
{
  return TkgbuttonCreate(clientData, interp, argc, argv, TYPE_TKGMENUBUTTON);
}


static int
TkgbuttonCreate(clientData, interp, argc, argv, type)
ClientData clientData;	/* Main window associated with
			 * interpreter. */
Tcl_Interp *interp;	/* Current interpreter. */
int argc;		/* Number of arguments. */
char **argv;		/* Argument strings. */
int type;		/* Type of button to create: TYPE_TKGBUTTON
			 * or TYPE_TKGMENUBUTTON. */
{
  register Tkgbutton *butPtr;
  Tk_Window tkwin = (Tk_Window) clientData;
  Tk_Window new;

  if (argc < 2) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " pathName ?options?\"", (char *) NULL);
    return TCL_ERROR;
  }

  /*
   * Create the new window.
   */

  new = Tk_CreateWindowFromPath(interp, tkwin, argv[1], (char *) NULL);
  if (new == NULL) {
    return TCL_ERROR;
  }

  /*
   * Initialize the data structure for the button.
   */

  butPtr = (Tkgbutton *) ckalloc(sizeof(Tkgbutton));
  butPtr->tkwin = new;
  butPtr->display = Tk_Display(new);
  butPtr->interp = interp;
  butPtr->widgetCmd = Tcl_CreateCommand(interp, Tk_PathName(butPtr->tkwin),
					TkgbuttonWidgetCmd, (ClientData) butPtr, TkgbuttonCmdDeletedProc);
  butPtr->type = type;
  butPtr->text = NULL;
  butPtr->numChars = 0;
  butPtr->underline = -1;
  butPtr->textVarName = NULL;
  butPtr->bitmap = None;
  butPtr->imageString = NULL;
  butPtr->image = NULL;
  butPtr->iconside = NULL;
  butPtr->menuName = NULL;
  butPtr->state = tkNormalUid;
  butPtr->normalBorder = NULL;
  butPtr->activeBorder = NULL;
  butPtr->borderWidth = 0;
  butPtr->relief = TK_RELIEF_FLAT;
  butPtr->highlightWidth = 0;
  butPtr->highlightBgColorPtr = NULL;
  butPtr->highlightColorPtr = NULL;
  butPtr->inset = 0;
  butPtr->fontPtr = NULL;
  butPtr->normalFg = NULL;
  butPtr->activeFg = NULL;
  butPtr->disabledFg = NULL;
  butPtr->normalTextGC = None;
  butPtr->activeTextGC = None;
  butPtr->gray = None;
  butPtr->disabledGC = None;
  butPtr->copyGC = None;
  butPtr->widthString = NULL;
  butPtr->heightString = NULL;
  butPtr->width = 0;
  butPtr->height = 0;
  butPtr->wrapLength = 0;
  butPtr->padX = 0;
  butPtr->padY = 0;
  butPtr->sep = 0;
  butPtr->textanchor = TK_ANCHOR_CENTER;
  butPtr->imageanchor = TK_ANCHOR_CENTER;
  butPtr->justify = TK_JUSTIFY_CENTER;
  butPtr->cursor = None;
  butPtr->command = NULL;
  butPtr->exec = NULL;
  butPtr->windowname = NULL;
  butPtr->staydown = 0;
  butPtr->takeFocus = NULL;
  butPtr->flags = 0;

  Tk_SetClass(new, classNames[type]);
  Tk_CreateEventHandler(butPtr->tkwin,
			ExposureMask|StructureNotifyMask|FocusChangeMask,
			TkgbuttonEventProc, (ClientData) butPtr);
  if (ConfigureTkgbutton(interp, butPtr, argc-2, argv+2,
			 configFlags[type]) != TCL_OK) {
    Tk_DestroyWindow(butPtr->tkwin);
    return TCL_ERROR;
  }

  interp->result = Tk_PathName(butPtr->tkwin);
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ButtonWidgetCmd --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a widget managed by this module.
 *
 * Results:
 *	A standard Tcl result.
 *
 *--------------------------------------------------------------
 */

static int
TkgbuttonWidgetCmd(clientData, interp, argc, argv)
ClientData clientData;	/* Information about button widget. */
Tcl_Interp *interp;	/* Current interpreter. */
int argc;		/* Number of arguments. */
char **argv;		/* Argument strings. */
{
  register Tkgbutton *butPtr = (Tkgbutton *) clientData;
  int result = TCL_OK;
  size_t length;
  int c;

  if (argc < 2) {
    sprintf(interp->result,
	    "wrong # args: should be \"%.50s option ?arg arg ...?\"",
	    argv[0]);
    return TCL_ERROR;
  }
  Tcl_Preserve((ClientData) butPtr);
  c = argv[1][0];
  length = strlen(argv[1]);
  if ((c == 'c') && (strncmp(argv[1], "cget", length) == 0)
      && (length >= 2)) {
    if (argc != 3) {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " cget option\"",
		       (char *) NULL);
      goto error;
    }
    result = Tk_ConfigureValue(interp, butPtr->tkwin, configSpecs,
			       (char *) butPtr, argv[2], configFlags[butPtr->type]);
  } else if ((c == 'c') && (strncmp(argv[1], "configure", length) == 0)
	     && (length >= 2)) {
    if (argc == 2) {
      result = Tk_ConfigureInfo(interp, butPtr->tkwin, configSpecs,
				(char *) butPtr, (char *) NULL, configFlags[butPtr->type]);
    } else if (argc == 3) {
      result = Tk_ConfigureInfo(interp, butPtr->tkwin, configSpecs,
				(char *) butPtr, argv[2],
				configFlags[butPtr->type]);
    } else {
      result = ConfigureTkgbutton(interp, butPtr, argc-2, argv+2,
				  configFlags[butPtr->type] | TK_CONFIG_ARGV_ONLY);
    }
  } else if ((c == 'f') && (strncmp(argv[1], "flash", length) == 0)
	     && (butPtr->type != TYPE_TKGMENUBUTTON)) {
    int i;
    if (argc > 2) {
      sprintf(interp->result,
	      "wrong # args: should be \"%.50s flash\"",
	      argv[0]);
      goto error;
    }
    if (butPtr->state != tkDisabledUid) {
      for (i = 0; i < 4; i++) {
	butPtr->state = (butPtr->state == tkNormalUid)
	  ? tkActiveUid : tkNormalUid;
	Tk_SetBackgroundFromBorder(butPtr->tkwin,
				   (butPtr->state == tkActiveUid) ? butPtr->activeBorder
				   : butPtr->normalBorder);
	DisplayTkgbutton((ClientData) butPtr);

	/*
	 * Special note: must cancel any existing idle handler
	 * for DisplayButton;  it's no longer needed, and DisplayButton
	 * cleared the REDRAW_PENDING flag.
	 */

	Tcl_CancelIdleCall(DisplayTkgbutton, (ClientData) butPtr);
	XFlush(butPtr->display);
	Tcl_Sleep(50);
      }
    }
  } else if ((c == 'i') && (strncmp(argv[1], "invoke", length) == 0)
	     && (butPtr->type != TYPE_TKGMENUBUTTON)) {
    if (argc > 2) {
      sprintf(interp->result,
	      "wrong # args: should be \"%.50s invoke\"",
	      argv[0]);
      goto error;
    }
    if (butPtr->state != tkDisabledUid) {
      result = InvokeTkgbutton(butPtr);
    }
  } else {
    sprintf(interp->result,
	    "bad option \"%.50s\": must be %s", argv[1],
	    optionStrings[0]);
    goto error;
  }
  Tcl_Release((ClientData) butPtr);
  return result;

error:
  Tcl_Release((ClientData) butPtr);
  return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * DestroyButton --
 *
 *	This procedure is invoked by Tcl_EventuallyFree or Tcl_Release
 *	to clean up the internal structure of a button at a safe time
 *	(when no-one is using it anymore).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the widget is freed up.
 *
 *----------------------------------------------------------------------
 */

static void
DestroyTkgbutton(butPtr)
Tkgbutton *butPtr;		/* Info about button widget. */
{
  /*
   * Free up all the stuff that requires special handling, then
   * let Tk_FreeOptions handle all the standard option-related
   * stuff.
   */

  if (butPtr->textVarName != NULL) {
    Tcl_UntraceVar(butPtr->interp, butPtr->textVarName,
		   TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		   TkgbuttonTextVarProc, (ClientData) butPtr);
  }
  if (butPtr->image != NULL) {
    Tk_FreeImage(butPtr->image);
  }
  if (butPtr->normalTextGC != None) {
    Tk_FreeGC(butPtr->display, butPtr->normalTextGC);
  }
  if (butPtr->activeTextGC != None) {
    Tk_FreeGC(butPtr->display, butPtr->activeTextGC);
  }
  if (butPtr->gray != None) {
    Tk_FreeBitmap(butPtr->display, butPtr->gray);
  }
  if (butPtr->disabledGC != None) {
    Tk_FreeGC(butPtr->display, butPtr->disabledGC);
  }
  if (butPtr->copyGC != None) {
    Tk_FreeGC(butPtr->display, butPtr->copyGC);
  }
  Tk_FreeOptions(configSpecs, (char *) butPtr, butPtr->display,
		 configFlags[butPtr->type]);
  Tcl_EventuallyFree((ClientData)butPtr, TCL_DYNAMIC);
}

/*
 *----------------------------------------------------------------------
 *
 * ConfigureButton --
 *
 *	This procedure is called to process an argv/argc list, plus
 *	the Tk option database, in order to configure (or
 *	reconfigure) a button widget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as text string, colors, font,
 *	etc. get set for butPtr;  old resources get freed, if there
 *	were any.  The button is redisplayed.
 *
 *----------------------------------------------------------------------
 */

static int
ConfigureTkgbutton(interp, butPtr, argc, argv, flags)
Tcl_Interp *interp;		/* Used for error reporting. */
register Tkgbutton *butPtr;	/* Information about widget;  may or may
				 * not already have values for some fields. */
int argc;			/* Number of valid entries in argv. */
char **argv;			/* Arguments. */
int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
  XGCValues gcValues;
  GC newGC;
  unsigned long mask;
  Tk_Image image;

  /*
   * Eliminate any existing trace on variables monitored by the button.
   */

  if (butPtr->textVarName != NULL) {
    Tcl_UntraceVar(interp, butPtr->textVarName, 
		   TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		   TkgbuttonTextVarProc, (ClientData) butPtr);
  }

  if (Tk_ConfigureWidget(interp, butPtr->tkwin, configSpecs,
			 argc, argv, (char *) butPtr, flags) != TCL_OK) {
    return TCL_ERROR;
  }

  /*
   * A few options need special processing, such as setting the
   * background from a 3-D border, or filling in complicated
   * defaults that couldn't be specified to Tk_ConfigureWidget.
   */

  if ((butPtr->state == tkActiveUid) && !Tk_StrictMotif(butPtr->tkwin)) {
    Tk_SetBackgroundFromBorder(butPtr->tkwin, butPtr->activeBorder);
  } else {
    Tk_SetBackgroundFromBorder(butPtr->tkwin, butPtr->normalBorder);
    if ((butPtr->state != tkNormalUid) && (butPtr->state != tkActiveUid)
	&& (butPtr->state != tkDisabledUid)) {
      Tcl_AppendResult(interp, "bad state value \"", butPtr->state,
		       "\": must be normal, active, or disabled", (char *) NULL);
      butPtr->state = tkNormalUid;
      return TCL_ERROR;
    }
  }

  if (butPtr->highlightWidth < 0) {
    butPtr->highlightWidth = 0;
  }

  gcValues.font = butPtr->fontPtr->fid;
  gcValues.foreground = butPtr->normalFg->pixel;
  gcValues.background = Tk_3DBorderColor(butPtr->normalBorder)->pixel;
    
  /*
   * Note: GraphicsExpose events are disabled in normalTextGC because it's
   * used to copy stuff from an off-screen pixmap onto the screen (we know
   * that there's no problem with obscured areas).
   */

  gcValues.graphics_exposures = False;
  newGC = Tk_GetGC(butPtr->tkwin,
		   GCForeground|GCBackground|GCFont|GCGraphicsExposures,
		   &gcValues);
  if (butPtr->normalTextGC != None) {
    Tk_FreeGC(butPtr->display, butPtr->normalTextGC);
  }
  butPtr->normalTextGC = newGC;

  gcValues.font = butPtr->fontPtr->fid;
  gcValues.foreground = butPtr->activeFg->pixel;
  gcValues.background = Tk_3DBorderColor(butPtr->activeBorder)->pixel;
  newGC = Tk_GetGC(butPtr->tkwin,
		   GCForeground|GCBackground|GCFont, &gcValues);
  if (butPtr->activeTextGC != None) {
    Tk_FreeGC(butPtr->display, butPtr->activeTextGC);
  }
  butPtr->activeTextGC = newGC;

  gcValues.font = butPtr->fontPtr-> fid;
  gcValues.background = Tk_3DBorderColor(butPtr->normalBorder)->pixel;
  if ((butPtr->disabledFg != NULL) && (butPtr->imageString == NULL)) {
    gcValues.foreground = butPtr->disabledFg->pixel;
    mask = GCForeground|GCBackground|GCFont;
  } else {
    gcValues.foreground = gcValues.background;
    if (butPtr->gray == None) {
      butPtr->gray = Tk_GetBitmap(interp, butPtr->tkwin,
				  Tk_GetUid("gray50"));
      if (butPtr->gray == None) {
	return TCL_ERROR;
      }
    }
    gcValues.fill_style = FillStippled;
    gcValues.stipple = butPtr->gray;
    mask = GCForeground|GCFillStyle|GCStipple;
  }
  newGC = Tk_GetGC(butPtr->tkwin, mask, &gcValues);
  if (butPtr->disabledGC != None) {
    Tk_FreeGC(butPtr->display, butPtr->disabledGC);
  }
  butPtr->disabledGC = newGC;

  if (butPtr->copyGC == None) {
    butPtr->copyGC = Tk_GetGC(butPtr->tkwin, 0, &gcValues);
  }

  if (butPtr->padX < 0) {
    butPtr->padX = 0;
  }
  if (butPtr->padY < 0) {
    butPtr->padY = 0;
  }
  if (butPtr->sep < 0) {
    butPtr->sep = 0;
  }

  /*
   * Get the images for the widget, if there are any.  Allocate the
   * new images before freeing the old ones, so that the reference
   * counts don't go to zero and cause image data to be discarded.
   */

  if (butPtr->imageString != NULL) {
    image = Tk_GetImage(butPtr->interp, butPtr->tkwin,
			butPtr->imageString, TkgbuttonImageProc, (ClientData) butPtr);
    if (image == NULL) {
      return TCL_ERROR;
    }
  } else {
    image = NULL;
  }
  if (butPtr->image != NULL) {
    Tk_FreeImage(butPtr->image);
  }
  butPtr->image = image;

  if (butPtr->textVarName != NULL) {
    /*
     * The button must display the value of a variable: set up a trace
     * on the variable's value, create the variable if it doesn't
     * exist, and fetch its current value.
     */

    char *value;

    value = Tcl_GetVar(interp, butPtr->textVarName, TCL_GLOBAL_ONLY);
    if (value == NULL) {
      if (Tcl_SetVar(interp, butPtr->textVarName, butPtr->text,
		     TCL_GLOBAL_ONLY|TCL_LEAVE_ERR_MSG) == NULL) {
	return TCL_ERROR;
      }
    } else {
      if (butPtr->text != NULL) {
	ckfree(butPtr->text);
      }
      butPtr->text = (char *) ckalloc((unsigned) (strlen(value) + 1));
      strcpy(butPtr->text, value);
    }
    Tcl_TraceVar(interp, butPtr->textVarName,
		 TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		 TkgbuttonTextVarProc, (ClientData) butPtr);
  }

  if (Tk_GetPixels(interp, butPtr->tkwin, butPtr->widthString,
		   &butPtr->width) != TCL_OK) {
		   widthError:
    Tcl_AddErrorInfo(interp, "\n    (processing -width option)");
    return TCL_ERROR;
  }
  if (Tk_GetPixels(interp, butPtr->tkwin, butPtr->heightString,
		   &butPtr->height) != TCL_OK) {
		   heightError:
    Tcl_AddErrorInfo(interp, "\n    (processing -height option)");
    return TCL_ERROR;
  }
  ComputeTkgbuttonGeometry(butPtr);

  /*
   * Lastly, arrange for the button to be redisplayed.
   */

  if (Tk_IsMapped(butPtr->tkwin) && !(butPtr->flags & REDRAW_PENDING)) {
    Tcl_DoWhenIdle(DisplayTkgbutton, (ClientData) butPtr);
    butPtr->flags |= REDRAW_PENDING;
  }

  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * DisplayButton --
 *
 *	This procedure is invoked to display a button widget.  It is
 *	normally invoked as an idle handler.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Commands are output to X to display the button in its
 *	current mode.  The REDRAW_PENDING flag is cleared.
 *
 *----------------------------------------------------------------------
 */

static void
DisplayTkgbutton(clientData)
ClientData clientData;	/* Information about widget. */
{
  register Tkgbutton *butPtr = (Tkgbutton *) clientData;
  GC gc;
  Tk_3DBorder border;
  Pixmap pixmap;
  int pixx = 0;			/* Initialization only needed to stop
				 * compiler warning. */
  int relief;
  int xskirt, yskirt;           /* Size of unusable edge of button window */
  register Tk_Window tkwin = butPtr->tkwin;
  int pixwidth, pixheight;      /* size of image */
  int pixxorig, pixyorig;       /* up-left corner of region in button for image */
  int pixwwidth, pixwheight;    /* size of region in button for image */
  int textwidth, textheight;      /* size of text */
  int textxorig, textyorig;       /* up-left corner of region in button for text */
  int textwwidth, textwheight;    /* size of region in button for text */
  int wwidth, wheight;            /* size of usable region in button */
  int drawimage, drawbitmap, drawtext;        /* do we draw an image, a bitmap, text? */
  char *side = butPtr->iconside; /* where does the image or bitmap go? */
  int offset;			/* we offset
				 * the text to make the button appear to
				 * move up and down as the relief changes. */

  butPtr->flags &= ~REDRAW_PENDING;
  if ((butPtr->tkwin == NULL) || !Tk_IsMapped(tkwin)) {
    return;
  }

  border = butPtr->normalBorder;
  if ((butPtr->state == tkDisabledUid) && (butPtr->disabledFg != NULL)) {
    gc = butPtr->disabledGC;
  } else if ((butPtr->state == tkActiveUid)
	     && !Tk_StrictMotif(butPtr->tkwin)) {
    gc = butPtr->activeTextGC;
    border = butPtr->activeBorder;
  } else {
    gc = butPtr->normalTextGC;
  }

  relief = butPtr->relief;
  offset = !Tk_StrictMotif(butPtr->tkwin); 

  /*
   * In order to avoid screen flashes, this procedure redraws
   * the button in a pixmap, then copies the pixmap to the
   * screen in a single operation.  This means that there's no
   * point in time where the on-sreen image has been cleared.
   */

  pixmap = Tk_GetPixmap(butPtr->display, Tk_WindowId(tkwin),
			Tk_Width(tkwin), Tk_Height(tkwin), Tk_Depth(tkwin));
  Tk_Fill3DRectangle(tkwin, pixmap, border, 0, 0, Tk_Width(tkwin),
		     Tk_Height(tkwin), 0, TK_RELIEF_FLAT);

  /*
   * Calculate regions for image and text
   */

  drawimage = drawbitmap = drawtext = 0;
  pixwidth = pixheight = textwidth = textheight = 0;
  if (butPtr->image != None) {
    drawimage = 1;
    Tk_SizeOfImage(butPtr->image, &pixwidth, &pixheight);
  } else if (butPtr->bitmap != None) {
    drawbitmap = 1;
    Tk_SizeOfBitmap(butPtr->display, butPtr->bitmap, &pixwidth, &pixheight);
  }
  if (*butPtr->text) {
    drawtext = 1;
    textwidth = butPtr->textWidth;
    textheight = butPtr->textHeight;
  }

  xskirt = butPtr->inset + offset + butPtr->padX;
  yskirt = butPtr->inset + offset + butPtr->padY;
  wwidth = Tk_Width(tkwin) - 2 * xskirt;
  wheight = Tk_Height(tkwin) - 2 * yskirt;
  pixxorig =  textxorig = xskirt;
  pixyorig =  textyorig = yskirt;
  pixwwidth = textwwidth = wwidth;
  pixwheight = textwheight = wheight;

  if (drawtext && (drawimage || drawbitmap)) {
    /* we're drawing both; divide extra space evenly*/
    if (!strcmp(side,LEFT) || !strcmp(side,RIGHT)) {
	pixwwidth = pixwidth + 
	  ((int) (wwidth - pixwidth - textwidth - butPtr->sep))/2;
	pixwwidth = max(pixwidth,pixwwidth);
	textwwidth = wwidth - pixwwidth - butPtr->sep;
	textwwidth = max(textwwidth,textwidth);
	if (!strcmp(side,LEFT)) {
	  textxorig = xskirt + pixwwidth +butPtr->sep;
	} else {
	  pixxorig = xskirt + textwwidth +butPtr->sep;
	}
    } else if (!strcmp(side,BOTTOM) || !strcmp(side,TOP)) {
      /* top or bottom */
	pixwheight = pixheight + 
	  ((int) (wheight - pixheight - textheight - butPtr->sep))/2;
	pixwheight = max(pixheight,pixwheight);
	textwheight = wheight - pixwheight - butPtr->sep;
	textwheight = max(textheight,textwheight);
	if (!strcmp(side,BOTTOM)) {
	  pixyorig = yskirt + textwheight + butPtr->sep;
	} else {
	  textyorig = yskirt + pixwheight + butPtr->sep;
	}
    }
  }
    
  if (drawimage || drawbitmap) {
    int x, y;
    switch (butPtr->imageanchor) {
    case TK_ANCHOR_NW: case TK_ANCHOR_W: case TK_ANCHOR_SW:
      x = pixxorig;
      break;
    case TK_ANCHOR_N: case TK_ANCHOR_CENTER: case TK_ANCHOR_S:
      x =  pixxorig + ((int) (pixwwidth - pixwidth))/2;
      break;
    default:
      x = pixwwidth - pixwidth;
      break;
    }
    switch (butPtr->imageanchor) {
    case TK_ANCHOR_NW: case TK_ANCHOR_N: case TK_ANCHOR_NE:
      y = pixyorig;
      break;
    case TK_ANCHOR_W: case TK_ANCHOR_CENTER: case TK_ANCHOR_E:
      y = pixyorig + (pixwheight - pixheight)/2;
      break;
    default:
      y = pixyorig + pixwheight - pixheight;
      break;
    }
    if (relief == TK_RELIEF_RAISED) {
      x -= offset;
      y -= offset;
    } else if (relief == TK_RELIEF_SUNKEN) {
      x += offset;
      y += offset;
    }
    if (drawimage) {
      Tk_RedrawImage(butPtr->image, 0, 0, pixwidth, pixheight, pixmap,
		     x, y);
    } else {
      XSetClipOrigin(butPtr->display, gc, x, y);
      XCopyPlane(butPtr->display, butPtr->bitmap, pixmap, gc, 0, 0,
		 (unsigned int) pixwidth, (unsigned int) pixheight, x, y, 1);
      XSetClipOrigin(butPtr->display, gc, 0, 0);
    }
  }
  if (drawtext) {
    int x, y;
    switch (butPtr->textanchor) {
    case TK_ANCHOR_NW: case TK_ANCHOR_W: case TK_ANCHOR_SW:
      x = textxorig;
      break;
    case TK_ANCHOR_N: case TK_ANCHOR_CENTER: case TK_ANCHOR_S:
      x = textxorig + ((int) (textwwidth - textwidth))/2;
      break;
    default:
      x = textxorig + textwwidth - textwidth;
      break;
    }
    switch (butPtr->textanchor) {
    case TK_ANCHOR_NW: case TK_ANCHOR_N: case TK_ANCHOR_NE:
      y = textyorig;
      break;
    case TK_ANCHOR_W: case TK_ANCHOR_CENTER: case TK_ANCHOR_E:
      y = textyorig + ((int) (textwheight - textheight))/2;
      break;
    default:
      y = textyorig + textwheight - textheight;
      break;
    }
    if (relief == TK_RELIEF_RAISED) {
      x -= offset;
      y -= offset;
    } else if (relief == TK_RELIEF_SUNKEN) {
      x += offset;
      y += offset;
    }
    TkDisplayText(butPtr->display, pixmap, butPtr->fontPtr,
		  butPtr->text, butPtr->numChars, x, y, textwidth,
		  butPtr->justify, butPtr->underline, gc);
    y += textheight/2;
  }

  /*
   * If the button is disabled with a stipple rather than a special
   * foreground color, generate the stippled effect.  
   */

  if ((butPtr->state == tkDisabledUid)
      && ((butPtr->disabledFg == NULL) || (butPtr->image != NULL))) {
    XFillRectangle(butPtr->display, pixmap, butPtr->disabledGC,
		   butPtr->inset, butPtr->inset,
		   (unsigned) (Tk_Width(tkwin) - 2*butPtr->inset),
		   (unsigned) (Tk_Height(tkwin) - 2*butPtr->inset));
  }

  /*
   * Draw the border and traversal highlight last.  This way, if the
   * button's contents overflow they'll be covered up by the border.
   */

  if (relief != TK_RELIEF_FLAT) {
    Tk_Draw3DRectangle(tkwin, pixmap, border,
		       butPtr->highlightWidth, butPtr->highlightWidth,
		       Tk_Width(tkwin) - 2*butPtr->highlightWidth,
		       Tk_Height(tkwin) - 2*butPtr->highlightWidth,
		       butPtr->borderWidth, relief);
  }
  if (butPtr->highlightWidth != 0) {
    GC gc;

    if (butPtr->flags & GOT_FOCUS) {
      gc = Tk_GCForColor(butPtr->highlightColorPtr, pixmap);
    } else {
      gc = Tk_GCForColor(butPtr->highlightBgColorPtr, pixmap);
    }
    Tk_DrawFocusHighlight(tkwin, gc, butPtr->highlightWidth, pixmap);
  }

  /*
   * Copy the information from the off-screen pixmap onto the screen,
   * then delete the pixmap.
   */

  XCopyArea(butPtr->display, pixmap, Tk_WindowId(tkwin),
	    butPtr->copyGC, 0, 0, (unsigned) Tk_Width(tkwin),
	    (unsigned) Tk_Height(tkwin), 0, 0);
  Tk_FreePixmap(butPtr->display, pixmap);
}

/*
 *--------------------------------------------------------------
 *
 * ButtonEventProc --
 *
 *	This procedure is invoked by the Tk dispatcher for various
 *	events on buttons.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When the window gets deleted, internal structures get
 *	cleaned up.  When it gets exposed, it is redisplayed.
 *
 *--------------------------------------------------------------
 */

static void
TkgbuttonEventProc(clientData, eventPtr)
ClientData clientData;	/* Information about window. */
XEvent *eventPtr;		/* Information about event. */
{
  Tkgbutton *butPtr = (Tkgbutton *) clientData;
  if ((eventPtr->type == Expose) && (eventPtr->xexpose.count == 0)) {
    goto redraw;
  } else if (eventPtr->type == ConfigureNotify) {
    /*
     * Must redraw after size changes, since layout could have changed
     * and borders will need to be redrawn.
     */

    goto redraw;
  } else if (eventPtr->type == DestroyNotify) {
    if (butPtr->tkwin != NULL) {
      butPtr->tkwin = NULL;
      Tcl_DeleteCommand(butPtr->interp,
			Tcl_GetCommandName(butPtr->interp, butPtr->widgetCmd));
    }
    if (butPtr->flags & REDRAW_PENDING) {
      Tcl_CancelIdleCall(DisplayTkgbutton, (ClientData) butPtr);
    }
    DestroyTkgbutton(butPtr);
  } else if (eventPtr->type == FocusIn) {
    if (eventPtr->xfocus.detail != NotifyInferior) {
      butPtr->flags |= GOT_FOCUS;
      if (butPtr->highlightWidth > 0) {
	goto redraw;
      }
    }
  } else if (eventPtr->type == FocusOut) {
    if (eventPtr->xfocus.detail != NotifyInferior) {
      butPtr->flags &= ~GOT_FOCUS;
      if (butPtr->highlightWidth > 0) {
	goto redraw;
      }
    }
  }
  return;

redraw:
  if ((butPtr->tkwin != NULL) && !(butPtr->flags & REDRAW_PENDING)) {
    Tcl_DoWhenIdle(DisplayTkgbutton, (ClientData) butPtr);
    butPtr->flags |= REDRAW_PENDING;
  }
}

/*
 *----------------------------------------------------------------------
 *
 * ButtonCmdDeletedProc --
 *
 *	This procedure is invoked when a widget command is deleted.  If
 *	the widget isn't already in the process of being destroyed,
 *	this command destroys it.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The widget is destroyed.
 *
 *----------------------------------------------------------------------
 */

static void
TkgbuttonCmdDeletedProc(clientData)
ClientData clientData;	/* Pointer to widget record for widget. */
{
  Tkgbutton *butPtr = (Tkgbutton *) clientData;
  Tk_Window tkwin = butPtr->tkwin;

  /*
   * This procedure could be invoked either because the window was
   * destroyed and the command was then deleted (in which case tkwin
   * is NULL) or because the command was deleted, and then this procedure
   * destroys the widget.
   */

  if (tkwin != NULL) {
    butPtr->tkwin = NULL;
    Tk_DestroyWindow(tkwin);
  }
}

/*
 *----------------------------------------------------------------------
 *
 * ComputeButtonGeometry --
 *
 *	After changes in a button's text or bitmap, this procedure
 *	recomputes the button's geometry and passes this information
 *	along to the geometry manager for the window.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The button's window may change size.
 *
 *----------------------------------------------------------------------
 */

static void
ComputeTkgbuttonGeometry(butPtr)
register Tkgbutton *butPtr;	/* Tkgbutton whose geometry may have changed. */
{
  int pixwidth, pixheight, textwidth, textheight, width, height;
  int drawimage, drawtext;	/* Are we drawing an image? text? */
  char *side;

  pixwidth = pixheight = textwidth = textheight = 
    width = height = drawimage = drawtext = 0;

  if (butPtr->highlightWidth < 0) {
    butPtr->highlightWidth = 0;
  }
  butPtr->inset = butPtr->highlightWidth + butPtr->borderWidth;
  if (butPtr->image != NULL) {
    Tk_SizeOfImage(butPtr->image, &pixwidth, &pixheight);
    drawimage = 1;
  } else if (butPtr->bitmap != None) {
    Tk_SizeOfBitmap(butPtr->display, butPtr->bitmap, &pixwidth, &pixheight);
    drawimage = 1;
  } 
  if ((butPtr->numChars = strlen(butPtr->text))) {
    TkComputeTextGeometry(butPtr->fontPtr, butPtr->text,
			  butPtr->numChars, butPtr->wrapLength, &butPtr->textWidth,
			  &butPtr->textHeight);
    textwidth = butPtr->textWidth;
    textheight = butPtr->textHeight;
    drawtext = 1;
  }
  side = butPtr->iconside;
  if (!strcmp(side,LEFT) || !strcmp(side,RIGHT)) {
    height = max(pixheight,textheight);
    width = pixwidth + textwidth;
    if (drawimage && drawtext) {
      width += butPtr->sep;
    }
  } else if (!strcmp(side,BOTTOM) || !strcmp(side,TOP)) {
    width = max(pixwidth,textwidth);
    height = pixheight + textheight;
    if (drawimage && drawtext) {
      height += butPtr->sep;
    }
  } else {
    width = max(pixwidth,textwidth);
    height = max(pixheight,textheight);
  }

  width += 2*butPtr->padX;
  height += 2*butPtr->padY;

  if (!Tk_StrictMotif(butPtr->tkwin)) {
    width += 2;
    height += 2;
  }

  if (butPtr->width > 0) {
    width = butPtr->width;
  }
  if (butPtr->height > 0) {
    height = butPtr->height;
  }

  Tk_GeometryRequest(butPtr->tkwin, (int) (width 
					   + 2*butPtr->inset), (int) (height + 2*butPtr->inset));
  Tk_SetInternalBorder(butPtr->tkwin, butPtr->inset);
}

/*
 *----------------------------------------------------------------------
 *
 * InvokeButton --
 *
 *	This procedure is called to carry out the actions associated
 *	with a button, such as invoking a Tcl command or setting a
 *	variable.  This procedure is invoked, for example, when the
 *	button is invoked via the mouse.
 *
 * Results:
 *	A standard Tcl return value.  Information is also left in
 *	interp->result.
 *
 * Side effects:
 *	Depends on the button and its associated command.
 *
 *----------------------------------------------------------------------
 */

static int
InvokeTkgbutton(butPtr)
register Tkgbutton *butPtr;		/* Information about button. */
{
  if (butPtr->command != NULL) {
    return TkCopyAndGlobalEval(butPtr->interp, butPtr->command);
  }
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ButtonTextVarProc --
 *
 *	This procedure is invoked when someone changes the variable
 *	whose contents are to be displayed in a button.
 *
 * Results:
 *	NULL is always returned.
 *
 * Side effects:
 *	The text displayed in the button will change to match the
 *	variable.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static char *
TkgbuttonTextVarProc(clientData, interp, name1, name2, flags)
ClientData clientData;	/* Information about button. */
Tcl_Interp *interp;	/* Interpreter containing variable. */
char *name1;		/* Not used. */
char *name2;		/* Not used. */
int flags;		/* Information about what happened. */
{
  register Tkgbutton *butPtr = (Tkgbutton *) clientData;
  char *value;

  /*
   * If the variable is unset, then immediately recreate it unless
   * the whole interpreter is going away.
   */

  if (flags & TCL_TRACE_UNSETS) {
    if ((flags & TCL_TRACE_DESTROYED) && !(flags & TCL_INTERP_DESTROYED)) {
      Tcl_SetVar(interp, butPtr->textVarName, butPtr->text,
		 TCL_GLOBAL_ONLY);
      Tcl_TraceVar(interp, butPtr->textVarName,
		   TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		   TkgbuttonTextVarProc, clientData);
    }
    return (char *) NULL;
  }

  value = Tcl_GetVar(interp, butPtr->textVarName, TCL_GLOBAL_ONLY);
  if (value == NULL) {
    value = "";
  }
  if (butPtr->text != NULL) {
    ckfree(butPtr->text);
  }
  butPtr->text = (char *) ckalloc((unsigned) (strlen(value) + 1));
  strcpy(butPtr->text, value);
  ComputeTkgbuttonGeometry(butPtr);

  if ((butPtr->tkwin != NULL) && Tk_IsMapped(butPtr->tkwin)
      && !(butPtr->flags & REDRAW_PENDING)) {
    Tcl_DoWhenIdle(DisplayTkgbutton, (ClientData) butPtr);
    butPtr->flags |= REDRAW_PENDING;
  }
  return (char *) NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * ButtonImageProc --
 *
 *	This procedure is invoked by the image code whenever the manager
 *	for an image does something that affects the size of contents
 *	of an image displayed in a button.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Arranges for the button to get redisplayed.
 *
 *----------------------------------------------------------------------
 */

static void
TkgbuttonImageProc(clientData, x, y, width, height, imgWidth, imgHeight)
ClientData clientData;		/* Pointer to widget record. */
int x, y;			/* Upper left pixel (within image)
				 * that must be redisplayed. */
int width, height;		/* Dimensions of area to redisplay
				 * (may be <= 0). */
int imgWidth, imgHeight;	/* New dimensions of image. */
{
  register Tkgbutton *butPtr = (Tkgbutton *) clientData;

  if (butPtr->tkwin != NULL) {
    ComputeTkgbuttonGeometry(butPtr);
    if (Tk_IsMapped(butPtr->tkwin) && !(butPtr->flags & REDRAW_PENDING)) {
      Tcl_DoWhenIdle(DisplayTkgbutton, (ClientData) butPtr);
      butPtr->flags |= REDRAW_PENDING;
    }
  }
}


int 
Tkgbutton_Init(interp)
Tcl_Interp *interp;		/* Interpreter in which the package is
				 * to be made available. */
{
  int code;
  Tk_Window tkwin;

  if (!(tkwin = Tk_MainWindow(interp))) {
    return TCL_ERROR;
  }
  Tcl_CreateCommand(interp, "tkgbutton", TkgbuttonCmd, 
		    (ClientData) tkwin, (void (*)()) NULL);
  Tcl_CreateCommand(interp, "tkgmenubutton", TkgmenubuttonCmd, 
		    (ClientData) tkwin, (void (*)()) NULL);
  code = Tcl_PkgProvide(interp, "Tkgbutton", "4.1");
  if (code != TCL_OK) {
    return code;
  }
  return TCL_OK;
}
