# Hierarchical listbox pseudowidget
# data in $name-hldata
# each item gets a list:
#  0 -- "mode" (whatever)
#  1 -- text
#  2 -- list of options
#  3 -- list of child items
#  4 -- open/closed
# many features unimplemented
# see prefs-config.tcl for an example of use.

proc HList {name action args} {
    switch $action {
	create {
	    HListCreate $name $args
	} insert {
	    eval [concat HListInsert $name $args]
	} delete {
	    eval [concat HListDelete $name $args]
	} select {
	    eval [concat HListSelect $name $args]
	} default {
	    bgerror "bad action switch to HList"
	}
    }
}

##########
# Creation

set TKG(switches,HListCreate) {
    pathname
    {data ""}
    menu3
    emptycommand
    {double1 break}
    closedimagefile
    openimagefile
    iconcommand
    {selectmode single}
    selectcommand
    deletecommand
    {font $TKG(textfont)}
    {width 60}
    {height 20}
    {foreground $TKG(foreground)}
    {background $TKG(textbackground)}
    {selectforeground $TKG(foreground)}
    {selectbackground white}
    {activeforeground $TKG(activeforeground)}
    {activebackground $TKG(activebackground)}
}

set TKG(switches,HListCreate,ms) {
    closedimagefile openimagefile iconcommand selectcommand deletecommand
    movecommand menu3
}

proc HListCreate {name args} {
    upvar #0 $name-hlparams P
    set P(modes) ""
    set P(iconcounter) 0
    set args [lindex $args 0]
    # Parse switches
    global TKG
    set switches $TKG(switches,HListCreate)
    set modeswitches $TKG(switches,HListCreate,ms)
    TKGParseArgs $name-hlparams $args\
	$switches $modeswitches "HListCreate"
    TKGSetSwitchDefaults $name-hlparams $switches
    TKGSetMSSwitchDefaults $name-hlparams $switches
    foreach type {closed open} {
	foreach mode $P(modes) {
	    if [info exists P(${type}imagefile,$mode)] {
		set imagefile $P(${type}imagefile,$mode)
		set P(${type}image,$mode) HL$imagefile
		if {[lsearch [image names] HL$imagefile] == -1} {
		    SetImage HL$imagefile $imagefile
		}
	    }
	}
	if [info exists P(${type}imagefile)] {
	    set imagefile $P(${type}imagefile)
	    set P(${type}image) HL$imagefile
	    if {[lsearch [image names] HL$imagefile] == -1} {
		SetImage HL$imagefile $imagefile
	    }
	}
    }
    text $P(pathname) -font $P(font) -height $P(height) -width $P(width)\
	-foreground $P(foreground) -background $P(background)\
	-tabs {0 26 46} -cursor top_left_arrow -state disabled
    bind $P(pathname) <Motion> break
    bind $P(pathname) <1> "HListCheckEmpty $name;break"
    bind $P(pathname) <Leave> "HListLeave $name"
    if ![Empty [info procs HListInit-$name]] HListInit-$name
    HListMenus $name
    HListDraw $name
}

proc HListCheckEmpty {name} {
    upvar \#0 $name-hlparams P
    if ![llength $P(data)] {
	$P(emptycommand)
    } 
}

#########################
# Draw the whole hlist

proc HListDraw {name} {
    upvar \#0 $name-hlparams P
    $P(pathname) configure -state normal
    $P(pathname) delete 1.0 end
    $P(pathname) configure -state disabled
    set ped 0
    set index 1
    foreach item [set P(data)] {
	HListWidgetInsertAtIndex $name $ped $item $index.0
	set index [HListDrawKids $name $ped $item $index]
	incr index
	incr ped
    }
    HListTagPeds $name
    set P(selection) ""
}

proc HListDrawKids {name ped item index} {
    set kidnum 0
    foreach child [lindex $item 3] {
	incr index
	HListWidgetInsertAtIndex $name [concat $ped $kidnum] $child $index.0
	set index [HListDrawKids $name [concat $ped $kidnum] $child $index]
	incr kidnum
    }
    return $index
}

########################
# Insertion and deletion

proc HListInsert {name ped item} {
    HListDataReplace $name $ped "" $item
    HListDraw $name
}

proc HListWidgetInsertAtIndex {name ped item index} {
    upvar \#0 $name-hlparams P
    set joinped [join $ped -]
    set mode [lindex $item 0]
    set label [lindex $item 1]
    set options [lindex $item 2]
    set open [expr ![string match [lindex item 4] closed]]
    if $open {set oc open} {set oc closed}
    # Determine tags
    set tags [list item]
    if {[set i [lsearch $options -T]] != -1} {
	lappend tags [lindex $options [expr $i + 1]]
    }
    # Enable writing
    $P(pathname) configure -state normal
    # Indent
    set tabs "\t"
    for {set i 1} {$i < [llength $ped]} {incr i} {
	append tabs "\t"
    }
    $P(pathname) insert $index "$tabs"
    # Maybe insert an icon
    if [info exists P(${oc}image,$mode)] {
	set image [set P(${oc}image,$mode)]
    } elseif [info exists P(${oc}image)] {
	set image [set P(${oc}image)]
    }
    if [info exists image] {
	set iconpath $P(pathname).i[incr P(iconcounter)]
	label $iconpath -image $image -background $P(background)
	set iconindex [$P(pathname) index "$index lineend"]
	$P(pathname) window create $iconindex -window $iconpath
	if [info exists P(iconcommand,[lindex $item 0])] {
	    bind $iconpath <1> \
		"$P(iconcommand,[lindex $item 0]) $name $ped; break"
	    bind $iconpath <Double-1> break
	}
    }
    # Always write the label
    $P(pathname) insert "$index lineend" "\t${label}\n" $tags
    # Disable writing
    $P(pathname) configure -state disabled
}

proc HListDelete {name ped1 {ped2 {}}} {
    if [Empty $ped2] {set ped2 $ped1}
    upvar \#0 $name-hlparams P
    HListDataReplace $name $ped1 $ped2
    HListDraw $name
}

proc HListDataReplace {name ped1 {ped2 {}} {insertitem {}}} {
    if [Empty $ped2] {set ped2 $ped1}
    upvar \#0 $name-hlparams P
    if {[llength $ped1] == 1} {
	if [Empty $insertitem] {
	    set P(data) [lreplace $P(data) $ped1 $ped2]
	} else {
	    set P(data) [linsert $P(data) $ped1 $insertitem]
	}
	return
    }
    # ok, we have to dissect the hierachy...
    set end [expr [llength $ped1] - 1]
    set item(0) $P(data)
    set i [lindex $ped1 0]
    set item(1) [lindex $item(0) $i]
    for {set level 2} {$level <= $end} {incr level} {
	set i [lindex $ped1 [expr $level - 1]]
	set item($level) [lindex [lindex $item([expr $level - 1]) 3] $i]
    }
    set i [lindex $ped1 end]
    set j [lindex $ped2 end]
    if [Empty $insertitem] {
	set item($end) [lreplace $item($end) 3 3 \
			    [lreplace [lindex $item($end) 3] $i $j]]
    } else {
	set item($end) [lreplace $item($end) 3 3 \
			    [linsert [lindex $item($end) 3] $i $insertitem]]
    }
    for {set level [expr $end - 1]} {$level > 0} {incr level -1} {
	set i [lindex $ped1 $level]
	set newkids [lreplace [lindex $item($level) 3] $i $i $item([expr $level + 1])]
	set item($level) [lreplace $item($level) 3 3 $newkids]
    }
    set i [lindex $ped1 0]
    set P(data) [lreplace $item(0) $i $i $item(1)]
}

##################################
# Entering and leaving underlining

proc HListItemEnter {name ped} {
    upvar \#0 $name-hlparams P
    set joinped [join $ped -]
    if [info exists P(pointeronped)] {
	$P(pathname) tag configure  \
	    Ped-[join $P(pointeronped) -] -underline 0
    }
    $P(pathname) tag configure Ped-$joinped -underline 1
    set P(pointeronped) $ped
}

proc HListLeave {name} {
    upvar \#0 $name-hlparams P
    if [info exists P(pointeronped)] {
	$P(pathname) tag configure  \
	    Ped-[join $P(pointeronped) -] -underline 0
    }
}

#################################
# Tags

proc HListTagPeds {name} {
    upvar \#0 $name-hlparams P
    set w $P(pathname)
    foreach tag [$w tag names] {
	if [string match Ped-* $tag] {
	    $w tag delete $tag
	}
    }
    set D $P(data)
    set line 1
    for {set i 0} {$i < [llength $D]} {incr i} {
	set item [lindex $D $i]
	HListTagWithPed $name $w $line $i [lindex $item 0]
	set line [HListTagKids $name $item $w [expr $line + 1] $i]
    }
}

proc HListTagKids {name item w line ped} {
    set kids [lindex $item 3]
    if [string match [lindex $item 4] closed] {return $line}
    for {set i 0} {$i < [llength $kids]} {incr i} {
	set kid [lindex $kids $i]
	HListTagWithPed $name $w $line [concat $ped $i] [lindex $kid 0]
	set line [HListTagKids $name $kid $w [expr $line + 1] [concat $ped $i]]
    }
    return $line
}

proc HListTagWithPed {name w line ped mode} {
    upvar \#0 $name-hlparams P
    set jp [join $ped -]
    $w tag add Ped-$jp $line.0 [$w index "$line.0 lineend"]
    $w tag bind Ped-$jp <Enter> [list HListItemEnter $name $ped]
    $w tag bind Ped-$jp <1> [list HList1 $name $ped]
    $w tag bind Ped-$jp <B1-Motion> [list HList1Motion $name %x %y]
    $w tag bind Ped-$jp <ButtonRelease-1> [list HList1Release $name $ped %x %y]
    $w tag bind Ped-$jp <3> [list HList3 %X %Y $name $ped $mode]
    $w tag bind Ped-$jp <Double-1> $P(double1)
}    
	
##########################################
# Getting items and indices

proc HListGetItem {name ped} {
    upvar \#0 $name-hlparams P
    set data [lindex $P(data) [lindex $ped 0]]
    set ped [lreplace $ped 0 0]
    while {[llength $ped] != 0} {
	set data [lindex [lindex $data 3] [lindex $ped 0]]
	set ped [lreplace $ped 0 0]
    }
    return $data
}

proc HListGetIconPath {name ped} {
    upvar \#0 $name-hlparams P
    set range [$P(pathname) tag nextrange Ped-[join $ped -] 1.0]
    set dump [eval $P(pathname) dump -window $range]
    return [lindex $dump 1]
}

proc HListPedLess {ped1 ped2} {
    for {set i 0} {$i < [llength $ped1]} {incr i} {
	set e1 [lindex $ped1 $i]
	if [Empty [set e2 [lindex $ped2 $i]]] {return 0}
	if ![string match $e1 $e2] {
	    return [expr $e1 < $e2]
	}
    }
    return [expr [llength $ped1] < [llength $ped2]]
}

################
# Selection

proc HList1 {name ped} {
    upvar \#0 $name-hlparams P
    if ![In $ped $P(selection)] {
	HListItemSelect $name $P(pointeronped)
    } else {
	$P(pathname) configure -cursor crosshair
    }
}

proc HList1Motion {name x y} {
    upvar \#0 $name-hlparams(pathname) P
    if [regexp {Ped-([-0-9]*)} [$P tag names [$P index @$x,$y]] v v] {
	set ped [split $v -]
	HListItemEnter $name $ped
	if ![string match crosshair [$P cget -cursor]] {
	    HListItemSelect $name $ped
	}
    }
}

proc HList1Release {name ped x y} {
    upvar \#0 $name-hlparams P
    if ![string match crosshair [$P(pathname) cget -cursor]] return
    if [regexp {Ped-([-0-9]*)} [$P(pathname) tag names [$P(pathname) index @$x,$y]] v v] {
	set newped [split $v -]
	if ![In $newped $P(selection)] {
	    HListDragDrop $name $P(selection) $newped
	}
    }
    $P(pathname) configure -cursor top_left_arrow
}    

proc HListDragDrop {name selection destped} {
    upvar \#0 $name-hlparams P
    set ped1 [lindex $P(selection) 0]
    set item1 [HListGetItem $name $ped1]
    set item2 [HListGetItem $name $destped]
    if [HListPedLess $ped1 $destped] {
	HList $name insert $destped $item1
	HList $name delete $ped1
    } else {
	HList $name delete $ped1
	HList $name insert $destped $item1
    }
}

proc HListItemSelect {name ped} {
    upvar \#0 $name-hlparams P
    if [string match $P(selectmode) single] {
	HListClearSelection $name
    }
    if ![In $ped $P(selection)] {
	lappend P(selection) $ped
    }
    $P(pathname) tag configure Ped-[join $ped -] \
	-foreground $P(selectforeground) -background $P(selectbackground)
    if ![Empty [set w [HListGetIconPath $name $ped]]] {
	$w configure\
	    -foreground $P(selectforeground) -background $P(selectbackground)
    }
}   

proc HListClearSelection {name} {
    upvar \#0 $name-hlparams P
    if [string match $P(selectmode) single] {
	foreach ped $P(selection) {
	    HListItemDeselect $name $ped
	}
    } else {
    }
}

proc HListItemDeselect {name ped} {
    upvar \#0 $name-hlparams P
    set i [lsearch $P(selection) $ped]
    set P(selection) [lreplace $P(selection) $i $i]
    $P(pathname) tag configure Ped-[join $ped -] \
	-foreground $P(foreground) -background $P(background)
    if ![Empty [set w [HListGetIconPath $name $ped]]] {
	$w configure\
	    -foreground $P(foreground) -background $P(background)
    }
}   

#################
# Mouse-3 menus

proc HList3 {x y name ped mode} {
    HList1 $name $ped
    set w .hlmenu-$name
    if [winfo exists $w$mode] {
	tk_popup $w$mode $x $y
    } elseif [winfo exists $w] {
	tk_popup $w $x $y
	grab $w
    }
}

proc HListMenus {name} {
    upvar \#0 $name-hlparams P
    foreach mode [concat {{}} $P(modes)] {
	set m .hlmenu-$name$mode
	if ![Empty $mode] {
	    set itemsvar P(menu3,mode)
	} else {
	    set itemsvar P(menu3)
	}
	if ![info exists $itemsvar] return
	set items [set $itemsvar]
	catch {destroy $m}
	menu $m -tearoff 0
	foreach item $items {
	    $m add command -label [lindex $item 0]\
		-command [lindex $item 1]
	}
    }
}

