#
#  ItemInspector.tcl ---
#  
#  Part of the Whiteboard application. 
#  It lets the user inspect and configure item options in the canvas.
#  The options are organized into a list as:
#  listOfAllOptions = {{-option oldValue entryWidget} {...} ...}
#   
#  The whiteboard application has been developed by:
#  
#	Mats Bengtsson, 
#	Hamngatan 21,
#	SE-58226, Linkoping Sweden,
#	matben@lin.foa.se,
#	phone: +46 13 136114
# 
#  It is distributed under the standard GPL.
#  See the README file for further details.
#  
#------------------------------------------------------------------------

proc ItemInspector  { wCan }  {
    global  sysFont finishedItemInspect itInspect prefs debugLevel
    
    set nl_ {\\n}
    if {$debugLevel >= 2}  {  
	puts "ItemInspector (entry)::"
    }
    # First get the item id.
    set itemId [$wCan find withtag current]
    if {[llength $itemId] == 0}  {
	return
    }
    set itPrefNo [CanvasGetItnoFrom $wCan current]
    if {[llength $itPrefNo] == 0}  {
	return
    }
    # Movies may not be selected this way; temporary solution?
    if {[lsearch [$wCan gettags $itPrefNo] "movie"] >= 0}  {
	return
    }
    
    # Write this container as a simple proc with automatic sizing.
    set w .itinsp
    catch {toplevel $w}
    wm title $w "Item Inspector"
    
    # Global frame.
    pack [frame $w.frall -borderwidth 1 -relief raised] -fill both -expand 1
    set w1 [frame $w.frall.fr1]    
    set wcont1 [LabeledFrame $w1 "Item Options"]

    # Overall frame for whole container.
    set frtot [frame $wcont1.frin]
    pack $frtot -padx 10 -pady 10
    
    # List available options of the option menus.
    array set theMenuOpts {arrow {none first last both}  \
      capstyle {butt projecting round} joinstyle {bevel miter round}  \
      smooth {false true} stipple {none gray75 gray50 gray25 gray12}  \
      outlinestipple {none gray75 gray50 gray25 gray12}}
    set listOfAllOptions {}
    
    # Item type.
    set iLine 0
    set itemType [$wCan type $itemId]
    label $frtot.lbl$iLine -text "Item type:" -font $sysFont(sb)
    entry $frtot.ent$iLine -width 30 -font $sysFont(s)
    $frtot.ent$iLine insert end $itemType
    $frtot.ent$iLine configure -state disabled
    grid $frtot.lbl$iLine -column 0 -row $iLine -sticky e -padx 2 -pady 1
    grid $frtot.ent$iLine -column 1 -row $iLine -sticky w -padx 2 -pady 1
    lappend listOfAllOptions [list type $itemType $frtot.ent$iLine]
    
    # Coordinates.
    incr iLine
    label $frtot.lbl$iLine -text "Coordinates:" -font $sysFont(sb)
    entry $frtot.ent$iLine -width 30 -font $sysFont(s)
    set theCoords [$wCan coords $itemId]
    $frtot.ent$iLine insert end $theCoords
    $frtot.ent$iLine configure -state disabled
    grid $frtot.lbl$iLine -column 0 -row $iLine -sticky e -padx 2 -pady 1
    grid $frtot.ent$iLine -column 1 -row $iLine -sticky w -padx 2 -pady 1
    lappend listOfAllOptions [list coords $theCoords $frtot.ent$iLine]

    # Get all item options.
    set opts [$wCan itemconfigure $itemId]
    foreach opt $opts {
	incr iLine
	set op [lindex $opt 0]
	set val [lindex $opt 4]
	# If multine text, encode as one line with explicit "\n".
	if {$op == "-text"}  {
	    regsub -all "\n" $val $nl_ oneliner
	    regsub -all "\r" $oneliner $nl_ oneliner
	    set val $oneliner
	}
	set opname [string trim $op -]
	label $frtot.lbl$iLine -text "$opname:" -font $sysFont(sb)
	#puts "op=$op, val=$val"
	
	# Intercept some options for nontext output.
	if {$op == "-fill" || $op == "-outline"}  {
	    frame $frtot.ent$iLine
	    if {$val == ""}  {
		set itInspect($opname) transparent
	    } else {
		set itInspect($opname) fill
	    }
	    set wMenu [tk_optionMenu $frtot.menu$iLine itInspect($opname)  \
	      transparent fill]
	    $wMenu configure -font $sysFont(sb)
	    $frtot.menu$iLine configure -font $sysFont(sb) -highlightthickness 0  \
	      -background $prefs(bgColGeneral) -foreground black
	    entry $frtot.entent$iLine -font $sysFont(s) -width 4 -state disabled
	    if {$val != ""}  {
		$frtot.entent$iLine configure -background $val
	    }
	    pack $frtot.menu$iLine -in $frtot.ent$iLine -side left
	    pack $frtot.entent$iLine -in $frtot.ent$iLine  \
	      -side left -fill x -expand 1
	    bind $frtot.entent$iLine <Double-Button-1>   \
	      "ChooseItemColor $frtot.entent$iLine"
	} elseif {$op == "-tags"}  {
	    entry $frtot.ent$iLine -width 30 -font $sysFont(s) 
	    $frtot.ent$iLine insert end $val
	    $frtot.ent$iLine configure -state disabled
	    
	    # Pure menu options.
	} elseif {$op == "-arrow" || $op == "-capstyle" || $op == "-joinstyle" \
	  || $op == "-smooth" || $op == "-stipple" || $op == "-outlinestipple"}  {
	    if {$val == ""}  {
		set itInspect($opname) "none"
	    } else  {
		set itInspect($opname) $val
	    }
	    set wMenu [eval {tk_optionMenu $frtot.ent$iLine itInspect($opname)}  \
	      $theMenuOpts($opname)]
	    $wMenu configure -font $sysFont(sb) 
	    $frtot.ent$iLine configure -font $sysFont(sb) -highlightthickness 0  \
	      -background $prefs(bgColGeneral) -foreground black
	} else  {
	    entry $frtot.ent$iLine -width 30 -font $sysFont(s) 
	    $frtot.ent$iLine insert end $val
	}
	grid $frtot.lbl$iLine -column 0 -row $iLine -sticky e -padx 2 -pady 1
	if {$op == "-fill" || $op == "-outline"}  {
	    grid $frtot.ent$iLine -column 1 -row $iLine -sticky ew -padx 2 -pady 1
	} else  {
	    grid $frtot.ent$iLine -column 1 -row $iLine -sticky w -padx 2 -pady 1
	}
	if {$op == "-fill" || $op == "-outline"}  {
	    lappend listOfAllOptions [list $op $val $frtot.entent$iLine]
	} else  {
	    lappend listOfAllOptions [list $op $val $frtot.ent$iLine]
	}
    }
    pack $w1 -fill x
    
    # Button part.
    set frbot [frame $w.frall.frbot -borderwidth 0]
    pack [button $frbot.btconn -text "   Save   " -default active  \
      -command [list CanvasConfigureItem $wCan $itemId $listOfAllOptions]]  \
      -side right -padx 5 -pady 5
    pack [button $frbot.btcancel -text " Cancel "  \
      -command {set finishedItemInspect 0}]  \
      -side right -padx 5 -pady 5
    pack $frbot -side top -fill both -expand 1 -in $w.frall  \
      -padx 8 -pady 6
    
    wm resizable $w 0 0
    bind $w <Return> "$frbot.btconn invoke"
    
    # grab and focus
    focus $w
    catch {grab $w}
    tkwait variable finishedItemInspect
    catch {grab release $w}
    destroy $w
    if {$finishedItemInspect == 1}  {
	return {}
    } else {
	return {}
    }
}

proc CanvasConfigureItem  { wCan itemId listOfAllOptions }  {
    global  finishedItemInspect itInspect allIPnumsTo ipNum2Socket
    
    
    set itPrefNo [CanvasGetItnoFrom $wCan $itemId]

    # Loop through all options. Assemble a configure list.
    set allNewOpts {}
    foreach opt $listOfAllOptions {
	set op [lindex $opt 0]
	set val [lindex $opt 1]
	set entWid [lindex $opt 2]
	set opname [string trim $op -]
	#puts "op=$op, val=$val, entWid=$entWid"
	# Intercept to handle special cases.
	if {[string compare $op "type"] == 0}  {
	    # Do nothing
	    continue
	} elseif {[string compare $op "coords"] == 0}  {
	    
	    continue
	} elseif {([string compare $op "-fill"] == 0) ||  \
	  ([string compare $op "-outline"] == 0)}  {
	    set newOpt $itInspect($opname)
	    if {[string compare $newOpt "transparent"] == 0}  {
		set newVal {}
	    } else  {
		set newVal [$entWid cget -background]
	    }
	} elseif {([string compare $op "-arrow"] == 0) ||  \
	  ([string compare $op "-capstyle"] == 0) ||  \
	  ([string compare $op "-joinstyle"] == 0) ||  \
	  ([string compare $op "-smooth"] == 0)}  {
	    set newVal $itInspect($opname)
	} elseif {([string compare $op "-stipple"] == 0) ||  \
	  ([string compare $op "-outlinestipple"] == 0)}  {
	    set newOpt $itInspect($opname)
	    if {[string compare $newOpt "none"] == 0}  {
		set newVal {}
	    } else  {
		set newVal $itInspect($opname)
	    }
	} else  {
	    set newVal [$entWid get]
	}
	#puts "newVal=$newVal"
	# If new different from old, reconfigure. Reinterpret \n"
	if {[string compare $val $newVal] != 0}  {
	    $wCan itemconfigure $itemId $op  \
	      [subst -nocommands -novariables $newVal]
	    lappend allNewOpts $op $newVal
	}
    }    
    # Let other also know.
    set cmd [concat itemconfigure $itPrefNo $allNewOpts]
    #puts "CANVAS: $cmd"
    foreach ip $allIPnumsTo  {
	puts $ipNum2Socket($ip) "CANVAS: $cmd"
    }
    
    set finishedItemInspect 1 
}

proc ChooseItemColor  { wEntry }  {
    
    set col [$wEntry cget -background]
    set col [tk_chooseColor -initialcolor $col]
    if {[string length $col] > 0}	 {
 	$wEntry configure -background $col
    }
}

#--------------------------------------------------------------------
