#  TinyHttpd.tcl ---
#  
#  Part of the Whiteboard application. 
#  Implements tiny part of a http server for the movies,
#  although it may be used as a standard http server.
#   
#  The whiteboard application has been developed by:
#  
#	Mats Bengtsson, 
#	Hamngatan 21,
#	SE-58226, Linkoping Sweden,
#	matben@lin.foa.se,
#	phone: +46 13 136114
# 
#  It is distributed under the standard GPL.
#  See the README file for further details.
#  
#------------------------------------------------------------------------

#  Set the base path of the http path. It is relative 'thisPath'.
set httpdRootPath httpd

#  Do we want a fixed base path are relative the unix '~' home path?
set httpdFixedPathQ 1

#  Make sure that we are in the directory of the application itself.
set thisPath [info script]
if {$thisPath != ""}  {
    cd [file dirname $thisPath]
}
array set ext2MimeType {.txt text/plain .html text/html .gif image/gif  \
  .jpeg image/jpeg .jpg image/jpeg .mov video/quicktime   \
  .sdp application/sdp .rtp application/x-rtsp .rtsp application/x-rtsp  \
  .mpg video/mpeg .mpeg video/mpeg .avi video/x-msvideo  \
  .aif audio/aiff .au audio/basic .wav audio/wav .png image/png  \
  .mid audio/midi}
set binExts {.gif .jpg .jpeg .mov .sdp .rtp .rtsp .mpg .mpeg .avi .aif .au  \
  .wav .png .mid}
array set httpMsg {200 OK 404 {File not found on server.}  \
  401 {No supported MIME type.}}
set debugServerLevel 2

proc DoStartHttpServer  { thisServPort }  {
    if {[catch {socket -server SetupHttpChannel $thisServPort} msg]}  {
	tk_messageBox -message "Couldn't start server socket: $msg." \
	  -icon error -type ok
    }	
}

proc SetupHttpChannel { channel ip port } {
    global  debugServerLevel 
    
    fileevent $channel readable "HandleHttpRequest $channel $ip $port"
    if {$debugServerLevel >= 2}  {
	puts "SetupHttpChannel:: channel=$channel, ip=$ip, port=$port"
    }
    # Everything should be done with 'fileevent'.
    fconfigure $channel -blocking 0
    fconfigure $channel -buffering line
}

proc HandleHttpRequest { channel ip port }  {
    global  debugServerLevel httpdRootPath ext2MimeType binExts httpMsg   \
      tcl_platform httpdFixedPathQ
    
    set path_ {[^ ]*}
    set blank_ {[ \n]+}
    set chunk 1024

    # If client closes socket.
    if {[eof $channel]} {
	if {$debugServerLevel >= 2}  {
	    #puts "HandleHttpRequest:: eof channel=$channel"
	}

    } elseif {[gets $channel line] != -1} {
	if {$debugServerLevel >= 2}  {
	    #puts "--->$ip:${port}:: $line"
	}	
	# We only implement the GET and HEAD operations.
	if {[regexp "^(GET|HEAD) +($path_) +HTTP" $line junk cmd path]} {
	    if {$debugServerLevel >= 2}  {
		puts "$cmd: path=$path, from [lindex [fconfigure $channel -peername] 1]"
	    }
	    # Remove fileevent to not mess up 'fcopy'.
	    fileevent $channel readable {}
	    
	    # Skip until first blank line.
	    set nbytes [gets $channel line]
	    #puts "$line"
	    while {![regexp "^$blank_$" $line] && $nbytes > 0}  {
		set nbytes [gets $channel line]
		#puts "$line"
	    }
	    # Trim leading '/'.
	    set path [string trim $path /]
	    
	    # Relative a fixed (still relative) path, or a unix-style (~) path.
	    if {$httpdFixedPathQ}  {
		set filePath [file join $httpdRootPath $path]
	    } elseif {$tcl_platform(platform) == "unix"}  {
		set filePath [glob -nocomplain $path]
	    } else {
		puts stderr "TinyHttpd: inconsistent file path option."
	    }
	    set fext [string tolower [file extension $filePath]]
	    set timeTxt [clock format [clock seconds] -format "%A, %d-%b-%y  %X"]
	    if {$debugServerLevel >= 2}  {
		puts "filePath=$filePath"
	    }
	    # Check that the file is there and opens correctly.
 	    if {$filePath == "" || [catch {open $filePath r} fid]}  {
		if {$debugServerLevel >= 2}  {
		    puts stderr "TinyHttpd: open $filePath failed"
		}
		puts $channel "HTTP/1.0  404 $httpMsg(404)\n"
		set filePath [file join $httpdRootPath msg404.html]
		catch {open $filePath r} fid
	    } elseif {![info exists ext2MimeType($fext)]}  {
		
		# Use extension to find mime type and transfer mode.
		if {$debugServerLevel >= 2}  {puts stderr "TinyHttpd: no mime type"}
		puts $channel "HTTP/1.0 401 $httpMsg(401)\n"
		set filePath [file join $httpdRootPath msg401.html]
		catch {open $filePath r} fid
	    } else  {
		
		# Put stuff.
		if {$debugServerLevel >= 2}  {puts "TinyHttpd: HTTP/1.0 200 OK"}
		puts $channel "HTTP/1.0 200 OK"
		puts $channel "server: MatsTinyHttpd/1.0"
		puts $channel "last-modified: $timeTxt"
		puts $channel "content-type: $ext2MimeType($fext)\n"
		flush $channel
	    }
	    if {$cmd == "HEAD"}  {
		close $channel
		close $fid
		return
	    }
	    # If binary data.
	    if {[lsearch $binExts $fext] >= 0}  {
		if {$debugServerLevel >= 2}  {puts "TinyHttpd: binary"}
	    	fconfigure $fid -translation binary
	    	fconfigure $channel -translation binary
	    }
	    # Background copy. Be sure to switch off all fileevents on channel.
	    fileevent $channel readable {}
	    fileevent $channel writable {}
	    set total 0
	    fcopy $fid $channel -command  \
		    [list HttpdCopyMore $fid $channel $chunk $total] -size $chunk
	}
    }
}

proc HttpdCopyMore {in out chunk total bytes {error {}}} {
    global  debugServerLevel 
    
    if {$debugServerLevel >= 2}  {
	puts "HttpdCopyMore:: total=$total, bytes=$bytes"
    }
    incr total $bytes
    if {([string length $error] != 0) || [eof $in]}  {
	puts "Reached eof or error=$error."
	catch {puts "\nFinished: $total bytes to [lindex [fconfigure $out -peername] 1]"}
	close $in
	close $out
    } else {
	puts -nonewline "."
	#update idletasks
	update
	fcopy $in $out -command  \
		[list HttpdCopyMore $in $out $chunk $total] -size $chunk
    }
}

proc Parse { url }  {

    set serv_ {[^:/]+}
    set port_ {:[0-9]+}
    set path_ {[^ ]*}
    regexp -nocase "^(http://)?($serv_)($port_)?(/$path_)$" $url  \
      match protocol domain port path
}

DoStartHttpServer 8011

#---------------------------------------------------------------------
