#!/bin/sh
# the next line restarts using wish \
	exec wish "$0" "$@"

#------------------------------------------------------------------------
#
#  Whiteboard090.tcl ---
#  
#  This is a whiteboard application, that is, a simple drawing program
#  which also includes text in addition to the usual elements like
#  lines, rectangles, ellipsis etc. The application includes its own
#  server, so that several clients can be connected in a pairwise
#  manner. This way a completely symmetric communication state is 
#  achieved. The protocol is built on top of the built in TCP/IP,
#  and consists, with a few exceptions, only of a canvas command.
#  
#  This whiteboard application has been developed by:
#  
#	Mats Bengtsson, 
#	Hamngatan 21,
#	SE-58226, Linkoping Sweden,
#	matben@lin.foa.se,
#	phone: +46 13 136114
# 
#  It is distributed under the standard GPL.
#  See the README file for further details.
#  
#------------------------------------------------------------------------

#  The ip number is central to all administration of connections.
#  Each connection has a unique ip number from which all other necessary
#  variables are looked up using arrays:
#  
#  ipNum2Name: maps ip number to the specific domain name.
#  ipName2Num: inverse of above.
#  ipNum2Socket: maps ip number to the specific socket.
#  ipNum2ServPort: maps ip number to the specific remote server port number.
#  ipNum2nEnt: maps ip number to the entry line in the connect panel from
#     top (0) and down.
#  ipNum2User: maps ip number to the user name.
#
#  To keep track of where images/movies exist at this clients disk
#  we have two synced lists 'knownFiles' and 'knownPaths' where an element
#  in 'knownFiles' which contains a files tail corresponds precisely
#  to this files full path in 'knownPaths'.
#  Also we have an array 'imageItno2FileTail' which gives the tail of 
#  the file from the items 'itpref/itno'.

#  Make sure that we are in the directory of the application itself.
set thisPath [info script]
if {$thisPath != ""}  {
    cd [file dirname $thisPath]
}
set thisPath [pwd]
set launchSecs [clock seconds]

# These are auxilary procedures that make there own top windows.
foreach sourceName {  \
  mkOpenConnection.tcl   \
  mkProgressWindow.tcl   \
  mkShowInfoServer.tcl   \
  mkShowInfoClients.tcl  \
  mkGetCanvas.tcl        \
  mkEditShortcuts.tcl    \
  mkOpenMulticast.tcl    \
  PreferencesUtils.tcl   \
  ItemInspector.tcl}  {
    if {[catch {source $sourceName} msg]}  {
	tk_messageBox -message "Error:  $msg" -icon error -type ok
	exit
    }    
}
# Quick time package if available.
if {$tcl_platform(platform) == "macintosh" || $tcl_platform(platform) == "windows"}  {
    if {![catch {package require QuickTimeTcl} msg]}  {
	set prefs(QT) 1
    } else {
	set prefs(QT) 0
	after idle {tk_messageBox -message   \
	  "There exists a very cool QuickTime plugin for Macs and Windows."\
	  -icon info -type ok}
    }
} else  {
    set prefs(QT) 0
}
if {$prefs(QT) && [catch {source MovieController.tcl} msg]}  {
    tk_messageBox -message "Error:  $msg" \
       -icon error -type ok
    exit
}
# Http package can be useful if using QuickTime.
if {![catch {package require http} msg]}  {
    set prefs(http) 1
} else {
    set prefs(http) 0
}
# Mac Speech via PlainTalk if available.
if {$tcl_platform(platform) == "macintosh"}  {
    if {![catch {package require MacSpeech} msg]}  {
	set prefs(MacSpeech) 1
	#speak -voice Victoria {This is Victoria speaking. \
	#  Welcome to this fabolous program.}
    } else {
	set prefs(MacSpeech) 0
    }
} else  {
    set prefs(MacSpeech) 0
}
# ip numbers, port numbers, and names.
set thisServPort 1025
set remoteServPort 1025
set internalIPnum 127.0.0.1
set internalIPname "localhost"
# Set our IP number temporarily.
set thisIPnum $internalIPnum 

# Find user name.
if {[info exists env(USER)]}  {
    set thisUserName $env(USER)
} elseif {[info exists env(LOGIN)]}  {
    set thisUserName $env(LOGIN)
} elseif {[info exists env(USERNAME)]}  {
    set thisUserName $env(USERNAME)
} else {
    set thisUserName "Unknown"
}

# Keep lists of ip numbers for connected clients and servers.
set allIPnums {}
set allIPnumsTo {}

# Keep synced lists of all known file names (tails) and their corresponding
# paths. The position of a pair name-path in each list must be identical!
set knownFiles {}
set knownPaths {}
    
# Other inits.
set listCommTo {}
set listCommFrom {}
# Status message shown in the status line.
set statMess {This line contains the status message!}
# Main canvas widget path.
set wCan .fmain.can
set servCan $wCan
set specServCan .serv.ca
set btFrame .fmain.frleft.frbt
set commFrame .fcomm.ent
set wStatMess .fcomm.stat.lbl
# Toplevel for progress window.
set wProgWin .pr
# Canvas size; these are also min sizes.
set dims(hCanOri) 328
set dims(wCanOri) 350
set dims(x) 30
set dims(y) 30
# Total size of the application (not including menu); only temporary values.
set dims(wRoot) 1    
set dims(hRoot) 1
# As above but including the menu.
set dims(wTot) 1    
set dims(hTot) 1
# Running tag number must be unique for each item.
# It is always used in conjunction with the local prefix as $prefix/$itno.
# It is *never* reused during the lifetime of the application.
# It is updated *only* when writing to own canvas.
# When the server part writes to the canvas it is the prefix/no of the
# remote client that is used. Perhaps [clock seconds] may be used? 
# Should be a positive number though.
set itno 0
# Unique tag prefix for items created by this client.
set myItpref [info hostname]
if {$myItpref == ""}  {
    set myItpref $internalIPname
}
# On multiuser platforms (unix) prepend the user name; no spaces allowed.
if {$tcl_platform(platform) == "unix" && [info exists env(USER)]}  {
    set user $thisUserName
    regsub -all " " $user "" user
    set myItpref ${user}@$myItpref
}
# Running number for entries in the comm frame.
set nEnt 0

# Mappings file extension to transfer mode; binary or text.
# Supported binary files, that is, images movies etc.
# Start with the core Tk supported formats.
set supExts(txt) {".txt" ".tcl" ".TXT" ".TCL"}
set supExts(image) {".gif" ".GIF"}
set supExts(movie) {}
set supExts(bin) [concat $supExts(image) $supExts(movie)]
set supExts(all) [concat $supExts(txt) $supExts(bin)]

# Mac "TYPE".
set supTypes(txt) {TEXT}
set supTypes(image) {GIFf GIFF}
set supTypes(movie) {}

# For QuickTime: (there are many more...)
if {$prefs(QT)}  {
    set supExts(image) [concat $supExts(image)  \
      {".jpg" ".jpeg" ".tif" ".tiff" ".png" \
      ".JPG" ".JPEG" ".TIF" ".TIFF" ".PNG"}]
    set supExts(movie) [concat $supExts(movie)  \
      {".mov" ".mp3" ".mpg" ".mpeg" ".au" \
      ".MOV" ".MP3" ".MPG" ".MPEG" ".AU"}]
    set supExts(bin) [concat $supExts(image) $supExts(movie)]
    # QuickTime should support every format given above.
    set supExts(all) [concat $supExts(txt) $supExts(bin)]
    # Mac "TYPE".
    set supTypes(image) [concat $supTypes(image) {JPEG PNGf}]
    set supTypes(movie) [concat $supTypes(movie) {MooV PLAY ULAW}]
} 

# Add more supported filetypes as additional extensions and Mac types.

# Mapping from Mac "TYPE" to file extension.
# Important: each mac type above *must* appear as an element in 'macType2Ext'.
set macType2Ext(TEXT) ".txt"
set macType2Ext(GIFf) ".gif"
set macType2Ext(GIFF) ".gif"
set macType2Ext(JPEG) ".jpg"
set macType2Ext(MooV) ".mov"
set macType2Ext(PLAY) ".mp3"
set macType2Ext(ULAW) ".au"
set macType2Ext(PNGf) ".png"

# Useful time constants in seconds.
set tmsec(min) 60
set tmsec(hour) [expr 60*$tmsec(min)]
set tmsec(day) [expr 24*$tmsec(hour)]
set tmsec(week) [expr 7*$tmsec(day)]
set tmsec(30days) [expr 30*$tmsec(day)]

#--- Preferences are set here --------------------------------------
# These are the hardcoded, application default, values, and can be
# overridden by the ones in user default file.
# The tool buttons.
set prefs(btState) 00
set prefs(btStateOld) 00
#
set prefs(bgColCanvas) #dedede
set prefs(bgColGeneral) #dedede
# fg and bg colors set in color selector; bgCol always white.
set prefs(fgCol) black
set prefs(bgCol) white
# Line thickness.
set prefs(penThick) 1	
# Brush thickness.
set prefs(brushThick) 8	
# Fill color for circles, polygons etc.
set prefs(fill) 0
# If and how polygons should be smoothed.
set prefs(smooth) 0
set prefs(splinesteps) 0
# Offset when duplicating canvas items and when opening images and movies.
set prefs(offsetCopy) 16
# Present anchor coordinates when importing images and movies.
# Gets translated with 'prefs(offsetCopy)' for each new imported item.
set importAnchor(x) $prefs(offsetCopy)
set importAnchor(y) $prefs(offsetCopy)
# Side of selecting box .
set prefs(aBBox) 2
# Font prefs set in menus. Sizes according to html.
set prefs(fontSize) 2
set prefs(font) Helvetica
set prefs(fontWeight) normal
# Html sizes or point sizes when transferring text items?
set prefs(useHtmlSizes) 1
# Grid on or off.
set prefs(canGridOn) 0                  
# Grid spacing.
set prefs(gridDist) 40                 
# Only manipulate own items?
set prefs(privacy) 0                
# Mark bounding box (1) or each coords (0).
set prefs(bboxOrCoords) 0
# Connect automatically to connecting clients.
set prefs(autoConnect) 1                
# Disconnect automatically to disconnecting clients.
set prefs(autoDisconnect) $prefs(autoConnect)	
# When connecting to other client, connect automatically to all *its* clients.
set prefs(multiConnect) 1
# Start server when launching application?
set autoStartServer 1             
# Open connection in async mode.
set prefs(asyncOpen) 1
# Safe server interpretator? (not working)
set makeSafeServ 0	
# Maximum time to wait for server to respond.
set prefs(timeout) 20
# When and how old is a cached file allowed to be before downloading a new?
set prefs(checkCache) "never"
# Animated 'Coccinella'?
if {$prefs(QT)}  {
    set prefs(coccinellaMovie) 0
} else  {
    set prefs(coccinellaMovie) 0
}
# If it is the first time the application is launched, then welcome.
set prefs(firstLaunch) 1
# The file name of the welcoming canvas.
set prefs(welcomeFile) myWelcome.can
#---- Shortcuts -----------------------------------------------------------
#----   domain names for open connection ----------------------------------
set shortcuts [list [list  \
  "user specified" mbmac laelia atlas vanessa other]   \
  [list  \
  remote.computer.name mbmac.lin.foa.se laelia.lin.foa.se   \
  atlas.sto.foa.se vanessa.lin.foa.se other.computer.name]]
#----   url's for streaming live movies -----------------------------------
set shortsMulticastQT {{   \
  "user specified"   \
  Bloomberg          \
  "Hard Radio"       \
  NPR  \
  "BBC World TV" }  {  \
  {}  \
  www.apple.com/quicktime/showcase/radio/bloomberg/bloombergradio.mov  \
  www.apple.com/quicktime/showcase/radio/hardradio/hardradio.mov  \
  www.apple.com/quicktime/showcase/radio/npr/npr.mov  \
  www.apple.com/quicktime/favorites/bbc_world1/bbc_world1.mov}}
#--------------------------------------------------------------------------
set compNameOrNum [lindex [lindex $shortcuts 1] 0]
# Number of bytes in copy buffert.
set chunkSize 4096
set chunkSize 8192
# Use a very unlikely combination for the separator of items in clipboard.
set clipItemSep " ANDqzU\06 "
# Default for 'clipToken' should always be "string" to be prepared
# for imports from other apps.
set clipToken "string"
# Level of detail for printouts. >= 2 for my outputs.
set debugLevel 0
# Level of detail for printouts for server. >= 2 for my outputs.
set debugServerLevel 0
# Certain special non-printable chars for signals in PUT/GET mode.
set putCode(ACK) \x06
set putCode(FIL) \x03
set putCode(EXT) \x04
set putCode(NAK) \x15
# Messages connected to these codes.
set putCode2Msg(ACK) "Correct response. Accepts file."
set putCode2Msg(FIL) "File already cached."
set putCode2Msg(EXT) "File type not supported."
set putCode2Msg(NAK) "Error: other server problems."

if {$tcl_platform(platform) == "macintosh" && $debugLevel == 0 &&  \
  $debugServerLevel == 0}  {
    console hide
}
# Modifier keys and meny height (guess); add canvas border as well.
# System fonts used. Other system dependent stuff.
if {$tcl_platform(platform) == "unix"}	{
    set osprefs(mod) Control
    set sysFont(s) {Helvetica 10 normal}
    set sysFont(sb) {Helvetica 10 bold}
    array set fontSize2Points {1 10 2 12 3 14 4 18 5 24 6 36}
    array set fontPoints2Size {10 1 12 2 14 3 18 4 24 5 36 6}
    set clockClicksPerSec 1000000
    set prefs(userPrefsFilePath) "~/.whiteboard"
} elseif {$tcl_platform(platform) == "macintosh"}  {
    set osprefs(mod) Command
    set sysFont(s) {Geneva 10 normal}
    set sysFont(sb) {Geneva 10 bold}
    array set fontSize2Points {1 10 2 12 3 14 4 18 5 24 6 36}
    array set fontPoints2Size {10 1 12 2 14 3 18 4 24 5 36 6}
    set clockClicksPerSec 1000000
    if {[info exists env(PREF_FOLDER)]}  {
	set prefs(userPrefsFilePath) [file join $env(PREF_FOLDER) "Whiteboard Prefs"]
    } else  {
	set prefs(userPrefsFilePath) "Whiteboard Prefs"
    }
} elseif {$tcl_platform(platform) == "windows"}	 {
    set osprefs(mod) Control
    set sysFont(s) {Arial 7 normal}
    set sysFont(sb) {Arial 7 bold}
    array set fontSize2Points {1 6 2 8 3 9 4 11 5 14 6 20}
    array set fontPoints2Size {6 1 8 2 9 3 11 4 14 5 20 6}
    set clockClicksPerSec 1000000
    if {[info exists env(USERPROFILE)]}  {
	set prefs(userPrefsFilePath) [file join $env(USERPROFILE) "Whiteboard Prefs"]
    } else  {
	set prefs(userPrefsFilePath) "Whiteboard Prefs"
    }
}

# Set default light gray background.
if {$tcl_platform(platform) == "macintosh" || $tcl_platform(platform) == "unix"  \
  || $tcl_platform(platform) == "windows"}  {
    option add *Canvas.Background $prefs(bgColGeneral)
    option add *Frame.Background $prefs(bgColGeneral)
    option add *Label.Background $prefs(bgColGeneral)
    option add *Text.Background $prefs(bgColGeneral)
    option add *Message.Background $prefs(bgColGeneral)
    option add *Radiobutton.Background $prefs(bgColGeneral)
    option add *Entry.Background white
    option add *Entry.BorderWidth 1
}
if {$tcl_platform(platform) == "macintosh"}  {
    option add *Checkbutton.Background $prefs(bgColGeneral)
    option add *Button.HighlightBackground $prefs(bgColGeneral)
    #option add *Entry.Font application
    option add *Entry.Font $sysFont(s)
    option add *Listbox.Font $sysFont(s)
}
#tk_setPalette background $prefs(bgColGeneral) activeForeground white  \
#  selectBackground black selectForeground white highlightColor black
#option add *Entry.Background white
#option add *Listbox.Background white

# Manage the user preferences. Start by reading the preferences file.
PreferencesInit

# Set the user preferences from the preferences file if they are there,
# else take the hardcoded defaults.
# 'thePrefs': a list of lists where each sublist defines an item in the
# following way:  {theVarName itsResourceName itsHardCodedDefaultValue}.
PreferencesAdd [list  \
  [list prefs(btState)       prefs_btState       $prefs(btState)]  \
  [list prefs(bgColCanvas)   prefs_bgColCanvas   $prefs(bgColCanvas)]  \
  [list prefs(fgCol)         prefs_fgCol         $prefs(fgCol)]  \
  [list prefs(penThick)      prefs_penThick      $prefs(penThick)]  \
  [list prefs(brushThick)    prefs_brushThick    $prefs(brushThick)]  \
  [list prefs(fill)          prefs_fill          $prefs(fill)]  \
  [list prefs(fontSize)      prefs_fontSize      $prefs(fontSize)] \
  [list prefs(font)          prefs_font          $prefs(font)]  \
  [list prefs(fontWeight)    prefs_fontWeight    $prefs(fontWeight)]  \
  [list prefs(smooth)        prefs_smooth        $prefs(smooth)]  \
  [list prefs(splinesteps)   prefs_splinesteps   $prefs(splinesteps)]  \
  [list prefs(canGridOn)     prefs_canGridOn     $prefs(canGridOn)]  \
  [list prefs(privacy)       prefs_privacy       $prefs(privacy)]   \
  [list prefs(checkCache)    prefs_checkCache    $prefs(checkCache)]  \
  [list shortcuts            shortcuts           $shortcuts]  \
  [list shortsMulticastQT    shortsMulticastQT   $shortsMulticastQT]  \
  [list dims(x)              dims_x              $dims(x)]  \
  [list dims(y)              dims_y              $dims(y)]  \
  [list dims(wRoot)          dims_wRoot          $dims(wRoot)]  \
  [list dims(hRoot)          dims_hRoot          $dims(hRoot)]  \
  [list prefs(firstLaunch)   prefs_firstLaunch   $prefs(firstLaunch)]   \
  [list prefs(MacSpeech)     prefs_MacSpeech     $prefs(MacSpeech)]  ]  

# Function definitions
#--------------------------------------------------------------------------

#   LabeledFrame ---
#
#   A small utility that makes a nice frame with a label.
#   'wpath' is the widget path of the parent (it should be a frame); 
#   the return value is the widget path to the interior of the container.

proc LabeledFrame { wpath txt {opts {}} }  {
    global  sysFont
    
    pack [frame $wpath.st -borderwidth 0]  \
      -side top -fill both -pady 2 -padx 2 -expand true
    pack [frame $wpath.st.fr -relief groove -bd 2]  \
      -side top -fill both -expand true -padx 10 -pady 10 -ipadx 0 -ipady 0  \
      -in $wpath.st
    place [label $wpath.st.lbl -text $txt -font $sysFont(sb) -bd 0 -padx 6]  \
      -in $wpath.st -x 20 -y 14 -anchor sw
    return $wpath.st.fr
}

proc max { a b }  {
    return [expr ($a >= $b) ? $a : $b]
}

proc min { a b }  {
    return [expr ($a <= $b) ? $a : $b]
}

proc CreateAllButtons {	 }	{
    global  btFrame btNo2Name btName2No idColSel prefs wColSel
    
    array set btNo2Name	 \
      {00 point 01 move 10 line 11 arrow 20 rect 21 oval 30 pen 31 brush  \
      40 text 41 del 50 paint 51 poly}
    array set btName2No	 \
      {point 00 move 01 line 10 arrow 11 rect 20 oval 21 pen 30 brush 31  \
      text 40 del 41 paint 50 poly 51}
    
    for {set icol 0} {$icol <= 1} {incr icol}  {
	for {set irow 0} {$irow <= 5} {incr irow}  {
	    set fname bt_$btNo2Name($irow$icol)_off.gif
	    image create photo im_off$irow$icol -format gif	 \
	      -file [file join images $fname]
	    set fname bt_$btNo2Name($irow$icol)_on.gif
	    image create photo im_on$irow$icol -format gif	\
	      -file [file join images $fname]
	    set lwi [label $btFrame.bt$irow$icol -image im_off$irow$icol \
	      -borderwidth 0]
	    grid $lwi -row $irow -column $icol -padx 0 -pady 0
	    bind $lwi <Button-1> "ClickToolButton %W $btNo2Name($irow$icol)"
	}
    }
    #puts "CreateAllButtons:: set bindings ClickToolButton"
    
    # Color selector.
    image create photo imcolor -format gif -file [file join images col_switch.gif]
    set imheight [image height imcolor]
    set wColSel [canvas $btFrame.cacol -width 56 -height $imheight  \
      -highlightthickness 0]
    $btFrame.cacol create image 0 0 -anchor nw -image imcolor
    set idColSel [$btFrame.cacol create rect 7 7 33 30	\
      -fill $prefs(fgCol) -outline {}]
    $btFrame.cacol bind $idColSel <Button-1> {ColorSelector $prefs(fgCol)}
    
    # Black and white reset rectangle.
    image create photo bwrect -format gif -file [file join images transparent_rect.gif]
    #	set idColBW [$btFrame.cacol create rect 38 4 52 18]
    set idBWReset [$btFrame.cacol create image 4 34 -anchor nw -image bwrect]
    $btFrame.cacol bind $idBWReset <Button-1>  \
      "$wColSel itemconfigure $idColSel -fill black;  \
      set prefs(fgCol) black; set prefs(bgCol) white"
    
    # bg and fg switching.
    set idBWSwitch [$btFrame.cacol create image 38 4 -anchor nw -image bwrect]
    $btFrame.cacol bind $idBWSwitch <Button-1> \
      "SwitchBgAndFgCol $wColSel $idColSel"
    grid $btFrame.cacol -  -padx 0 -pady 0
}

proc SwitchBgAndFgCol { wColSel idColSel }  {
    global  prefs
    
    $wColSel itemconfigure $idColSel -fill $prefs(bgCol)
    set tmp $prefs(fgCol)
    set prefs(fgCol) $prefs(bgCol)
    set prefs(bgCol) $tmp
}

#   StartStopAnimatedWave, AnimateWave ---
#
#   Utility routines for animating the wave in the status message frame.

proc StartStopAnimatedWave  { w start }  {
    global  animateWave
    
    # Define speed and update frequency. Pix per sec and times per sec.
    set speed 150
    set freq 16

    if {$start}  {
	# Check if not already started.
	if {[info exists animateWave]}  {
	    return
	}
	set animateWave(pix) [expr int($speed/$freq)]
	set animateWave(wait) [expr int(1000.0/$freq)]
	set animateWave(id) [$w create image 0 0 -anchor nw -image im_wave]
	$w lower $animateWave(id)
	set animateWave(x) 0
	set animateWave(dir) 1
	set animateWave(killId) [after $animateWave(wait) [list AnimateWave $w]]
    } elseif {[info exists animateWave(killId)]}  {
	after cancel $animateWave(killId)
	$w delete $animateWave(id)
	catch {unset animateWave}
    }
}

proc AnimateWave  { w }  {
    global  dims animateWave
    
    set deltax [expr $animateWave(dir) * $animateWave(pix)]
    incr animateWave(x) $deltax
    if {$animateWave(x) > [expr $dims(wStatMess) - 80]}  {
	set animateWave(dir) -1
    } elseif {$animateWave(x) <= -60}  {
	set animateWave(dir) 1
    }
    $w move $animateWave(id) $deltax 0
    set animateWave(killId) [after $animateWave(wait) [list AnimateWave $w]]
}
    
proc StartStopMovie  { w start }  {
    global  startStopMovie

    # .fmain.frleft.padphoto
    set movieTime [lindex [lindex [$w gettime] 0] 1]
    set movieDuration [lindex [lindex [$w gettime] 1] 1]
    set movieTimeScale [lindex [lindex [$w gettime] 2] 1]
    set startStopMovie($w,wait) [expr int(1000.0*$movieDuration/($movieTimeScale+0.0))]
    if {$start}  {
	if {$movieTime == 0}  {
	    $w play
	} else  {
	    $w settime 0
	}
	set startStopMovie($w,killId)  \
	  [after $startStopMovie($w,wait) [list StartStopMovie $w 1]]
    } else  {
	if {[info exists startStopMovie($w,killId)]}  {
	    after cancel $startStopMovie($w,killId)
	}
    }
}

#   ClickToolButton ----
#
#   Uhhh...  When a tool button is clicked. Mainly sets all button specific
#   bindings.

proc ClickToolButton { w btName }  {
    global  prefs btName2No wCan btFrame statMess debugLevel
    
    if {$debugLevel >= 2}  {
	#puts "ClickToolButton:: w=$w, btName=$btName"
    }
    set prefs(btState) $btName2No($btName)
    set irow [string index $prefs(btState) 0]
    set icol [string index $prefs(btState) 1]
    $btFrame.bt$irow$icol configure -image im_on$irow$icol
    if {$prefs(btState) != $prefs(btStateOld)}  {
	set irow [string index $prefs(btStateOld) 0]
	set icol [string index $prefs(btStateOld) 1]
	$btFrame.bt$irow$icol configure -image im_off$irow$icol
    }
    set prefs(btStateOld) $prefs(btState)
    RemoveAllBindings $wCan
    
    # Deselect text items.
    if {$btName != "text"}  {
	$wCan select clear
    }
    if {$btName == "del" || $btName == "text"}  {
	DeselectAll $wCan
    }
    # Cancel any outstanding polygon drawings.
    PolyFinish $wCan -10 -10
    
    $wCan config -cursor {}
    
    switch $btName {
	point {
	    bind $wCan <Button-1>  \
	      {MarkBbox %W 0; BoxBegin %W [%W canvasx %x] [%W canvasy %y] rect}
	    bind $wCan <Shift-Button-1>	{MarkBbox %W 1}
	    bind $wCan <B1-Motion> { \
	      BoxDrag %W [%W canvasx %x] [%W canvasy %y] 0 rect 1}
	    bind $wCan <ButtonRelease-1>  \
	      {SendBox %W [%W canvasx %x] [%W canvasy %y] 0 rect 1}
	    bind $wCan <Double-Button-1> {ItemInspector %W}
	    set statMess "Point or drag to mark"
	}
	move {
	    # Bindings for moving items; movies need special class.
	    # The frame with the movie the mouse events, not the canvas.
	    bind $wCan <Button-1>  \
	      {InitXMove %W [%W canvasx %x] [%W canvasy %y]}
	    bind $wCan <B1-Motion> {DoXMove %W [%W canvasx %x] [%W canvasy %y]}
	    bind $wCan <ButtonRelease-1>  \
	      {SendXMove %W [%W canvasx %x] [%W canvasy %y]}
	    
	    # Moving single coordinates.
	    $wCan bind tbbox <Button-1>  \
	      {InitXMove %W [%W canvasx %x] [%W canvasy %y] point}
	    $wCan bind tbbox <B1-Motion>  \
	      {DoXMove %W [%W canvasx %x] [%W canvasy %y] point}
	    $wCan bind tbbox <ButtonRelease-1>  \
	      {SendXMove %W [%W canvasx %x] [%W canvasy %y] point}
	    
	    # Moving movies.
	    bind QTFrame <Button-1> {InitXMove $wCan  \
	      [$wCan canvasx [expr [winfo x %W] + %x]]  \
	      [$wCan canvasy [expr [winfo y %W] + %y]] movie}
	    bind QTFrame <B1-Motion> {DoXMove $wCan  \
	      [$wCan canvasx [expr [winfo x %W] + %x]]  \
	      [$wCan canvasy [expr [winfo y %W] + %y]] movie}
	    bind QTFrame <ButtonRelease-1> {SendXMove $wCan  \
	      [$wCan canvasx [expr [winfo x %W] + %x]]  \
	      [$wCan canvasy [expr [winfo y %W] + %y]] movie}
	    $wCan config -cursor hand2
	    set statMess "Move tool"
	}
	rect {
	    # Bindings for rectangle drawing.
	    bind $wCan <Button-1>  \
	      {BoxBegin %W [%W canvasx %x] [%W canvasy %y] rect}
	    bind $wCan <B1-Motion> { \
	      BoxDrag %W [%W canvasx %x] [%W canvasy %y] 0 rect}
	    bind $wCan <Shift-B1-Motion> { \
	      BoxDrag %W [%W canvasx %x] [%W canvasy %y] 1 rect}
	    bind $wCan <ButtonRelease-1>  \
	      {SendBox %W [%W canvasx %x] [%W canvasy %y] 0 rect}
	    bind $wCan <Shift-ButtonRelease-1>  \
	      {SendBox %W [%W canvasx %x] [%W canvasy %y] 1 rect}
	    set statMess "Rectangle tool"
	}
	oval {
	    bind $wCan <Button-1>  \
	      {BoxBegin %W [%W canvasx %x] [%W canvasy %y] oval}
	    bind $wCan <B1-Motion>	\
	      {BoxDrag %W [%W canvasx %x] [%W canvasy %y] 0 oval}
	    bind $wCan <Shift-B1-Motion>	\
	      {BoxDrag %W [%W canvasx %x] [%W canvasy %y] 1 oval}
	    bind $wCan <ButtonRelease-1>  \
	      {SendBox %W [%W canvasx %x] [%W canvasy %y] 0 oval}
	    bind $wCan <Shift-ButtonRelease-1>  \
	      {SendBox %W [%W canvasx %x] [%W canvasy %y] 1 oval}
	    set statMess "Oval tool"
	}
	text {
	    CanvasEditBind $wCan
	    set statMess "Text tool"
	}
	del {
	    bind $wCan <Button-1>  \
	      {DeleteItem %W [%W canvasx %x] [%W canvasy %y]}
	    bind QTFrame <Button-1> {DeleteItem $wCan  \
	      [$wCan canvasx [expr [winfo x %W] + %x]]  \
	      [$wCan canvasy [expr [winfo y %W] + %y]] movie %W}
	    set statMess "Click to delete"
	}
	pen {
	    bind $wCan <Button-1>  \
	      {StrokeBegin %W [%W canvasx %x] [%W canvasy %y]}
	    bind $wCan <B1-Motion>	\
	      {StrokeDrag %W [%W canvasx %x] [%W canvasy %y]}
	    bind $wCan <ButtonRelease-1>  \
	      {StrokeEnd %W [%W canvasx %x] [%W canvasy %y]}
	    set statMess "Pen tool"
	}
	brush {
	    bind $wCan <Button-1>  \
	      {StrokeBegin %W [%W canvasx %x] [%W canvasy %y]}
	    bind $wCan <B1-Motion>	\
	      {StrokeDrag %W [%W canvasx %x] [%W canvasy %y] $prefs(brushThick)}
	    bind $wCan <ButtonRelease-1>  \
	      {StrokeEnd %W [%W canvasx %x] [%W canvasy %y] $prefs(brushThick)}
	    set statMess "Brush tool"
	}
	line {
	    bind $wCan <Button-1>  \
	      {LineBegin %W [%W canvasx %x] [%W canvasy %y]}
	    bind $wCan <B1-Motion>	\
	      {LineDrag %W [%W canvasx %x] [%W canvasy %y] 0}
	    bind $wCan <Shift-B1-Motion>	\
	      {LineDrag %W [%W canvasx %x] [%W canvasy %y] 1}
	    bind $wCan <ButtonRelease-1>  \
	      {LineEnd %W [%W canvasx %x] [%W canvasy %y] 0}
	    bind $wCan <Shift-ButtonRelease-1>  \
	      {LineEnd %W [%W canvasx %x] [%W canvasy %y] 1}
	    set statMess "Line tool"
	}
	arrow {
	    bind $wCan <Button-1>  \
	      {LineBegin %W [%W canvasx %x] [%W canvasy %y] arrow}
	    bind $wCan <B1-Motion>	\
	      {LineDrag %W [%W canvasx %x] [%W canvasy %y] 0 arrow}
	    bind $wCan <Shift-B1-Motion>	\
	      {LineDrag %W [%W canvasx %x] [%W canvasy %y] 1 arrow}
	    bind $wCan <ButtonRelease-1>  \
	      {LineEnd %W [%W canvasx %x] [%W canvasy %y] 0 arrow}
	    bind $wCan <Shift-ButtonRelease-1>  \
	      {LineEnd %W [%W canvasx %x] [%W canvasy %y] 1 arrow}
	    set statMess "Arrow tool"
	}
	paint {
	    bind $wCan  <Button-1>  \
		    {DoPaint %W [%W canvasx %x] [%W canvasy %y]}
	    bind $wCan  <Alt-Button-1>  \
		    {DoPaint %W [%W canvasx %x] [%W canvasy %y] 1}
	    set statMess "Paint tool, alt-click to transparency (not working properly yet)"
	}
	poly {
            bind $wCan  <Button-1>  \
                    {PolySetPoint %W [%W canvasx %x] [%W canvasy %y]}
	    set statMess "Polygon tool, spacebar to cancel"
        }       
    }
    
}

proc CanvasEditBind { c }  {
    bind $c <Button-1> \
      {CanvasFocus %W [%W canvasx %x] [%W canvasy %y]}
    bind $c <Button-2> \
      {CanvasTextPaste %W [%W canvasx %x] [%W canvasy %y]}
    #	bind $c <<Cut>> {CanvasTextCopy %W; CanvasTextDelete %W}
    #	bind $c <<Copy>> {CanvasTextCopy %W}
    #	bind $c <<Paste>> {CanvasTextPaste %W}
    $c bind text <Button-1>	 \
      {CanvasTextHit %W [%W canvasx %x] [%W canvasy %y]}
    $c bind text <B1-Motion>  \
      {CanvasTextDrag %W [%W canvasx %x] [%W canvasy %y]}
    $c bind text <Double-Button-1>  \
      {CanvasTextSelectWord %W [%W canvasx %x] [%W canvasy %y]}
    $c bind text <Delete> {CanvasTextDelete %W}
    #$c bind text <Control-Key-d> {CanvasDelChar %W}
    $c bind text <Control-Key-d> {CanvasTextBackSpace %W}
    $c bind text <BackSpace> {CanvasTextBackSpace %W}
    $c bind text <Return> {CanvasNewLine %W}
    $c bind text <Any-Key> {CanvasTextInsert %W %A}
    $c bind text <Control-Any-Key> {CanvasTextInsert %W ""}
    $c bind text <Key-Right> {CanvasTextMoveRight %W}
    $c bind text <Key-Left> {CanvasTextMoveLeft %W}
    
    # Need some new string functions here.
    if {[info tclversion] >= 8.2}  {
	$c bind text <Key-Up> {CanvasTextMoveUpOrDown %W up}
	$c bind text <Key-Down> {CanvasTextMoveUpOrDown %W down}
    }
}

proc RemoveAllBindings { w }  {
    global  debugLevel
    
    if {$debugLevel >= 2}  {
	puts "RemoveAllBindings::"
    }
    $w bind text <Button-1> {}
    $w bind text <B1-Motion> {}
    $w bind text <Any-Key> {}
    $w bind text <Double-Button-1> {}
    
    # Remove bindings on markers on selected items.
    $w bind tbbox <Button-1> {}
    $w bind tbbox <B1-Motion> {}
    $w bind tbbox <ButtonRelease-1> {}
    
    bind $w <Button> {}
    bind $w <Button-1> {}
    bind $w <Button-Motion> {}
    bind $w <ButtonRelease> {}
    bind $w <Shift-Button-1> {}
    bind $w <Double-Button-1> {}
    bind $w <Any-Key> {}
    bind $w <ButtonRelease-1> {}
    bind $w <B1-Motion> {}
    bind $w <Shift-B1-Motion> {}
    bind $w <Shift-ButtonRelease-1> {}
    bind QTFrame <Button-1> {}
    bind QTFrame <B1-Motion> {}
    bind QTFrame <ButtonRelease-1> {}
    focus .
}

proc TraceStatusMessage { varName junk op }  {
    global  statMess wStatMess
    
    $wStatMess itemconfigure stattxt -text $statMess
}

#   TracePreferences ---
#
#   When certain preferences are set, they should trigger some change.

proc TracePreferences { varName index op }  {
    global  prefs dims wCan wColSel idColSel
    
    if {[string compare $varName "prefs"] == 0}  {
	switch $index {
	    bgColCanvas  {
		$wCan configure -bg $prefs(bgColCanvas)
	    }
	    fgCol  {
		$wColSel itemconfigure $idColSel -fill $prefs(fgCol)
		$wColSel raise $idColSel
	    }
	    canGridOn  {
		DoCanvasGrid
	    }
	}
    } elseif {[string compare $varName "dims"] == 0}  {
    
    }
}

#   InitXMove ---
#
#   'what' = "item": move an ordinary item.
#   'what' = "point": move one single point. Has always first priority.
#   'what' = "movie": QuickTime movie, make ghost rectangle instead.

proc InitXMove { w x y {what item} }  {
    global  xDrag debugLevel
    
    if {$debugLevel >= 2}  {
	puts "InitXMove:: w=$w, x=$x, y=$y, what=$what"
    }
    # If more than one item triggered, choose the "point".
    if {[info exists xDrag(what)] && $xDrag(what) == "point"}  {
	if {$debugLevel >= 2}  {
	    puts "InitXMove:: rejected"
	}
	return
    }
    set id_ {[0-9]+}
    set xDrag(what) $what
    set xDrag(baseX) $x
    set xDrag(baseY) $y
    set xDrag(anchorX) $x
    set xDrag(anchorY) $y
    set xDrag(type) [$w type current]
    if {$debugLevel >= 2}  {
	puts "InitXMove:: xDrag(type)=$xDrag(type), current=[CanvasGetItnoFrom $w current]"
    }    
    # If movie then make ghost rectangle. 
    # Movies do not obey the usual stacking order!
    if {$what == "movie"}  {
	set id [FindTypeFromOverlapping $w $x $y "movie"]
	if {$id == ""}  {
	    return
	}
	set it [CanvasGetItnoFrom $w $id]
	if {$it == ""}  {
	    return
	}
	$w addtag selectedmovie withtag $id
	set bbox [$w bbox $id]
	set x1 [expr [lindex $bbox 0] - 1]
	set y1 [expr [lindex $bbox 1] - 1]
	set x2 [expr [lindex $bbox 2] + 1]
	set y2 [expr [lindex $bbox 3] + 1]
	$w create rectangle $x1 $y1 $x2 $y2 -outline gray50 -width 3 \
	  -stipple gray50 -tags "ghostrect"

	# Highlight marker.
    } elseif {$what == "point"}  {
	
	# 'current' must be a marker with tag 'tbbox'.
	set id [$w find withtag current]
	$w addtag hitBbox withtag $id
	#puts "InitXMove:: w gettags current=[$w gettags current]"
	
	# Find associated id for the actual item. Saved in the tags of the marker.
	if {![regexp " +id($id_)" [$w gettags current] match theItemId]}  {
	    #puts "no match: w gettags current=[$w gettags current]"
	    return
	}
	set xDrag(type) [$w type $theItemId]
	if {($xDrag(type) == "text") || ($xDrag(type) == "image")}  {
	    #unset xDrag
	    return
	}
	# Make a highlightbox at the 'hitBbox' marker.
	set bbox [$w bbox $id]
	set x1 [expr [lindex $bbox 0] - 1]
	set y1 [expr [lindex $bbox 1] - 1]
	set x2 [expr [lindex $bbox 2] + 1]
	set y2 [expr [lindex $bbox 3] + 1]
	$w create rectangle $x1 $y1 $x2 $y2 -outline black -width 1 \
	  -tags "lightBbox id$theItemId" -fill white
	
	# Get the index of the coordinates that was 'hit'. Then update only
	# this coordinate when moving.
	# For rectangle and oval items a list with all four coordinates is used,
	# but only the hit corner and the diagonally opposite one are kept.
	set oldCoords [$w coords $theItemId]
	#puts "x= $x, y=$y, oldCoords=$oldCoords"
	if {([string compare $xDrag(type) "rectangle"] == 0) ||  \
	  ([string compare $xDrag(type) "oval"] == 0)}  {
	    
	    # Need to reconstruct all four coordinates as: 0---1
	    #                                              |   |
	    #                                              2---3
	    set fullListCoords [concat   \
	      [lindex $oldCoords 0] [lindex $oldCoords 1]  \
	      [lindex $oldCoords 2] [lindex $oldCoords 1]  \
	      [lindex $oldCoords 0] [lindex $oldCoords 3]  \
	      [lindex $oldCoords 2] [lindex $oldCoords 3] ]
	} else  {
	    set fullListCoords $oldCoords
	}
	# Find the one closest to the hit marker.
	set n [llength $fullListCoords]
	set minDist 1000
	for {set i 0} {$i < $n} {incr i 2}  {
	    set len [expr hypot([lindex $fullListCoords $i] - $x,  \
	      [lindex $fullListCoords [expr $i + 1]] - $y)]
	    if {$len < $minDist}  {
		set ind $i
		set minDist $len
	    }
	}
	set ptInd [expr $ind/2]
	#puts "ind=$ind, ptInd=$ptInd"
	if {([string compare $xDrag(type) "rectangle"] == 0) ||  \
	  ([string compare $xDrag(type) "oval"] == 0)}  {
	    
	    # Keep only hit corner and the diagonally opposite one.
	    set coords [concat [lindex $fullListCoords $ind]  \
	      [lindex $fullListCoords [expr $ind + 1]] ]
	    if {$ptInd == 0}  {
		set coords [lappend coords    \
		  [lindex $fullListCoords 6] [lindex $fullListCoords 7] ]
	    } elseif {$ptInd == 1}  {
		set coords [lappend coords    \
		  [lindex $fullListCoords 4] [lindex $fullListCoords 5] ]
	    } elseif {$ptInd == 2}  {
		set coords [lappend coords    \
		  [lindex $fullListCoords 2] [lindex $fullListCoords 3] ]
	    } elseif {$ptInd == 3}  {
		set coords [lappend coords    \
		  [lindex $fullListCoords 0] [lindex $fullListCoords 1] ]
	    }	    
	    set ind 0
	    set fullListCoords $coords
	}
	if {$debugLevel >= 2}  {
	    puts "ind=$ind, ptInd=$ptInd, fullListCoords=$fullListCoords"
	}
	set xDrag(hitInd) $ind
	set xDrag(coords) $fullListCoords
    } elseif {$what == "item"}  {
	
	# Add specific tag to the item being moved for later use.
	set id [$w find withtag current]
	$w addtag ismoved withtag $id
    }
}

#   DoXMove ---
#
#   If selected items, move them, else move current item if exists.
#   It uses the xDrag array to keep track of start and current position.
#   If 'what' = "point", then drag the single point.

proc DoXMove { w x y {what item} }	{
    global  xDrag
    
    if {![info exists xDrag]}  {
	return
    }
    # If we drag a point, then reject events triggered by non-point events.
    if {$xDrag(what) == "point" && $what != "point"}  {
	#puts "DoXMove:: rejected, what=$what"
	return
    }
    # If dragging 'point' (marker) of a fixed size item, return.
    if {$what == "point" && ($xDrag(type) == "text" || $xDrag(type) == "image")}  {
	#puts "DoXMove:: rejected, what=$what"
	return
    }
    set id_ {[0-9]+}
    
    # First, get canvas objects with tag 'selected'.
    set ids [$w find withtag selected]
    #puts "DoXMove:: ids=$ids, what=$what"
    if {$what == "item"}  {
	# If no selected items.
	if {$ids == ""}  {
	    # Be sure to exclude nonmovable items.
	    set tagsCurrent [$w gettags current]
	    set it [CanvasGetItnoFrom $w current]
	    #puts "DoXMove:: tagsCurrent=$tagsCurrent, it=$it"
	    if {$it == ""}  {
		return
	    }
	    #puts "DoXMove:: tagsCurrent=$tagsCurrent"
	    if { [lsearch $tagsCurrent grid] >= 0 }  {
		return
	    }
	    $w move current [expr $x - $xDrag(baseX)] [expr $y - $xDrag(baseY)]
	} else  {
	    # If selected, move all items and markers.
	    foreach id $ids  {
		set it [CanvasGetItnoFrom $w $id]
		if {$it != ""}   {
		    $w move $id [expr $x - $xDrag(baseX)] [expr $y - $xDrag(baseY)]
		}
	    }
	    # Move markers with them.
	    $w move tbbox [expr $x - $xDrag(baseX)] [expr $y - $xDrag(baseY)]
	} 
    } elseif {$what == "movie"}  {
	$w move ghostrect [expr $x - $xDrag(baseX)] [expr $y - $xDrag(baseY)]
    } elseif {$what == "point"}  {
	
	# Find associated id for the actual item. Saved in the tags of the marker.
	if {![regexp " +id($id_)" [$w gettags hitBbox] match theItemId]}  {
	    #puts "no match: w gettags hitBbox=[$w gettags hitBbox]"
	    return
	}
	if {[lsearch [$w gettags current] hitBbox] == -1}  {
	    #puts "DoXMove:: Warning, no match"
	    return
	}
	set ind $xDrag(hitInd)
	set newCoords [lreplace $xDrag(coords) $ind [expr $ind + 1] $x $y]
	eval $w coords $theItemId $newCoords
	$w move hitBbox [expr $x - $xDrag(baseX)] [expr $y - $xDrag(baseY)]
	$w move lightBbox [expr $x - $xDrag(baseX)] [expr $y - $xDrag(baseY)]
    }
    set xDrag(baseX) $x
    set xDrag(baseY) $y
}

#   SendXMove ---
#
#   Finished moving using DoXMove. Make sure that all connected clients
#   also moves either the selected clients or the current item.

proc SendXMove { w x y {what item} }  {
    global  xDrag allIPnumsTo ipNum2Socket debugLevel
    
    if {$debugLevel >= 2}  {
	puts "SendXMove:: what=$what, [info exists xDrag]"
    }
    if {![info exists xDrag]}  {
	return
    }
    # If we drag a point, then reject events triggered by non-point events.
    if {$xDrag(what) == "point" && $what != "point"}  {
	return
    }
    set id_ {[0-9]+}

    # Get item(s).
    # First, get canvas objects with tag 'selected' or 'ismoved'.
    set ids [$w find withtag selected]
    set id [$w find withtag ismoved]
    set theItno [CanvasGetItnoFrom $w $id]
    if {$debugLevel >= 2}  {
	puts "SendXMove:: ids=$ids, id=$id, theItno=$theItno, x=$x, y=$y"
    }
    if {$what == "item" && $ids == "" && $theItno == ""}  {
	return
    }
    # Find item tags ('theItno') for the items being moved.
    if {$what == "item" || $what == "movie"}  {

	# If no selected items.
	if {$ids == ""}  {
	    
	    # We already have 'theItno' from above.
	} else  {
	    # If selected, move all items.
	    set theItno {}
	    foreach id $ids  {
		set it [CanvasGetItnoFrom $w $id]
		lappend theItno $it
	    }
	} 
	
	# Dragging corner points.
    } elseif {$what == "point"}  {

	# Find associated id for the actual item. Saved in the tags of the marker.
	if {![regexp " +id($id_)" [$w gettags current] match theItemId]}  {
	    #puts "no match: w gettags current=[$w gettags current]"
	    return
	}
	set theItno [CanvasGetItnoFrom $w $theItemId]
	if {$theItno == ""}  {
	    return
	}
	# If endpoints overlap in line item, make closed polygon.
	# Find out if closed polygon or open line item. If closed, remove duplicate.
	set isClosed 0
	if {$xDrag(type) == "line"} {
	    set n [llength $xDrag(coords)]
	    set len [expr hypot(  \
	      [lindex $xDrag(coords) [expr $n - 2]] -  \
	      [lindex $xDrag(coords) 0],  \
	      [lindex $xDrag(coords) [expr $n - 1]] -  \
	      [lindex $xDrag(coords) 1] )]
	    if {$len < 8}  {
		# Make the line segments to a closed polygon.
		#puts "isClosed"
		set isClosed 1
		# Get all actual options.
		set opcmd [CanvasItemConfigure $w $theItemId]
		set theCoords [$w coords $theItemId]
		set polyCoords [lreplace $theCoords end end]
		set polyCoords [lreplace $polyCoords end end]
		set cmd1 [list $w delete $theItemId]
		eval $cmd1
		# Make the closed polygon. Get rid of non-applicable options.
		foreach op {"-arrow" "-arrowshape" "-capstyle" "-joinstyle"}  {
		    set ind [lsearch -exact $opcmd $op]
		    if {$ind >= 0}  {
			set opcmd [lreplace $opcmd $ind [expr $ind + 1]]
		    }
		}
		# Replace -fill with -outline.
		set ind [lsearch -exact $opcmd "-fill"]
		if {$ind >= 0}  {
		    set opcmd [lreplace $opcmd $ind $ind "-outline"]
		}
		set opcmd [concat $opcmd "-fill {}"]

		# Update the new item id.
		set cmd2 "$w create polygon $polyCoords $opcmd"
		set theItemId [eval $w create polygon $polyCoords $opcmd]
	    }
	}
	if {!$isClosed}  {
	    set cmd "coords $theItno [$w coords $theItno]"
	}
	# Move all markers along.
	$w delete id$theItemId
	MarkBbox $w 0 $theItemId
    }
    # For QT movies.
    if {$what == "movie"}  {
	$w move selectedmovie [expr $x - $xDrag(anchorX)]  \
	  [expr $y - $xDrag(anchorY)]
	$w dtag selectedmovie selectedmovie
    }
    # Delete the ghost rect or highlighted marker if any. Remove temporary tags.
    $w delete ghostrect
    $w delete lightBbox
    $w dtag all hitBbox 
    $w dtag all ismoved

    # Do send to all connected.
    foreach ip $allIPnumsTo  {
	set s $ipNum2Socket($ip)
	if {$what == "point"}  {
	    if {$isClosed}  {
		puts $s "CANVAS: $cmd1"
		puts $s "CANVAS: $cmd2"
	    } else  {
		puts $s "CANVAS: $cmd"
	    }
	} else  {
	    foreach it $theItno  {
		puts $s "CANVAS: move $it [expr $x - $xDrag(anchorX)]  \
		  [expr $y - $xDrag(anchorY)]"
	    }
	}
    }
    catch {unset xDrag}
}

proc BoxBegin { w x y type }	{
    global  theBox

    set theBox($w,anchor) [list $x $y]
    catch {unset theBox($w,last)}
}

#   BoxDrag ---
#   
#   If 'shift' constrain to square or circle.
#   If 'mark', then draw a temporary marking rectangle.
#   If not 'mark', then draw ordinary rectangle if 'type' is rect,
#   or oval if 'type' is oval.

proc BoxDrag { w x y shift type {mark 0} }  {
    global  theBox itno myItpref prefs debugLevel
    
    if {$debugLevel >= 2}  {
	#puts "BoxDrag (entry):: type=$type, mark=$mark"
    }
    catch {$w delete $theBox($w,last)}
    # If not set anchor, just return.
    if {![info exists theBox($w,anchor)]}  {
	return
    }
    set boxOrig $theBox($w,anchor)
    # If 'shift' constrain to square or circle.
    if {$shift}  {
	set box [eval "ConstrainedBoxDrag $theBox($w,anchor) $x $y $type"]
	set boxOrig [lrange $box 0 1]
	set x [lindex $box 2]
	set y [lindex $box 3]
    }
    # Either mark rectangle or draw rectangle.
    if {$mark == 0}  {
	if {$prefs(fill) == 0}  {
	    set theBox($w,last) [eval {$w create $type} $boxOrig  \
	      {$x $y -outline $prefs(fgCol) -width $prefs(penThick)  \
	      -tags [list $type $myItpref/$itno] }]
	} else  {
	    set theBox($w,last) [eval {$w create $type} $boxOrig  \
	      {$x $y -outline $prefs(fgCol) -fill $prefs(fgCol)  \
	      -width $prefs(penThick) -tags [list $type $myItpref/$itno] }]
	}
    } else {
	set theBox($w,last) [eval {$w create $type} $boxOrig	\
	  {$x $y -outline gray50 -stipple gray50 -width 2 -tags "markbox" }]
    }
}

#   SendBox ---
#
#   If 'shift' constrain to square or circle.
#   If 'mark', then make the overlapping objects selected and delete the
#   temporary rectangle. If not 'mark', then let all other clients
#   write a rectangle or oval.
#   'type' should be 'rect' or 'oval'.

proc SendBox { w x y shift type {mark 0} }  {
    global  theBox itno myItpref prefs debugLevel  \
      allIPnumsTo ipNum2Socket
    
    # If no theBox($w,anchor) defined just return.
    if {![info exists theBox($w,anchor)]}   {
	return
    }
    if {[lindex $theBox($w,anchor) 0] == $x && [lindex $theBox($w,anchor) 1] == $y}  {
	set nomove 1
	return
    } else {
	set nomove 0
    }
    if {$mark}  {
	set ids [eval {$w find overlapping} $theBox($w,anchor) {$x $y}]
	if {$debugLevel >= 2} {
	    puts "SendBox:: ids=$ids"
	}
	foreach id $ids {
	    MarkBbox $w 1 $id
	}
	$w delete withtag markbox
    }
    # Create real objects.
    if {!$mark && !$nomove && [llength $allIPnumsTo] > 0}  {
	set boxOrig $theBox($w,anchor)
	# If 'shift' constrain to square or circle.
	if {$shift}  {
	    set box [eval "ConstrainedBoxDrag $theBox($w,anchor) $x $y $type"]
	    set boxOrig [lrange $box 0 1]
	    set x [lindex $box 2]
	    set y [lindex $box 3]
	}
	if {$prefs(fill) == 0}  {
	    set cmd "create $type $boxOrig $x $y	\
	      -tags {$type $myItpref/$itno} -outline $prefs(fgCol)  \
	      -width $prefs(penThick)"
	} else  {
	    set cmd "create $type $boxOrig $x $y	\
	      -tags {$type $myItpref/$itno} -outline $prefs(fgCol)  \
	      -width $prefs(penThick) -fill $prefs(fgCol)"
	}
	foreach ip $allIPnumsTo  {
	    set s $ipNum2Socket($ip)
	    puts $s "CANVAS: $cmd"
	}
    }
    # Only increment counter for real objects; not when marking.
    if {!$mark}  {
	if {$debugLevel >= 2} {
	    puts "SendBox:: incr itno from $itno"
	}
	incr itno
    }
    catch {unset theBox}
}

#   ConstrainedBoxDrag ---
#
#   With the 'shift' key pressed, the rect and oval items are contrained
#   to squares and circles respectively.

proc ConstrainedBoxDrag  { xanch yanch x y type }  {
    
    set deltax [expr $x - $xanch]
    set deltay [expr $y - $yanch]
    set prod [expr $deltax * $deltay]
    if {$type == "rect"}  {
	set boxOrig [list $xanch $yanch]
	if {$prod != 0}  {
	    set sign [expr $prod / abs($prod)]
	} else  {
	    set sign 1
	}
	if {[expr abs($deltax)] > [expr abs($deltay)]}  {
	    set x [expr $sign * ($y - $yanch) + $xanch]
	} else  {
	    set y [expr $sign * ($x - $xanch) + $yanch]
	}
	
	# A pure circle is not made with the bounding rect model.
	# The anchor and the present x, y define the diagonal instead.
    } elseif {$type == "oval"}  {
	set r [expr hypot($deltax, $deltay)/2.0]
	set midx [expr ($xanch + $x)/2.0]
	set midy [expr ($yanch + $y)/2.0]
	set boxOrig [list [expr int($midx - $r)] [expr int($midy - $r)]]
	set x [expr int($midx + $r)]
	set y [expr int($midy + $r)]
    }
    return [concat $boxOrig $x $y]
}

#   DoPaint ---
#
#   Fills item with the foreground color. If 'alt', then transparent.

proc DoPaint  { w x y {alt 0}}  {
    global  prefs allIPnumsTo ipNum2Socket

    #puts "DoPaint:: w=$w, x=$x, y=$y, alt=$alt"
    # Find items overlapping x and y.
    #set ids [$w find overlapping $x $y $x $y]
    set ids [$w find all]
    #puts "ids=$ids"
    foreach id $ids {
	set theType [$w type $id]
	#puts "id=$id, theType=$theType"
	# Sort out uninteresting items early.
	if {$theType != "rectangle" && $theType != "oval"}  {
	    continue
	}
	# Must be in bounding box.
	set theBbox [$w bbox $id]
	#puts "theBbox=$theBbox"
	if {$x >= [lindex $theBbox 0] && $x <= [lindex $theBbox 2] &&  \
	  $y >= [lindex $theBbox 1] && $y <= [lindex $theBbox 3]}  {
	    # OK, inside!
	    # Allow privacy.
	    set theItno [CanvasGetItnoFrom $w $id]
	    if {$theItno == ""}  {
		continue
	    }
	    set cmd {}
	    if {$theType == "rectangle"}  {
		if {$alt == 0}  {
		    set cmd "itemconfigure $theItno -fill $prefs(fgCol)"
		} elseif {$alt == 1}  {
		    set cmd "itemconfigure $theItno -fill {}"
		}
		eval $w $cmd
	    } elseif {$theType == "oval"}  {
		# Use ellipsis equation (1 = x^2/a^2 + y^2/b^2) to find if inside.
		set centx [expr ([lindex $theBbox 0] + [lindex $theBbox 2])/2.0]
		set centy [expr ([lindex $theBbox 1] + [lindex $theBbox 3])/2.0]
		set a [expr abs($centx - [lindex $theBbox 0])]
		set b [expr abs($centy - [lindex $theBbox 1])]
		if {[expr ($x-$centx)*($x-$centx)/($a*$a) +   \
		  ($y-$centy)*($y-$centy)/($b*$b)] <= 1}  {
		    # Inside!
		    if {$alt == 0}  {
			set cmd "itemconfigure $theItno -fill $prefs(fgCol)"
		    } elseif {$alt == 1}  {
			set cmd "itemconfigure $theItno -fill {}"
		    }
		    eval $w $cmd
		}
	    }	    
	    if {[string length $cmd] > 1}  {
		foreach ip $allIPnumsTo  {
		    puts $ipNum2Socket($ip) "CANVAS: $cmd"
		}
	    }
	}
    }

}

#   PolySetPoint, PolyDrag, PolyFinish ---
#
#   Polygon drawing routines.

proc PolySetPoint  { w x y }  {
    global  thePoly

    #puts "PolySetPoint::"
    if {![info exists thePoly(0)]}  {
	# First point.
	catch {unset thePoly}
	set thePoly(N) 0
	set thePoly(0) [list $x $y]
    } elseif {[expr   \
      hypot([lindex $thePoly(0) 0] - $x, [lindex $thePoly(0) 1] - $y)] < 6}  {
	# If this point close enough to 'thePoly(0)', close polygon.
	PolyDrag $w [lindex $thePoly(0) 0] [lindex $thePoly(0) 1]
	set thePoly(last) {}
	incr thePoly(N)
	set thePoly($thePoly(N)) $thePoly(0)
	PolyFinish $w [lindex $thePoly(0) 0] [lindex $thePoly(0) 1]
	return
    } else  {
	set thePoly(last) {}
	incr thePoly(N)
	#set thePoly($thePoly(N)) [list $x $y]
	set thePoly($thePoly(N)) $thePoly(xy)
    }
    #puts "[parray thePoly]"
    # Let the latest line segment follow the mouse movements.
    focus $w
    bind $w <Motion> {PolyDrag %W [%W canvasx %x] [%W canvasy %y]}
    bind $w <Shift-Motion> {PolyDrag %W [%W canvasx %x] [%W canvasy %y] 1}
    bind $w <KeyPress-space> {PolyFinish %W [%W canvasx %x] [%W canvasy %y]}
}               

proc PolyDrag  { w x y {shift 0} }  {
    global  thePoly prefs

    # Move one end point of the latest line segment of the polygon.
    # If anchor not set just return.
    if {![info exists thePoly(0)]}  {
	return
    }
    catch {$w delete $thePoly(last)}
    # Verical or horizontal.
    if {$shift}  {
	set anch $thePoly($thePoly(N))
	if {[expr abs($x - [lindex $anch 0])] >  \
	  [expr abs($y - [lindex $anch 1])]}  {
	    set y [lindex $anch 1]
	} else  {
	    set x [lindex $anch 0]
	}
    }
    # Keep track of last coordinates. Important for 'shift'.
    set thePoly(xy) [list $x $y]
    set thePoly(last) [eval {$w create line} $thePoly($thePoly(N))  \
      {$x $y -tags "polylines" -fill $prefs(fgCol)  \
      -width $prefs(penThick)}]
}

proc PolyFinish { w x y }  {
    global  thePoly myItpref itno prefs allIPnumsTo ipNum2Socket
    
    #puts "[parray thePoly]"
    bind $w <Motion> {}
    bind $w <KeyPress-space> {}
    # If anchor not set just return.
    if {![info exists thePoly(0)]}  {
	return
    }
    # If too few segment.
    if {$thePoly(N) <= 1}  {
	$w delete polylines
	catch {unset thePoly}
	return
    }
    # Delete last line segment.
    catch {$w delete $thePoly(last)}
    
    # Find out if closed polygon or open line item. If closed, remove duplicate.
    set isClosed 0
    if {[expr   \
      hypot([lindex $thePoly(0) 0] - $x, [lindex $thePoly(0) 1] - $y)] < 4}  {
	set isClosed 1
	unset thePoly($thePoly(N))
	incr thePoly(N) -1
    }
    # Transform the set of lines to a polygon (or line) item.
    set coords {}
    for {set i 0} {$i <= $thePoly(N)} {incr i}  {
	append coords $thePoly($i) " "
    }
    $w delete polylines
    if {$prefs(fill) == 0}  {
	set theFill "-fill {}"
    } else  {
	set theFill "-fill $prefs(fgCol)"
    }
    if {$isClosed}  {
	set cmd "create polygon $coords -tags {poly $myItpref/$itno}  \
	  -outline $prefs(fgCol) $theFill -width $prefs(penThick)  \
	  -smooth $prefs(smooth) -splinesteps $prefs(splinesteps)"
    } else  {
	set cmd "create line $coords -tags {poly $myItpref/$itno}  \
	  -fill $prefs(fgCol) -width $prefs(penThick)  \
	  -smooth $prefs(smooth) -splinesteps $prefs(splinesteps)"
    }
    eval $w $cmd
    catch {unset thePoly}
    
    # Let al other clients know.
    foreach ip $allIPnumsTo  {
	set s $ipNum2Socket($ip)
	puts $s "CANVAS: $cmd"
    }
    incr itno
}

#   DeleteItem ---
#
#   Delete item in canvas 'w' with coordinates 'x' and 'y'.
#   'id' can be "current", "selected", "movie" or just an id number.

proc DeleteItem	 { w x y {id current} {moviePath {}} }  {
    global  debugLevel allIPnumsTo ipNum2Socket
    
    if {$debugLevel >= 2}  {
	puts "DeleteItem:: w=$w, id=$id, moviePath=$moviePath"
    }
    # Get item.
    if {$id == "current"}  {
	set theItno [CanvasGetItnoFrom $w current]
	if {$theItno == ""}  {
	    return
	}
	$w delete current
    } elseif {$id == "selected"}  {
	# First, get canvas objects with tag 'selected'.
	set ids [$w find withtag selected]
	if {[llength $ids] == 0}  {
	    return
	}
	set theItno ""
	foreach id $ids  {
	    set it [CanvasGetItnoFrom $w $id]
	    if {$it != ""}  {
		lappend theItno $it
		$w delete $it
	    }
	}
	# Remove select marks.
	DeselectAll $w
    } elseif {$id == "movie"}  {
	# QT movie.
	set id [lindex [$w find closest $x $y 3] 0]
	set theItno [CanvasGetItnoFrom $w $id]
	#puts "   id=$id, theItno=$theItno"
	if {$theItno == ""}  {
	    return
	}
	# Delete both the window item and the movie.
	$w delete $id
	catch {destroy ${moviePath}.m}
    } else  {
	# 'id' is an actual item number.
	set theItno [CanvasGetItnoFrom $w $id]
	if {$theItno == ""}  {
	    return
	}
	$w delete $theItno
    }
    if {$theItno == ""}	  {
	return
    }
    foreach ip $allIPnumsTo  {
        set s $ipNum2Socket($ip)
	foreach it $theItno  {
	    puts $s "CANVAS: delete $it"
	}
    }
}

proc FindTypeFromOverlapping { c x y type }  {    
    set ids [$c find overlapping [expr $x-2] [expr $y-2]  \
      [expr $x+2] [expr $y+2]]
    set id {}
    # Choose the first item with tags $type.
    foreach i $ids  {
	if {[lsearch [$c gettags $i] $type] >= 0}  {
	    # Found "$type".
	    set id $i
	    break
	}
    }
    #puts "FindTypeFromOverlapping:: ids=$ids, id=$id"
    return $id
}

#   CanvasFocus ---
#
#   Puts a text insert bar in the canvas. If already text item under the mouse
#   then give focus to that item.

proc CanvasFocus { c x y }	{
    global  prefs itno myItpref debugLevel allIPnumsTo ipNum2Socket   \
      fontSize2Points
    
    focus $c
    set id [FindTypeFromOverlapping $c $x $y "text"]
    #puts "CanvasFocus:: id=$id"
    if {($id == {}) || ([$c type $id] != "text")}  {
	set cmd "create text $x $y -text {} -tags {text $myItpref/$itno} \
	  -anchor nw -fill $prefs(fgCol) \
	  -font {$prefs(font) $fontSize2Points($prefs(fontSize)) $prefs(fontWeight)}"
	if {$debugLevel >= 2}  {
	    puts "$c $cmd"
	}
	set t [eval $c $cmd]
	$c focus $t
	$c select clear
	$c icursor $t 0
	
	# If 'useHtmlSizes', then transport the html sizes instead of point sizes.
	if {$prefs(useHtmlSizes)}  {
	    set cmd "create text $x $y -text {} -tags {text $myItpref/$itno} \
	      -anchor nw -fill $prefs(fgCol) \
	      -font {$prefs(font) $prefs(fontSize) $prefs(fontWeight)}"
	}
	foreach ip $allIPnumsTo  {
	    set s $ipNum2Socket($ip)
	    puts $s "CANVAS: $cmd"
	}
	incr itno
    }
}

proc CanvasTextHit { c x y {select 1} }	 {
    $c focus current
    $c icursor current @$x,$y
    $c select clear
    $c select from current @$x,$y
}

proc CanvasTextDrag { c x y }  {
    global  tcl_platform
    
    DeselectAll $c
    $c select to current @$x,$y
    # Mac text bindings.????
    if {$tcl_platform(platform) == "macintosh"}  {
	#$c focus
    }
    # menus
    .menu.edit entryconfigure *Cut* -state normal
    .menu.edit entryconfigure *Copy* -state normal
}

proc CanvasTextSelectWord { c x y }  {
    global  tcl_platform
    
    DeselectAll $c
    $c focus current
    # Mac text bindings.????
    if {$tcl_platform(platform) == "macintosh" ||   \
      $tcl_platform(platform) == "unix" ||  \
      $tcl_platform(platform) == "windows"}  {
	set id [$c find withtag current]
	if {$id == ""}  {
	    return
	}
	if {[$c type $id] != "text"}  {
	    return
	}
	set txt [$c itemcget $id -text]
	set ind [$c index $id @$x,$y]
	#puts "CanvasTextSelectWord:: txt=$txt, ind=$ind"
	# Find the boundaries of the word and select word.
	$c select from $id [string wordstart $txt $ind]
	$c select adjust $id [expr [string wordend $txt $ind] - 1]
	# menus
	.menu.edit entryconfigure *Cut* -state normal
	.menu.edit entryconfigure *Copy* -state normal
    }    
}

proc CanvasTextDelete { c }	 {
    global	debugLevel allIPnumsTo ipNum2Socket
    
    if {$debugLevel >= 2}  {
	puts "CanvasTextDelete"
    }
    set idfocus [$c focus]
    if {[$c select item] != {}}	 {
	set theItno [CanvasGetItnoFrom $c focus]
	if {$theItno == ""}  {
	    return
	}
	set sfirst [$c index $idfocus sel.first]
	set slast [$c index $idfocus sel.last]
	$c dchars [$c select item] sel.first sel.last
	if {$debugLevel >= 2}  {
	    puts "sfirst: $sfirst, slast: $slast"
	}
        foreach ip $allIPnumsTo  {
            set s $ipNum2Socket($ip)
	    puts $s [list "CANVAS:" dchars $theItno $sfirst $slast]
	}
    } elseif {$idfocus != {}}  {
	set theItno [CanvasGetItnoFrom $c focus]
	if {$theItno == ""}  {
	    return
	}
	set ind [expr [$c index $idfocus insert] - 1]
	$c dchars $idfocus $ind
	if {[llength $allIPnumsTo] > 0}	{
	    # Find the 'itno'.
	    set theItno [CanvasGetItnoFrom $c focus]
	    if {$debugLevel >= 2}  {
		puts "theItno: $theItno, ind: $ind"
	    }
	    if {$theItno == ""}	 {
		return
	    }
	    foreach ip $allIPnumsTo  {
		set s $ipNum2Socket($ip)
		puts $s [list "CANVAS:" dchars $theItno $ind]
	    }
	}
    }
}

proc CanvasDelChar { c }  {
    #puts "CanvasDelChar::"
    set theItno [CanvasGetItnoFrom $c focus]
    if {$theItno == ""}  {
	return
    }
    set idfocus [$c focus]
    if {$idfocus != {}}	 {
	$c dchars $idfocus [expr [$c index $idfocus insert] -1]
    }
}

#   CanvasTextBackSpace ---
#
#   A backspace if selected text deletes that text.
#   A backspace if text item has focus deletes text left of insert cursor.

proc CanvasTextBackSpace { c }	{
    
    #puts "CanvasTextBackSpace"
    # Just return if any *items* selected.
    if {[llength [$c find withtag selected]] > 0}  {
	return
    }
    set selit [$c select item]
    if {$selit != {}} {
	set theItno [CanvasGetItnoFrom $c $selit]
	if {$theItno == ""}  {
	    return
	}
	$c dchars [$c select item] sel.first sel.last
    } else  {
	set theItno [CanvasGetItnoFrom $c focus]
	if {$theItno == ""}  {
	    return
	}
	set t [$c focus]
	$c icursor $t [expr [$c index $t insert] -1]
	$c dchars $t insert
    }
}

proc CanvasTextErase { c }	{
    #puts "CanvasTextErase"
    $c delete [$c focus]
}

proc CanvasNewLine { c }  {
    global	debugLevel allIPnumsTo ipNum2Socket
    
    set nl_ "\\n"
    $c insert [$c focus] insert \n
    # find the 'itno' 
    set theItno [CanvasGetItnoFrom $c focus]
    if {$theItno == ""}	 {
	return
    }
    set ind [expr [$c index [$c focus] insert] - 1]
    if {$debugLevel >= 2}  {
	puts "theItno = $theItno, ind = $ind"
	puts "CANVAS: insert $theItno $ind $nl_"
    }
    foreach ip $allIPnumsTo  {
	set s $ipNum2Socket($ip)
	puts $s [list "CANVAS:" insert $theItno $ind $nl_]
    }
}

#   CanvasGetItnoFrom ---
#   
#   Finds the specific item identifier from 'fromWhat'.
#   If 'privacy' then returns empty string of not item made here.
#   fromWhat = "current": picks the current item
#   fromWhat = "focus": picks the item that has focus
#   fromWhat = canvas id number: takes the specified canvas id

proc CanvasGetItnoFrom { c {fromWhat current}}	 {
    global	myItpref prefs debugLevel
    
    # find the 'itno'
    set pre_ {[^/ ]+}
    set digit_ {[0-9]}
    if {[string compare $fromWhat "current"] == 0}  {
	set tcurr [$c gettags current]
    } elseif {[string compare $fromWhat "focus"] == 0}  {
	set tcurr [$c gettags [$c focus]]
    } else {
	set tcurr [$c gettags $fromWhat]
    }
    if {$debugLevel >= 2}  {
	#puts "CanvasGetItnoFrom:: tcurr=$tcurr"
    }
    if {$tcurr == ""}  {
	return {}
    }
    if {$prefs(privacy)}  {
	if {[regexp "($myItpref/$digit_+)" "$tcurr" theItno] == 0}  {
	    return {}
	}
    } else  {
	if {[regexp "(^| )($pre_/$digit_+)" "$tcurr" junk junk2 theItno] == 0}  {
	    return {}
	}
    }
    return $theItno
}

#   CanvasTextInsert ---
#
#   Inserts text string 'char' at the insert point of the text item
#   with focus. Handles newlines as well.

proc CanvasTextInsert { c char }  {
    global  debugLevel allIPnumsTo ipNum2Socket tcl_platform prefs
        
    set punct {[.,;?!]}
    set nl_ "\\n"
    # Find the 'itno'.
    set theItno [CanvasGetItnoFrom $c focus]
    if {$theItno == "" || $char == ""}	 {
	if {$debugLevel >= 2}  {
	    puts "CanvasTextInsert:: theItno == {}"
	}
	return
    }
    # The index of the insertion point.
    set ind [$c index [$c focus] insert]

    # Mac text bindings: delete selection before inserting.
    if {$tcl_platform(platform) == "macintosh"}  {
	if {![catch {selection get} s]}  {
	    if {[llength $s] > 0}  {
		CanvasTextDelete $c
		selection clear
	    }
	}
    }

    # The actual canvas text insertion; note that 'ind' is found before.
    $c insert [$c focus] insert $char
    if {$debugLevel >= 2}  {
	puts "CanvasTextInsert:: theItno = $theItno, ind = $ind, char: $char"
    }
    # If speech, speech last sentence if finished.
    if {$prefs(MacSpeech)}  {
	if {[string match *${punct}* $char]}  {
	    speak [$c itemcget $theItno -text]
	}
    }
    # Need to treat the case with actual newlines in char string.
    # Write to all other clients; need to make a one liner first.
    if {[llength $allIPnumsTo] > 0}  {
	regsub -all "\n" $char $nl_ oneliner
	if {$debugLevel >= 2}  {
	    puts "CanvasTextInsert oneliner=$oneliner"
	}
    }
    foreach ip $allIPnumsTo  {
        set s $ipNum2Socket($ip)
	puts $s [list "CANVAS:" insert $theItno $ind $oneliner]
    }	
}

proc CanvasTextMoveRight { c }	{
    global  tcl_platform
    
    set foc [$c focus]
    
    # Mac text bindings: remove selection then move insert to end.
    if {$tcl_platform(platform) == "macintosh" ||  \
      $tcl_platform(platform) == "windows"}  {
	# If selection.
	if {![catch {selection get} s]}  {
	    if {[llength $s] > 0}  {
		$c icursor $foc [expr [$c index $foc sel.last] + 1]
		$c select clear
	    }
	} else  {
	    $c icursor $foc [expr [$c index $foc insert] + 1]
	}
    } else  {
	$c icursor $foc [expr [$c index $foc insert] + 1]
    }
}

proc CanvasTextMoveLeft { c }  {
    global  tcl_platform
    
    set foc [$c focus]
    
    # Mac text bindings: remove selection then move insert to first.
    if {$tcl_platform(platform) == "macintosh" ||  \
      $tcl_platform(platform) == "windows"}  {
	# If selection.
	if {![catch {selection get} s]}  {
	    if {[llength $s] > 0}  {
		$c icursor $foc [expr [$c index $foc sel.first] + 0]
		$c select clear
	    }
	} else  {
	    $c icursor $foc [expr [$c index $foc insert] - 1]
	}
    } else  {
	$c icursor $foc [expr [$c index $foc insert] - 1]
    }
}

proc CanvasTextMoveUpOrDown { c upOrDown }  {
    global  debugLevel
    
    if {$debugLevel >= 2}  {
	puts "CanvasTextMoveUpOrDown:: upOrDown=$upOrDown"
    }
    set foc [$c focus]
    # Find index of new character. Only for left justified text.
    set ind [$c index $foc insert]
    set theText [$c itemcget $foc -text]
    #puts "ind=$ind"
    if {[string compare $upOrDown "up"] == 0}  {
	
	# Up one line. String operations.
	set indPrevNL [string last \n $theText [expr $ind - 1]]
	set indPrev2NL [string last \n $theText [expr $indPrevNL - 1]]
	#puts "indPrevNL=$indPrevNL, indPrev2NL=$indPrev2NL"
	# If first line.
	if {$indPrevNL == -1}  {
	    return
	}
	set ncharLeft [expr $ind - $indPrevNL - 1]
	set newInd [min [expr $indPrev2NL + $ncharLeft + 1] $indPrevNL]
	#puts "ncharLeft=$ncharLeft, newInd=$newInd"
	$c icursor $foc $newInd
    } else  {
	
	# Down one line.
	set indPrevNL [string last \n $theText [expr $ind - 1]]
	set indNextNL [string first \n $theText [expr $indPrevNL + 1]]
	set indNext2NL [string first \n $theText [expr $indNextNL + 1]]
	# If last line.
	if {$indNextNL == -1}  {
	    return
	}
	set ncharLeft [expr $ind - $indPrevNL - 1]
	if {$indNext2NL == -1}  {
	    # Move to last line.
	    set newInd [expr $indNextNL + $ncharLeft + 1]
	} else  {
	    set newInd [min [expr $indNextNL + $ncharLeft + 1] $indNext2NL]
	}
	#puts "indPrevNL=$indPrevNL, indNextNL=$indNextNL, indNext2NL=$indNext2NL,  \
	#  ncharLeft=$ncharLeft, newInd=$newInd"
	$c icursor $foc $newInd
    }
}

#   StrokeBegin, StrokeDrag, StrokeEnd ---
#
#   Handles drawing of an arbitrary line. Uses global 'stroke' variable
#   to store all intermediate points on the line, and stroke(N) to store
#   the number of such points. If 'thick'=-1, then use 'prefs(penThick)',
#   else use the 'thick' argument as line thickness.

proc StrokeBegin { w x y }	{
    global  stroke
    
    catch {unset stroke}
    set stroke(N) 0
    set stroke(0) [list $x $y]
}

proc StrokeDrag { w x y {thick -1} }  {
    global  stroke prefs itno myItpref
    
    # If stroke not set just return.
    if {![info exists stroke(N)]}  {
	return
    }
    set coords $stroke($stroke(N))
    lappend coords $x $y
    incr stroke(N)
    set stroke($stroke(N)) [list $x $y]
    if {$thick == -1}  {
	set thisThick $prefs(penThick)
    } else {
	set thisThick $thick
    }
    eval {$w create line} $coords  \
	    {-tags [list segments $myItpref/$itno] -fill $prefs(fgCol)  \
	    -width $thisThick}
}

proc StrokeEnd { w x y {thick -1} }	 {
    global	stroke prefs itno myItpref debugLevel allIPnumsTo ipNum2Socket
    
    # If stroke not set just return.
    set coords {}
    if {![info exists stroke(N)]}  {
	return
    }
    for {set i 0} {$i <= $stroke(N)} {incr i}  {
	append coords $stroke($i) " "
    }
    $w delete segments
    if {[llength $coords] <= 2}  {
	return
    }
    if {$thick == -1}  {
	set thisThick $prefs(penThick)
    } else {
	set thisThick $thick
    }
    set cmd "create line $coords  \
      -tags {line $myItpref/$itno} -joinstyle round  \
      -smooth $prefs(smooth) -splinesteps $prefs(splinesteps) \
      -fill $prefs(fgCol) -width $thisThick"
    if {$debugLevel >= 2}  {
	puts "StrokeEnd:: cmd=$cmd"
    }
    eval $w $cmd
    foreach ip $allIPnumsTo  {
        set s $ipNum2Socket($ip)
	puts $s "CANVAS: $cmd"
    }
    incr itno
    catch {unset stroke}
}

#   LineBegin, LineDrag, LineEnd ---
#
#   Handles drawing of a straight line. Uses global 'theLine' variable
#   to store anchor point and end point of the line.
#   If 'shift' constrain the line to be vertical or horizontal.
#   If 'opt'=arrow draw an arrow at the final line end.

proc LineBegin { w x y {opt 0} }  {
    global  theLine
    
    set theLine($w,anchor) [list $x $y]
    catch {unset theLine($w,last)}
}

proc LineDrag { w x y shift {opt 0} }	 {
    global  theLine itno myItpref prefs
    
    # If anchor not set just return.
    if {![info exists theLine($w,anchor)]}  {
	return
    }

    catch {$w delete $theLine($w,last)}
    if {$opt == "arrow"}  {
	set extras [list -arrow last]
    } else	{
	set extras ""
    }
    # Verical or horizontal.
    if {$shift}  {
	if {[expr abs($x - [lindex $theLine($w,anchor) 0])] >  \
	  [expr abs($y - [lindex $theLine($w,anchor) 1])]}  {
	    set y [lindex $theLine($w,anchor) 1]
	} else  {
	    set x [lindex $theLine($w,anchor) 0]
	}
    }
    set theLine($w,last) [eval {$w create line} $theLine($w,anchor)  \
      {$x $y -tags "line $myItpref/$itno" -fill $prefs(fgCol)  \
      -width $prefs(penThick)} $extras]
}

proc LineEnd { w x y shift {opt 0}}  {
    global  itno myItpref theLine prefs allIPnumsTo ipNum2Socket
    
    # If anchor not set just return.
    if {![info exists theLine($w,anchor)]}  {
	return
    }
    if {$opt == "arrow"}  {
	set extras [list -arrow last]
    } else	{
	set extras ""
    }
    # Verical or horizontal.
    if {$shift}  {
	if {[expr abs($x - [lindex $theLine($w,anchor) 0])] >  \
	  [expr abs($y - [lindex $theLine($w,anchor) 1])]}  {
	    set y [lindex $theLine($w,anchor) 1]
	} else  {
	    set x [lindex $theLine($w,anchor) 0]
	}
    }
    if {[llength $allIPnumsTo] > 0}  {
	set cmd "create line $theLine($w,anchor) $x $y	\
	  -tags {line $myItpref/$itno} -joinstyle round	\
	  -smooth true -fill $prefs(fgCol) -width $prefs(penThick) $extras"
	foreach ip $allIPnumsTo  {
	    set s $ipNum2Socket($ip)
	    puts $s "CANVAS: $cmd"
	}
    }
    incr itno
    catch {unset theLine}
}

proc DoUndo  {  }  {
    tk_messageBox -message "Chicken!!! \nUndo is not for real men." \
      -icon info -type ok
}

proc SelectAll { w }  {
    global  itno myItpref debugLevel
    
    set ids [$w find all]
    if {$debugLevel >= 2}  {
	puts "SelectAll:: ids=$ids"
    }
    foreach id $ids {
	MarkBbox $w 1 $id
    }
}

proc DeselectAll { w }  {
    $w delete withtag tbbox
    $w dtag all selected
    # menus
    .menu.edit entryconfigure *Cut* -state disabled
    .menu.edit entryconfigure *Copy* -state disabled
    .menu.edit entryconfigure *Raise* -state disabled
    .menu.edit entryconfigure *Lower* -state disabled
}

#   MarkBbox ---
#
#   Makes four tiny squares at the corners of the specified items.
#   'which' can either be "current" or an id.

proc MarkBbox { w shift {which current}}  {
    global	prefs debugLevel
    
    if {$debugLevel >= 2}  {
	puts "MarkBbox (entry):: w=$w, shift=$shift, which=$which"
    }
    set a $prefs(aBBox)
    # If no shift key, deselect all.
    if {$shift == 0}  {
	DeselectAll $w
    }
    if {$which == "current"}  {
	set thebbox [$w bbox current]
    } else {
	set thebbox [$w bbox $which]
    }
    if {[llength $thebbox] == 0}  {
	return
    }
    if {$which == "current"}  {
	set itno [CanvasGetItnoFrom $w current]
	set id [$w find withtag current]
    } else  {
	set itno [CanvasGetItnoFrom $w $which]
	set id $which
    }
    if {[llength $itno] == 0 || [llength $id] == 0}  {
	return
    }
    # Movies may not be selected this way; temporary solution?
    if {[lsearch [$w gettags $itno] "movie"] >= 0}  {
	return
    }
    if {$debugLevel >= 2}  {
	puts "MarkBbox:: itno=$itno, id=$id"
    }
    # Add tag 'selected' to the selected item. Indicate to which item id
    # a marker belongs with adding a tag 'id$id'.
    set theType [$w type $which]
    $w addtag "selected" withtag $itno
    set theMarkTags "tbbox id$id"
    
    # If mark the bounding box. Also for all "regular" shapes.
    if {$prefs(bboxOrCoords) || ($theType == "oval") || ($theType == "text")  \
      || ($theType == "rectangle") || ($theType == "image")  \
      || ($theType == "window")}  {
	set x1 [lindex $thebbox 0]
	set y1 [lindex $thebbox 1]
	set x2 [lindex $thebbox 2]
	set y2 [lindex $thebbox 3]
	$w create rect [expr $x1-$a] [expr $y1-$a] [expr $x1+$a] [expr $y1+$a] \
	  -tags $theMarkTags -fill white
	$w create rect [expr $x1-$a] [expr $y2-$a] [expr $x1+$a] [expr $y2+$a] \
	  -tags $theMarkTags -fill white
	$w create rect [expr $x2-$a] [expr $y1-$a] [expr $x2+$a] [expr $y1+$a] \
	  -tags $theMarkTags -fill white
	$w create rect [expr $x2-$a] [expr $y2-$a] [expr $x2+$a] [expr $y2+$a] \
	  -tags $theMarkTags -fill white
    } else  {
	
	# Mark each coordinate. {x0 y0 x1 y1 ... }
	set theCoords [$w coords $which]
	set n [llength $theCoords]
	for {set i 0} {$i < $n} {incr i 2}  {
	    set x [lindex $theCoords $i]
	    set y [lindex $theCoords [expr $i + 1]]
	    $w create rect [expr $x-$a] [expr $y-$a] [expr $x+$a] [expr $y+$a] \
	      -tags $theMarkTags -fill white
	}
    }
    # Bind key to be able to delete selected item.
    #bind . <BackSpace> "DeleteItem $w %x %y selected"
    #bind . <Control-Key-d> "DeleteItem $w %x %y selected"
    
    # Enable cut and paste.
    .menu.edit entryconfigure *Cut* -state normal
    .menu.edit entryconfigure *Copy* -state normal
    .menu.edit entryconfigure *Raise* -state normal
    .menu.edit entryconfigure *Lower* -state normal
}

proc RaiseOrLowerItems { w {what raise} }  {
    global  prefs allIPnumsTo ipNum2Socket
    
    if {$what == "raise"}  {
	set doWhat $what
	set which all
    } elseif {$what == "lower"}  {
	if {$prefs(canGridOn)}  {
	    set doWhat "raise"   
	    set which grid
	} else  {
	    set doWhat $what
	    set which all
	}
    }
    $w $doWhat selected $which
    $w $doWhat tbbox $which
    
    # if connected
    if {[llength $allIPnumsTo] > 0}  {
	# first, get canvas objects with tag 'selected'
	set ids [$w find withtag selected]
	set theItno {}
	foreach id $ids  {
	    lappend theItno [CanvasGetItnoFrom $w $id]
	}
	foreach ip $allIPnumsTo  {
	    set s $ipNum2Socket($ip)
	    foreach it $theItno  {
		puts $s "CANVAS:  $what $it all"
	    }   
	}
    }
}

#   CopySelectedToClipboard ---
#
#   Copies the selection, either complete items or pure text, to the clipboard.
#   If there are no selected items, pure text is copied.
#   Set a flag 'clipToken' to tell which; "string" or "item".
#   The items are copied one by one using 'CopySingleItemToClipboard'.
#   doWhat: "cut" or "copy".
 
proc CopySelectedToClipboard  { w doWhat }  {
    global  debugLevel clipItemSep clipToken
    
    if {$debugLevel >= 2}  {
	puts "CopySelectedToClipboard:: doWhat=$doWhat"
    }

    # First, get canvas objects with tag 'selected'.
    set ids [$w find withtag selected]	
    clipboard clear
    
    # If selected text within text item.
    if {[llength $ids] == 0}  {
	CanvasTextCopy $w
	if {$doWhat == "cut"}	 {
	    CanvasTextDelete $w
	}
	set clipToken "string"
    } else {
	# Loop over all selected items, use 'clipItemSep' as separator.
	foreach id $ids  {
	    CopySingleItemToClipboard $w $doWhat $id
	    if {[lindex $ids end] != $id}  {
		clipboard append $clipItemSep
	    }
	}
	set clipToken "item"
    }
    .menu.edit entryconfigure *Paste* -state normal
}

#   CopySingleItemToClipboard ---
#
#   Copies the item given by 'id' to the clipboard.
#   doWhat: "cut" or "copy".

proc CopySingleItemToClipboard  { w doWhat id }  {
    global  debugLevel
    
    if {$debugLevel >= 2}  {
	puts "CopySingleItemToClipboard:: id=$id"
    }
    if {[llength $id] == 0}  {
	return
    }
    set theTags [$w gettags $id]

    # Get all actual options.
    set opcmd [CanvasItemConfigure $w $id]
    set itemType [$w type $id]
    set co [$w coords $id]
    set cmd [concat "create" $itemType $co $opcmd]
    if {$debugLevel >= 2}  {
	puts "CopySingleItemToClipboard:: cmd=$cmd"
    }
    # Copy the canvas object to the clipboard.
    clipboard append $cmd
    
    # If cut then delete items.
    if {$doWhat == "cut"}	 {
	DeleteItem $w 0 0 $id
	$w delete withtag tbbox
    } elseif {$doWhat == "copy"}	 {
	
    }
    .menu.edit entryconfigure *Paste* -state normal
}

#   CanvasItemConfigure ---
#
#   As canvas itemconfigure but only the actual options.

proc CanvasItemConfigure { w id }  {
    set opcmd {}
    set opts [$w itemconfigure $id]
    foreach opt $opts {
	set op [lindex $opt 0]
	set val [lindex $opt 4]
	lappend opcmd $op $val
    }
    return $opcmd
}

#   CanvasTextCopy ---
#  
#   Just copies text from text items. If selected text, copy that,
#   else if text item has focus copy complete text item.

proc CanvasTextCopy { c }  {
    global  debugLevel
    
    if {$debugLevel >= 2}  {
	puts "CanvasTextCopy::"
    }
    if {[$c select item] != {}}	 { 
	clipboard clear
	set t [$c select item]
	set text [$c itemcget $t -text]
	set start [$c index $t sel.first]
	set end [$c index $t sel.last]
	clipboard append [string range $text $start $end]
	if {$debugLevel >= 2}  {
	    puts "CanvasTextCopy:: select item] != {}, text=$text"
	}
    } elseif {[$c focus] != {}}	 {
	clipboard clear
	set t [$c focus]
	set text [$c itemcget $t -text]
	clipboard append $text
	if {$debugLevel >= 2}  {
	    puts "CanvasTextCopy:: focus] != {}, text=$text"
	}
    }
}

#   PasteFromClipboardToCanvas ---
#
#   Depending on 'clipToken', either paste simple text string, or complete item(s).
#   Items are pasted one by one using 'PasteSingleFromClipboardToCanvas'.

proc PasteFromClipboardToCanvas { w }  {
    global  debugLevel clipItemSep clipToken
    
    $w delete withtag tbbox
    $w dtag all selected
    
    # Pick apart the clipboard content with the 'clipItemSep' separator.
    set cmds [selection get -sel CLIPBOARD]
    if {$debugLevel >= 2}  {
	puts "PasteFromClipboardToCanvas:: clipToken=$clipToken, cmds=$cmds"
    }
    $w delete withtag tbbox
    
    # Try to figure out if pute text string (clipToken="string") or complete
    # canvas create item command (clipToken="item").
    set tmpCmds $cmds
    # Check first if it has the potential of a canvas command.
    if {[regexp  "^create " $cmds]}  {
	set sep [string trim $clipItemSep]
	set firstCmd [CmdToken tmpCmds $sep]
	# Then check if it really is a canvas command.
	#puts "it has the potential of a canvas command:: firstCmd=$firstCmd"
	if {[info complete ".junk $firstCmd"]}  {
	    #puts "it is a canvas item command"
	    set clipToken "item"
	} else  {
	    set clipToken "string"	    
	} 
    } else  {
	set clipToken "string"	    
    }
        
    # Depending on clipToken, either paste simple text string, or complete item(s).
    if {$clipToken == "string"}  {
	CanvasTextInsert $w $cmds
	
    } elseif {$clipToken == "item"}  {
	set sep [string trim $clipItemSep]
	set firstCmd [CmdToken cmds $sep]
	if {$debugLevel >= 2}  {
	    puts "PasteFromClipboardToCanvas:: firstCmd=$firstCmd, sep=$sep"
	}
	while {$firstCmd != -1}  {
	    PasteSingleFromClipboardToCanvas $w $firstCmd
	    set firstCmd [CmdToken cmds $sep]
	}
    }
    # Default for 'clipToken' should always be "string" to be prepared
    # for imports from other apps. Not 100% foolproof.
    set clipToken "string"
}

#   PasteSingleFromClipboardToCanvas ---
#
#   Evaluates the canvas create command given by 'cmd', but at a coordinate
#   offset, makes it the new selection and copies it again to clipboard.
#   Be sure to treat newlines correctly when sending command to clients.

proc PasteSingleFromClipboardToCanvas	 { w cmd }	{
    global	itno myItpref prefs debugLevel allIPnumsTo ipNum2Socket

    set nl_ "\\n"
    if {$debugLevel >= 2}  {
	puts "PasteSingleFromClipboardToCanvas:: cmd=$cmd"
    }
    # add new tags
    set itemType [lindex $cmd 1]
    set theTags [list $itemType $myItpref/$itno]
    lappend cmd -tags $theTags
    
    # make coordinate offset, first get coords
    set ind1 [lsearch $cmd \[0-9.\]*]
    set ind2 [expr [lsearch $cmd -*\[a-z\]*] - 1]
    #	puts "ind1=$ind1, ind2=$ind2"
    set theCoords [lrange $cmd $ind1 $ind2]
    #	puts "theCoords=$theCoords"
    set cooOffset {}
    foreach coo $theCoords {
	lappend cooOffset [expr $coo + $prefs(offsetCopy)]
    }
    
    # paste back coordinates in cmd
    set newcmd [concat [lrange $cmd 0 [expr $ind1 - 1]] $cooOffset  \
	    [lrange $cmd [expr $ind2 + 1] end]]
    if {$debugLevel >= 2}  {
	puts "PasteSingleFromClipboardToCanvas:: newcmd=$newcmd"
    }
    # Actually draw canvas command.
    set id [eval $w $newcmd]
    if {$debugLevel >= 2}  {
	puts "PasteSingleFromClipboardToCanvas:: id=$id"
    }
    
    # Create new bbox and select item.
    MarkBbox $w 1 $id
    
    # Copy the newly pasted object to clipboard.
    CopySelectedToClipboard  $w copy
    
    # Write to all other clients; need to make a one liner first.
    regsub -all "\n" $newcmd $nl_ oneliner
    if {$debugLevel >= 2}  {
	puts "PasteSingleFromClipboardToCanvas:: oneliner=$oneliner"
    }
    foreach ip $allIPnumsTo  {
	set s $ipNum2Socket($ip)
	puts $s "CANVAS: $oneliner"
    }
    incr itno
}

#   CanvasTextPaste ---
#
#   Unix style paste using button 2.

proc CanvasTextPaste { c {x {}} {y {}} }  {
    global	debugLevel allIPnumsTo ipNum2Socket
    
    if {$debugLevel >= 2}  {
	puts "CanvasTextPaste::"
    }
    # If no selection just return.
    if {[catch {selection get} _s] &&
    [catch {selection get -selection CLIPBOARD} _s]}  {
	if {$debugLevel >= 2}  {
	    puts "CanvasTextPaste:: no selection"
	}
	return
    }
    # Once the text string is found use...
    CanvasTextInsert $c $_s
    return

    # old surplus code
    set id [$c focus]
    if {[string length $id] == 0}  {
	set id [$c find withtag current]
    }
    if {[string length $id] == 0}  {
	# no object under the mouse
	if {[string length $x] == 0}  {
	    # keyboard paste
	    set x [expr [winfo pointerx $c] - [winfo rootx $c]]
	    set y [expr [winfo pointery $c] - [winfo rooty $c]]
	}
	CanvasFocus $c $x $y
    } else {
	$c focus $id
    }
    set theItno [CanvasGetItnoFrom $c focus]
    if {$theItno == ""}	 {
	if {$debugLevel >= 2}  {
	    puts "CanvasTextPaste:: theItno={}"
	}
	return
    }
    # The actual insert command.
    #puts "CanvasTextPaste:: focus=[$c focus],  \
    #	    insertindex=[$c index [$c focus] insert], _s=$_s, theItno=$theItno"
    $c insert [$c focus] insert $_s
    set ind [expr [$c index [$c focus] insert] - 0]
    foreach ip $allIPnumsTo  {
        set s $ipNum2Socket($ip)
	puts $s [list "CANVAS:" insert $theItno $ind $_s]
    }	
}

proc ColorSelector { col }	{
    global  prefs wColSel idColSel
    
    set col [tk_chooseColor -initialcolor $col]
    if {[string length $col] > 0}	 {
	# The change should be triggered automatically through the trace.
	set prefs(fgCol) $col
	#$wColSel itemconfigure $idColSel -fill $prefs(fgCol)
	#$wColSel raise $idColSel
    }
}

proc SetCanvasBgColor { w }	 {
    global  prefs
    
    set col [tk_chooseColor -initialcolor $prefs(bgColCanvas)]
    if {[string length $col] > 0}  {
	# The change should be triggered automatically through the trace.
	set prefs(bgColCanvas) $col
	#$w configure -bg $prefs(bgColCanvas)
    }
}

proc DoCanvasGrid  {  }	 {
    # make a grid in the canvas; uses prefs(canGridOn) to toggle grid
    global  wCan prefs
    
    set length 1200
    set gridDist $prefs(gridDist)
    set canGridOn $prefs(canGridOn)
    if {$canGridOn == 0}  {
	$wCan delete grid
	return
    }
    for {set x $gridDist} {$x <= $length} {set x [expr $x + $gridDist]}	 {
	$wCan create line $x 0 $x $length	\
		-width 1 -fill gray50 -stipple gray50 -tags grid
    }
    for {set y $gridDist} {$y <= $length} {set y [expr $y + $gridDist]}  {
	$wCan create line 0 $y $length $y  \
		-width 1 -fill gray50 -stipple gray50 -tags grid
    }
    $wCan lower grid
}

#   DoCloseConnection ---
#
#   Handle everything to close a connection.

proc DoCloseConnection { ipNum }  {
    global  prefs allIPnumsTo ipNum2Socket ipNum2Name
        
    # If it is not there, just return.
    set ind [lsearch $allIPnumsTo $ipNum]
    if {$ind == -1}  {
	return
    }
    # Do the actual closing.
    catch {close $ipNum2Socket($ipNum)}

    # Update the communication frame; remove connection 'to'.
    if {$prefs(autoDisconnect)}  {
	MakeCommEntry $ipNum  0 0
    } else  {
	MakeCommEntry $ipNum  0 -1
    }
    
    if {[llength $allIPnumsTo] == 0}  {
	.menu.file entryconfigure "*Put File*" -state disabled
	.menu.file entryconfigure "*Put Canvas*" -state disabled
	.menu.file entryconfigure "*Get Canvas*" -state disabled
	#.menu.info entryconfigure "*On Server*" -state disabled
	.menu.info entryconfigure "*On Clients*" -state disabled
    }
}

proc SavePostscript { w }  {
    set typelist {
	{"Postscript File" {".ps"}}
    }
    set ans [tk_getSaveFile -title "Save As Postscript"	 \
	    -filetypes $typelist -defaultextension ".ps"]
    if {$ans != ""}	 {
	$w postscript -pageheight 20c -pagewidth 16c -pageanchor nw -file $ans
    }
}

proc NewImportAnchor  {  }  {
    global  importAnchor prefs dims
    
    set x $importAnchor(x)
    set y $importAnchor(y)
    
    # Update 'importAnchor'.
    incr importAnchor(x) $prefs(offsetCopy)
    incr importAnchor(y) $prefs(offsetCopy)
    if {$importAnchor(x) > [expr $dims(wRoot) - 60] 
    || $importAnchor(y) > [expr $dims(hRoot) - 60]} {
	set importAnchor(x) $prefs(offsetCopy)
	set importAnchor(y) $prefs(offsetCopy)
    }
    return "$x $y"
}

#   ImportImageOrMovieDlg ---
#
#   Handles the dialog of opening a file ans then lets 'DoImportImageOrMovie' 
#   do the rest.

proc ImportImageOrMovieDlg { w }  {
    global  prefs supExts supTypes tcl_platform
    
    if {$prefs(QT)}  {
	if {$tcl_platform(platform) == "macintosh"}  {
	    set typelist [list   \
	      [list "Image" * $supTypes(image)]  \
	      [list "QT Movie" * $supTypes(movie)]  \
	      [list "Any File" *]]	    
	} else  {
	    set typelist [list  \
	      [list "Image" $supExts(image) $supTypes(image)] \
	      [list "QT Movie" $supExts(movie) $supTypes(movie)]]
	}
    } else {	
	if {$tcl_platform(platform) == "macintosh"}  {
	    set typelist [list [list "Image" * $supTypes(image)]]
	} else  {
	    set typelist [list [list "Image" $supExts(image) $supTypes(image)]]
	}
    }
    #puts "typelist=$typelist"
    set ans [tk_getOpenFile -title "Open Image/Movie" -filetypes $typelist]
    if {$ans == ""} {
	return
    }
    set fileName $ans
    
    # Once the file name is chosen continue...
    set anchor [NewImportAnchor]
    DoImportImageOrMovie $w $fileName "$anchor"
}

#   DoImportImageOrMovie ---
#   
#   Opens an image in the canvas and puts it to all other clients.
#   If QuickTime is supported, a movie can be opened as well.
#   'supExts(bin)' and 'supExts(movie)' tells which formats that are supported
#   on this platform.
#   'itspecs' = {x y} is a list of the x and y coordinates, or
#   'itspecs' = {x y idtag} where idtag is the item specific tag is used.
#   If idtag is missing then use own.
#   where = "all": write to this canvas and all others.
#   where = "other": write only to remote client canvases.
#   where = ip number: write only to this remote client canvas and not to own.
#   where = "own": write only to this canvas and not to any other.

proc DoImportImageOrMovie { w fileName itspecs {where all} }  {
    global  myItpref itno debugLevel allIPnumsTo prefs supExts macType2Ext  \
      tcl_platform
    
    set dot_ {\.}

    set fileTail [file tail $fileName]
    set fext [string tolower [file extension $fileTail]]
    if {$debugLevel >= 2}  {
	puts "DoImportImageOrMovie::fileName=$fileName, itspecs=$itspecs, \
	  where=$where, fext=$fext"
    }
    # The Mac allows the possibility of no extension; find one!
    if {$tcl_platform(platform) == "macintosh" && $fext == ""}  {
	set macType [file attributes $fileName -type]
	#puts "macType=$macType"
	set fext $macType2Ext($macType)
    }
    set x [lindex $itspecs 0]
    set y [lindex $itspecs 1]
    if {[llength $itspecs] == 3}  {
	set useTag [lindex $itspecs 2]
    } else  {
	set useTag $myItpref/$itno
    }
    
    # Depending on the file extension do different things.
    # Image:
    if {[lsearch $supExts(image) $fext] >= 0}  {
	
	if {$where == "all" || $where == "own"}  {
	    if {[lsearch -exact [image names] im_$fileTail] < 0}  {
		image create photo im_$fileTail -file $fileName
	    }
	    set cmd "create image $x $y   \
	      -image im_$fileTail -anchor nw -tags {image $useTag}"
	    eval $w $cmd
	}
	# Transfer image file to all other servers.
	# The optional parameters is a list: {idtag x y}
	if {$where != "own"}  {
	    if {[llength $allIPnumsTo] > 0}  { 
		PutFile $fileName $where "$x $y $useTag"
	    }	
	}
	
	# QuickTime movies. Movies are put in frame with class 'QTFrame'
	# in order to catch mouse events.
	# Strip dots in filename, " " to _; small chars in pathname.
    } elseif {[lsearch $supExts(movie) $fext] >= 0}  {
	set fileRoot [string tolower [file rootname [file tail $fileName]]]
	regsub -all $dot_ $fileRoot "" tmp
	regsub -all " " $tmp "_" newName
	set fr ${w}.fr_${newName}${itno}
	if {$where == "all" || $where == "own"}  {
	    
	    # Make a frame for the movie; need special class to catch mouse events.
	    frame $fr -height 1 -width 1 -bg gray40 -class QTFrame
	    
	    # Import the movie only if not exists already. ???
	    if {[catch {Movie ${fr}.m -file $fileName -progress MovieProgressProc} \
	      msg]}  {
		tk_messageBox -message "Error:  $msg" \
		  -icon error -type ok
		catch {destroy $fr}
		return
	    }
	    set mpath $msg	
	    $w create window $x $y -anchor nw -window $fr -tags "movie $useTag"
	    place $mpath -in $fr -anchor nw -x 3 -y 3
	    update idletasks
	    set height [winfo height $mpath]
	    set width [winfo width $mpath]
	    if {$debugLevel >= 2}  {
		puts "mpath=$mpath, newName=$newName, height=$height, width=$width"
	    }
	    $fr configure -width [expr $width + 6] -height [expr $height + 6]
	    $mpath play    
	}
	
	# Transfer movie file to all other servers.
	# Several options possible:
	#   1) flatten, put in httpd directory, and transfer via http.
	#   2) make hint track and serve using RTP.
	#   3) put as an ordinary binary file, perhaps flattened.
	if {($where != "own") && ([llength $allIPnumsTo] > 0)}  {
		
	    # Need to flatten QT movie first?
	    set tmpflat "flatten_$fileRoot"
	    $mpath flatten $tmpflat
	    
	    # Do the actual transfer.
	    #PutFile $tmpflat $where "$x $y $useTag"
	    
	    # Serve via http?
	    
	    # Serve via RTP?
	    
	}
    }
    # Add to the lists of known files.
    AddToKnownFiles $fileTail $fileName "$useTag"
    
    # Update 'itno' only when also writing to own canvas!
    if {$where == "all" || $where == "own"}  {
	incr itno
    }
}

#   ResizeImage ---
#
#   Uhh.. resizes the selected images. 'zoomFactor' is 0,1 for no resize,
#   2 for an enlargement with a factor of two, and
#   -2 for a size decrease to half size.   
#   'which' = "sel": selected images, or a specific image with tag 'which'.
#   'newTag' = "auto": use 'myItpref' and running 'itno', 
#   else 'newTag' is the tag to use.
#   where = "all": write to this canvas and all others.
#   where = "other": write only to remote client canvases.
#   where = ip number: write only to this remote client canvas and not to own.
#   where = "own": write only to this canvas and not to any other.

proc ResizeImage { w zoomFactor which newTag {where all} }  {
    global  itno myItpref ipNum2Socket allIPnumsTo
    
    #puts "ResizeImage:: w=$w, zoomFactor=$zoomFactor"
    set scaleFactor 2
    set int_ {[-0-9]+}
    
    # Compute total resize factor.
    if {$zoomFactor >= 0 && $zoomFactor <= 1 }  {
	return
    } elseif {$zoomFactor == 2}  {
	set theScale 2
    } elseif {$zoomFactor == -2}  {
	set theScale 0.5
    } else  {
	return
    }
    if {$which == "sel"}  {
	# Choose selected images only.
	set ids [$w find withtag selected]
    } else  {
	# Choose image with tag 'which'.
	set ids [$w find withtag $which]
	if {[llength $ids] <= 0}  {
	    return
	}
    }
    set idsNew {}
    foreach id $ids  {
	
	if {$where == "all" || $where == "own"}  {
	    
	    set theType [$w type $id]
	    #puts "ResizeImage:: id=$id, theType=$theType, theScale=$theScale"
	    if {[string compare $theType "image"] != 0}  {
		continue
	    }
	    # Check if no privacy problems. Only if 'which' is the selected.
	    set itOrig [CanvasGetItnoFrom $w $id]
	    if {$which == "sel" && $itOrig == ""}  {
		continue
	    }
	    set theCoords [$w coords $id]
	    set theIm [$w itemcget $id -image]
	    # Resized photos add tag to name '_zoom2' for double size,
	    # '_zoom-2' for half size etc.
	    if {[regexp "_zoom(${int_})$" $theIm match sizeNo]}  {
		# This image already resized.
		#puts "# This image already resized."
		if {$zoomFactor == 2}  {
		    if {$sizeNo >= 2}  {
			set newSizeNo [expr $sizeNo * $zoomFactor]
		    } elseif {$sizeNo == -2} {
			set newSizeNo 0
		    } else {
			set newSizeNo [expr $sizeNo/$zoomFactor]
		    }
		} elseif {$zoomFactor == -2}  {
		    if {$sizeNo <= -2}  {
			set newSizeNo [expr -$sizeNo * $zoomFactor]
		    } elseif {$sizeNo == 2} {
			set newSizeNo 0
		    } else {
			set newSizeNo [expr -$sizeNo/$zoomFactor]
		    }
		}
		#puts "newSizeNo=$newSizeNo"
		if {$newSizeNo == 0}  {
		    # Get original image. Strip off the _zoom tag.
		    regsub "_zoom$sizeNo" $theIm  "" newImName
		} else  {
		    regsub "_zoom$sizeNo" $theIm "_zoom$newSizeNo" newImName
		}
	    } else  {
		# Add tag to name indicating that it has been resized.
		#puts "# Add tag to name indicating that it has been resized."
		set newSizeNo $zoomFactor
		set newImName ${theIm}_zoom${newSizeNo}
	    }
	    #puts "ResizeImage:: theCoords=$theCoords, theIm=$theIm, \
	    #  newImName=$newImName, newSizeNo=$newSizeNo"
	    
	    # Create new image for the scaled version if it does not exist before.
	    if {[lsearch -exact [image names] $newImName] < 0}  {
		#puts "new photo: zoomFactor=$zoomFactor"
		# Make new.
		image create photo $newImName
		if {$zoomFactor > 0}  {
		    $newImName copy $theIm -zoom $theScale
		} else  {
		    $newImName copy $theIm -subsample [expr round(1.0/$theScale)]
		}
	    }
	    # Choose this clients automatic tags or take 'newTag'.
	    if {$newTag == "auto"}  {
		set useTag $myItpref/$itno
	    } else  {
		set useTag $newTag
	    }
	    set cmd "create image $theCoords   \
	      -image $newImName -anchor nw -tags {image $useTag}"
	    set newId [eval $w $cmd]
	    lappend idsNew $newId
	}
	# This ends works on resizing "own" photos.
	
	# Make a list with all ip numbers to put file to.
	if {$where != "own"}  {
	    if {$where == "other" || $where == "all"}  {
		set allPutIP $allIPnumsTo
	    } else  {
		set allPutIP $where
	    }    
	    # Loop over all connected servers or only the specified one.
	    foreach ip $allPutIP  {
		puts $ipNum2Socket($ip)  \
			"RESIZE IMAGE: $itOrig $useTag $zoomFactor"
	    }
	}
	# Remove old.
	$w delete $id
	incr itno
    }
    DeselectAll $w
    # Mark the new ones.
    foreach id $idsNew  {
	MarkBbox $w 1 $id
    }
}

#   DoEraseAll ---
#
#   where = "all": erase this canvas and all others.
#   where = "other": erase only client canvases.
#   where = "own": erase only own canvas.

proc DoEraseAll { w { where all } }	 {
    global  allIPnumsTo ipNum2Socket
    
    DeselectAll $w
    set ids [$w find all]
    foreach id $ids {
	# do not erase grid
	set theTags [$w gettags $id]
	if {[lsearch $theTags grid] >= 0}  {
	    continue
	}
	set it [CanvasGetItnoFrom $w $id]
	if {$it == ""}  {
	    continue
	}
	if {$where != "other"}  {
	    $w delete $id
	}
	if {$where != "own"}  {
	    foreach ip $allIPnumsTo  {
		set s $ipNum2Socket($ip)
		puts $s "CANVAS: delete $it"
	    }
	}
    }
}

#   DoOpenConnection ---
#
#   Starts process to open a connection to another client.
#   Needs procedures from the mkOpenConnection.tcl file.
#   When opening in async mode, the calling sequence is:
#      DoOpenConnection -> mkOpenConnection -> PushBtConnect ->
#         -> DoConnect -> WhenSocketOpensInits,
#   where the last sequence is triggered by a fileevent.

proc DoOpenConnection {	 }	{
    global  servCan specServCan wCan internalIPname debugLevel makeSafeServ  \
      debugLevel thisServPort remoteServPort internalIPnum prefs
    
    if {$debugLevel >= 2}  {
	puts "DoOpenConnection:: enter"
    }
    # Complete dialog for choosing address etc.
    set ans [mkOpenConnection]
    if {$debugLevel >= 2}  {
	puts "DoOpenConnection:: ans=$ans"
    }
}

#   IsConnectedToQ ---
#
#   Finds if connected to 'ipNameOrNum' by searching 'allIPnumsTo'.
#   It always answers that it is connected to itself ('thisIPnum').
#   This is also true if 'thisIPnum' equal to the standard 127.0.0.1.

proc IsConnectedToQ { ipNameOrNum }  {
    global  ipName2Num allIPnumsTo thisIPnum internalIPnum

    set ip_ {[a-zA-Z]+}
    # Find out if 'ipNameOrNum' is name or number.
    # If any character in 'ipNameOrNum' then assume it is a name.
    if {[regexp $ip_ $ipNameOrNum]}  {
	#puts "IsConnectedToQ:: ipNameOrNum=$ipNameOrNum, its a name!"
	if {[info exists ipName2Num($ipNameOrNum)]}  {
	    set ipnum $ipName2Num($ipNameOrNum)
	} else  {
	    return 0
	}
    } else  {
	set ipnum $ipNameOrNum
    }
    if {$ipnum == $internalIPnum}  {
	return 1
    } elseif {[lsearch -exact "$allIPnumsTo $thisIPnum" $ipnum] >= 0}  {
	return 1
    } else  {
	return 0
    }
}

#   Sets auto disconnect identical to autoConnect.

proc DoAutoConnect  {  }  {
    global  prefs
    set prefs(autoDisconnect) $prefs(autoConnect)
}

#   CmdToken ---
#   
#   Returns part of 'cmdName' up to 'separator' and deletes that part 
#   from 'cmdName'.

proc CmdToken { cmdName separator }  {
    upvar $cmdName theCmd
    
    # If nothing then return -1.
    if {[llength $theCmd] == 0} {
	return -1
    }
    
    set indSep [lsearch -exact $theCmd $separator]
    # If no separator then just return the remaining part.
    if {$indSep == -1}  {
	set firstPart $theCmd
	set theCmd {}
	return $firstPart
    }
    # If separator in -text then ???.
    if {[lindex $theCmd [expr $indSep - 1]] != "-text"}  {
	set firstPart [lrange $theCmd 0 [expr $indSep - 1]]
    } else {
	puts "Warning in CmdToken: -text part wrong"
    }
    set theCmd [lrange $theCmd [expr $indSep + 1] end]
    return $firstPart
}

#   MakeCommEntry ---
#
#   Adds, removes or updates an entry in the communications frame.
#   If to or from is -1 then disregard this variable.

proc MakeCommEntry  { ipNum to from }  { 
    global  commFrame commTo commFrom commIP listCommTo listCommFrom \
	    nEnt ipNum2nEnt ipName2Num dims tcl_platform debugLevel  \
	    allIPnums allIPnumsTo ipNum2Name ipNum2User sysFont prefs wCan
    
    # Need to check if already exist before adding a completely new entry.
    set ind [lsearch $allIPnums $ipNum]
    if {$ind >= 0}  {
	set alreadyThere 1
    } else {
	set alreadyThere 0
    }
    if {!$alreadyThere}  {
	lappend allIPnums $ipNum
	set commIP($ipNum) $ipNum2Name($ipNum)
	# Just init arrays.
	set commTo($ipNum) 0		
	set commFrom($ipNum) 0
    } 
    if {$to >= 0}  {
	set commTo($ipNum) $to
    }
    if {$from >= 0}  {
	set commFrom($ipNum) $from
    }
    if {$debugLevel >= 2}  {
	puts "MakeCommEntry:: alreadyThere=$alreadyThere, ipNum=$ipNum, ind=$ind"
    }
    # If it is not there and shouldnt be added, just return.
    if {!$alreadyThere && $commTo($ipNum) == 0 && $commFrom($ipNum) == 0}  {
	if {$debugLevel >= 2}  {
	  puts "MakeCommEntry:: it is not there and shouldnt be added"
	}
	return
    }
    # Update allIPnumsTo to contain each ip num connected to.
    set ind [lsearch $allIPnumsTo $ipNum]
    if {$ind == -1 && $commTo($ipNum) == 1}  {
	lappend allIPnumsTo $ipNum
    } elseif {$ind >= 0 && $commTo($ipNum) == 0}  {
	set allIPnumsTo [lreplace $allIPnumsTo $ind $ind]
    }
    # Keep lists to find out which checkbutton triggered.
    if {!$alreadyThere}  {	
	lappend listCommTo $commTo($ipNum)
	lappend listCommFrom $commFrom($ipNum)
    } elseif {$alreadyThere} {
	set listCommTo [lreplace $listCommTo $ind $ind $commTo($ipNum)]
	set listCommFrom [lreplace $listCommFrom $ind $ind $commFrom($ipNum)]
    }
    # If both to and from 0 then remove entry.
    if {$commTo($ipNum) == 0 && $commFrom($ipNum) == 0}  {
	CommEntryRemove $ipNum
    }
    # If already exist dont add widgets.
    if {$alreadyThere}  {
	return
    }
    # Size administration is very tricky; blood, sweat and tears...
    # Fix the canvas size to relax wm geometry. (Don't forget the border)
    $wCan configure -height [expr [winfo height $wCan] - 2]  \
      -width [expr [winfo width $wCan] - 2]
    # Switch off the geometry constraint to let resize automatically.
    wm geometry . {}
    
    # Add new status line.
    entry $commFrame.ad$nEnt -width 26 -font $sysFont(s)  \
	    -textvariable commIP($ipNum) -relief sunken   \
	    -bg $prefs(bgColGeneral)
    $commFrame.ad$nEnt configure -state disabled
    entry $commFrame.us$nEnt -width 16 -font $sysFont(s)   \
	    -textvariable ipNum2User($ipNum) -relief sunken  \
	    -bg $prefs(bgColGeneral)
    $commFrame.us$nEnt configure -state disabled
    checkbutton $commFrame.to$nEnt -variable commTo($ipNum)   \
	    -highlightthickness 0 -state normal -command CheckCommTo
    checkbutton $commFrame.from$nEnt -variable commFrom($ipNum)  \
	    -highlightthickness 0 -state disabled
    grid $commFrame.ad$nEnt $commFrame.us$nEnt $commFrame.to$nEnt   \
      $commFrame.from$nEnt -padx 4
    update idletasks
    
    # Organize the new geometry. First fix using wm geometry, then relax
    # canvas size.
    set newGeom [ParseWMGeometry .]
    #puts "newGeom=$newGeom"
    wm geometry . [lindex $newGeom 0]x[lindex $newGeom 1]
    $wCan configure -height 1 -width 1
    
    # Geometry considerations. Update geometry vars and set new minsize.
    after idle {
	set dims(hComm) [winfo height $commFrame]
	set dims(hMinRoot) [expr $dims(hMinCanvas) + $dims(hStatus) + $dims(hComm)]
	set dims(hMinTot) [expr $dims(hMinRoot) + $dims(hMenu)]
	if {$debugLevel >= 2}  {
	    puts "MakeCommEntry:: dims(hComm)=$dims(hComm),  \
	      dims(hMinRoot)=$dims(hMinRoot), dims(hMinTot)=$dims(hMinTot)"
	}
	# Note: wm minsize is *with* the menu!!!
	wm minsize . $dims(wMinTot) $dims(hMinTot)
    }	    
    # Map ip name to nEnt.
    if {$debugLevel >= 2}  {
	puts "MakeCommEntry:: nEnt=$nEnt"
    }
    set ipNum2nEnt($ipNum) $nEnt
    # Step up running index.
    incr nEnt
}

# debug stuff....
proc junkCommEntry  { ipNum }  {
    global ipNum2Name ipNum2User ipNum2Socket
    
    set ipNum2Name($ipNum) Mats$ipNum
    set ipNum2User($ipNum) "Mats Be"
    set ipNum2Socket($ipNum) sock99
    MakeCommEntry $ipNum 1 1
}

#   CheckCommTo ---
#
#   This is the callback function when the checkbutton 'To' has been trigged.

proc CheckCommTo {  }  {
    global  commTo listCommTo ipName2Num debugLevel allIPnums ipNum2Name  \
      ipNum2ServPort
    
    # Find out which button triggered the event.
    set ind -1
    set trigInd -1
    set trigIP ""
    foreach ip $allIPnums  {
	incr ind
	set newTo $commTo($ip)
	set oldTo [lindex $listCommTo $ind]
	if {$debugLevel >= 2}  {
	    puts "CheckCommTo:: newTo=$newTo, oldTo=$oldTo"
	}
	if {$newTo != $oldTo}  {
	    set trigInd $ind
	    set trigIP $ip	
	}
    }
    if {$debugLevel >= 2}  {
	puts "CheckCommTo:: trigInd=$trigInd, trigIP=$trigIP"
    }
    if {$trigInd == -1}  {
	return
    }
    # Sync commTo and listCommTo.
    set listCommTo [lreplace $listCommTo $trigInd $trigInd $commTo($trigIP)]
    
    # Close connection.
    if {$commTo($trigIP) == 0}  {
	set res [tk_messageBox -message "Are you sure that you want\
	  to disconnect $ipNum2Name($trigIP)?" \
	  -icon warning -type yesno -default yes]
	if {$res == "no"}  {
	    # reset 
	    set commTo($trigIP) 1
	    set listCommTo [lreplace $listCommTo $trigInd $trigInd $commTo($trigIP)]
	    return
	} elseif {$res == "yes"}  {
	    DoCloseConnection $trigIP
	}
    } elseif {$commTo($trigIP) == 1}  {
	# Open connection. Let propagateSizeToClients = true.
	DoConnect $trigIP $ipNum2ServPort($trigIP) 1
	MakeCommEntry $trigIP 1 -1
    }
}

proc CommEntryRemove  { ipNum }   {
    global  commFrame listCommTo listCommFrom commTo commFrom  \
      commIP ipNum2nEnt debugLevel wCan  \
      allIPnums allIPnumsTo ipName2Num ipNum2Name dims
    
    # Find widget paths from ipNum and remove the entries .
    # Also update all variables that stores info on this connection.
    set no $ipNum2nEnt($ipNum)
    set ind [lsearch $allIPnums $ipNum]
    set ind3 [lsearch $allIPnumsTo $ipNum]
    if {$debugLevel >= 2}  {
	puts "CommEntryRemove:: no=$no, ind=$ind"
    }
    if {$ind >= 0}  {
	set allIPnums [lreplace $allIPnums $ind $ind]
	set listCommTo [lreplace $listCommTo $ind $ind]
	set listCommFrom [lreplace $listCommFrom $ind $ind]
    }
    if {$ind3 >= 0}  {
	set allIPnumsTo [lreplace $allIPnumsTo $ind3 $ind3]
    }    
    # Size administration is very tricky; blood, sweat and tears...
    # Fix the canvas size to relax wm geometry.
    $wCan configure -height [expr [winfo height $wCan] - 2]  \
      -width [expr [winfo width $wCan] - 2]
    # Switch off the geometry constraint to let resize automatically.
    wm geometry . {}

    # Remove the widgets.
    grid forget $commFrame.ad$no $commFrame.us$no $commFrame.to$no   \
      $commFrame.from$no
    destroy $commFrame.ad$no $commFrame.us$no $commFrame.to$no   \
      $commFrame.from$no
    update idletasks
    
    # Organize the new geometry. First fix using wm geometry, then relax
    # canvas size.
    set newGeom [ParseWMGeometry .]
    #puts "newGeom=$newGeom"
    wm geometry . [lindex $newGeom 0]x[lindex $newGeom 1]
    $wCan configure -height 1 -width 1

    # Geometry considerations. Update geometry vars and set new minsize.
    after idle {
	set dims(hComm) [winfo height $commFrame]
	set dims(hMinRoot) [expr $dims(hMinCanvas) + $dims(hStatus) + $dims(hComm)]
	set dims(hMinTot) [expr $dims(hMinRoot) + $dims(hMenu)]
	if {$debugLevel >= 2}  {
	    puts "CommEntryRemove:: dims(hComm)=$dims(hComm),  \
	      dims(hMinRoot)=$dims(hMinRoot), dims(hMinTot)=$dims(hMinTot)"
	}
	# Note: wm minsize is *with* the menu!!!
	wm minsize . $dims(wMinTot) $dims(hMinTot)
    }	    
    # Update the comm to/from arrays.
    catch {unset commIP commTo commFrom}
    foreach ip $allIPnums {
	set commIP($ip) $ipNum2Name($ip)
    }
    foreach to $listCommTo ip $allIPnums {
	set commTo($ip) $to
    }
    foreach from $listCommFrom ip $allIPnums {
	set commFrom($ip) $from
    }
    if {$debugLevel >= 2}  {
	puts "exit CommEntryRemove"
    }
}

#   PutFileDlg ---
#
#   Opens a file in a dialog and lets 'PutFile' do the job of transferring
#   the file to all other clients.

proc PutFileDlg  {  }  {
    global  debugLevel allIPnumsTo prefs supExts supTypes tcl_platform
    
    if {[llength $allIPnumsTo] == 0}  {
	return
    }
    
    if {$prefs(QT)}  {
	if {$tcl_platform(platform) == "macintosh"}  {
	    set typelist [list   \
	      [list "Text File" * $supTypes(txt)]  \
	      [list "Image" * $supTypes(image)]  \
	      [list "QT Movie" * $supTypes(movie)]]	    
	} else  {
	    set typelist [list  \
	      [list "Text File" $supExts(txt)]  \
	      [list "Image" $supExts(image) $supTypes(image)] \
	      [list "QT Movie" $supExts(movie) $supTypes(movie)]]
	}
    } else {	
	if {$tcl_platform(platform) == "macintosh"}  {
	    set typelist [list   \
	      [list "Text File" * $supTypes(txt)]  \
	      [list "Image" * $supTypes(image)]]
	} else  {
	    set typelist [list   \
	      [list "Text File" $supExts(txt)]  \
	      [list "Image" $supExts(image) $supTypes(image)]]
	}
    }
    set ans [tk_getOpenFile -title "Put Image/Movie" -filetypes $typelist]
    if {$ans == ""}	 {
	return
    }
    set fileName $ans
    
    # Do the actual putting once the file is chosen.
    PutFile $fileName "all"
}

#   PutFile ---
#   
#   Transfers a file to all remote servers. It needs some negotiation to work.
#   'fileName' is the path to the file to be put.
#   If no 'fileName' given, open dialog to select one.
#   'optList' contains extra info, for instance {x y idtag}, which must be 
#   understood by the server.
#   where = "other" or "all": put only to remote clients.
#   where = ip number: put only to this remote client.
#   The calling sequence is:
#   
#   'PutFileOpenAndConnect'  opens file and socket, requests a
#   connection and sets up event handler for clients answer.
#   
#   'PutFileWaitingForResponse'  waits until the client answers and
#   act accordingly; if ok it sets up the next event handler which
#   deals with the copying itself.
#
#   'PutFileCallback'  copies the file from disk to socket
#   until end of file using event handlers, when it closes things etc.

proc PutFile  { fileName where {optList {}} }  {
    global  debugLevel allIPnumsTo ipNum2ServPort  \
      prefs supExts macType2Ext tcl_platform
    
    if {[llength $allIPnumsTo] == 0}  {
	return
    }
    if {$debugLevel >= 2}  {
	puts "PutFile:: fileName=$fileName, optList=$optList"
    }
    # Use extension to hint transfer mode for *file read*.
    set fileTail [file tail $fileName]
    set fext [string tolower [file extension $fileTail]]

    # The Mac allows the possibility of no (or any) extension; find one!
    if {$tcl_platform(platform) == "macintosh"}  {
	set macType [file attributes $fileName -type]
	set fext $macType2Ext($macType)
    }
    # Make a list with all ip numbers to put file to.
    if {$where == "other" || $where == "all"}  {
	set allPutIP $allIPnumsTo
    } else  {
	set allPutIP $where
    }    
    # Loop over all connected servers or only the specified one.
    foreach ip $allPutIP  {
	
	# All file copying to the sockets through event handlers.
	# See above for the calling sequence.
	PutFileOpenAndConnect $ip $fileName $fext $optList
    }
}

#   PutFileOpenAndConnect ---
#
#   recordPutFile(sock): {fileId filePath}; just to keep track of things.

proc PutFileOpenAndConnect  { ip fileName fext {optList {}} }  {
    global  supExts statMess ipNum2ServPort nameOrIPToKillerId tcl_platform  \
      debugLevel ipNum2Name debugLevel recordPutFile
    
    if {$debugLevel >= 2}  {
	puts "PutFileOpenAndConnect:: ip=$ip, fileName=$fileName, optList=$optList"
    }
    # Open the file all over for each connected client.
    if {[catch {open $fileName r} orig]}  {
	tk_messageBox -message "Failed when trying to open $fileName." \
	  -icon error -type ok
	# Handle error.
	catch {close $orig}
	return 
    }
    set fileTail [file tail $fileName]
    set totBytes [file size $fileName]
    
    # Regardless of the mac extension, give the file its normal extension
    # when putting.
    set fileTailRemote $fileTail
    if {$tcl_platform(platform) == "macintosh"}  {
	set fileTailRemote [file rootname $fileTail]$fext
    }
    # Use extension to hint transfer mode for *file read*.
    if {[lsearch $supExts(bin) $fext] >= 0}  {
	fconfigure $orig -translation {binary binary}
    } elseif {[lsearch $supExts(txt) $fext] >= 0}  {
	fconfigure $orig -translation auto
    }
    if {$debugLevel >= 2}  {
	puts "PutFileOpenAndConnect:: fileTail=$fileTail, fext=$fext, \
	  fileTailRemote=$fileTailRemote"
    }
    # Open new temporary socket only for this put operation.
    set statMess "Contacting client $ipNum2Name($ip). Waiting for response..."
    if {$debugLevel >= 2}  {
	puts "statMess=$statMess, clock=[clock second]"
    }
    update idletasks
    
    # The actual socket is opened; this should be done -async in the future.
    if {[catch {socket $ip $ipNum2ServPort($ip)} s]}  {
	tk_messageBox -message   \
	  "Failed when trying to connect to $ipNum2Name($ip). $s" \
	  -icon error -type ok
	# Handle error.
	PutFileShutDown $orig $s
	return
    }    
    # Set in nonblocking mode and register the next event handler.
    fconfigure $s -blocking 0
    fconfigure $s -buffering line
    # BUG TRACKING!!!!
    flush $s
    # BUG TRACKING!!!!
    
    # Store a record for this pair of socket fileId.
    set recordPutFile($s) [list $orig $fileName]
        
    # Schedule timeout event.
    PutFileScheduleKiller $orig $s $ip "$fileTailRemote"
    
    # Set up event handler to wait for server response.
    fileevent $s readable [list PutFileWaitingForResponse   \
      $ip $orig $s $totBytes $fileTailRemote]
    set statMess "Client contacted: $ipNum2Name($ip); negotiating..."
    update idletasks
    if {$debugLevel >= 2}  {
	puts "PutFileOpenAndConnect:: fconfigure: [fconfigure $s]"
	puts "statMess=$statMess, clock=[clock second]"
    }    

    # This command should be understood by the server.
    puts  $s [list "PUT:" $fileTailRemote $totBytes $optList]
    flush $s

    # In order for the server to read a complete line, binary mode
    # must wait until the line oriented part is completed.
    # Use extension to hint transfer mode for *socket write*.
    if {[lsearch $supExts(bin) $fext] >= 0}  {
        fconfigure $s -translation {binary binary}
    } elseif {[lsearch $supExts(txt) $fext] >= 0}  {
        fconfigure $s -translation auto
    }
}

proc PutFileWaitingForResponse { ip orig s totBytes fileTailRemote }  {
    global  statMess debugLevel chunkSize putCode putCode2Msg ipNum2Name
    
    # Get server response.
    set resp [read $s 1]
    set statMess "Client at $ipNum2Name($ip) responded."
    update idletasks
    if {$debugLevel >= 2}  {
	puts "PutFileWaitingForResponse:: just read server response"
	puts "statMess=$statMess, clock=[clock second]"
    }
    if {[eof $s]}  {
	# Shutdown!
	PutFileShutDown $orig $s
	return
    }
    # Catch problems.
    if {$resp != $putCode(ACK)}  {

	# Close temporary socket and the source file; get on with next ip.
	PutFileShutDown $orig $s
	
	# Start handling the "known" error codes.
	if {$resp == $putCode(NAK)}  {
	    if {$debugLevel >= 2}  {
		puts "PutFileWaitingForResponse:: $putCode2Msg(NAK)"
	    }
	    set statMess  \
	      "Client at $ipNum2Name($ip) couldn't accept file $fileTailRemote"
	    update idletasks
	} elseif {$resp == $putCode(FIL)}  {
	    if {$debugLevel >= 2}  {
		puts "PutFileWaitingForResponse:: $putCode2Msg(FIL)"
	    }
	    set statMess  \
	      "Client at $ipNum2Name($ip) already has file $fileTailRemote"
	    update idletasks
	} elseif {$resp == $putCode(EXT)}  {
	    if {$debugLevel >= 2}  {
		puts "PutFileWaitingForResponse:: $putCode2Msg(EXT)"
	    }
	    set statMess  \
	      "Client at $ipNum2Name($ip) does not support file type\
	      $fileTailRemote"
	    update idletasks
	    
	    # Unknown error code.
	} elseif {$resp != $putCode(ACK)}  {
	    if {$debugLevel >= 2}  {
		puts "PutFile:: Client didn't respond correctly"
	    }
	    set statMess "Client at $ipNum2Name($ip) didn't respond correctly"
	    update idletasks
	}
	return
    }
    # Do the actual transfer. fcopy registers 'PutFileCallback'.
    if {$debugLevel >= 2}  {
	puts "PutFileWaitingForResponse:: start transfer"
    }
    # Be sure to switch off any fileevent before fcopy. BUG SEARCH!!!!
    fileevent $s readable {}
    fileevent $s writable {}
    # BUG TRACKING!!!!
    
    # Schedule timeout event.
    PutFileScheduleKiller $orig $s $ip "$fileTailRemote"

    fcopy $orig $s -command   \
      [list PutFileCallback $ip $orig $s $fileTailRemote $totBytes 0]  \
      -size $chunkSize
}

#   PutFileCallback ---
#
#   Callback for file copying from file to network socket.

proc PutFileCallback { ip orig s name totBytes sumBytes bytes {error {}} }  {
    global  chunkSize ipNum2Name statMess debugLevel
    
    incr sumBytes $bytes
    set percent [format "%3.0f" [expr 100 - 100*$sumBytes/($totBytes + 1.0)]]
    set statMess "Putting file $name to $ipNum2Name($ip) (${percent}% left)"
    update idletasks
    if {$debugLevel >= 2}  {
	puts "PutFileCallback:: (entry) error=$error, name=$name, bytes=$bytes, err=$error"
	puts "statMess=$statMess, clock=[clock second]"
    }
    # Catch an error if the socket is already closed. Buggy!!!!!
    if {[catch {eof $s}]}  {
	return
    }
    # If error or file end.
    if {[string length $error] != 0} {
	if {$debugLevel >= 2}  {
	    puts "PutFileCallback:: Error!, err=$error"
	}
	PutFileShutDown $orig $s
    } elseif {[eof $orig]} {
	if {$debugLevel >= 2}  {
	    puts "PutFileCallback:: end transfer eof on disk file, err=$error"
	}
	PutFileShutDown $orig $s
    } elseif {[eof $s]} {
	if {$debugLevel >= 2}  {
	    puts "PutFileCallback:: end transfer eof on socket, err=$error"
	}
	PutFileShutDown $orig $s
	set statMess "Network error when putting file $name on $ipNum2Name($ip)."
    } else {
	# Schedule timeout event.
	PutFileScheduleKiller $orig $s $ip "$name"
	# Rebind copy callback.
	fcopy $orig $s -command   \
	  [list PutFileCallback $ip $orig $s $name $totBytes $sumBytes]  \
	  -size $chunkSize
	if {$debugLevel >= 2}  {
	    puts "PutFileCallback:: just after new fcopy callback"
	}
    }
}

#   PutFileScheduleKiller, PutFileKill, PutFileShutDown ---
#
#   Utility routines to handle timeout events on get file operations.

proc PutFileScheduleKiller { orig s ip fileName }  {
    global  putFileKillerId prefs
    
    if {[info exists putFileKillerId($s)]}  {
	after cancel $putFileKillerId($s)
    }
    set putFileKillerId($s) [after [expr 1000*$prefs(timeout)]   \
      [list PutFileKill $orig $s $ip $fileName]]
}

proc PutFileKill { orig s ip fileName }  {
    global  putFileKillerId prefs ipNum2Name
    
    if {![info exists putFileKillerId($s)]}  {
	return
    }
    tk_messageBox -message   \
      "Timout when waiting for data for file $fileName from $ipNum2Name($ip)" \
      -icon error -type ok
    PutFileShutDown $orig $s
}

proc PutFileShutDown { orig s }  {
    global  statMess debugLevel putFileKillerId recordPutFile

    if {$debugLevel >= 2}  {
	puts "PutFileShutDown::"
    }
    #  Clean.
    catch {unset recordPutFile($s)}
    if {[info exists putFileKillerId($s)]}  {
	after cancel $putFileKillerId($s)
    }
    catch {unset putFileKillerId($s)}
    set statMess {}
    catch {close $orig}
    catch {close $s}
}

#   recordPutFile(sock): {fileId filePath}; just to keep track of things.
#   recordGetFile(sock): {fileId filePath}; just to keep track of things.

proc CancelAllPutGetPendingOpen  {  }  {
    global  recordPutFile recordGetFile openConnectionKillerId debugLevel
    
    if {$debugLevel >= 2}  {
	puts "CancelAllPutGetPendingOpen::"
	catch {parray recordPutFile}
	catch {parray recordGetFile}
    }
    # Close and clean up. Make temporary records since they are unset in
    # the shutdown routines.
    # The Put part:
    if {[info exists recordPutFile]}  {
	array set tmpRecordPutFile [array get recordPutFile]
	foreach s [array names tmpRecordPutFile]  {
	    PutFileShutDown [lindex $tmpRecordPutFile($s) 0] $s
	}
    }
    # The Get part (reversed order!):
    if {[info exists recordGetFile]}  {
	array set tmpRecordGetFile [array get recordGetFile]
	foreach s [array names tmpRecordGetFile]  {
	    GetFileShutDown $s [lindex $tmpRecordGetFile($s) 0]
	}
	# Perhaps the files themselves should also be deleted?
	foreach s [array names tmpRecordGetFile]  {
	    file delete [lindex $tmpRecordGetFile($s) 1]
	}
    }    
    # Pending Open connection:
    if {[info exists openConnectionKillerId]}  {
	foreach s [array names openConnectionKillerId]  {
	    # Be sure to cancel any timeout events first.
	    after cancel $openConnectionKillerId($s)
	    # Then close socket.
	    catch {close $s}
	}
    }
}

#   FontHtmlSizeToPointSize ---
#
#   Change the -font option in a canvas command from html size type
#   to point size.

proc FontHtmlSizeToPointSize { canCmd }  {
    global  fontSize2Points
    
    set ind [lsearch -exact $canCmd "-font"]
    if {$ind >= 2}  {
	set fontSpec [lindex $canCmd [expr $ind +1]]
	set htmlFontSize [lindex $fontSpec 1]
	# Check that it is between 1 and 6.
	if {$htmlFontSize >= 1 && $htmlFontSize <= 6}  {
	    set newFontSpec   \
	      [lreplace $fontSpec 1 1 $fontSize2Points($htmlFontSize)]
	    # Replace font specification in drawing command.
	    set canCmd [lreplace $canCmd [expr $ind + 1]   \
	      [expr $ind + 1] $newFontSpec]
	}
    }
    return $canCmd
}

#   FullFilePathToRelative ---
#
#   Trims the 'basePath' from 'fullPath', and makes it a relative path.
#   This could perhaps be generalized.

proc FullFilePathToRelative { fullPath basePath }  {
    global  tcl_platform
    
    #puts "FullFilePathToRelative:: fullPath=$fullPath, basePath=$basePath"
    # Is already relative path?
    if {[file pathtype $fullPath] == "relative"}  {
	return $fullPath
    }
    set any_ {.*}
    if {![regexp "${basePath}($any_)" $fullPath match relPath]}  {
	puts stderr "FullFilePathToRelative:: path problems writing image."
	return $fullPath
    }
    #puts "match=$match, relPath=$relPath"
    
    # Be sure to make it a relative path (prepend ':', trim '/' etc.).
    if {[string compare [file pathtype $relPath] "absolute"] == 0}  {
	if {$tcl_platform(platform) == "macintosh"}  {
	    set relPath ":$relPath"
	} elseif {$tcl_platform(platform) == "unix"}  {
	    set relPath [string trimleft $relPath /]
	} elseif {$tcl_platform(platform) == "windows"}  {
	    # Don't know. This?
	    set relPath [string trimleft $relPath /]
	}
    }
    return $relPath
}
    
#   FileToCanvas ---
#   
#   Reads line by line from file. Each line contains an almost complete 
#   canvas command except for the widget path. 
#   Lines can also contain 'image create ...' commands.
#   The file must be opened and file id given as 'fileId'.
#   where = "all": write to this canvas and all others.
#   where = "other": write only to remote client canvases.
#   where = ip number: write only to this remote client canvas and not to own.
#   where = "own": write only to this canvas and not to any other.

proc FileToCanvas { w fileId {where all} }  {
    global  itno myItpref allIPnumsTo ipNum2Socket imageItno2FileTail  \
      prefs
    
    # Should file names in file be translated to native path?
    set fileNameToNative 1
    
    #puts "FileToCanvas:: where=$where"
    set pre_ {[^/ ]+}
    set ino_ {[0-9]+}
    set nl_ "\\n"
    
    # New freash 'itno' only if writing to own canvas as well.
    if {$where == "all" || $where == "own"}  {
	set updateItnos 1
    } else  {
	set updateItnos 0
    }
    
    # Read line by line; each line contains an almost complete canvas command.
    # Item prefix and item numbers need to be taken care of.
    while {[gets $fileId line] >= 0}  { 
	
	#puts "FileToCanvas:: line=$line"
	set previousImageOrMovieCmd {}
	
	# Figure out if image create command...or movie command.
	if {([string compare [lindex $line 0] "image"] == 0) ||  \
	  ([string compare [lindex $line 0] "Movie"] == 0)}  {
	    set previousImageOrMovieCmd $line
	    # Get the next line as well.
	    gets $fileId line
	} 
	
	# This must be a canvas command.
	set type [lindex $line 1]
	# This can be a bug if text contains "-tags"???
	set ind [lsearch -exact $line "-tags"]
	if {$ind >= 0}  {
	    set tagInd [expr $ind + 1]
	    set theTags [lindex $line $tagInd]
	} else  {
	    continue
	}
	# The it tags must get new numbers since 'itno' may never be reused.
	# This is only valid when writing on this canvas, see 'updateItnos'.
	# Be sure to replace any other prefix with own prefix; else 
	# a complete mess is achieved.
	if {![regexp "(^| )($pre_)/($ino_)" $theTags match junk prefix oldno]} {
	    puts "FileToCanvas:: Warning, didn't match tags! theTags=$theTags"
	    continue
	}
	if {$updateItnos}  {
	    regsub "(^| )($pre_)/($ino_)" $theTags  " $myItpref/$itno" newTags
	    # Replace tags.
	    #puts "old line=$line"
	    set line [lreplace $line $tagInd $tagInd $newTags]
	    #puts "new line=$line"
	    set oldittag $prefix/$oldno
	    set ittag $myItpref/$itno
	} elseif {!$updateItnos}  {
	    set ittag $prefix/$oldno
	    set oldittag $ittag
	}
	
	# Make newline substitutions.
	set newcmd [subst -nocommands -novariables $line]
	
	# Images and movies. Handle in a different manner.
	if {$type == "image" || $type == "window"}  {
	    
	    #puts "FileToCanvas:: updateItnos=$updateItnos, oldittag=$oldittag, $ittag"
	    # Do not forget to update links to files through 'imageItno2FileTail'.
	    if {$updateItnos && [info exists imageItno2FileTail($oldittag)]}  {
	        set imageItno2FileTail($ittag) $imageItno2FileTail($oldittag)
	    }	    
	    # First, try to localize the original file. If unknown just skip.
	    # Use the image create command on the previous line if exist.
	    # Extract the complete file path.
	    if {($type == "image" || $type == "window") &&   \
	      ([llength $previousImageOrMovieCmd] > 0)}  {
		set ind [lsearch -exact $previousImageOrMovieCmd "-file"]
		if {$ind >= 0}  {
		    set filePath [lindex $previousImageOrMovieCmd [expr $ind + 1]]
		}
		# Translate to native file path? Useful if want to have
		# platform independent canvas file format.
		# Bug if path contains '/' on Mac, or ':' on unix?
		if {$fileNameToNative}  {
		    set filePath [file nativename $filePath]
		}
		#puts "previousImageOrMovieCmd:: ind=$ind, filePath=$filePath"
	    } elseif {[info exists imageItno2FileTail($ittag)]}  {
	        set filePath [GetKnownPathFromTail $imageItno2FileTail($ittag)]
	    } else {
	        puts "FileToCanvas:: couldn't localize image/window $oldittag"
	        continue
	    }
	    # Need to know the coordinates.
	    set x [lindex $line 2]
	    set y [lindex $line 3]
	    DoImportImageOrMovie $w $filePath "$x $y $ittag" $where
	    
	    
	} else  {
	    # Draw ordinary item not image nor window (movie).
	    if {$where == "all" || $where == "own"}  {
	
		# If html font sizes, then translate these to point sizes.
		if {$prefs(useHtmlSizes) && $type == "text"}  {
		    catch {eval $w [FontHtmlSizeToPointSize $newcmd]} id
		} else  {
		    catch {eval $w $newcmd} id
		}
		# Speak...
		if {$prefs(MacSpeech) && $type == "text"}  {
		    speak [$w itemcget $id -text]
		}
	    }
	    # Encode all newlines as \n .
	    regsub -all "\n" $line $nl_ newcmd
	    
	    # Write to all other clients.
	    if {$where == "all" || $where == "other"}  {
		foreach ip $allIPnumsTo  {
		    set s $ipNum2Socket($ip)
		    puts $s "CANVAS: $newcmd"
		}
	    } else  {
		# Write only to specified client with ip number 'where'.
		set s $ipNum2Socket($where)
		puts $s "CANVAS: $newcmd"
	    }
	}
	# Update 'itno' only when also writing to own canvas!
	if {$updateItnos}  {
	    incr itno
	}
    }
}

#   CanvasToFile ---
#   
#   Writes line by line to file. Each line contains an almost complete 
#   canvas command except for the widget path. 
#   The file must be opened and file id given as 'fileId'.
#   'saveImageOrMovieCmds' = 0: writes canvas commands to file except for images
#   or movies.
#   'saveImageOrMovieCmds' = 1: prepends each line with an image or movie canvas
#   command with the create commands. If 'filePathsAbsolute' the -file option
#   contains the full path name, else relative path name to this script path.

proc CanvasToFile { w fileId {saveImageOrMovieCmds 0} }  {
    global  thisPath prefs fontPoints2Size
    
    # When saving images or movies, save relative or absolute path names?
    # It is perhaps best to choose a path relative the actual file path of the file?
    set filePathsAbsolute 0
    set nl_ {\\n}
    set ids [$w find all]
    foreach id $ids {
	
	# Do not save grid or markers.
	set theTags [$w gettags $id]
	if {[lsearch $theTags grid] >= 0 || [lsearch $theTags tbbox] >= 0}  {
	    continue
	}
	# If image and if 'saveImageOrMovieCmds', then save the image creating
	# command before the item create command.
	set theType [$w type $id]
	#puts "theType=$theType"
	if {$saveImageOrMovieCmds && ([string compare $theType "image"] == 0)}  {
	    # Get image name.
	    set imageName [$w itemcget $id -image]
	    # ...and its type; bitmap or photo.
	    set imageType [image type $imageName]
	    set opts [$imageName configure]
	    #puts "opts=$opts"
	    set opcmd {}
	    foreach opt $opts {
		set op [lindex $opt 0]
		set val [lindex $opt 4]
		#puts "op=$op, val=$val"
		# Intercept file path if not 'filePathsAbsolute', and make it 
		# relative instead.
		if {!$filePathsAbsolute && ([string compare $op "-file"] == 0)}  {
		    set val [FullFilePathToRelative $val $thisPath]
		}
		lappend opcmd $op $val
	    }
	    set cmd [concat "image create $imageType $imageName" $opcmd]
	    puts $fileId $cmd    
	} elseif {$saveImageOrMovieCmds && ([string compare $theType "window"] == 0)} {

	    # Save the movie.
	    set windowName [$w itemcget $id -window]
	    set movieName ${windowName}.m
	    set opts [$movieName configure]
	    set opcmd {}
	    foreach opt $opts {
		set op [lindex $opt 0]
		set val [lindex $opt 4]
		# Intercept file path if not 'filePathsAbsolute', and make it 
		# relative instead.
		if {!$filePathsAbsolute && ([string compare $op "-file"] == 0)}  {
		    set val [FullFilePathToRelative $val $thisPath]
		}
		lappend opcmd $op $val
	    }
	    set cmd [concat "Movie $movieName" $opcmd]
	    puts $fileId $cmd    
	}
	# The actual canvas item.
	set opts [$w itemconfigure $id]
	set opcmd {}
	foreach opt $opts {
	    set op [lindex $opt 0]
	    set val [lindex $opt 4]
	    
            # If multine text, encode as one line with explicit "\n".
	    if {[string compare $op "-text"] == 0}  {
	        regsub -all "\n" $val $nl_ oneliner
	        regsub -all "\r" $oneliner $nl_ oneliner
		set val $oneliner
	    }
	    # If using 'useHtmlSizes', translate font  point size to html size.
	    # Needs perhaps some more error checking here.
	    if {$prefs(useHtmlSizes) && ([string compare $op "-font"] == 0)}  {
		set val [lreplace $val 1 1 $fontPoints2Size([lindex $val 1])]
	    }
	    lappend opcmd $op $val
	}
	set co [$w coords $id]
	set cmd [concat "create" $theType $co $opcmd]
	puts $fileId $cmd
    }
}

#   DoOpenCanvasFile ---
#
#   Creates a standard file open dialog, opens the file, and draws to
#   canvas via 'FileToCanvas'. If 'filePath' given, dont show file
#   open dialog.

proc DoOpenCanvasFile { w {filePath {}}}  {
    
    if {[string length $filePath] == 0}  {
	set typelist {
	    {"Canvas" {".can"}}
	    {"Text" {".txt"}}
	}
	set ans [tk_messageBox -message   \
	  "Opening a canvas file erases everything in present canvas!" \
	  -icon warning -type okcancel -default ok]
	if {$ans == "cancel"}  {
	    return
	}
	set ans [tk_getOpenFile -title "Open Canvas" -filetypes $typelist  \
	  -defaultextension ".can"]
	if {$ans == ""}  {
	    return
	}
	set fileName $ans
    } else  {
	set fileName $filePath
    }    
    # Opens the data file.
    if [catch [list open $fileName r] fileId]  {
	tk_messageBox -message   \
	  "Cannot open $fileName for reading." \
	  -icon error -type ok
    }
    DoEraseAll $w     
    FileToCanvas $w $fileId
    close $fileId
}

proc DoSaveCanvasFile { w }  {
    set typelist {
	{"Canvas" {".can"}}
	{"Text" {".txt"}}
    }
    set ans [tk_getSaveFile -title "Save Canvas" -filetypes $typelist  \
	    -defaultextension ".can"]
    if {$ans == ""}  {
	return
    }
    set fileName $ans
    # Opens the data file.
    if {[catch [list open $fileName w] fileId]}  {
	tk_messageBox -message   \
	  "Cannot open $fileName for writing." \
	  -icon error -type ok
	return
    }
    puts "DoSaveCanvasFile:: calling CanvasToFile"
    CanvasToFile $w $fileId 1
    close $fileId
}

proc DoPutCanvasDlg  { w }  {
    
    set ans [tk_messageBox -message   \
      "Warning! Syncing this canvas first erases all client canvases." \
      -icon warning -type okcancel -default ok]
    if {$ans != "ok"}  {
	return
    }
    # Erase all other client canvases.
    DoEraseAll $w "other"

    # Put this canvas to all others.
    DoPutCanvas $w all
}
    
#   DoPutCanvas ---
#   
#   Synchronizes, or puts, this canvas to all others. 
#   It uses a temporary file. Images don't work automatically.
#   If 'toIPnum' then put canvas 'w' only to that ip number.

proc DoPutCanvas { w {toIPnum all} }  {

    set tmpFile ".tmp943621.can"

    # Save canvas to temporary file.
    if [catch [list open $tmpFile w] fileId]  {
	tk_messageBox -message   \
	  "Cannot open $tmpFile for writing." \
	  -icon error -type ok
    }
    CanvasToFile $w $fileId
    catch {close $fileId}

    if [catch [list open $tmpFile r] fileId]  {
	tk_messageBox -message   \
	  "Cannot open $tmpFile for reading." \
	  -icon error -type ok
    }
    # Distribute to all other client canvases.
    if {$toIPnum == "all"}  {
	FileToCanvas $w $fileId "other"
    } else  {
	FileToCanvas $w $fileId $toIPnum
    }
    catch {close $fileId}

    # Finally delete the temporary file.
    file delete $tmpFile
}

#   DoGetCanvas ---
#
#   Fetches the content of the canvas of a specified client.

proc DoGetCanvas { w }  {
    global  debugLevel ipNum2Name ipName2Num ipNum2Socket
    
    # The dialog to select remote client.
    set getCanIPNum [mkGetCanvas .getcan]
    if {$debugLevel >= 2}  {
	puts "DoGetCanvas:: getCanIPNum=$getCanIPNum"
    }
    if {$getCanIPNum == ""}  {
	return
    }    
    # Erase everything in own canvas.
    DoEraseAll $w "own"
    
    # GET CANVAS.
    catch {puts $ipNum2Socket($getCanIPNum) "GET CANVAS:"}
}

#   FindWidgetGeometryAtLaunch ---
#
#   Just after launch, find and set various geometries of the application.
#   'hRoot' excludes the menu height, 'hTot' includes it.
#   Note: [winfo height .#menu] gives the menu height when the menu is in the
#   root window; [wm geometry .] gives and sets dimensions without the menu;
#   [wm minsize .] gives and sets dimensions with the menu included.

proc FindWidgetGeometryAtLaunch {  }  {
    global  dims wCan commFrame wStatMess tcl_platform
    
    # The actual dimensions.
    set dims(wRoot) [winfo width .]
    set dims(hRoot) [winfo height .]
    set dims(hStatus) [winfo height .fcomm.st]
    set dims(hComm) [winfo height $commFrame]
    set dims(wStatMess) [winfo width $wStatMess]
    if {$tcl_platform(platform) != "macintosh"}  {
	set dims(hMenu) [winfo height .#menu]
    } else  {
	set dims(hMenu) 0
    }
    set dims(wCanvas) [winfo width $wCan]
    set dims(hCanvas) [winfo height $wCan]
    set dims(wTot) $dims(wRoot)
    set dims(hTot) [expr $dims(hRoot) + $dims(hMenu)]
    # Position of root window.
    set dimList [ParseWMGeometry .]
    set dims(x) [lindex $dimList 2]  
    set dims(y) [lindex $dimList 3]  

    # The minimum dimensions.
    set dims(wMinCanvas) [expr $dims(wCanOri) + 2]
    set dims(hMinCanvas) [expr $dims(hCanOri) + 2]
    set dims(wMinRoot) [expr $dims(wMinCanvas) + 56]
    set dims(hMinRoot) [expr $dims(hMinCanvas) + $dims(hStatus) + $dims(hComm)]
    set dims(wMinTot) $dims(wMinRoot)
    set dims(hMinTot) [expr $dims(hMinRoot) + $dims(hMenu)]
    
    # The minsize when no connected clients. Is updated when connect/disconnect.
    wm minsize . $dims(wMinTot) $dims(hMinTot)
}

proc ParseWMGeometry { w }  {
    set int_ {[0-9]+}
    set sign_ {\+|\-}
    regexp "(${int_})x(${int_})($sign_)(${int_})($sign_)(${int_})"   \
      [wm geometry $w] match wid hei junk1 x junk2 y
    return [list $wid $hei $x $y]
}

#   CanvasSizeChange ---
#   
#   If size change in canvas (application), then let other clients know.
#   But only if 'propagateToClients'.

proc CanvasSizeChange { propagateToClients {force 0} }  {
    global  allIPnumsTo ipNum2Socket debugLevel wCan dims wStatMess
    
    # Get new sizes.
    update idletasks
    
    # Sizes without any menu.
    set w [winfo width .]
    set h [winfo height .]
    set wCanvas [winfo width $wCan]
    set hCanvas [winfo height $wCan]
    
    if {$debugLevel >= 2}  {
	puts "CanvasSizeChange:: w=$w, h=$h, wCanvas=$wCanvas, hCanvas=$hCanvas"
	puts "CanvasSizeChange:: force=$force, wm geometry .=[wm geometry .]"
    }
    # Only if size changed or if force.
    if {$propagateToClients}  {
	if {($dims(wCanvas) != $wCanvas) || ($dims(hCanvas) != $hCanvas)   \
	  || $force }  {
	    #puts "CanvasSizeChange:: propagate"
	    foreach ip $allIPnumsTo  {
		set s $ipNum2Socket($ip)
		puts $s "RESIZE: $wCanvas $hCanvas"
	    }
	}
    }
    # Update actual size values. 'Root' no menu, 'Tot' with menu.
    set dims(wStatMess) [winfo width $wStatMess]
    set dims(wRoot) $w
    set dims(hRoot) $h
    set dims(wTot) $dims(wRoot)
    set dims(hTot) [expr $dims(hRoot) + $dims(hMenu)]
    set dims(wCanvas) $wCanvas
    set dims(hCanvas) $hCanvas
}

proc SetCanvasSize { cw ch }  {
    global  dims
        
    # Compute new root size from the desired canvas size.
    set hRootFinal [expr $ch + $dims(hStatus) + $dims(hComm)]
    set wRootFinal [expr $cw + 56]
    #puts "SetCanvasSize:: cw=$cw, ch=$ch, hRootFinal=$hRootFinal, wRootFinal=$wRootFinal"
    wm geometry . ${wRootFinal}x${hRootFinal}
}

#   GetTransferRateFromTiming ---
#
#   From 'listClicksBytes' which is a list of {clockClicks bytes}, the transfer
#   rate is computed as bytes per seconds.
#   A moving average of the latest elements is used.

proc GetTransferRateFromTiming  { listClicksBytes }  {
    global  clockClicksPerSec debugLevel
    
  if {$debugLevel >= 2}  {
      #puts "GetTransferRateFromTiming:: entry"
  }
    set n [llength $listClicksBytes]
    set nAve 3
    set istart [expr $n - $nAve]
    if {$istart < 0}  {
	set istart 0
    }
    set iend [expr $n - 1]
    set sumBytes [expr [lindex [lindex $listClicksBytes $iend] 1] -  \
      [lindex [lindex $listClicksBytes $istart] 1]]
    set sumClicks [expr [lindex [lindex $listClicksBytes $iend] 0] -  \
      [lindex [lindex $listClicksBytes $istart] 0]]
    # Treat the case with wrap around. (Guess)
    if {$sumClicks <= 0}  {
	set sumClicks $clockClicksPerSec
    }
    set aveBytesPerClick [expr $sumBytes / ($sumClicks + 1.0)]
    set aveBytesPerSec [expr $aveBytesPerClick * $clockClicksPerSec]
    return $aveBytesPerSec
}

#   BytesPerSecFormatted ---
#
#   Returns the transfer rate as a nicely formatted text string.

proc BytesPerSecFormatted  { bytesPerSec }  {

    # Find format: bytes or k.
    if {$bytesPerSec < 1000}  {
	set txtRate "$bytesPerSec bytes/sec"
    } elseif {$bytesPerSec < 1000000}  {
	set txtRate [list [format "%.1f" [expr $bytesPerSec/1000.0] ]Kb/sec]
    } else {
	set txtRate [list [format "%.1f" [expr $bytesPerSec/1000000.0] ]Mb/sec]
    }
    return $txtRate
}

#   AddToKnownFiles, GetKnownPathFromTail ---
#
#   Keeps track of already opened or received images/movies files
#   through the synced lists 'knownFiles' and 'knownPaths'.
#   The 'fileTail' name is always the native file name which on the mac
#   my lack an extension.

proc AddToKnownFiles { fileTail filePath it }  {
    global  knownFiles knownPaths imageItno2FileTail
 
    # Check first if its not already there.
    if {[GetKnownPathFromTail $fileTail] == ""}  {
	lappend knownFiles "$fileTail"
	lappend knownPaths "$filePath"
    }
    set imageItno2FileTail($it) $fileTail
}
		
proc GetKnownPathFromTail { fileTail }  {
    global  knownFiles knownPaths tcl_platform
   
    set dot_ {\.}
    set ind [lsearch -exact $knownFiles $fileTail]
    # On mac it is only necessary that the rootnames agree.
    if {$tcl_platform(platform)  == "macintosh"}  {
	set fileRoot [file rootname $fileTail]
	set ind [lsearch -regexp $knownFiles "^${fileRoot}$dot_*|^${fileRoot}$"]	
    }
    # Return nothing if its not there.
    if {$ind < 0}  {
	return ""
    } else  {
	set path [lindex $knownPaths $ind]
	# Check if the file exists.
	if {[file exists $path]}  {
	    return $path
	} else  {
	    return ""
	}
    }
}

#   IsFileInCacheQ ---
#
#   Checks if file with tail of file name 'fileTail' exist in the incoming
#   directory taking into account the constraints of 'prefs(checkCache)'.

proc IsFileInCacheQ { fileTail }  {
    global  prefs thisPath
    
    if {$prefs(checkCache) == "never"}  {
	return 0
    }
    set fullName [file join $thisPath incoming $fileTail]

    set ans [file exists $fullName]
    if {$ans == 1}  {
	set ans [file isfile $fullName]
    } else  {
	return 0
    }
    # At this stage we are sure that the file is actually there.
    if {$prefs(checkCache) == "always"}  {
	return 1
    }
    # Check if the time limit has passed.
    if {![FileOlderThan $fullName $prefs(checkCache)]}  {
	return 1
    } else  {
	return 0
    }
}

#   FileOlderThan ---
#
#   Find out if file older than 'timespan'.
#   'timespan' can be: "launch", "min", "hour", "day", "30days".

proc FileOlderThan { filePath timespan }  {
    global  tmsec launchSecs
    
    if {[lsearch -exact {"launch" "min" "hour" "day" "30days"} $timespan] < 0}  {
	return 1
    }
    set fileTime [file mtime $filePath]
    set thisTime [clock seconds]
    set ans 1
    if {$timespan == "launch"}  {
	if {$fileTime > $launchSecs}  {
	    set ans 0
	}
    } else  {
	if {[expr $thisTime - $fileTime] < $tmsec($timespan)}  {
	    set ans 0
	}
    }
}

#   AppGetFocus ---
#
#   Check clipboard and activate corresponding menus.    

proc AppGetFocus {  }  {
    
    #puts "AppGetFocus::"
    # Check the clipboard or selection.
    if {[catch {selection get -selection CLIPBOARD} sel]}  {
	return
    } elseif {[string length $sel] > 0}  {
	.menu.edit entryconfigure *Paste* -state normal
   }
}

proc DoQuit  {  }  {
    global  dims
    
    # Before quitting, save user preferences. 
    # Need to collect some of them first.
    # Position of root window.
    set dimList [ParseWMGeometry .]
    set dims(x) [lindex $dimList 2]
    set dims(y) [lindex $dimList 3]
    PreferencesSaveToFile
    exit
}
    
# This belongs to the server part, but is necessary for autoStartServer.

proc DoStartServer  { thisServPort }  {
    if {[catch {socket -server SetupChannel $thisServPort} msg]}  {
	tk_messageBox -message   \
	  "Couldn't start server socket: $msg." \
	  -icon error -type ok
    }	
}

#--- The Menus ------------------------------------------------------------------

menu .menu -tearoff 0
#--- File ------------------------------------------------------------
set m [menu .menu.file -tearoff 0]
.menu add cascade -label "File " -menu $m
if {!$autoStartServer}  {
    $m add command -label "Start Server"	\
      -command "DoStartServer $thisServPort" 
}
$m add command -label "Open Connection..."	\
  -command DoOpenConnection -accelerator $osprefs(mod)+O
$m add separator
$m add command -label "Open Image/Movie..." -command "ImportImageOrMovieDlg $wCan" \
  -accelerator $osprefs(mod)+I
if {$prefs(QT) && $prefs(http)}  {
    $m add command -label "Open URL Stream..." -command {mkOpenMulticast}
}
$m add separator
$m add command -label "Put Canvas" -command "DoPutCanvasDlg $wCan"  \
  -state disabled
$m add command -label "Get Canvas..." -command "DoGetCanvas $wCan"  \
  -state disabled
$m add command -label "Put File..." -command "PutFileDlg" -state disabled
$m add command -label "Stop Put/Get/Open" -command "CancelAllPutGetPendingOpen"   \
  -state normal
$m add separator
$m add command -label "Open Canvas..." -command "DoOpenCanvasFile $wCan"
$m add command -label "Save Canvas..." -command "DoSaveCanvasFile $wCan"  \
  -accelerator $osprefs(mod)+S
$m add command -label "Save Postscript..." -command "SavePostscript $wCan"  \
  -accelerator $osprefs(mod)+P
$m add command -label "Quit" -command DoQuit -accelerator $osprefs(mod)+Q
#--- Edit ------------------------------------------------------------
set m [menu .menu.edit -tearoff 0]
.menu add cascade -label "Edit " -menu $m
$m add command -label "All" -command "SelectAll $wCan"  \
	-accelerator $osprefs(mod)+A
$m add command -label "Erase All" -command "DoEraseAll $wCan"
$m add separator
$m add command -label "Cut" -command "event generate $wCan <<Cut>>"  \
	-accelerator $osprefs(mod)+X -state disabled
$m add command -label "Copy" -command "event generate $wCan <<Copy>>"	\
	-accelerator $osprefs(mod)+C -state disabled
$m add command -label "Paste" -command "event generate $wCan <<Paste>>"  \
	-accelerator $osprefs(mod)+V -state disabled
$m add separator
$m add command -label "Raise" -command "RaiseOrLowerItems $wCan raise"  \
  -accelerator $osprefs(mod)+R -state disabled
$m add command -label "Lower" -command "RaiseOrLowerItems $wCan lower"  \
  -accelerator $osprefs(mod)+L -state disabled
$m add separator
$m add command -label "Image Larger" -command "ResizeImage $wCan 2 sel auto"   \
  -accelerator $osprefs(mod)+>
$m add command -label "Image Smaller" -command "ResizeImage $wCan -2 sel auto"   \
  -accelerator $osprefs(mod)+<
#--- Prefs ------------------------------------------------------------
set m [menu .menu.prefs -tearoff 0]
.menu add cascade -label "Preferences " -menu $m
$m add command -label "Background Color..." -command "SetCanvasBgColor $wCan"
$m add checkbutton -label "Grid" -variable prefs(canGridOn)  \
  -onvalue 1 -offvalue 0
set mt [menu .menu.prefs.thick -tearoff 0]
$m add cascade -label "Thickness" -menu .menu.prefs.thick
$mt add radio -label 1 -variable prefs(penThick)
$mt add radio -label 2 -variable prefs(penThick)
$mt add radio -label 4 -variable prefs(penThick)
$mt add radio -label 6 -variable prefs(penThick)
set mt [menu .menu.prefs.brthick -tearoff 0]
$m add cascade -label "Brush Thickness" -menu .menu.prefs.brthick
$mt add radio -label 8 -variable prefs(brushThick)
$mt add radio -label 10 -variable prefs(brushThick)
$mt add radio -label 12 -variable prefs(brushThick)
$mt add radio -label 16 -variable prefs(brushThick)
$m add checkbutton -label "Fill" -variable prefs(fill)   
set mt [menu .menu.prefs.smooth -tearoff 0]
$m add cascade -label "Line Smoothness" -menu .menu.prefs.smooth
$mt add radio -label None -value 0 -variable prefs(splinesteps)  \
  -command {set prefs(smooth) 0}
$mt add radio -label 2 -value 2 -variable prefs(splinesteps)  \
  -command {set prefs(smooth) 1}
$mt add radio -label 4 -value 4 -variable prefs(splinesteps)  \
  -command {set prefs(smooth) 1}
$mt add radio -label 6 -value 6 -variable prefs(splinesteps)  \
  -command {set prefs(smooth) 1}
$mt add radio -label 8 -value 8 -variable prefs(splinesteps)  \
  -command {set prefs(smooth) 1}
$mt add radio -label 10 -value 10 -variable prefs(splinesteps)  \
  -command {set prefs(smooth) 1}
$m add sep
set mt [menu .menu.prefs.font -tearoff 0]
$m add cascade -label "Font" -menu $mt
$mt add radio -label "Times" -variable prefs(font)
$mt add radio -label "Helvetica" -variable prefs(font)
$mt add radio -label "Courier" -variable prefs(font)
set mt [menu .menu.prefs.size -tearoff 0]
$m add cascade -label "Size" -menu $mt
$mt add radio -label "1" -variable prefs(fontSize)
$mt add radio -label "2" -variable prefs(fontSize)
$mt add radio -label "3" -variable prefs(fontSize)
$mt add radio -label "4" -variable prefs(fontSize)
$mt add radio -label "5" -variable prefs(fontSize)
$mt add radio -label "6" -variable prefs(fontSize)
set mt [menu .menu.prefs.weight -tearoff 0]
$m add cascade -label "Weight" -menu $mt
$mt add radio -label "Normal" -variable prefs(fontWeight) -value normal
$mt add radio -label "Bold" -variable prefs(fontWeight) -value bold
$m add sep
if {$prefs(MacSpeech)}  {
    $m add checkbutton -label "Speech" -variable prefs(MacSpeech)
}
$m add checkbutton -label "Auto Connect" -variable prefs(autoConnect)  \
  -command DoAutoConnect
$m add checkbutton -label "Multi Connect" -variable prefs(multiConnect)
$m add checkbutton -label "Privacy" -variable prefs(privacy) -onvalue 1 -offvalue 0
set mt [menu .menu.prefs.cache -tearoff 0]
$m add cascade -label "Check Cache" -menu $mt
$mt add radio -label "Always" -value "always" -variable prefs(checkCache)
$mt add radio -label "At Launch" -value "launch" -variable prefs(checkCache)
$mt add radio -label "> 1 Hour" -value "hour" -variable prefs(checkCache)
$mt add radio -label "> 1 Day" -value "day" -variable prefs(checkCache)
$mt add radio -label "> 1 Week" -value "week" -variable prefs(checkCache)
$mt add radio -label "Never" -value "never" -variable prefs(checkCache)
$m add command -label "Edit Shortcuts..."   \
  -command "mkEditShortcuts .tshcts shortcuts"
$m add sep
set mt [menu .menu.prefs.revert -tearoff 0]
$m add cascade -label "Revert To" -menu $mt
$mt add command -label "User Defaults" -command PreferencesResetToUserDefaults
$mt add command -label "Application Defaults"   \
  -command PreferencesResetToAppDefaults
#--- Info ------------------------------------------------------------
set m [menu .menu.info -tearoff 0]
.menu add cascade -label "Info " -menu $m
$m add command -label "On Server..."  \
	-command {mkShowInfoServer $thisIPnum} -state normal
$m add command -label "On Clients..."   \
	-command {mkShowInfoClients $allIPnumsTo} -state disabled
#---------------------------------------------------------------------

. configure -menu .menu

# Make the button pad.
pack [frame .fmain] -side top -fill both -expand true
pack [frame .fmain.frleft] -side left -fill y
pack [frame $btFrame] -side top
pack [label .fmain.frleft.pad -relief raised -borderwidth 1]   \
  -fill both -expand true
# The 'Coccinella'.
if {$prefs(coccinellaMovie)}  {
    pack [Movie .fmain.frleft.padphoto -controller 0   \
      -file [file join images beetle<->igelpiga.mov]]   \
      -in .fmain.frleft.pad -side bottom
} else  {
    image create photo igelpiga -format gif -file [file join images igelpiga.gif]
    #image create photo igelpiga -format gif -file [file join images beetle2ind.gif]
    pack [label .fmain.frleft.padphoto -borderwidth 0 -image igelpiga]   \
      -in .fmain.frleft.pad -side bottom
}
  
#CreateAllButtons

canvas $wCan -height $dims(hCanOri) -width $dims(wCanOri) -relief raised -bd 1	\
  -highlightthickness 0 -background $prefs(bgColCanvas)
pack $wCan -fill both -expand true -side right

CreateAllButtons

ClickToolButton $wCan $btNo2Name($prefs(btState))

# Make the connection frame.
#puts "Make the connection frame."
image create photo im_handle -format gif -file [file join images handtag2.gif]
pack [frame .fcomm] -side top -fill x

# Status message part.
#puts "Status message part."
pack [frame .fcomm.st -relief raised -borderwidth 1]  \
  -side top -fill x -pady 0
pack [frame .fcomm.stat -relief groove -bd 2]  \
  -side top -fill x -padx 10 -pady 2 -in .fcomm.st
image create photo im_wave -format gif -file [file join images wave3.gif]
pack [canvas $wStatMess -bd 0 -highlightthickness 0 -height 14]   \
  -side left -pady 1 -padx 6 -fill x -expand true
$wStatMess create text 0 0 -anchor nw -text $statMess -font $sysFont(s)  \
  -tags stattxt

# The labels in comm frame.
#puts "The labels in comm frame."
pack [frame $commFrame  -relief raised -borderwidth 1] -side left
label $commFrame.comm -text "  Remote address:" -width 22 -anchor w
label $commFrame.user -text "  User:" -width 14 -anchor w
label $commFrame.to -text "   To     "
label $commFrame.from -text "From "
grid $commFrame.comm $commFrame.user $commFrame.to $commFrame.from -sticky w
pack [frame .fcomm.pad -relief raised -borderwidth 1] -side top   \
  -fill both -expand true
pack [label .fcomm.pad.hand -relief flat -borderwidth 0 -image im_handle]  \
  -side right -anchor sw

# Trigger changes when certain variables are changed or set.
#puts "Trigger changes when certain variables are changed or set."
trace variable statMess w TraceStatusMessage
trace variable prefs w TracePreferences

# Add things that are defined in the prefs file and not updated else.
#puts "Add things that are defined in the prefs file and not updated else."
DoCanvasGrid

# Setting the window position never hurts.
#puts "Setting the window position never hurts."
wm geometry . +$dims(x)+$dims(y)

# Setting total (root) size however, should only be done if set in pref file!
if {$dims(wRoot) > 1 && $dims(hRoot) > 1}  {
    wm geometry . $dims(wRoot)x$dims(hRoot)
}

# Add virtual events.
#puts "Add virtual events."
event add <<Cut>> <$osprefs(mod)-Key-x>
event add <<Copy>> <$osprefs(mod)-Key-c>
event add <<Paste>> <$osprefs(mod)-Key-v>
bind $wCan <<Cut>> "CopySelectedToClipboard $wCan cut"
bind $wCan <<Copy>> "CopySelectedToClipboard $wCan copy"
bind $wCan <<Paste>> "PasteFromClipboardToCanvas $wCan"

# bindings on all canvas items
#$wCan bind all <BackSpace> "DeleteItem $wCan %x %y selected"
#$wCan bind all <BackSpace> "puts "BackSpace:: %W %x %y selected"
#bind $wCan <BackSpace> "DeleteItem $wCan %x %y selected"

# Various bindings.
#puts "Various bindings."
bind . <BackSpace> "DeleteItem $wCan %x %y selected"
bind . <Control-Key-d> "DeleteItem $wCan %x %y selected"
# This should be used for showing, for instance, text tracks of mp3.
#bind QTFrame <Enter> {puts "QTFrame <Enter>, text tracks of mp3"}

# Start the server. It was necessary to have an 'update idletasks' command here
# because when starting the script directly, and not from within wish, somehow
# there was a timing problem in 'DoStartServer'.
#puts "Start the server."
update idletasks
if {$autoStartServer}  {
    DoStartServer $thisServPort
}

#puts "update idletasks 2"
update idletasks
wm title . "Whiteboard"

# A trick to let the window manager be finished before getting the geometry.
#puts "A trick to let the window manager be finished before getting the geometry."
after idle FindWidgetGeometryAtLaunch

bind . <FocusIn> AppGetFocus

# Update size info when application is resized.
bind $wCan <Configure> {CanvasSizeChange 1}

# Is it the first time it is launched, then show the welcome canvas.
if {$prefs(firstLaunch)}  {
    eval [list DoOpenCanvasFile $wCan $prefs(welcomeFile)]
}
set prefs(firstLaunch) 0

#*** The server part ***************************************************
#
#	The canvas drawing instructions are evaluated in a safe interpreter
#	so that no mean commands may be evaluated.
#
# Create the interpreter.
if {$makeSafeServ}  {
    #	interp create -safe serverInterp
    interp create serverInterp
    if {$debugServerLevel >= 3}  {
	puts "issafe: [interp issafe serverInterp]"
    }
    interp eval serverInterp [list set servCan $servCan]
}

#---------------------------------------------------------------------
#   HandleClientRequest ---
#
#   This is the actual server that reads client requests. 
#   The most important is the CANVAS command which is a complete
#   canvas command that is prefixed only by the widget path.
#
proc HandleClientRequest {channel ip port} {
    global  wCan servCan debugLevel tempChannel ipNum2Name  \
      makeSafeServ chunkSize debugServerLevel putCode  \
      myItpref remoteServPort ipNum2User ipNum2Socket  \
      ipNum2ServPort prefs allIPnumsTo dims  \
      supExts thisIPnum thisUserName recordGetFile fontSize2Points
    
    # regexp patterns.
    set wrd_ {[^ ]+}
    set optwrd_ {[^ ]*}
    set optlist_ {.*}
    set any_ {.+}
    set nothing_ {}
    # Matches list with braces.  
    # ($llist_|$wrd_)  :should match single item list or multi item list.
    set llist_ {\{[^\}]+\}}
    set pre_ {[^/ ]+}
    set portwrd_ {[0-9]+}
    set int_ {[0-9]+}
    set signint_ {[-0-9]+}
    set nl_ {\\n}
    
    # If client closes socket.
    if {[eof $channel]} {
	if {$debugServerLevel >= 2}  {
	    puts "HandleClientRequest:: eof channel=$channel"
	}
	# Update entry only for nontemporary channels.
	if {![info exists tempChannel($channel)]}  {
	    if {$prefs(autoDisconnect)}  {
		DoCloseConnection $ip
	    } else  {
		MakeCommEntry $ip -1 0
	    }
	} else {
	    unset tempChannel($channel)
	}
	catch {close $channel}
	
	# Read one line at the time and find out what to do from the
	# leading word.
    } elseif {[gets $channel line] != -1} {
	if {$debugServerLevel >= 2}  {
	    puts "--->$ip:${port}:: $line"
	}
	
	# Regular drawing commands in the canvas.
	if {[regexp "^CANVAS: +(.*)$" $line junk instr]} {
	    
	    # Make newline substitutions.
	    set newinstr [subst -nocommands -novariables $instr]
	    if {$debugServerLevel >= 3}  {
		puts "--->newinstr: $newinstr"
	    }
	    # Intercept the canvas command in certain cases:
	    # If moving a selected item, be sure to move the markers with it.
	    # The item can either be selected by remote client or here.
	    if {[string compare [lindex $newinstr 0] "move"] == 0}  {
		set theItno [lindex $newinstr 1]
		set id [$servCan find withtag $theItno]
		set idsMarker [$servCan find withtag id$id]
		if {[string length $idsMarker] > 0}  {
		    set dx [lindex $newinstr 2]
		    set dy [lindex $newinstr 3]
		    foreach id $idsMarker {
			$servCan move $id $dx $dy
		    }
		}
	    }
	    # If html sizes in text items, be sure to translate them into
	    # platform specific point sizes.
	    if {$prefs(useHtmlSizes)}  {
		if {([string compare [lindex $newinstr 0] "create"] == 0)  &&  \
		  ([string compare [lindex $newinstr 1] "text"] == 0)}  {
		    set newinstr [FontHtmlSizeToPointSize $newinstr]
		}
	    }
	    if {!$makeSafeServ}  {
		if {[catch {eval $servCan $newinstr} id]}  {
		    puts "--->error: did not understand: $id"
		}
	    } else {
		#  Safe interpreter.
		interp eval serverInterp [list set newinstr $newinstr]
		serverInterp eval {puts "serverInterp:: newinstr=$newinstr"}
		# The problem seems to be the canvas path is unknown to the slave.
		if {[catch {interp eval serverInterp $servCan $newinstr} id]}  {
		    puts "--->error: did not understand: $id"
		}
	    }
	    # If text then speak up to last punct.
	    set type [$servCan type $id]
	    if {$prefs(MacSpeech) && $type == "text"}  {
		speak [$w itemcget $id -text]
	    }
	    
	    # A client tells which server port number it has, its item prefix
	    # and its user name.
	} elseif {[regexp "^IDENTITY: +($portwrd_) +($pre_) +($llist_|$wrd_)$" \
	  $line junk remPort id user]}  {
	    if {$debugServerLevel >= 2 }  {
		puts "HandleClientRequest:: IDENTITY: remPort=$remPort, \
		  id=$id, user=$user"
	    }
	    # Save port in array.
	    set ipNum2ServPort($ip) $remPort
	    # If user is a list remove braces.
	    set ipNum2User($ip) [lindex $user 0]
	    # Other stuff to save?

	    # Check that not own ip and user.
	    if {$ip == $thisIPnum &&   \
	      [string compare [string tolower $user]  \
	      [string tolower $thisUserName]] == 0}  {
		tk_messageBox -message   \
		  "A connecting client has chosen an ip number  \
		  and user name identical to your own." \
		  -icon warning -type ok
	    }
	    # If auto connect, then make a connection to the client as well.
	    if {$prefs(autoConnect)  && [lsearch $allIPnumsTo $ip] == -1}  {
		if {$debugServerLevel >= 2}  {
		    puts "HandleClientRequest:: autoConnect:  \
		      name=$ipNum2Name($ip), remPort=$remPort"
		}
		# Handle the complete connection process.
		# Let propagateSizeToClients = false.
		DoConnect $ipNum2Name($ip) $ipNum2ServPort($ip) 0
	    }
	    # Let connecting client now this 'allIPnumsTo' (for multi connect).
	    #puts $ipNum2Socket($ip) "IPS CONNECTED: $allIPnumsTo"
	    # Moved to WhenSocketOpensInits; problem in async open.
	    
	    # A client tells which other ips it is connected to.
	    # 'remListIPandPort' contains: ip1 port1 ip2 port2 ...
	} elseif {[regexp "^IPS CONNECTED: +($any_|$nothing_)$" \
	  $line junk remListIPandPort]}  {
	    if {$debugServerLevel >= 2 }  {
		puts "HandleClientRequest:: IPS CONNECTED:  \
		  remListIPandPort=$remListIPandPort"
	    }
	    # If multi connect then connect to all other 'remAllIPnumsTo'.
	    if {$prefs(multiConnect)}  {
		# Make temporary array that maps ip to port.
		array set arrayIP2Port $remListIPandPort
		foreach ipNum [array names arrayIP2Port] {
		    if {![IsConnectedToQ $ipNum]}  {		
			
			# Handle the complete connection process.
			# Let propagateSizeToClients = false.
			DoConnect $ipNum $arrayIP2Port($ipNum) 0
		    }
		}
	    }
	    
	    # Received resize request.
	} elseif {[regexp "^RESIZE: +($int_) +($int_)$" $line match w h]} {
	    if {$debugServerLevel >= 2 }  {
		puts "HandleClientRequest:: RESIZE: w=$w, h=$h"
	    }
	    # Change total size of application so that w and h is the canvas size.
	    # Be sure to not propagate this size change to other clients.
	    bind $wCan <Configure> {CanvasSizeChange 0}
	    SetCanvasSize $w $h
	    update idletasks
	    bind $wCan <Configure> {CanvasSizeChange 1}
	    
	    # Put file to receive file; handles via temporary socket.
	} elseif {[regexp "^(PUT|GET): +($llist_|$wrd_) +($wrd_) *($optlist_)$" \
	  $line   what cmd fileName totBytes optList]} {
	    if {$cmd == "PUT"}  {
		if {$debugServerLevel >= 2 }  {
		    puts "HandleClientRequest:: PUT: cmd=$cmd, channel=$channel"
		    puts "   fileName=$fileName, totBytes=$totBytes, \
		      optList=$optList"
		}
		# Be sure to strip off any path. (thisPath)???
		set fileTail [file tail $fileName]
		set fext [string tolower [file extension $fileTail]]
		if {![file isdirectory incoming]}  {
		    file mkdir incoming
		}
		set inFile [file join incoming $fileTail]
		#puts "   inFile=$inFile"
		# Strip off one level of lists.
		set optList [lindex $optList 0]
		
		# Mark channel as temporary.
		set tempChannel($channel) 1
		# Disable callback for this channel. Important for fcopy!
		fileevent $channel readable {}
		# Is this really necessary? Shouldn't fcopy do that?
		#fconfigure $channel -blocking 1
		# If filePath == {}, then unknown (check below).
		set filePath [GetKnownPathFromTail $fileTail]

		# Before doing anything more check if the extension is supported.
		if {[lsearch $supExts(all) $fext] < 0}  {
		    puts "Error:: file extension is not supported."
		    puts -nonewline $channel $putCode(EXT)
		    flush $channel
		    catch {close $channel}
		    return
		}
		# Before doing anything more check if the file is cached.
		if {[IsFileInCacheQ $fileTail]}  {
		    #puts "Info:: IsFileInCacheQ==1"
		    puts -nonewline $channel $putCode(FIL)
		    flush $channel
		    catch {close $channel}
		    DoImportImageOrMovie $servCan $inFile $optList "own"
		    return

		    # Before doing anything more check if the file is "known".
		    # This part executes if file not cached but known.
		} elseif {[llength $filePath] > 0}  {
		    #puts "Info:: Known file, filePath=$filePath"
		    puts -nonewline $channel $putCode(FIL)
		    flush $channel
		    catch {close $channel}
		    DoImportImageOrMovie $servCan $filePath $optList "own"
		    return
		}		
		# Check that the destination file opens correctly.
		if {[catch {open $inFile w} dest]}  {
		    puts "Error:: file does not opens correctly."
		    puts -nonewline $channel $putCode(NAK)
		    flush $channel
		    catch {close $channel}
		    tk_messageBox -message   \
		      "Server failed when trying to open $fileName" \
		      -icon error -type ok
		    return
		}
		# Use extension to hint transfer mode.
		if {[lsearch $supExts(bin) $fext] >= 0}  {
		    fconfigure $channel -translation {binary binary}
		    fconfigure $channel -buffering full
		    fconfigure $dest -translation {binary binary}
		    # Be sure to switch off any encoding for binary.
		    if {[info tclversion] >= 8.1}  {
			catch {fconfigure $channel -encoding {}}
		    } 
		} elseif {[lsearch $supExts(txt) $fext] >= 0}  {
		    fconfigure $channel -translation auto
		    fconfigure $dest -translation auto
		    set optList junk
		}
		# Here we answer that it's ok to get on with the file transfer.
		puts -nonewline $channel $putCode(ACK)
		flush $channel
		
		# recordGetFile(sock): {fileId filePath}; 
		# just to keep track of things.
		set recordGetFile($channel) [list $dest $inFile]
		
		# Schedule timeout killer event.
		GetFileScheduleKiller $channel $dest $ip $fileTail
		
		fconfigure $channel -blocking 0
		# Do the actual transfer.
		if {$debugServerLevel >= 2}  {
		    puts "HandleClientRequest::  \
		      start transfer [fconfigure $channel]"
		}
		fcopy $channel $dest -size $chunkSize -command  \
		  [list FileCopyCallback $channel $dest $ip $fileTail  \
		  $totBytes 0 $optList]
		if {$debugServerLevel >= 2}  {
		    puts "HandleClientRequest:: fcopy, registered handler"
		}
		
	    } elseif {$cmd == "GET"}  {
		# Not implemented.
		puts "Warning: GET not implemented"
	    }
	    
	    # The present client requests to put this canvas.
	} elseif {[regexp "^GET CANVAS:$" $line]} {
	    if {$debugServerLevel >= 2}  {
		puts "--->GET CANVAS:"
	    }
	    DoPutCanvas $wCan $ip
	    
	    # Image (photo) resizing.
	} elseif {[regexp "^RESIZE IMAGE: +($wrd_) +($wrd_) +($signint_)$"   \
	  $line match itOrig itNew zoomFactor]}  {
	    if {$debugServerLevel >= 2}  {
		puts "--->RESIZE IMAGE: itOrig=$itOrig, itNew=$itNew, \
		  zoomFactor=$zoomFactor"
	    }
	    ResizeImage $servCan $zoomFactor $itOrig $itNew "own"
	    
	    # This is probably unnecessary since caught by eof above.
	} elseif {[regexp "^EOF" $line junk]} {
	    if {$debugServerLevel >= 2}  {
		puts "--->Closed connection"
		puts "HandleClientRequest:: EOF read on channel=$channel"
	    }
	    # Update entry only for nontemporary channels.
	    if {![info exists tempChannel($channel)]}  {
		MakeCommEntry $ip  -1 0
	    } else {
		unset tempChannel($channel)
	    }
	    catch {close $channel}
	} else  {
	    if {$debugServerLevel >= 2}  {
		puts "---> unknown instruktion"
	    }
	}
    }
}

#   SetupChannel ---
#   
#   Handles remote connections to the server port.
#   Sets up the callback routine.

proc SetupChannel { channel ip port } {
    global  debugServerLevel ipNum2Name ipName2Num ipNum2Socket   \
      ipNum2ServPort thisIPnum
    
    fileevent $channel readable "HandleClientRequest $channel $ip $port"

    # Everything should be done with 'fileevent'.
    fconfigure $channel -blocking 0

    # Everything is lineoriented except binary transfer operations.
    fconfigure $channel -buffering line
    
    # For nonlatin characters to work be sure to use Unicode/UTF-8.
    if {[info tclversion] >= 8.1}  {
        catch {fconfigure $channel -encoding utf-8}
    }
    if {$debugServerLevel >= 2}  {
	puts "SetupChannel:: --->Connection made to $ip:${port} on \
	  channel $channel. Its characteristics are:"
	puts [fconfigure $channel]
    }
    # Save ip nums and names etc in arrays.
    set peername [fconfigure $channel -peername]
    set sockname [fconfigure $channel -sockname] 
    set ipNum [lindex $peername 0]
    set ipName [lindex $peername 1]
    set ipNum2Name($ipNum) $ipName
    set ipName2Num($ipName) $ipNum
    set thisIPnum [lindex $sockname 0]
    
    # Be sure to add own serv port and socket as well. temporary channels???
    if {$debugServerLevel >= 2}  {
	puts "SetupChannel:: thisIPnum=$thisIPnum, [lindex $sockname 2]"
    }
    if {![info exists ipNum2ServPort($thisIPnum)]}  {
	set ipNum2ServPort($thisIPnum) [lindex $sockname 2]
    }
    if {![info exists ipNum2Socket($thisIPnum)]}  {
	set ipNum2Socket($thisIPnum) $channel
    }
    # Add entry in the communication frame.
    MakeCommEntry $ipNum  -1 1
    .menu.info entryconfigure "*On Server*" -state normal
    .menu.info entryconfigure "*On Clients*" -state normal
}

proc MakeSpecialServerCanvas { wtop wcan } {
    global  dims

    catch {toplevel $wtop}
    canvas $wcan -height $dims(hCanOri) -width $dims(wCanOri)   \
      -relief raised -bd 1 -scrollregion "0 0 $dims(wCanOri) $dims(hCanOri)" \
      -highlightthickness 0
    pack $wcan
    wm title $wtop "Whiteboard: server"
    wm resizable $wtop 0 0
}

#   FileCopyCallback ---
#
#   Callback function to handel the server part of putting a file.
#   'in': socket, 'out': fileId, 'fileTail': tail of file name,
#   'totoBytes': total size of file,...

proc FileCopyCallback { in out ip fileTail totBytes sumBytes  \
  optList bytes {error {}} }  {
    global  chunkSize debugServerLevel wProgWin statMess ipNum2Name   \
      timingGetFile skitVarWait
    
    if {$debugServerLevel >= 2}  {
	puts "FileCopyCallback:: (entry) error=$error, bytes=$bytes, \
	  optList=$optList"
    }
    if {$error != {}}  {
	tk_messageBox -message   \
	  "File transfer of file $fileTail failed with error: $error" \
	  -icon error -type ok
	catch "destroy $wProgWin$in"
	# Close socket and file.
	GetFileShutDown $in $out
    }
    # Check if socket already closed, perhaps the user pushed the cancel bt.
    if {[catch {eof $in}]}  {
	#puts "FileCopyCallback:: catch eof in"
	return
    }
    set progWin 0
    if {$totBytes > [expr 4*$chunkSize]}  {
	set progWin 1
    }
    if {$progWin && $sumBytes == 0} {
	mkProgressWindow $wProgWin$in Transfer "$fileTail"    \
	  "GetFileShutDown $in $out" -1
    }
    # Reset timing list. (microseconds on Mac)
    if {$sumBytes == 0}  {
	catch {unset timingGetFile($in)}
    }
    incr sumBytes $bytes
    # Store timing data in a list with each element a list {clicks sumBytes}.
    lappend timingGetFile($in) [list [clock clicks] $sumBytes]
    # Get transfer statistics.
    set bytesPerSec [GetTransferRateFromTiming  $timingGetFile($in)]
    set txtBytesPerSec [BytesPerSecFormatted $bytesPerSec]
    set percent [format "%3.0f" [expr 100*$sumBytes/($totBytes + 1.0)]]
    if {$sumBytes > [expr 3*$chunkSize]}  {
 	set secsLeft  \
	  [expr int(ceil(($totBytes - $sumBytes)/($bytesPerSec + 1.0)))]
	set txtTimeLeft ", $secsLeft secs remaining"
    } else  {
	set txtTimeLeft ""
    }
    if {$debugServerLevel >= 2}  {
	puts "FileCopyCallback:: sumBytes=$sumBytes,  percent=$percent"
    }
    if {$progWin}  {
        #puts -nonewline "FileCopyCallback:: before after 500; "
	mkProgressWindow $wProgWin$in Transfer "$fileTail"   \
	  "GetFileShutDown $in $out" $percent
	# Delay just to see what happens. 
	# This way the event loop is not blocked during the time delay.
	if {$debugServerLevel >= 2}  {
	    set skitVarWait 0
	    after 2000 {set skitVarWait 1}
	    vwait skitVarWait
	}
	#puts "after 'after 2000'"
    }    
    # Status message.
    set statMess  \
      "Getting file: $fileTail from $ipNum2Name($ip) \
      (at ${txtBytesPerSec}${txtTimeLeft})"
    if {[eof $in] }  {

	# Consistency checking: totBytes must be equal to the actual bytes
	# received.
	if {$totBytes != $sumBytes}  {
	    puts "FileCopyCallback:: eof in: totBytes=$totBytes, sumBytes=$sumBytes"
	}
	CleanupAfterFileCopy $in $out $fileTail $optList $bytes
	
	# Rebind this procedure.
    } else  {
	# Schedule timeout killer event.
	GetFileScheduleKiller $in $out $ip $fileTail
	
	# Not finished; rebind this callback.
	fcopy $in $out -command  \
	  [list FileCopyCallback $in $out $ip $fileTail $totBytes $sumBytes $optList] \
	  -size $chunkSize
    }
}

#   CleanupAfterFileCopy ---
#
#   Closes sockets and files. Calls to 'DoImportImageOrMovie' to actually show
#   image or movie in canvas.
#   'optList' = {x y idtag}

proc CleanupAfterFileCopy { in out fileTail optList bytes {error {}} }  {
    global  servCan debugServerLevel wProgWin supExts statMess timingGetFile
    
    if {$debugServerLevel >= 2}  {
	puts "CleanupAfterFileCopy:: (entry) bytes=$bytes, optList=$optList"
    }
    # Close socket and file.
    GetFileShutDown $in $out

    if {[string length $error] != 0}  {
	puts "error during file copy: $error"
	return
    }
    # File extension hints what supported file type.
    set fext [string tolower [file extension $fileTail]]
    
    set inFile [file join [info script] incoming $fileTail]
    set x [lindex $optList 0]
    set y [lindex $optList 1]
    set tag [lindex $optList 2]
    
    if {$debugServerLevel >= 2}  {
	puts "CleanupAfterFileCopy:: inFile=$inFile"
    }
    # Cleanup window and reset byte counter.
    catch "destroy $wProgWin$in"
    set statMess {}
    
    # Do the actual work of showing the image/movie.
    DoImportImageOrMovie $servCan $inFile "$x $y $tag" "own"
}

#   GetFileScheduleKiller, GetFileKill, GetFileShutDown ---
#
#   Utility routines to handle timeout events on get file operations.

proc GetFileScheduleKiller { sock fid ip fileName }  {
    global  getFileKillerId prefs
    
    if {[info exists getFileKillerId($sock)]}  {
	after cancel $getFileKillerId($sock)
    }
    set getFileKillerId($sock) [after [expr 1000*$prefs(timeout)]   \
      [list GetFileKill $sock $fid $ip $fileName]]
}

proc GetFileKill { sock fid ip fileName }  {
    global  getFileKillerId prefs ipNum2Name
    
    if {![info exists getFileKillerId($sock)]}  {
	return
    }
    tk_messageBox -message   \
      "Timout when waiting for data for file $fileName from $ipNum2Name($ip)" \
      -icon error -type ok
    GetFileShutDown $sock $fid
}

proc GetFileShutDown { sock fid }  {
    global  timingGetFile getFileKillerId recordGetFile
    
    #  Clean.
    catch {unset recordGetFile($s)}
    catch {unset timingGetFile($sock)}
    if {[info exists getFileKillerId($sock)]}  {
	after cancel $getFileKillerId($sock)
    }
    catch {unset getFileKillerId($sock)}
    # Close.
    catch {close $sock}
    catch {close $fid}
}

#--------------------------------------------------------------------------
