#------------------------------------------------------------------------
#
#  mkOpenConnection.tcl ---
#  
#  Part of the Whiteboard application. 
#  It creates a dialog for connecting to the server via TCP/IP,
#  and provide some procedures to make the connection.
#   
#  The whiteboard application has been developed by:
#  
#	Mats Bengtsson, 
#	Hamngatan 21,
#	SE-58226, Linkoping Sweden,
#	matben@lin.foa.se,
#	phone: +46 13 136114
# 
#  It is distributed under the standard GPL.
#  See the README file for further details.
#  
#------------------------------------------------------------------------

proc mkOpenConnection { }  {
    global  selCompName internalIPnum remoteServPort sysFont prefs  \
	    shortcuts compNameOrNum compPort finishedOpenConn connIPnum

    set w .opc
    set finishedOpenConn -1
    catch {toplevel $w}
    wm title $w "Open Connection"
    
    # Global frame.
    pack [frame $w.frall -borderwidth 1 -relief raised] -fill both -expand 1
    
    # Ip part.
    frame $w.frip -borderwidth 0
    set wcont1 [LabeledFrame $w.frip "Connect to"]
    
    # Overall frame for whole container.
    set frtot [frame $wcont1.fr]
    pack $frtot
    message $frtot.msg -width 230 -font $sysFont(s) -text  \
      "Connect to a remote computer. Write remote computer name\
      or choose shortcut from the popup menu.\
      If necessary choose new remote port number."
    label $frtot.lblip -text "Shortcut:"
    entry $frtot.entip -width 30 -textvariable compNameOrNum
    
    # The option menu.
    set shorts [lindex $shortcuts 0]
    eval {tk_optionMenu $frtot.optm selCompName} $shorts
    $frtot.optm configure -highlightthickness 0  \
      -background $prefs(bgColGeneral) -foreground black
    grid $frtot.msg -column 0 -row 0 -columnspan 2 -sticky w -padx 6 -pady 2
    grid $frtot.lblip -column 0 -row 1 -sticky w -padx 6 -pady 2
    grid $frtot.optm -column 1 -row 1 -sticky e -padx 6 -pady 2
    grid $frtot.entip -column 0 -row 2 -sticky ew -padx 10 -columnspan 2
    pack $w.frip -side top -fill both -ipadx 10 -ipady 6 -in $w.frall

    # Port part.
    set ofr [frame $w.frport -borderwidth 0]
    set wcont2 [LabeledFrame $ofr "Port number"]
    label $wcont2.lport -text "Remote server port:"
    entry $wcont2.entport -width 6 -textvariable compPort
    set compPort $remoteServPort
    grid $wcont2.lport -row 0 -column 0 -padx 6
    grid $wcont2.entport -row 0 -column 1 -padx 6
    pack $ofr -side top -fill both -ipadx 10 -ipady 6 -in $w.frall

    # Button part.
    frame $w.frbot -borderwidth 0
    pack [button $w.btconn -text "Connect" -default active  \
      -command PushBtConnect]  \
      -in $w.frbot -side right -padx 5 -pady 5
    pack [button $w.btcancel -text " Cancel " -command {set finishedOpenConn 0}]  \
      -in $w.frbot -side right -padx 5 -pady 5
    pack $w.frbot -side top -fill both -expand 1 -in $w.frall  \
      -padx 8 -pady 6

    bind $w <Return> PushBtConnect
    trace variable selCompName w TraceOpenConnect
    
    wm resizable $w 0 0
    focus $w
    catch {grab $w}
    tkwait variable finishedOpenConn
    catch {grab release $w}
    destroy $w

    if {$finishedOpenConn == 1}  {
    	return $connIPnum
    } else {
    	return ""
    }
}

proc PushBtConnect {  }  {
    global  finishedOpenConn connIPnum compNameOrNum compPort internalIPnum  \
      internalIPname

    #puts "PushBtConnect:: compNameOrNum=$compNameOrNum"
    # Always allow connections to 'internalIPname'.
    # This is because 'IsConnectedToQ' always answers this question with true.
    if {$compNameOrNum == $internalIPnum || $compNameOrNum == $internalIPname} {
	set connIPnum [DoConnect $compNameOrNum $compPort 1]
	set finishedOpenConn 1
	return
    }
    # Check if not already connected to the ip in question.
    if {[IsConnectedToQ $compNameOrNum]}  {
	set finishedOpenConn 0
	return
    }
    set connIPnum [DoConnect $compNameOrNum $compPort 1]
    set finishedOpenConn 1
}

proc TraceOpenConnect { name junk1 junk2 }  {
    upvar #0 $name locName
    global  shortcuts compNameOrNum
    
    # 'compNameOrNum' is textvariable in entry widget
    set ind [lsearch [lindex $shortcuts 0] $locName]
    set compNameOrNum [lindex [lindex $shortcuts 1] $ind]
}

#   'DoConnect' handles the complete connection process.
#   It makes the actual connection to a given ip address and
#   port number. It sets some arrays to keep track of each connection.
#   If open socket async, then need 'WhenSocketOpensInits' as callback.
#   If 'propagateSizeToClients', then let other clients know this canvas size,
#   which is the case if interactive open, else not (weird things happen).

proc DoConnect { toNameOrNum toPort {propagateSizeToClients 1} }  {
    global  internalIPnum debugLevel prefs internalIPname errorCode statMess
    
    set nameOrIP $toNameOrNum
    set remoteServPort $toPort
    
    if {$debugLevel >= 2}  {
	puts "DoConnect:: nameOrIP: $nameOrIP, remoteServPort: $remoteServPort"
    }
    set statMess "Contacted $nameOrIP. Waiting for response..."
    StartStopAnimatedWave .fcomm.stat.lbl 1
    update idletasks
    
    # Handle the TCP/IP channel; if internal pick internalIPnum
    if {[string compare $nameOrIP $internalIPnum] == 0 ||
    [string compare $nameOrIP $internalIPname] == 0}  {
	if {$prefs(asyncOpen)}  {
	    set res [catch {socket -async -myaddr $internalIPnum  \
	      $internalIPnum $remoteServPort} server]
	} else  {
	    set res [catch {socket -myaddr $internalIPnum  \
	      $internalIPnum $remoteServPort} server]
	}
    } else {
	if {$prefs(asyncOpen)}  {
	    set res [catch {socket -async $nameOrIP $remoteServPort} server]
	} else  {
	    set res [catch {socket $nameOrIP $remoteServPort} server]
	}
    }
    if {$debugLevel >= 2}  {
	puts "DoConnect:: res=$res"
    }
    if {$res}  {
	tk_messageBox -message   \
	  "Couldn't create socket to server.\
	  Maybe networking is not functioning or server is down.\
	  The error code is: $errorCode" \
	  -icon error -type ok
	set statMess ""
	StartStopAnimatedWave .fcomm.stat.lbl 0
	update idletasks
	return {}
    }
    # Write line by line; encode newlines in text items as \n.
    fconfigure $server -buffering line
    
    # When socket writable the connection is opened.
    # Needs to be in nonblocking mode.
    fconfigure $server -blocking 0
    
    # For nonlatin characters to work be sure to use Unicode/UTF-8.
    if {[info tclversion] >= 8.1}  {
	catch {fconfigure $server -encoding utf-8}
    }
    # If open socket in async mode, need to wait for fileevent.
    if {$prefs(asyncOpen)}  {
	fileevent $server writable   \
	  "WhenSocketOpensInits $server $remoteServPort $propagateSizeToClients"
	# Set up timer event for timeouts.
	OpenConnectionScheduleKiller $server
	set ans ""
	
	# Else, it is already open.
    } else  {
	set ans [WhenSocketOpensInits   \
	  $server $remoteServPort $propagateSizeToClients]
    }
    return $ans
}

#   WhenSocketOpensInits ---
#
#   When socket is writable, it is open. Do all the necessary initializations.
#   If 'propagateSizeToClients', then let other clients know this canvas size.

proc WhenSocketOpensInits  { server remoteServPort {propagateSizeToClients 1} }  {
    global  ipNum2Name ipName2Num ipNum2Socket ipNum2ServPort thisServPort  \
      myItpref thisUserName openConnectionKillerId debugLevel  \
      internalIPnum wCan statMess allIPnumsTo

    if {$debugLevel >= 2}  {
	puts "WhenSocketOpensInits:: (entry)"
    }
    StartStopAnimatedWave .fcomm.stat.lbl 0

    # No more event handlers here.
    fileevent $server writable {}

    #  Cancel timeout killer.
    if {[info exists openConnectionKillerId($server)]}  {
	after cancel $openConnectionKillerId($server)
    }
    # Check if something went wrong first.
    if {[catch {fconfigure $server -sockname} sockname]}  {
	tk_messageBox -message "Something went wrong. $sockname" \
	  -icon error -type ok
	set statMess ""
	return {}
    }
    if {[catch {fconfigure $server -peername} peername]}  {
	tk_messageBox -message "Something went wrong. $peername" \
	  -icon error -type ok
	set statMess ""
	return {}
    }
    # Save ip nums and names etc in arrays.
    if {$debugLevel >= 2}  {
	puts "WhenSocketOpensInits:: peername=$peername"
    }
    set ipNum [lindex $peername 0]
    set ipName [lindex $peername 1]
    set ipNum2Name($ipNum) $ipName
    set ipName2Num($ipName) $ipNum
    set ipNum2Socket($ipNum) $server
    set ipNum2ServPort($ipNum) $remoteServPort
    set statMess "Client $ipName responded."

    # Let the remote computer know port and itpref used by this client.
    puts $server [list "IDENTITY:" $thisServPort $myItpref $thisUserName]
    
    # Let connecting client now this 'allIPnumsTo' (for multi connect).????
    # 'allIPnumsTo' not yet updated with the new ip; doesn't matter!
    set listIPandPort {}
    foreach ip $allIPnumsTo {
	lappend listIPandPort $ip $ipNum2ServPort($ipNum)
    }
    puts $server "IPS CONNECTED: $listIPandPort"

    # If internal then open server toplevel canvas for writing
    # else write in the original canvas.
    if {$ipNum == $internalIPnum && $remoteServPort == $thisServPort}  {
	set servCan $specServCan
	# Let the safe interpreter know also. Not working!!!
	if {$makeSafeServ}  {
	    interp eval serverInterp [list set servCan $servCan]
	}
	if {![winfo exists .serv]}  {
	    MakeSpecialServerCanvas .serv $servCan
	}
    } else {
	set servCan $wCan
    }
    # Add line in the communication entry. Also updates 'allIPnumsTo'.
    MakeCommEntry $ipNum 1 -1    
    
    # Let all other now about the size change. propagateToClients and force.
    CanvasSizeChange $propagateSizeToClients 1

    # Fix menus.
    .menu.file entryconfigure "*Put File*" -state normal
    .menu.file entryconfigure "*Put Canvas*" -state normal
    .menu.file entryconfigure "*Get Canvas*" -state normal
    
    return $ipNum
}

#   OpenConnectionScheduleKiller, OpenConnectionKill ---
#
#   Cancel 'OpenConnection' process if timeout.

proc OpenConnectionScheduleKiller  { sock }  {
    global  openConnectionKillerId prefs

    if {[info exists openConnectionKillerId($sock)]}  {
	after cancel $openConnectionKillerId($sock)
    }
    set openConnectionKillerId($sock) [after [expr 1000*$prefs(timeout)]   \
      "OpenConnectionKill $sock"]
}

proc OpenConnectionKill  { sock }  {
    global  openConnectionKillerId statMess
    
    catch {close $sock}
    set statMess "Timout when waiting for connection to open."
    if {[info exists openConnectionKillerId($sock)]}  {
	after cancel $openConnectionKillerId($sock)
    }
    catch {unset openConnectionKillerId($sock)}
    tk_messageBox -message   \
      "Timout when waiting for connection to open." \
      -icon error -type ok
}

#---------------------------------------------------------------------