# copyright (C) 1997-1999 Jean-Luc Fontaine (mailto:jfontain@multimania.com)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

set rcsId {$Id: freetext.tcl,v 1.14 1998/12/26 10:43:44 jfontain Exp $}

class freeText {

    proc freeText {this parentPath args} composite {
        [new text $parentPath -font $font::(mediumNormal) -wrap word -borderwidth 0 -highlightthickness 0] $args
    } viewer {} {
        viewer::setupDropSite $this $widget::($this,path)                                            ;# allow dropping of data cells
        set freeText::($this,labels) {}
        composite::complete $this
    }

    proc ~freeText {this} {
        if {[info exists freeText::($this,drag)]} {
            delete $freeText::($this,drag)
        }
        eval delete $freeText::($this,labels)
        if {[info exists freeText::($this,selector)]} {
            delete $freeText::($this,selector)
        }
        if {[string length $composite::($this,-deletecommand)]>0} {
            uplevel #0 $composite::($this,-deletecommand)                                   ;# always invoke command at global level
        }
    }

    proc iconData {} {
        return {
            R0lGODdhKAAoAIQAAHh4eMjMyAAAANDU0Pj8+Hh8eDg4OEBIQLi8uFAIYHAoeHAweGgYeIgQoLgo0KAYuLAYyLAg0GggeNg4+KgYyNhA+LgY2OBI+OBQ+Ngo
            +AAAAAAAAAAAAAAAAAAAAAAAACwAAAAAKAAoAAAF/iAgjmRpnugYAELrvnAsz+0qBHgeDDi/6z6fbjhkBQjIJBKgbDKbyecSaTtCCVJo1ql82gZXQiEsJo+j
            VKOowG673/B4W2UMn693aN7LAuPNgGgEVUg0hodYaTciMGSOTQJzX4Uvj5YEAmWDdZiVl46ZZ5OdLp+gI5uLLJ6mVwIGBwcAhKQtZLaWAqijjWEtCLm7nL21
            lAi4xYXCqr2/x5mlts64rwYGs8OsCMC/0wLPwATbmMt+xDDgnelKuiKpVs3f4L/q9OzljJ703dLf+5mYYF2jda4fv3EvEC6Dx0rcuG0PwzkcR84dwYat7llc
            JSOjqxc2Eogk5jGJAAULcxYwsNGgJcmSmFAucGDjAYQIL0sKkKBAwQQbFCBQOERUhswKQClEsMA0pdOnUKNKdXoBqAOmTadq3ZoSKQusWLmKlYrBBtizaNOq
            RetVwNq3cMFWmGsjg927ePPq3at3boa6fAMLDgx4sGHDK4goXsyYcQgAOw==
        }
    }

    proc options {this} {
        # force size values
        return [list\
            [list -cellindices cellIndices CellIndices {} {}]\
            [list -deletecommand {} {}]\
            [list -draggable draggable Draggable 0 0]\
            [list -endtext endText EndText {} {}]\
            [list -height height Height 1]\
            [list -width width Width 40]\
        ]
    }

    proc set-cellindices {this value} {                           ;# indices of soon to be created cells when initializing from file
        if {$composite::($this,complete)} {
            error {option -cellindices cannot be set dynamically}
        }
        set freeText::($this,nextCellIndex) 0                            ;# initialize cell insertion index index in list of indices
    }

    proc set-endtext {this value} {
        $widget::($this,path) insert end $value
    }

    proc set-deletecommand {this value} {}

    proc set-draggable {this value} {
        if {$composite::($this,complete)} {
            error {option -draggable cannot be set dynamically}
        }
        if {!$value} return                                                                                           ;# no dragging
        set freeText::($this,drag) [new dragSite -path $widget::($this,path) -validcommand "freeText::validateDrag $this 0"]
        dragSite::provide $freeText::($this,drag) OBJECTS "freeText::dragData $this"

        set freeText::($this,selector) [new selector -selectcommand "freeText::setLabelsState $this"]
    }

    foreach option {-height -width} {
        proc set$option {this value} "\$widget::(\$this,path) configure $option \$value"
    }

    proc dragData {this format} {
        switch $format {
            OBJECTS {
                set list [selector::selected $freeText::($this,selector)]
                if {[llength $list]>0} {
                    return $list                                                          ;# return selected labels if there are any
                } elseif {[empty $this]} {
                    return $this                                                               ;# return text object itself if empty
                } else {
                    return {}                                                                            ;# return nothing otherwise
                }
            }
            DATACELLS {
                return [cellsFromLabels $this [selector::selected $freeText::($this,selector)]]
            }
        }
    }

    proc validateDrag {this label x y} {
        if {($label==0)&&[empty $this]} {                                                                 ;# dragging from text area
            return 1                                                                       ;# empty viewer may be dragged into trash
        } elseif {[lsearch -exact [selector::selected $freeText::($this,selector)] $label]>=0} {
            return 1                                                                      ;# allow dragging from selected label only
        } else {
            return 0
        }
    }

    proc supportedTypes {this} {
        return {ascii dictionary integer real}
    }

    proc monitorCell {this array row column} {
        viewer::registerTrace $this $array
        set cell ${array}($row,$column)
        if {[lsearch -exact [cellsFromLabels $this $freeText::($this,labels)] $cell]>=0} return          ;# already displayed, abort
        set path $widget::($this,path)
        if {[info exists freeText::($this,nextCellIndex)]} {      ;# recreate data cell labels placement from recorded configuration
            set index [lindex $composite::($this,-cellindices) $freeText::($this,nextCellIndex)]
            if {[string length $index]==0} {                  ;# indices list exhausted: we are done initializing from recorded data
                unset freeText::($this,nextCellIndex)
                set index insert                                                         ;# position cell window at insertion cursor
            } else {
                incr freeText::($this,nextCellIndex)                                                  ;# get ready for upcoming cell
            }
        } else {
            set index insert                                                ;# insert cell label text and window at insertion cursor
            $path insert $index "[viewer::label $array $row $column]: "
        }
        set label [new label $path $cell]
        set labelPath $label::($label,path)
        # keep track of label existence
        switched::configure $label -deletecommand "freeText::deletedLabel $this $array $label"
        if {$composite::($this,-draggable)} {                                              ;# setup dragging and selection for label
            set drag [new dragSite -path $labelPath -validcommand "freeText::validateDrag $this $label"]
            dragSite::provide $drag OBJECTS "freeText::dragData $this"
            dragSite::provide $drag DATACELLS "freeText::dragData $this"
            set freeText::($this,drag,$label) $drag
            set selector $freeText::($this,selector)
            selector::add $selector $label
            bind $labelPath <ButtonRelease-1> "selector::select $selector $label"
            bind $labelPath <Control-ButtonRelease-1> "selector::toggle $selector $label"
            bind $labelPath <Shift-ButtonRelease-1> "freeText::extendSelection $this $label"
        }
        lappend freeText::($this,labels) $label
        $path window create $index -window $labelPath
        set freeText::($this,cell,$label) $cell
    }

    proc update {this array args} {                              ;# update display using cells data. ignore eventual trace arguments
        foreach label $freeText::($this,labels) {
            if {[catch {set $freeText::($this,cell,$label)} value]} {                                        ;# handle invalid cells
                switched::configure $label -text ?
            } else {
                switched::configure $label -text $value
            }
        }
    }

    proc deletedLabel {this array label} {
        if {$composite::($this,-draggable)} {
            delete $freeText::($this,drag,$label)
            selector::remove $freeText::($this,selector) $label
        }
        viewer::unregisterTrace $this $array                                          ;# trace may no longer be needed on this array
        ldelete freeText::($this,labels) $label
        unset freeText::($this,cell,$label)
    }

    proc cellsFromLabels {this labels} {
        set cells {}
        foreach label $labels {
            lappend cells $freeText::($this,cell,$label)
        }
        return $cells
    }

    proc cells {this} {
        return [cellsFromLabels $this $freeText::($this,labels)]
    }

    proc setLabelsState {this labels select} {
        foreach label $labels {
            label::select $label $select
        }
    }

    proc extendSelection {this endLabel} {
        set selector $freeText::($this,selector)
        if {[info exists selector::($selector,lastSelected)]} {                             ;# extend from previously selected label
            # build path to label mapping table (reasonable since it is likely that there is only a few embedded labels in the text)
            foreach label $freeText::($this,labels) {
                set labelFromPath($label::($label,path)) $label
            }
            # build ordered label list from windows returned ordered according to their postion (index) in the text
            set list {}
            foreach {key path index} [$widget::($this,path) dump -window 1.0 end] {
                if {[string length $path]==0} continue                                                     ;# ignore deleted windows
                lappend list $labelFromPath($path)
            }
            set start [lsearch -exact $list $selector::($selector,lastSelected)]
            set end [lsearch -exact $list $endLabel]
            if {$end<$start} {                                                           ;# make sure limits are in increasing order
                set index $start
                set start $end
                set end $index
            }
            selector::clear $selector
            selector::set $selector [lrange $list $start $end] 1
        } else {
            selector::select $selector $endLabel
        }
    }

    proc empty {this} {                                                      ;# if no labels exist and there is no visible text left
        return [expr\
            {([llength $freeText::($this,labels)]==0)&&([string length [string trim [$widget::($this,path) get 1.0 end]]]==0)}\
        ]
    }

    proc initializationConfiguration {this} {
        set options {}
        set text {}
        foreach {key string index} [$widget::($this,path) dump -text 1.0 end] {
            append text $string
        }
        lappend options -endtext $text
        set indices {}
        foreach {key path index} [$widget::($this,path) dump -window 1.0 end] {
            if {[string length $path]==0} continue                                                         ;# ignore deleted windows
            lappend indices $index
        }
        if {[llength $indices]>0} {
            lappend options -cellindices $indices
        }
        return $options
    }

}

class freeText {

    class label {

        proc label {this parentPath cell args} switched {$args} {
            set label [new label $parentPath\
                -font $font::(mediumBold) -relief sunken -padx 0 -pady 0 -borderwidth 1 -cursor top_left_arrow\
            ]
            # keep track of label existence as it may be deleted by directly editing in the parent text widget
            bind $widget::($label,path) <Destroy> "delete $this"
            set ($this,path) $widget::($label,path)
            set ($this,label) $label
            set ($this,cell) $cell
            switched::complete $this
        }

        proc ~label {this} {
            bind [set ($this,path)] <Destroy> {}                                                ;# remove binding to avoid recursion
            delete [set ($this,label)]
            if {[string length $switched::($this,-deletecommand)]>0} {
                uplevel #0 $switched::($this,-deletecommand)                                ;# always invoke command at global level
            }
        }

        proc options {this} {
            return [list\
                [list -deletecommand {} {}]\
                [list -text {} {}]\
            ]
        }

        proc set-deletecommand {this value} {}                                                   ;# data is stored at switched level

        proc set-text {this value} {
            [set ($this,path)] configure -text $value
        }

        proc select {this select} {
            if {$select} {
                [set ($this,path)] configure -background white
            } else {
                [set ($this,path)] configure -background $widget::(default,ButtonBackgroundColor)
            }
        }

    }

}
