#!/bin/sh
# the next line restarts using the interpreter \
exec wish "$0" "$@"

# copyright (C) 1997-1999 Jean-Luc Fontaine (mailto:jfontain@multimania.com)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

source utility.tcl

startGatheringRCSIds
startGatheringPackageDirectories

set rcsId {$Id: moodss.tcl,v 1.144 1999/01/17 20:14:09 jfontain Exp $}

set applicationVersion 6.2

source getopt.tcl

proc printUsage {exitCode} {
    puts stderr "Usage: $::argv0 \[OPTION\]... \[MODULE\] \[MODULE\]..."
    puts stderr {  -f, --file       configuration file name}
    puts stderr {  -h, --help       display this help and exit}
    puts stderr {  -p, --poll-time  poll time in seconds}
    puts stderr {  -r, --read-only  disable viewer creation, editing, ...}
    puts stderr {  -S, --static     disable internal window manager sizing and moving}
    puts stderr {  --show-modules   try to find available moodss modules}
    puts stderr {  --version        output version information and exit}
    exit $exitCode
}

if {[catch\
    {\
        set argv [parseCommandLineArguments\
            {-f 1 --file 1 -h 0 --help 0 -p 1 --poll-time 1 -r 0 --read-only 0 -S 0 --static 0 --show-modules 0 --version 0} $argv\
            arguments\
        ]\
    }\
    message\
]} {
    puts stderr $message
    printUsage 1
}

foreach {short long} {-f --file -h --help -p --poll-time -r --read-only -S --static} {
    catch {set arguments($short) $arguments($long)}                                          ;# long version if present has priority
}

if {[info exists arguments(--version)]} {
    puts "moodss (a Modular Object Oriented Dynamic SpreadSheet) version $applicationVersion"
    exit
}
if {
    [info exists arguments(-h)]||
    (([llength $argv]==0)&&![info exists arguments(-f)]&&![info exists arguments(--show-modules)])
} {
    printUsage 1
}

lappend auto_path /usr/lib/moodss .    ;# search in current directory sub-directories for development and rpm installation directory

if {[catch {package require stooop 3.7}]} {
    source stooop.tcl                                                                     ;# in case stooop package is not installed
}
namespace import stooop::*
if {[catch {package require switched 1.4}]} {                                           ;# in case switched package is not installed
    source switched.tcl
}
source modules.tcl

if {[info exists arguments(--show-modules)]} {
    modules::printAvailable
    exit                                                                                                                     ;# done
}

wm command . [concat $argv0 $argv]                                   ;# for proper window manager (windowmaker for example) behavior
wm group . .

proc setPollTimes {modules {commandLineTime {}}} {                                                      ;### move to modules.tcl ###
    global pollTimes pollTime

    set default 0
    set minimum 0
    foreach module $modules {
        set times [set ${module}::data(pollTimes)]
        if {[llength $times]==0} {
            error "module $module poll times list is empty"
        }
        # for an asynchronous module, the sole time value would be negative and is used as graph interval, for example
        set time [lindex $times 0]
        if {$time<0} {                                              ;# asynchronous module, poll time is a viewer interval (negated)
            set intervals($time) {}
            continue
        }
        if {$time>$default} {                                                                  ;# default value is the first in list
            set default $time                                                        ;# keep the greater default time of all modules
        }
        set times [lsort -integer $times]                                                                         ;# sort poll times
        set time [lindex $times 0]
        if {$time>$minimum} {
            set minimum $time                                                        ;# keep the greater minimum time of all modules
            set minimumModule $module
        }
        foreach time $times {                                        ;# poll times list is the combination of all modules poll times
            set data($time) {}
        }
    }
    set pollTimes [lsort -integer [array names data]]         ;# sort and restrict poll times above maximum module minimum poll time
    set pollTimes [lrange $pollTimes [lsearch -exact $pollTimes $minimum] end]
    set pollTime $default
    if {[string length $commandLineTime]>0} {                                           ;# eventually validate command line override
        if {$commandLineTime<$minimum} {
            puts stderr "$::argv0: minimum poll time is $minimum seconds for module $minimumModule"
            exit 1
        }
        set pollTime $commandLineTime
    }
    if {$pollTime==0} { 
        # all modules are asynchronous, so use an average time as a viewer interval for viewers that need it, such as graphs.
        # the poll times list is empty at this point so the user cannot change the poll time.
        # note that the viewer interval can still be forced by the command line poll time option.
        set sum 0
        set number 0
        foreach interval [array names intervals] {
            incr sum $interval
            incr number
        }
        set pollTime [expr {round(double($sum)/-$number)}]
    }
}

if {[catch {package require scwoop 2.6}]} {
    source scwoutil.tcl
    source scwoop.tcl                                                                     ;# in case scwoop package is not installed
    source bindings.tcl
    source widgetip.tcl
    source arrowbut.tcl
    source spinent.tcl
    source panner.tcl
}

source record.tcl

# load uninitialized modules 1st so that their tables are placed properly
modules::initializeNext $argv                                                                                           ;# recursive

if {[info exists arguments(-f)]} {                                                              ;# configuration file name specified
    set saveFile $arguments(-f)
    set initializer [new record -file $saveFile]
    record::read $initializer
    modules::initializeNext [record::modulesWithArguments $initializer]
    set modules::(initialized) [record::modules $initializer]
} else {
    set saveFile {}
}

wm iconname . "moodss $modules::(all)"                                                            ;# give the icon a meaningful name
set readOnly [info exists arguments(-r)]
set static [info exists arguments(-S)]

set modules::(synchronous) {}                                                                               ;# look at synchronicity
foreach module $modules::(all) {
    if {[lindex [set ${module}::data(pollTimes)] 0]>0} {                                                 ;# if module is synchronous
        lappend modules::(synchronous) $module
    }
}
set modules::(synchronous,string) [commaSeparatedString $modules::(synchronous)]

if {[info exists arguments(-p)]} {                                                     ;# command line argument has highest priority
    setPollTimes $modules::(all) $arguments(-p)
} elseif {[info exists initializer]} {                                                                  ;# then stored configuration
    setPollTimes $modules::(all) [record::pollTime $initializer]
} else {                                                                                                       ;# use modules values
    setPollTimes $modules::(all)
}

if {[catch {package require Tktable 2}]} {                                               ;# in case tkTable package is not installed
    switch $tcl_platform(platform) {
        unix {
            load ./Tktable.so.2.4
        }
        windows {
            load ./tktable.dll
        }
    }
}

if {[catch {package require BLT}]} {                                                         ;# in case BLT package is not installed
    switch $tcl_platform(platform) {
        unix {
            load ./libBLT.so.2.4g
        }
        windows {
            load ./blt80.dll                                                  ;# so far, only unofficial 8.0 is supported on Windows
        }
    }
}
set officialBLT [expr {$blt_version!=8.0}]                                               ;# remember which BLT version is being used

if {[catch {package require tkpiechart 5.2.2}]} {                                     ;# in case tkpiechart package is not installed
    source pielabel.tcl
    source boxlabel.tcl
    source canlabel.tcl
    source labarray.tcl
    source perilabel.tcl
    source slice.tcl
    source selector.tcl
    source objselec.tcl
    source pie.tcl
}

source config.tcl
source font.tcl
source scroll.tcl
source lifolbl.tcl
source keyslink.tcl
source gui.tcl
source dialog.tcl
source viewer.tcl
source blt2d.tcl
source tablesel.tcl
source datatab.tcl
source datagraf.tcl
source databar.tcl
source datapie.tcl
source lastwish.tcl
source sumtable.tcl
source freetext.tcl
source htmllib.tcl                                                ;# Tcl HTML library from Sun, used for viewing HTML help documents
source htmldata.tcl
source html.tcl                                            ;# must be sourced after HTML library since some procedures are redefined
source help.tcl
source drag.tcl
source drop.tcl
source canvhand.tcl
source canvaswm.tcl


proc createCellsViewer {class cells draggable static {pollTime {}}} {
    global canvas

    set viewer [new $class $canvas -draggable $draggable]
    if {[string length $pollTime]>0} {
        composite::configure $viewer -interval $pollTime
    }
    viewer::view $viewer $cells
    manageViewer $viewer 1 -static $static
    return $viewer
}

proc manageViewer {viewer destroyable args} {                         ;# arguments are window manager configuration switched options
    global windowManager

    set path $widget::($viewer,path)
    canvasWindowManager::manage $windowManager $path
    eval canvasWindowManager::configure $windowManager $path $args
    if {$destroyable} {
        composite::configure $viewer -deletecommand "canvasWindowManager::unmanage $windowManager $path"
    }
}

proc save {ask} {                               ;# save current configuration in user defined file or currently defined storage file
    global saveFile messenger

    if {$ask||([string length $saveFile]==0)} {
        set file [tk_getSaveFile\
            -title {moodss: Save} -initialdir [pwd] -defaultextension .moo -filetypes {{{moodss data} .moo}} -initialfile $saveFile
        ]
        if {[string length $file]==0} return                                                                 ;# user canceled saving
        set saveFile $file
    }
    lifoLabel::flash $messenger "saving in $saveFile..."                ;# flash instead of pushing as that goes too fast to be seen
    set record [new record -file $saveFile]
    record::write $record
    delete $record
}

proc refresh {} {
    global updateEvent messenger pollTime

    catch {after cancel $updateEvent}                                                             ;# eventually cancel current event

    if {[llength $modules::(synchronous)]==0} return                                                                ;# nothing to do

    # we do not know when data will actually be updated
    lifoLabel::push $messenger "launching $modules::(synchronous,string) data update..."
    update idletasks
    foreach module $modules::(synchronous) {
        ${module}::update                                                                   ;# ask module to update its dynamic data
    }
    lifoLabel::pop $messenger
    set updateEvent [after [expr {1000*$pollTime}] refresh]                                               ;# convert to milliseconds
}

createMenuWidget . $readOnly [llength $pollTimes]
pack [createMessageWidget .] -side bottom -fill x
updateTitle

set scroll [new scroll canvas .]
set canvas $composite::($scroll,scrolled,path)
set width [winfo screenwidth .]
set height [winfo screenheight .]
$canvas configure -background white -width $width -height $height -scrollregion [list 0 0 $width $height] -highlightthickness 0\
    -takefocus 0

set windowManager [new canvasWindowManager $canvas]

if {[info exists ::geometry]} {                                                               ;# command line geometry was specified
    wm geometry . $::geometry
} elseif {[info exists initializer]} {
    foreach {width height} [record::sizes $initializer] {}                                   ;# used stored geometry for main window
    wm geometry . ${width}x$height
} else {
    wm geometry . 400x300
}

image create photo applicationIcon -data [dataGraph::iconData]                                ;# use data graph icon for application
wm iconwindow . [toplevel .icon]
pack [label .icon.image -image applicationIcon]

if {!$readOnly} {
    pack [createDragAndDropZone .] -fill x
}
pack $widget::($scroll,path) -fill both -expand 1

set draggable [expr {!$readOnly}]

set x 0.0
set y 0.0
foreach module $modules::(all) {
    if {[info exists ${module}::data(views)]} {                                                        ;# user overrode default view
        set viewMembers [set ${module}::data(views)]                                      ;# create and manage a table for each view
    } else {
        set viewMembers {{}}                                                           ;# there is a single table (the default view)
    }
    set index 0
    foreach members $viewMembers {
        if {[llength $members]>0} {                                                                                  ;# it is a view
            array set view $members
            set table [new dataTable $canvas -data ${module}::data -view ::view -draggable $draggable]
            unset view
        } else {                                                                                     ;# use single default data view
            set table [new dataTable $canvas -data ${module}::data -draggable $draggable]
        }
        if {[info exists initializer]&&([lsearch -exact $modules::(initialized) $module]>=0)} {  ;# setup initialized modules tables
            eval composite::configure $table [record::tableOptions $initializer $module $index]
            # use stored window manager initialization data for table
            foreach {x y width height level} [record::tableWindowManagerData $initializer $module $index] {}
            manageViewer $table 0 -static $static -setx $x -sety $y -setwidth $width -setheight $height -level $level
        } else {
            manageViewer $table 0 -static $static -setx $x -sety $y
        }
        set x [expr {$x+$configuration::(xWindowManagerInitialOffset)}]                   ;# offset tables to achieve a nicer layout
        set y [expr {$y+$configuration::(yWindowManagerInitialOffset)}]
        incr index                                                                                      ;# next view for initializer
    }
}

refresh                                                                                                ;# initialize refresh process

if {[info exists initializer]} {                  ;# stored configuration, now that modules data is initialized, create some viewers
    foreach {class cells x y width height level switchedOptions} [record::viewersData $initializer] {
        set viewer [eval new $class $canvas $switchedOptions -draggable $draggable]
        foreach list [composite::configure $viewer] {
            if {[string compare [lindex $list 0] -interval]==0} {                                 ;# viewer supports interval option
                composite::configure $viewer -interval $pollTime                                          ;# so use current interval
                break                                                                                                        ;# done
            }
        }
        viewer::view $viewer $cells
        manageViewer $viewer 1 -static $static -setx $x -sety $y -setwidth $width -setheight $height -level $level
    }
}
