# copyright (C) 1997-1999 Jean-Luc Fontaine (mailto:jfontain@multimania.com)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

set rcsId {$Id: viewer.tcl,v 1.21 1999/01/04 14:34:28 jfontain Exp $}

class viewer {                                                                             ;# handle viewers related functionalities
    # viewers do not derive from a common interface class but rather support a common set a options through the composite class

    set viewer::(list) {}

    proc viewer {this} {
        lappend viewer::(list) $this
    }

    proc ~viewer {this} {
        variable ${this}traces

        foreach array [array names ${this}traces] {                                                       ;# remove all array traces
            trace vdelete ${array}(updates) w "viewer::update $this $array"
        }
        catch {unset ${this}traces}
        if {[info exists viewer::($this,drop)]} {
            delete $viewer::($this,drop)
        }
        ldelete viewer::(list) $this
    }

    virtual proc supportedTypes {this}

    proc view {this cells} {                                                                  ;# cells is a list of data array cells
        foreach cell [parseCells $this $cells] {
            eval monitorCell $this $cell
            set arrays([lindex $cell 0]) {}                                                                        ;# remember array
        }
        foreach array [array names arrays] {                                        ;# update viewer with current values immediately
            update $this $array
        }
    }

    virtual proc monitorCell {this array row column}

    proc parseCells {this cells} {
        if {[llength $cells]==0} return
        set parsed {}
        foreach cell $cells {
            parse $cell array row column type
            if {[lsearch -exact [supportedTypes $this] $type]<0} {
                lifoLabel::flash $::messenger "cannot display data of type $type"
                bell
                return
            }
            lappend parsed [list $array $row $column]
        }
        return $parsed
    }

    proc parse {dataCell arrayName rowName columnName typeName} {
        upvar $arrayName array $rowName row $columnName column $typeName type

        if {![regexp {^(.*)\(([0-9]+),([0-9]+)\)$} $dataCell dummy array row column]||($row<0)||($column<0)} {
            error "\"$dataCell\" is not a valid array cell"
        }
        set type [set ${array}($column,type)]
    }

    proc updateInterval {value} {                                             ;# static procedure for updating all viewers intervals
        foreach viewer $viewer::(list) {
            catch {composite::configure $viewer -interval $value}                 ;# some viewers do not support the interval option
        }
    }

    proc label {array row column} {   ;# derive a label from cell using the tabular data index columns. array name must be qualified
        if {[catch {set ${array}(indexColumns)} columns]} {                                                      ;# no index columns
            set columns 0                                                                 ;# use first column as single index column
        }
        set label {}
        foreach index $columns {                                                                      ;# use index columns for label
            if {[catch {set ${array}($row,$index)} value]} {
                append label {? }               ;# cell may no longer exist if originating from a save file, give user some feedback
            } else {
                append label "$value "
            }
        }
        append label [set ${array}($column,label)]
        return $label
    }

    virtual proc update {this array args}                                                         ;# update display using cells data

    proc registerTrace {this array} {
        variable ${this}traces

        if {![info exists ${this}traces($array)]} {                                                          ;# if not monitored yet
            trace variable ${array}(updates) w "viewer::update $this $array"                  ;# refresh as module data gets updated
            set ${this}traces($array) 0                                                      ;# initialize number of traces on array
        }
        incr ${this}traces($array)                                                        ;# keep track of number of traces on array
    }

    proc unregisterTrace {this array} {
        variable ${this}traces

        if {[incr ${this}traces($array) -1]<=0} {
            trace vdelete ${array}(updates) w "viewer::update $this $array"                             ;# trace is no longer needed
            unset ${this}traces($array)
        }
    }

    virtual proc cells {this}

    virtual proc initializationConfiguration {this} { ;# configuration with switch / value option pairs for initialization from file
        return {}
    }

    proc setupDropSite {this path} {                                 ;# allow dropping of data cells, viewer mutation or kill action
        set viewer::($this,drop) [new dropSite -path $path -formats {DATACELLS VIEWER KILL} -command "viewer::handleDrop $this"]
    }

    proc handleDrop {this} {
        if {![catch {set dragSite::data(DATACELLS)} data]} {
            view $this $data
        } elseif {![catch {set dragSite::data(VIEWER)} data]} {
            mutate $this $data
        } elseif {[info exists dragSite::data(KILL)]} {
            delete $this                                                                                           ;# self destructs
        }
    }

    proc mutate {this class} {
        if {[string compare $class [classof $this]]==0} return                       ;# no need to create a viewer of the same class
        set viewer [eval new $class $::canvas -draggable [composite::cget $this -draggable]]
        foreach list [composite::configure $viewer] {
            if {[string compare [lindex $list 0] -interval]==0} {                                 ;# viewer supports interval option
                composite::configure $viewer -interval $::pollTime                                        ;# so use current interval
                break                                                                                                        ;# done
            }
        }
        view $viewer [cells $this]
        foreach {x y width height} [canvasWindowManager::getGeometry $::windowManager $widget::($this,path)] {}
        set level [canvasWindowManager::getStackLevel $::windowManager $widget::($this,path)]
        delete $this                                                                                       ;# delete existing viewer
        # viewer is as destroyable as previously deleted viewer
        manageViewer $viewer 1  -static $::static -setx $x -sety $y -setwidth $width -setheight $height -level $level
    }

}
