static char rcsid[] = "$Id: narray.c,v 1.5 1998/08/21 15:43:15 nickm Exp nickm $";
/*
 *
 *  NArray - a tcl extension for manipulating multidimensional arrays
 *
 *  Author: N. C. Maliszewskyj, NIST Center for Neutron Research, August 1998
 *          P. Klosowski        NIST Center for Neutron Research
 *  Original Author:
 *          S. L. Shen          Lawrence Berkeley Laboratory,     August 1994
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ****************************************************************************
 *
 *
 * This software is copyright (C) 1994 by the Lawrence Berkeley Laboratory.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

#include <assert.h>
#include <ctype.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#ifdef NO_TCL_H
#error Cannot find tcl.h
#else
#include <tcl.h>
#endif
#include "narray.h"
#include "narrayInt.h"

#define state narray_compile_state

int NArrayObjectCmd(ClientData, Tcl_Interp*, int, char**);
static void NArrayFree(ClientData data);

static int narray_flags;

/*
 * Assorted helper functions
 */
void
pdims (int idx[], NArray *array) {
  int i;
  for (i=0;i<array->n_dims;i++) {
    printf(" %2d",idx[i]);
  }
  printf("\n");
}

int 
IndexFromOffset (int offset, int idx[], NArray *array) {
  int i, result;

  for (i=array->n_dims-1; i >= 0; i--) {
    result = (offset % array->dim_length[i]) ;
    offset = (offset - result) / array->dim_length[i] ;
    idx[i] = result;
  }
  return 0; 
}

unsigned int 
OffsetFromIndex (int idx[], NArray *array) {
  int i;
  unsigned int offset;

  offset = idx[0];
  for (i=1;i<array->n_dims; i++) {
    offset = idx[i] + array->dim_length[i] * offset;
  }
  return offset;
}

/* Compute linear offset from indices passed as array of strings
 * using Tcl to convert from ASCII->binary
 */
unsigned int
linearIndexFromStr(Tcl_Interp* interp, char * argv[], NArray *array){
  int i, lastDim = array->n_dims-1;
  int offset,ind;

  if (Tcl_GetInt(interp, argv[0], &offset) != TCL_OK)
    return -1;
  if ((offset < 0) || (offset >= array->dim_length[0])){
    Tcl_AppendResult(interp,"Index value ", argv[0]," is out of bounds",0);
    return -1;
  }
  for (i=1; i <= lastDim; i++) {
    if (Tcl_GetInt(interp, argv[i], &ind) != TCL_OK) 
      return -1;
    if ((ind < 0) || (ind >= array->dim_length[i])){
      Tcl_AppendResult(interp,"Index value ", argv[i]," is out of bounds",0);
      return -1;
    }
    offset = ind + array->dim_length[i]* offset;
  }
  return offset;
}

/*
 * NArrayCollapse - "Collapse" one dimension of the narray by summing over
 *                  all elements along that dimension.
 *
 *                  The result is an narray with one fewer dimension, each
 *                  element of which contains the sums computed in the
 *                  process of collapsing that dimension.
 *
 *                  Check out the cartoon below:
 *         
 *
 *              1  2  3  4            10  
 *              1  2  3  4            10
 *              1  2  3  4    ====>   10   direction = 0 1
 *              1  2  3  4            10
 *              1  2  3  4            10
 *
 *                  | 
 *                  | direction = 1 0
 *                  |
 *                  V
 *              5 10 15 20
 */
int 
NArrayCollapse(Tcl_Interp * interp, int argc, char *argv[], NArray *array1) {
  Tcl_CmdInfo cmdInfo;
  NArray *array2;
  int i, j, listArgc, ndirs, tempval, resp;
  /* Dimensions & indices - should probably size these guys dynamically */
  int direction[10], dim1[10], dim2[10]; 
  char **listArgv;

  if (argc < 2) {
    Tcl_AppendResult(interp,"Usage: arrayname collapse direction result",
		  TCL_STATIC);
    return TCL_ERROR;
  }

  /* Set up Narray2 */
  if (Tcl_GetCommandInfo(interp,argv[1],&cmdInfo) != 1) {
    Tcl_AppendResult(interp,"Cannot get info for NArray ",
		     "\"",argv[1],"\"",(char *)NULL);
    return TCL_ERROR;
  }

  if (((array2 = (NArray*) (cmdInfo.clientData)) == NULL) 
      || (array2->closure.interp!= interp)) {
    Tcl_AppendResult(interp,"\"",argv[1],"\""," not recognized as an NArray",
		     (char *)NULL);
    return TCL_ERROR;
  }

  /* Process direction list */
  if (Tcl_SplitList(interp,argv[0],&listArgc,&listArgv) != TCL_OK) {
    Tcl_SetResult(interp,"Enter direction as list of integers",TCL_STATIC);
    ckfree(listArgv);
    return TCL_ERROR;
  }

  if (listArgc != array1->n_dims) {
    Tcl_SetResult(interp,
		  "Wrong number of dimensions specified for integration",
		  TCL_STATIC);
    ckfree(listArgv);
    return TCL_ERROR;
  }

  ndirs = 0;
  memset(direction, 0, sizeof(direction));
  memset(dim1,      0, sizeof(dim1));
  memset(dim2,      0, sizeof(dim2));
  for (i=0;i<listArgc;i++) {
    if ((resp = Tcl_GetInt(interp,listArgv[i],&tempval)) != TCL_OK) {
      ckfree(listArgv);
      return resp;
    }
    direction[i] = (tempval ? 1: 0);
    if (tempval) ndirs++;
  }
  ckfree(listArgv);

  if (ndirs != 1) {
    Tcl_SetResult(interp,"Specify a single direction for integration",
		  TCL_STATIC);
    return TCL_ERROR;
  }

  /* 
   * Resize narray2 appropriately - for the moment, require that
   * narray2 have one fewer dimension than narray1 
   */
  if (array2->n_dims != array1->n_dims-1) {
    Tcl_AppendResult(interp,"\"",argv[2],"\"",
		     " must have one fewer dimension than ","\"",
		     argv[0],"\"",(char) NULL);
    return TCL_ERROR;
  }
  
  /*
   * Map the uncollapsed dimensions to the resulting narray
   */
  ndirs = 0;
  tempval = 1;
  for (i=0;i<array1->n_dims;i++) {
    if (!direction[i]) { 
      array2->dim_length[ndirs] = array1->dim_length[i];
      tempval *= array2->dim_length[ndirs];
      ndirs++;
    }
  }

  /*
   * Check storage requirements of resulting narray 
   *        - reallocate if necessary
   *        - initialize all elements to zero
   */
  if (array2->length != tempval) {
    ckfree(array2->storage);
    array2->length = tempval;
    array2->storage =
	    (NArrayFloat*) ckalloc(sizeof(NArrayFloat) * array2->length);
  }
  memset(array2->storage, 0, sizeof(NArrayFloat) * array2->length);
  
  /*
   * Now that we're set up properly, go for the whole enchilada
   *
   * 1) Loop over all elements of array1
   * 2) Calculate current indices of each element from current offset
   * 3) Map indices of array1 element to corresponding array2 element index
   * 4) Calculate array2 offset from supplied indices
   * 5) Sum the two values and store in array2 element
   */
  
  for (i=0;i<array1->length;i++) {
    memset(dim1,      0, sizeof(dim1));
    memset(dim2,      0, sizeof(dim2));
    IndexFromOffset(i,dim1,array1);
    ndirs=0;
    for (j=0;j<array1->n_dims;j++) {
      if (!direction[j]) {
	dim2[ndirs++] = dim1[j];
      }
    }
    tempval = OffsetFromIndex(dim2,array2);
    if (array1->debug & DEBUG_TRACE) {
      printf("Offset = %5d\n",i); 
      printf(" dim1:"); pdims(dim1,array1);
      printf(" dim2:"); pdims(dim2,array2);
      printf(" off = %5d\n",tempval);
    }
    *(array2->storage + tempval) += *(array1->storage + i);
  }
  return TCL_OK;
}

/*
 * NArrayImportExport - 
 *                Extract a subset of an narray and store it in another.
 *                or
 *                Copy the contents of an narray into a subset of the parent
 *
 *                Todo: Need to make special provisions in the case where
 *                we export a subset of fewer dimensions than the parent.
 *                For example, we should be able to extract a vector (dim=1) 
 *                from a matrix (dim=2).
 *
 *                Check out the cartoon below:
 *            
 *
 *                 1  2  3  4            
 *                 1  2  3  4            1 2 3  offset = 0 0
 *                 1  2  3  4    ====>   1 2 3  size   = 3 3
 *                 1  2  3  4            1 2 3
 *                 1  2  3  4            
 *
 *                     | 
 *                     | offset = 0 0
 *                     | size   = 1 0
 *                     V
 *                 1  2  3  4
 */
int 
NArrayImportExport(Tcl_Interp * interp, int argc, char *argv[], 
		   NArray * array1) {
  Tcl_CmdInfo cmdInfo;
  NArray *array2;
  int i, j, listArgc, ndirs, tempval, resp, proceed;
  int len, export;
  /* Dimensions & indices - should probably size these guys dynamically */
  int offset[10], size[10], dim1[10], dim2[10];
  char **listArgv;

  if (argc < 1) return TCL_ERROR;
  len = strlen(argv[0]);
  if (!strncmp(argv[0],"export",len)) {
    export = 1;
    if (argc != 4) {
      Tcl_SetResult(interp,"Usage: narrayname export offset size result",
		    TCL_STATIC);
      return TCL_ERROR;
    }
  } else if (!strncmp(argv[0],"import",len)) {
    export = 0;
    if (argc != 3) {
      Tcl_SetResult(interp,"Usage: narrayname export offset source",
		    TCL_STATIC);
      return TCL_ERROR;
    }
  } else {
    Tcl_SetResult(interp,"Valid options: import export",TCL_STATIC);
    return TCL_ERROR;
  }

  /* Set up Narray2 */
  if (Tcl_GetCommandInfo(interp,argv[argc-1],&cmdInfo) != 1) {
    Tcl_AppendResult(interp,"Cannot get info for NArray ",
		     "\"",argv[argc-1],"\"",(char *)NULL);
    return TCL_ERROR;
  }

  if (((array2 = (NArray*) (cmdInfo.clientData)) == NULL) 
      || (array2->closure.interp!= interp)) {
    Tcl_AppendResult(interp,"\"",argv[argc-1],"\"",
		     " not recognized as an NArray",
		     (char *)NULL);
    return TCL_ERROR;
  }

  if (array1->n_dims != array2->n_dims) {
    if (export) {
      Tcl_AppendResult(interp,"exported narray must have same ",
		     "number of dimensions as parent",0);
    } else {
      Tcl_AppendResult(interp,"imported narray must have same ",
		     "number of dimensions as parent",0);
    }
    return TCL_ERROR;
  }

  memset(offset, 0, sizeof(offset));
  memset(size,   0, sizeof(size));
  memset(dim1,   0, sizeof(dim1));
  memset(dim2,   0, sizeof(dim2));

  /* Process offset list */
  if (Tcl_SplitList(interp,argv[1],&listArgc,&listArgv) != TCL_OK) {
    Tcl_SetResult(interp,"Enter offset as list of integers",TCL_STATIC);
    ckfree(listArgv);
    return TCL_ERROR;
  }

  if (listArgc != array1->n_dims) {
    Tcl_SetResult(interp,
		  "Specify complete offset index for subset extraction",
		  TCL_STATIC);
    ckfree(listArgv);
    return TCL_ERROR;
  }

  for (i=0;i<listArgc;i++) {
    if ((resp = Tcl_GetInt(interp,listArgv[i],&tempval)) != TCL_OK) {
      ckfree(listArgv);
      return resp;
    }
    offset[i] = tempval;
  }
  ckfree(listArgv);

  /* Process size list */
  if (export) {
    if (Tcl_SplitList(interp,argv[2],&listArgc,&listArgv) != TCL_OK) {
      Tcl_SetResult(interp,"Enter size as list of integers",TCL_STATIC);
      ckfree(listArgv);
      return TCL_ERROR;
    }

    if (listArgc != array1->n_dims) {
      Tcl_SetResult(interp,
		    "Specify complete size indices for subset extraction",
		    TCL_STATIC);
      ckfree(listArgv);
      return TCL_ERROR;
    }

    for (i=0;i<listArgc;i++) {
      if ((resp = Tcl_GetInt(interp,listArgv[i],&tempval)) != TCL_OK) {
	ckfree(listArgv);
	return resp;
      }
      size[i] = tempval;
    }
    ckfree(listArgv);

    /* Now do some bounds checking */
    for (i=0;i<array1->n_dims;i++) {
      if (array1->dim_length[i] < offset[i] + size[i]) {
	Tcl_SetResult(interp,"A subset of the specified size overflows bounds",
		      TCL_STATIC);
	return TCL_ERROR;
      }
    }

    /*
     * Check storage requirements of resulting narray 
     *        - reallocate if necessary
     *        - update dimension lengths
     *        - initialize all elements to zero
     */

    tempval = 1;
    for (i=0;i<array2->n_dims;i++) tempval *= size[i];
    if (array2->length != tempval) {
      ckfree(array2->storage);
      array2->length = tempval;
      array2->storage =
	(NArrayFloat*) ckalloc(sizeof(NArrayFloat) * array2->length);
    }
    for(i=0;i<array2->n_dims;i++) array2->dim_length[i] = size[i];
    memset(array2->storage, 0, sizeof(NArrayFloat) * array2->length);
  } else {
    for (i=0;i<array2->n_dims;i++) {
      size[i] = array2->dim_length[i];
    }

    /* Now do some bounds checking */
    for (i=0;i<array1->n_dims;i++) {
      if (array1->dim_length[i] < offset[i] + size[i]) {
	Tcl_SetResult(interp,"A subset of the specified size overflows bounds",
		      TCL_STATIC);
	return TCL_ERROR;
      }
    }
  }

  /*
   * Now that we're set up properly, go for the whole enchilada
   *
   * 1) Loop over all elements of array1
   * 2) Calculate current indices of each element from current offset
   * 3) Map indices of array1 element to corresponding array2 element index
   * 4) Check bounds of array2 index against offset and size
   * 5) Calculate array2 offset from supplied indices
   * 6) Copy contents of array1 element into array2 if exporting, the reverse
   *    if importing
   */
  
  for (i=0;i<array1->length;i++) {
    IndexFromOffset(i,dim1,array1);

    proceed = 1;
    for (j=0;j<array1->n_dims;j++) {
      dim2[j] = dim1[j] - offset[j];
      proceed *= (((dim2[j] >= 0) && (dim2[j] < size[j])) ? 1 : 0);
    }
    if (array1->debug & DEBUG_TRACE) {
      printf("Offset = %d\n",i);
      printf("  proc: %d\n",proceed);
      printf("  offs:"); pdims(offset,array1);
      printf("  size:"); pdims(size,array1);
      printf("  dim1:"); pdims(dim1,array1);
      printf("  dim2:"); pdims(dim2,array2);
    }
    if (proceed) {
      tempval = OffsetFromIndex(dim2,array2);
      if (export) {
	*(array2->storage + tempval) = *(array1->storage + i);
      } else {
	*(array1->storage + i) = *(array2->storage + tempval);
      }
    }
  }

  return TCL_OK;
}


/*
 * narray create name dim0 ?dim1? ...
 *
 * Create a new array named "name" with dimensions of length
 * dim0 ... dimN.  Creates a new command "name" that handles
 * operations on the narray.  Returns "name".
 *
 */

static int NArrayCmd(ClientData data, Tcl_Interp* interp,
		     int argc, char* argv[])
{
  NArray* array;
  int i, n, len;

  if (argc < 2) {
    Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
		     " option ?args ... ?\"", 0);
    return TCL_ERROR;
  }
  len = strlen(argv[1]);
  if (!strncmp(argv[1], "create", len)) {
    if (argc < 4) {
      Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
		       " create name dim0 ?dim1? ...\"", 0);
      return TCL_ERROR;
    }
    n = argc - 3;
    array = (NArray*) ckalloc(sizeof(NArray) + (n - 1) * sizeof(int));
    array->n_dims = n;
    array->length = 1;
    array->debug = 0;
    array->closure.interp = interp;
    array->closure.vars = 0;
    array->closure.double_table = 0;
    array->closure.n_doubles = 0;
    array->closure.id_table = 0;
    array->closure.alloced_ids = 0;
    for (i = 0; i < n; i++) {
      if (Tcl_GetInt(interp, argv[3+i], &array->dim_length[i]) != TCL_OK) {
	ckfree(array);
	return TCL_ERROR;
      }
      array->length *= array->dim_length[i];
    }
    array->storage =
      (NArrayFloat*) ckalloc(sizeof(NArrayFloat) * array->length);
    memset(array->storage, 0, sizeof(NArrayFloat) * array->length);
    Tcl_CreateCommand(interp, argv[2], NArrayObjectCmd,
		      (ClientData) array, NArrayFree);
    Tcl_SetResult(interp, argv[2], TCL_VOLATILE);
    return TCL_OK;
  }

  Tcl_SetResult(interp,"Options: create",TCL_STATIC);
  return TCL_ERROR;
}

static void NArrayFree(ClientData data)
{
  NArray* array = (NArray*) data;
  int i;
  if (array->closure.vars) {
    ckfree(array->closure.vars);
  }
  if (array->closure.double_table) ckfree(array->closure.double_table);
  if (array->closure.id_table) {
    for (i = 0; i < array->closure.alloced_ids; i++)
      if (array->closure.id_table[i].flags & NARRAY_SLOT_IN_USE)
	ckfree(array->closure.id_table[i].id);
    ckfree(array->closure.id_table);
  }
  ckfree(array->storage);
  ckfree(array);
}

/*
 * NArrayObjectCmd is the command that handles operations on
 * a narray.
 *
 * $na aref n0 ?n1? ...
 *    Return the value at (n0, n1, ..., nN).
 *
 * $na aset n0 ?n1? ... val
 *    Set (n0, n1, ..., nN) to val.
 *
 * $na vref var
 *    Return the value of var
 *
 * $na vset var val
 *    Set var to val.
 *
 * $na vars
 *    Return a list of variables in this narray.
 *
 * $na map code ?{var1 narray1} ...
 *    Map code over the each element, making narray1 available in var1.
 *
 * $na dimensions
 *    Return the length of each dimension in a list.
 *
 * $na collapse direction narray_result
 *    Sum narray elements along specified direction, returning the result
 *    in an narray with one fewer dimension.
 *
 * $na status
 *    Return some information about the array.
 */

int NArrayObjectCmd(ClientData data, Tcl_Interp* interp,
		    int argc, char* argv[])
{
  int len, n, i;
  int k;
  NArrayFloat* result;
  NArray* array = (NArray*) data;
  char buf[TCL_DOUBLE_SPACE];
  double d;
    
  if (argc == 1) {
    Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
		     " option ?args?\"", 0);
    return TCL_ERROR;
  }
  len = strlen(argv[1]);
  if (!strncmp(argv[1], "aref", len)) {
    if (argc < 3) {
      Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
		       " index n0 ?n1? ...\"", 0);
      return TCL_ERROR;
    }
    if ((argc - 2) != array->n_dims) {
      sprintf(buf, "%d", array->n_dims);
      Tcl_AppendResult(interp, "must specify all ",buf," indices", 0);
      return TCL_ERROR;
    }
    if ((n = linearIndexFromStr(interp,&argv[2],array)) == -1)
      return TCL_ERROR;
    if (n >= array->length) {
      Tcl_AppendResult(interp, "indexes are out of range", 0);
      return TCL_ERROR;
    }
    Tcl_PrintDouble(interp, array->storage[n], buf);
    Tcl_SetResult(interp, buf, TCL_VOLATILE);
    return TCL_OK;
  }
  if (!strncmp(argv[1], "aset", len)) {
    if (argc < 4) {
      Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
		       " set n0 ?n1? ... val\"", 0);
      return TCL_ERROR;
    }
    if ((argc - 3) != array->n_dims) {
      sprintf(buf, "%d", array->n_dims);
      Tcl_AppendResult(interp, "must specify all ", buf," indices",0);
      return TCL_ERROR;
    }
    if (Tcl_GetDouble(interp, argv[argc - 1], &d) != TCL_OK)
      return TCL_ERROR;

    if ((n = linearIndexFromStr(interp,&argv[2],array)) == -1)
      return TCL_ERROR;
    if (n >= array->length) {
      Tcl_AppendResult(interp, "indexes are out of range", 0);
      return TCL_ERROR;
    }
    array->storage[n] = d;
    return TCL_OK;
  }
  if (!strncmp(argv[1], "vref", len)) {
    if (argc != 3) {
      Tcl_AppendResult(interp, "wrong # args, should be \"",
		       argv[0], " vref var\"", 0);
      return TCL_ERROR;
    }
    for (i = 0; i < array->closure.alloced_ids; i++) {
      if (((array->closure.id_table[i].flags & NARRAY_SLOT_IN_USE)
	   && (array->closure.id_table[i].flags & NARRAY_SLOT_VARIABLE)
	   && !strcmp(argv[2], array->closure.id_table[i].id))) {
	Tcl_PrintDouble(interp, array->closure.vars[i], buf);
	Tcl_SetResult(interp, buf, TCL_VOLATILE);
	return TCL_OK;
      }
    }
    Tcl_AppendResult(interp, "variable \"", argv[2], "\" does not exist",
		     0);
    return TCL_ERROR;
  }
  if (!strncmp(argv[1], "vset", len)) {
    if (argc != 4) {
      Tcl_AppendResult(interp, "wrong # args, should be \"",
		       argv[0], " vset var val\"", 0);
      return TCL_ERROR;
    }
    if (Tcl_GetDouble(interp, argv[3], &d) != TCL_OK)
      return TCL_ERROR;
    i = NArray_CreateClosureIdSlot(&array->closure, argv[2]);
    array->closure.vars[i] = d;
    array->closure.id_table[i].flags |= NARRAY_SLOT_VARIABLE;
    Tcl_PrintDouble(interp, d, buf);
    Tcl_SetResult(interp, buf, TCL_VOLATILE);
    return TCL_OK;
  }
  if (!strncmp(argv[1], "vars", len)) {
    if (argc != 2) {
      Tcl_AppendResult(interp, "wrong # args, should be \"",
		       argv[0], " vars\"", 0);
      return TCL_ERROR;
    }
    Tcl_ResetResult(interp);
    for (i = 0; i < array->closure.alloced_ids; i++) {
      if (((array->closure.id_table[i].flags & NARRAY_SLOT_IN_USE)
	   && (array->closure.id_table[i].flags & NARRAY_SLOT_VARIABLE))) {
	Tcl_AppendElement(interp, array->closure.id_table[i].id);
      }
    }
    return TCL_OK;
  }
  if (!strncmp(argv[1], "map", len)) {
    Code* code;
    int n, result;
    char** split_argv;
    Tcl_CmdInfo info;
	
    if (argc < 3) {
      Tcl_AppendResult(interp, "wrong # args, should be\"", argv[0],
		       " map code ?{var0 arrary0} ...?\"", 0);
      return TCL_ERROR;
    }
    result = TCL_OK;
    array->closure.n_arrays = 0;
    for (i = 0; i < (argc - 3); i++) {
      if (Tcl_SplitList(interp, argv[3+i], &n, &split_argv) != TCL_OK) {
	result = TCL_ERROR;
	goto map_error;
      }

      if (!Tcl_GetCommandInfo(interp, split_argv[1], &info)
	  || info.proc != NArrayObjectCmd) {
	Tcl_SetResult(interp,"Bad narray argument.",TCL_STATIC);
	ckfree(split_argv);
	result = TCL_ERROR;
	goto map_error;
      }
      assert(i < NARRAY_MAX_BOUND_ARRAYS);
      array->closure.array_table[i].array = (NArray*) info.clientData;
      array->closure.array_table[i].name = ckalloc(strlen(split_argv[0])
						   + 1);
      strcpy(array->closure.array_table[i].name, split_argv[0]);
      ckfree(split_argv);
      array->closure.n_arrays = i + 1;
    }
    code = NArray_Compile(array, argv[2]);

    if (code == 0) {
      Tcl_AppendResult(interp, state->error_msg, " in compiling \"",
		       argv[2], "\"", 0);
      result = TCL_ERROR;
      goto map_error;
    }
    if (array->debug & DEBUG_DUMP) {
      printf("Compiled code:\n");
      NArray_PrintCode(array, code);
    }
    if (NArray_ApplyCode(array, code) == 0) {
      Tcl_AppendResult(interp, array->errmsg, 0);
      result = TCL_ERROR;
    }
    NArray_FreeCode(code);

  map_error:
    for (i = 0; i < array->closure.n_arrays; i++)
      ckfree(array->closure.array_table[i].name);
    return result;
  }
  if (!strncmp(argv[1], "dimensions", len)) {
    if (argc != 2) {
      Tcl_AppendResult(interp, "wrong # args, should be\"", argv[0],
		       " dimensions\"", 0);
      return TCL_ERROR;
    }
    Tcl_ResetResult(interp);
    for (i = 0; i < array->n_dims; i++) {
      sprintf(buf, "%d", array->dim_length[i]);
      Tcl_AppendElement(interp, buf);
    }
    return TCL_OK;
  }
  if (!strncmp(argv[1], "status", len)) {
    if (argc != 2) {
      Tcl_AppendResult(interp, "wrong # args, should be\"", argv[0],
		       " dimensions\"", 0);
      return TCL_ERROR;
    }
    Tcl_ResetResult(interp);
    sprintf(buf, "%8.2fKB used, debug %d",
	    (sizeof(NArray) + array->length * sizeof(NArrayFloat)
	     + (array->n_dims - 1) * sizeof(int)) / 1024.0,
	    array->debug);
    Tcl_SetResult(interp, buf, TCL_VOLATILE);
    return TCL_OK;
  }
  if (!strncmp(argv[1],"collapse",len)) {
    if (argc < 4) {
      Tcl_AppendResult(interp,"Usage: ","\"",argv[0],"\"",
		       " collapse direction narray_result", 0);
      return TCL_ERROR;
    }
    return NArrayCollapse(interp, (argc-2), (argv+2), array);
  }
  if (!strncmp(argv[1],"export",len)) {
    if (argc < 5) {
      Tcl_AppendResult(interp,"Usage: ","\"",argv[0],"\"",
		       " export offset size narray_result", 0);
      return TCL_ERROR;
    }
    return NArrayImportExport(interp, (argc-1), (argv+1), array);
  }
  if (!strncmp(argv[1],"import",len)) {
    if (argc < 4) {
      Tcl_AppendResult(interp,"Usage: ","\"",argv[0],"\"",
		       " import offset narray_source", 0);
      return TCL_ERROR;
    }
    return NArrayImportExport(interp, (argc-1), (argv+1), array);
  }
  if (!strncmp(argv[1], "debug", len)) {
    if (argc != 3) {
      Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
		       " debug level\"", 0);
      return TCL_ERROR;
    }
    if (Tcl_GetInt(interp, argv[2], &array->debug) != TCL_OK)
      return TCL_ERROR;
    return TCL_OK;
  }

  Tcl_AppendResult(interp, "unknown option \"", argv[1],
		   "\", should be one of: ",
		   "aref, aset, vref, vset, vars,",
		   "collapse, export,"
		   "dimensions, map, status, debug",
		   (char *)NULL);
  return TCL_ERROR;
}

int Narray_Init(Tcl_Interp* interp)
{
  char* lib_dir;
  NArray_CodeInit(interp);
  NArray_FunctionsInit(interp);
  Tcl_CreateCommand(interp, "narray", NArrayCmd, 0, 0);
  if ((lib_dir = getenv("NARRAY_LIBRARY")) == 0)
    lib_dir = LIBRARY_DIR;
  if (Tcl_SetVar(interp, "auto_path", lib_dir,
		 (TCL_APPEND_VALUE|TCL_LIST_ELEMENT|TCL_LEAVE_ERR_MSG
		  |TCL_GLOBAL_ONLY)) == 0)
    return TCL_ERROR;
  if (Tcl_SetVar(interp, "narray_library", lib_dir,
		 TCL_LEAVE_ERR_MSG|TCL_GLOBAL_ONLY) == 0)
    return TCL_ERROR;
  if (Tcl_SetVar(interp, "narray_version", NARRAY_VERSION,
		 TCL_LEAVE_ERR_MSG|TCL_GLOBAL_ONLY) == 0)
    return TCL_ERROR;
  return TCL_OK;
}
