###############################################################################
###############################################################################
#                               Getleft
###############################################################################
###############################################################################
# copyright (C) 1999 Andres Garcia Garcia - ornalux@redestb.es
# Distributed under the terms of the GPL
###############################################################################
###############################################################################
# Version 0.7.6
###############################################################################
###############################################################################
# This program recursively downloads Web sites, as it goes, it changes absolute
# paths in the links to relative ones, and links to active content pages get
# changed to the resulting pages.
###############################################################################
#
# The following lines look for the most recent wish version in your path,
# keep in mind this program only works with version 8.1 or newer
#
# \
if which wish8.2>/dev/null; then
# \
    exec wish8.2 -f $0
# \
elif which wish8.1>/dev/null; then
# \
    exec wish8.1 -f $0
# \
else
# \
    exec wish -f $0
# \
fi

wm withdraw .
if {$tcl_version<8.1} {
    tk_messageBox -icon error -type ok -message "This program only works with\n\
            Tcl/Tk v 8.1 or newer, yours is:\n\t $tcl_version" -title Error
    exit
}

# Okay, we made it this far, now we have to check whether the programs Getleft
# depends on are actually within reach: curl, cat, touch and sed.

set dependencies 0
catch {exec curl}
if {[lindex $errorCode 1]=="ENOENT"} {
    tk_messageBox -type ok -icon error -title "No curl" \
     -message "Getleft depends on program\n\tcurl\nPlease check the docs"
    set dependencies 1
}
catch {exec sed}
if {[lindex $errorCode 1]=="ENOENT"} {
    tk_messageBox -type ok -icon error -title "No sed" \
     -message "Getleft depends on the program sed,\nPlease check the docs"
    set dependencies 1
}
catch {exec touch}
if {[lindex $errorCode 1]=="ENOENT"} {
    tk_messageBox -type ok -icon error -title "No touch" \
     -message "Getleft depends on the program touch,\nPlease check the docs"
    set dependencies 1
}
catch {exec cat noSuchFileWithThisFilename}
if {[lindex $errorCode 1]=="ENOENT"} {
    tk_messageBox -type ok -icon error -title "No cat" \
     -message "Getleft depends on the program cat,\nPlease check the docs"
    set dependencies 1
}

if {$dependencies==1} {
    exit
}

# All is well, so we really start

wm deiconify .

###############################################################################
# ShowLinks
#    Show the links found in the last preprocessed page, it's only good for
#    debugging.
###############################################################################
proc ShowLinks {} {
    global nLinks linkFile linkDescription

    for {set i 1} {$i<$nLinks} {incr i} {
        puts "$i: $linkFile($i) $linkDescription($i)"
    }
    return
}

#############################################################################
# SetCoordinates
#    Reads the coordinates of the upper left corner of a window and puts
#    then in 'coord'.
#
# Parameter:
#    windows whose position you want to know, defaults to the main program
#    window
#
# Side effect:
#    coord(x): x coordinate
#    coord(y): y one
#############################################################################
proc SetCoordinates {{cual .} } {
    global coord

    regexp {(?:\+)([0-9]+)(?:\+)([0-9]+)} [wm geometry $cual] \
            nada coord(x) coord(y)

    return
}

###############################################################################
# TidyDir
#    Takes things like ".." and "." from the absolute path.
#
# Parameter:
#    File path.
#
# Rturns:
#    The tidied file path.
###############################################################################
proc TidyDir {path} {
    if {[regexp {\.$} $path]} {
        append path /
    }
    for {set a 1 ; set b 1} {($a>0)||($b>0)} {} {
        set a [regsub -all {/\./} $path {/} path]
        set b [regsub -all {([^./]+/\.\./)} $path {} path]
    }
    for {} {[regsub {^/\.\.} $path {} path]} {} {}

    return $path
}

###############################################################################
# InitializeDownload
#    Takes care of the tasks needed before actually starting to download.
#
# Parameter:
#    pagina: The requested Url.
###############################################################################
proc InitializeDownload {pagina} {
    global directories siteUrl options
    global labelTitles labelMessages
    global relocated

    if {![regexp -nocase {(^http://)(.*)} $pagina nada http url]} {
        set url $pagina
    }
    if {[string first / $url]==-1} {
       set pagina $pagina/
    }

    if {[regexp -nocase {(?:http://)?([^/]+)((?:~[^/]*)?(?:[^\?]*))(?:/)(.*)} \
        $pagina nada siteUrl(www) siteUrl(dir) nombre_fichero]} {
    } else {
        tk_messageBox -type ok -title $labelTitles(initError) -icon error \
            -message $labelMessages(formatError)
        return
    }

    set directories(base) [Dialogos::SelectDirectory]
    if {![string compare $directories(base) ""]} {
        return
    }
    set directories(local) $directories(base)

    if {$nombre_fichero==""} {
        set nombre_fichero index.html
        set esHtml 1
    } else {
        set esHtml [regexp -nocase {html?$} $nombre_fichero]
    }

    if {$esHtml==0} {
        ::Ventana::HeadDownloading $pagina "-"
        if {[regexp -nocase "text/html" $::Ventana::Rizo::meta(content)]} {
            set nombre_fichero [file join $siteUrl(dir) $nombre_fichero]
        }
    } else {
        set nombre_fichero [file join $siteUrl(dir) $nombre_fichero]
    }
    set siteUrl(base) $siteUrl(dir)

    set fichero [OpenFile [TidyNames $nombre_fichero] ]

    if {$fichero!=1} {
        set result [Download $pagina $fichero "-" $esHtml]
        catch {wm withdraw .bajando}
        if {$relocated(ok)==1} {
            set siteUrl(dir)  $relocated(dir)
            set siteUrl(base) $siteUrl(dir)
            set fichero $directories(base)$relocated(file).labelMessages(downloading)
        }
    } else {
        tk_messageBox -type ok -title $labelTitles(initError) -icon error \
                -message [concat $labelMessages(notOpen) $nombre_fichero]
        return
    }

    if {$result==2} {
        tk_messageBox -icon error -type ok -title $labelTitles(error) \
                 -message "$Ventana::Rizo::errorMessage"
        return
    }
    set fichero [file root $fichero]
    if {($result==1)&&($esHtml==0)} {
        file rename -force $fichero $fichero.html
        exec touch $fichero
        return $fichero.html
    }

    return $fichero
}

###############################################################################
# RelativePath
#    The function returns the relative path from the current directory to the
#    root directory of the download.
#
# Parameter:
#    The directory of the file which is going to be processed for links.
#
# Returns:
#    The relative path.
###############################################################################
proc RelativePath {path_absoluto} {
    global directories

    set patron (?:$directories(base)/)(.*)
    regexp $patron $path_absoluto nada path
    if {![info exists path]} return

    set cuenta [regsub -all {/} $path {} ignorar]

    for {set i 0;set rel ""} {$i<$cuenta} {incr i} {
        append rel ../
    }
    return $rel
}

###############################################################################
# Preprocessing
#    Reads the Web page passed as a parameter and proccess it to extract
#    all links to local files and images it has
#
# Parameters:
#    file: file which contains the page to process
#
# Side efects:
#    'linkFile' keeps the links
#    'linkDescription' the descripcion of the link
###############################################################################
proc Preprocessing {file} {
    global nLinks linkFile linkDescription siteUrl directories
    global options filesDone getleftState
    global labelTitles labelMessages labelDialogs

    if [string match $file ""] return

#puts "Fichero a preprocesar: $file"

    set filesDone($file) 1
    set directories(local) [file dirname $file]

    if {$getleftState(filesChosen)==1} {
        Ventana::ShowWindow  process
        Ventana::FileStrings $file
    }

    if {[file exists $file.orig]} {
        set fichero $file.orig
    } else {
        set fichero $file
    }
    set nLinks  1
    set enlaces_absolutos 0
    set hay_url 0
    if [catch {open $fichero r} leer] {
        tk_messageBox -title $labelTitles(preError) -icon error\
            -message [concat $leer\n$labelMessages(notOpen)\n$fichero]
        set nLinks 0
        return
    }
    for {set cosaOld "aaaa" ; set rep 0; set cosa [gets $leer]} \
            {(![eof $leer]) || ([string compare $cosa ""])} \
            {if {![eof $leer]} {append cosa " " [gets $leer] " "} } {
#puts "\nCosa: $cosa"
       if {$cosaOld==$cosa} {
            incr rep
            if {$rep==10} {
                break
            }
        } else {
            set rep 0
            set cosaOld $cosa
        }
        if {[regexp {^\ ?<!} $cosa]} {
            while {![regexp {(?:>)(.*)} $cosa nada cosa]} {
                append cosa " " [gets $leer]
            }
            continue
        }
        if {[regexp -nocase {^\ ?<script} $cosa]} {
            while {![regexp -nocase {(?:/script>)(.*)} $cosa nada cosa]} {
                append cosa " " [gets $leer]
            }
            continue
        }
        if {![regexp -nocase {((<)(a\ $|a[^>]+h|link|img|frame|area|!|script)(.*))} $cosa cosa]} {
            set cosa ""
            continue
        }
        if [regexp -nocase  \
            {((?:<a[^>]+href *)|(?:<img[^>]* src *)|(?:<link[^>]+href *)|(?:<frame[^>]+src *))((?:= *)(?:\")?([^#\"\ >]+))(?:.*?>)(.*)} \
			    $cosa nada diff alt nombre_fichero cosa] {
#puts "\nNombre fichero: $nombre_fichero\n"
            if [info exists repetidos($nombre_fichero)] continue
            if {[regexp -nocase {^mailto:|^news:} $nombre_fichero]} continue
            if {([regexp {\?} $nombre_fichero])&&($options(cgi)==0)} continue
            if {[regexp {<!} $diff]} continue
		    if {[regexp -nocase {(javascript)(.*)(\(\))} $nombre_fichero]} continue
            if [info exists filesDone([TidyNames [TidyDir [file join $directories(base)$siteUrl(dir) $nombre_fichero]]])] {
                continue
            }
            if [regexp -nocase {(?:http:)(/[^/].*)} $nombre_fichero nada nombre_fichero] {
                set hay_url 1
            }
            if {([string first :// $nombre_fichero]!=-1 ) } {
                regexp {(?:://)([^/]+)(?:/)?(.*)} $nombre_fichero nada direccion_www nombre_fichero
                if ![info exists direccion_www] {
                    catch {error "Invalid url" SERVER \
                            "Server \"$labelMessages(noWWW)\" 1"}
                    continue
                }
                if {[string compare [string tolower [string trimright $siteUrl(www) {/}]] [string tolower $direccion_www]]} continue
                set nombre_fichero /$nombre_fichero
                set hay_url 1
                set enlaces_absolutos 1
            }
            if [regexp -nocase {href} $diff] {
                if {($options(dir)==0)&&($siteUrl(base)!="")} {
                    if {![regexp -nocase "^(/)([string trimleft "$siteUrl(base)" "/"])"  \
                        [TidyDir [file join $siteUrl(dir) $nombre_fichero]]]} {
                        continue
                    }
                }
                if [regexp -nocase {^<area} $diff] {
                    set linkDescription($nLinks) "$labelDialogs(map)"
                } elseif {[regexp -nocase {^<link} $diff]} {
                    set linkDescription($nLinks) "$labelDialogs(css)"
                } else {
                    while {![regexp -nocase {(.*?)(</a *>)} $cosa nada descripcion]} {
                        if [eof $leer] break
                        append cosa " " [gets $leer]
                    }
                    set linkDescription($nLinks) [TidyDescription $descripcion]
                }
            } elseif [regexp -nocase {frame} $diff] {
                set linkDescription($nLinks) "$labelDialogs(frame): $nombre_fichero"
            } else {
                if {![regexp -nocase {(?:ALT *= *\")(.*?)(\")} $alt nada descrip]} {
                    if {![regexp -nocase {(?:src *= *\")(.*?)(\")} $alt nada descrip]} {
                        set descrip $nombre_fichero
                    }
	            }
                set linkDescription($nLinks) "$labelDialogs(image): $descrip"
            }
            set linkFile($nLinks) [TidyDir $nombre_fichero]
            set repetidos($nombre_fichero) 1
            if  [regexp {^/} $nombre_fichero] {
                set enlaces_absolutos 1
                set enlaces_completos($nLinks) $diff$nombre_fichero
            }
            incr nLinks
        } else {
            regexp -nocase {(?:[^<]*)(<.*)} $cosa nada cosa
        }
    }
    close $leer
#ShowLinks
    ChangePage $file $hay_url

    return
}

###############################################################################
# TidyNames
#    Removes from the name and path of a files things like '?' '~' '+' '-'
#
# Returns
#    The filename without those characters.
###############################################################################
proc TidyNames {nombre} {

    regsub -all {~}  $nombre {} nombre
    regsub -all {\*} $nombre {} nombre
    if {[regexp {(?:^.:)(.*)} $nombre nada filename]} {
        regsub -all {:}  $filename {} filename
	  set nombre $filename
    } else {
        regsub -all {:} $nombre {} nombre
    }
    if {[regexp {([^\?]+)(?:\?)(.*)} $nombre nada uno dos]} {
        regsub -all {\?} $dos {} dos
        regsub -all {\+} $dos {} dos
        regsub -all {/}  $dos {} dos
        regsub -all {\\} $dos {} dos
        set nombre $uno$dos
    }
    return $nombre
}

###############################################################################
# TidyDescription
#    Gets the description of the link and translates it for human eyes.
#
# Parameter:
#    The description to translate
#
# Returns:
#    The description translated
###############################################################################
proc TidyDescription {descripcion} {
    global labelDialogs

    if [regexp -nocase {^(<img)} $descripcion] {
        if {![regexp {(?:>)(.+)} $descripcion nada descriptionText]} {
            if {![regexp -nocase {(?:ALT *= *\")([^\"]+)} $descripcion nada descrip]} {
                regexp -nocase {(?:src *= *\"?)([^\"\ ]+)} $descripcion nada descrip
            }
            set descripcion "$labelDialogs(linkImage): $descrip"
        } else {
            set descripcion $descriptionText
        }
    }
    regsub -all {<.*?>}      $descripcion {}   descripcion
    regsub -all {(&quot);?}  $descripcion {\"} descripcion
    regsub -all {&#34;?}     $descripcion {\"} descripcion
    regsub -all {&#39;?}     $descripcion {'}  descripcion
    regsub -all {&#153;?}    $descripcion {TM} descripcion
    regsub -all {&#191;?}    $descripcion {} descripcion
    regsub -all {&lt;?}      $descripcion {<}  descripcion
    regsub -all {&gt;?}      $descripcion {>}  descripcion
    regsub -all {(&)(amp);?} $descripcion {\1} descripcion
    regsub -all {&ntilde;?}  $descripcion {}  descripcion
    regsub -all {&aacute;?}  $descripcion {}  descripcion
    regsub -all {&eacute;?}  $descripcion {}  descripcion
    regsub -all {&iacute;?}  $descripcion {}  descripcion
    regsub -all {&oacute;?}  $descripcion {}  descripcion
    regsub -all {&uacute;?}  $descripcion {}  descripcion

    return $descripcion
}

###############################################################################
# Chosen
#    Selects and deselects the chosen files.
#
# Parameters:
#    patron: regular expresion that identifies the files to (de)select.
#    cual: '1' to select, '0' to deselect.
#    map: 1 in case we are mapping the site.
###############################################################################
proc Chosen {patron cual {map 0} } {
    global linkFile nLinks linksOk siteMap

    for {set i 1} {$i<$nLinks} {incr i} {
        if {$map==0} {
            if {[regexp -nocase $patron $linkFile($i)]} {
                set linksOk($i) $cual
            }
        } else {
            if {[regexp -nocase $patron $siteMap($i,1)]} {
                set linksOk($i) $cual
            }
        }
    }
    return
}

###############################################################################
# ChooseDownloads
#    Shows a dialog with all the links in the entered Url so you can choose
#    which ones will be followed and which will be ignored.
#
# Parameter
#    map: '1' if we are getting the site map, defaults to '0'
#
###############################################################################
proc ChooseDownloads { {map 0} } {
    global linkFile nLinks linksOk linkDescription
    global done dirGetleft labelButtons labelTitles labelDialogs
    global siteIndex siteMap

    if {$nLinks==1} "return 0"
    set elegir  [toplevel .elegir]
    set marco   [frame $elegir.marco]
    set checks  [frame $elegir.checks]
    set botones [frame $elegir.botones]

    wm title $elegir $labelTitles(choose)

    set texto [text $marco.texto -yscrollcommand [list $marco.yscroll set] \
            -xscrollcommand [list $marco.xscroll set] -wrap none]
    set aceptar [button $botones.aceptar   -text $labelButtons(ok)     -width 9 \
            -command "set done 1"]
    set cancelar [button $botones.cancelar -text $labelButtons(cancel) -width 9 \
            -command "set done 0" ]

    set mas   [image create bitmap  -file [file join $dirGetleft(images) mas]]
    set menos [image create bitmap  -file [file join $dirGetleft(images) menos]]
    set lhtml [label  $checks.lhtml -text "  $labelDialogs(html)"]
    set htmla [button $checks.htmla -image $mas   -width 8 -height 8 \
            -command "Chosen {(html$)|(htm$)|(/$)} 1 $map"]
    set htmle [button $checks.htmle -image $menos -width 8 -height 8 \
            -command "Chosen {(html$)|(htm$)|(/$)} 0 $map"]
    set lima  [label $checks.lima   -text "  $labelDialogs(images)"]
    set imaa  [button $checks.imaa  -image $mas   -width 8 -height 8 \
            -command "Chosen {(gif$)|(jpg$)|(bmp$)|(xbm$)|(tiff$)} 1 $map"]
    set imae  [button $checks.imae  -image $menos -width 8 -height 8 \
            -command "Chosen {(gif$)|(jpg$)|(bmp$)|(xbm$)|(tiff$)} 0 $map"]
    set lcom  [label $checks.lcom   -text "  $labelDialogs(archives)"]
    set coma  [button $checks.cona  -image $mas   -width 8 -height 8 \
            -command "Chosen {(tar$)|(gz$)|(z$)|(zip$)} 1 $map"]
    set come  [button $checks.cone  -image $menos -width 8 -height 8 \
            -command "Chosen {(tar$)|(gz$)|(z$)|(zip$)} 0 $map"]
    set ltodo [label $checks.ltodo  -text "  $labelDialogs(all)"]
    set todoa [button $checks.todoa -image $mas   -width 8 -height 8 \
            -command "Chosen {.} 1 $map"]
    set todoe [button $checks.todoe -image $menos -width 8 -height 8 \
            -command "Chosen {.} 0 $map"]

    set color [$texto cget -background]

    if {$map==1} {
        set nLinks $siteIndex
    }
    for {set i 1} {$i<$nLinks} {incr i} {
        checkbutton $texto.enlaces$i -variable linksOk($i) -background $color
        set linksOk($i) 1

        $texto window create $i.0 -window $texto.enlaces$i
        if {$map==0} {
            $texto insert insert $linkDescription($i)\n
        } else {
            for {set j 0 ; set spc "   " ; set pre ""} {$j<$siteMap($i,0)} {incr j} {
                append pre $spc
            }
            $texto insert insert $pre$siteMap($i,2)\n
        }
    }

    set barray [scrollbar $marco.yscroll -orient vertical \
        -command [list $texto yview]]
    set barrax [scrollbar $marco.xscroll -orient horizontal  \
        -command [list $texto xview]]

    bind $elegir <KeyPress-Prior> "$texto yview scroll -1 pages"
    bind $elegir <KeyPress-Next>  "$texto yview scroll  1 pages"
    bind $elegir <Key-Up>         "$texto yview scroll -1 units"
    bind $elegir <Key-Down>       "$texto yview scroll  1 units"
    bind $texto  <Button-1>       {
        tkTextButton1 %W %x %y
        regexp {(.*)(\.)} [%W index insert] nada line
        if {$line<$nLinks} {
            %W.enlaces$line invoke
        }
        %W tag remove sel 0.0 end
    }

    grid $marco -sticky news
    grid $texto $barray -sticky news
    grid $barrax -sticky ew
    grid $checks -sticky ew
    grid $lhtml $htmla $htmle $lima $imaa $imae $lcom $coma $come \
        $ltodo $todoa $todoe
    grid $botones -sticky ew -ipady 4
    grid $aceptar $cancelar

    grid rowconfigure    $elegir 0 -weight 1
    grid columnconfigure $elegir 0 -weight 1
    grid rowconfigure    $marco  0 -weight 1
    grid columnconfigure $marco  0 -weight 1

    $texto configure -state disabled

    wm protocol $elegir WM_DELETE_WINDOW "
            set done 0
            destroy $elegir"

    tkwait visibility $elegir

    grab $elegir
    tkwait variable done
    grab release $elegir
    destroy $elegir

    return $done
}

###############################################################################
# OpenFile
#    Creates, if needed, the directory in which the link will be saved.
#
# Parameter:
#    fichero: file name
#
# Returns:
#    Local file name with a 'downloading' extension
#    1: In case of error
###############################################################################
proc OpenFile {fichero} {
    global directories
    global labelTitles labelMessages

    if [regexp {(?:^/)(.*)} $fichero nada fichero] {
        set directorio_a_usar $directories(base)
    } else {
        set directorio_a_usar $directories(local)
    }

    set directorio [file dirname $fichero]
    if [file exists [file join $directorio_a_usar $directorio]] {
        if {![file isdirectory [file join $directorio_a_usar $directorio]]} {
            tk_messageBox -type ok -icon error \
                    -message "La hemos fastidiado con los nombres: $fichero"
            return 1
        }
     } else {
        if [catch {file mkdir [file join $directorio_a_usar $directorio]} error] {
            tk_messageBox -title $labelTitles(error) -icon error \
                    -message $error
            return 1
        }
     }

     set fichero_a_bajar \
        [file join $directorio_a_usar ${fichero}.$labelMessages(downloading)]

     return $fichero_a_bajar
}

###############################################################################
# Download
#    Downloads a link
#
# Parameter:
#    enlace:  link to download
#    fichero: handle to the file in which it will be saved
#    madre:  'mother' page
#    sufix:  whether the file ends in 'htm?' (1) or not (0)
#
# Returns:
#    - '4' in case of a relocation and a HTML page
#    - '3' in case of a relocation and not a HTML page
#    - '2' in case of error
#    - '1' if it is a HTML page
#    - '0' if it is not
###############################################################################
proc Download {enlace fichero madre sufix} {
    global siteUrl options errorCode
    global Ventana::Rizo::meta
    global relocated directories
    global labelMessages

    set fichero [TidyDir $fichero]
    set enlace  [TidyDir $enlace]

    set root [file root $fichero]
    if {([file exists $root])&&($options(update)==0)} {
        set relocated(ok) 0
        if {$sufix==0} {
            if {([file isdirectory $root])&&(![regexp {/$} $root])} {
                if [file exists $root/index.html] {
                    set relocated(ok) 1
                    set relocated(file) $root/index.html
                    regexp -nocase "(?:^$directories(base))(.*)" \
                            $relocated(file) nada relocated(file)
                    set relocated(file) [file join [RelativePath $relocated(file)] \
                            [string trimleft $relocated(file)]]
                    regexp {(?:[^/]+)(.*)} $enlace nada enlace
                    set relocated(url) $enlace/
                    set relocated(dir) $enlace
                    return 4
                }
            } elseif [file exists [file root $fichero].html] {
                return 1
            } else {
                return 0
            }
        } else {
            return 1
        }
    }
    for {set relocated(ok) 0} {1==1} {} {
        Ventana::HeadDownloading $enlace $madre
        if {$::Ventana::Rizo::curlError!=0} {
            ::Herramientas::ErrorLog $fichero $enlace
            return 2
        }
        if {$meta(relocate)!=""} {
            if {([regexp {\?} $meta(relocate)])} {
                return 0
            }
            regexp -nocase {(?:http://)?([^/]+)((?:~[^/]*)?(?:[^\?]*))(?:/)(.*)} \
                    $meta(relocate) nada direccionWWW direccionDir filename
            if {$direccionWWW!=$siteUrl(www)} {
                set Ventana::Rizo::errorMessage "Relocated link\n$meta(relocate)"
                return 2
            }
            set relocated(url) $direccionDir/$filename
            set relocated(dir) $direccionDir
            if {$filename==""} {
                set filename index.html
            }
            set filename [TidyNames [file join $direccionDir $filename]]
            set fileToDownload [OpenFile $filename]
            if {[file exists [file root $fileToDownload]]} {
                if {([regexp -nocase {html?$} $fileToDownload])||([file exists $fileToDownload.labelMessages(downloading)])} {
                    set isHtml 1
                } else {
                    set isHtml 0
                }
                return [expr 3 + $isHtml]
            }
            if {$fileToDownload==1} {
                return 2
            }
            set enlace $meta(relocate)
            set relocated(file) [file join [RelativePath $filename] \
                    [string trimleft $filename]]
            set fichero $fileToDownload
            set relocated(ok) 1
        } else {
            break
        }
    }
    if {([info exists meta(totalBytes)])&&($meta(totalBytes)==0)} {
        catch {error "Sin contenido" GETLEFT \
                "Getleft \"$labelDialogs(noContent)\" 002"}
        ::Herramientas::ErrorLog $fichero $enlace
        return 2
    }

    set filename [file root $fichero]
    if {($options(update)==1)&&([file exists $filename])} {
        if [file exists $filename.orig] {
            file stat $filename.orig oldFile
        } else {
            file stat $filename oldFile
        }
        if {[catch "set meta(lastModified)"]} {
            set lastChange 0
        } else {
            set lastChange [clock scan $meta(lastModified)]
        }
        if {$lastChange<$oldFile(mtime)} {
            if {$sufix==0} {
                if [file exists [file root $fichero].html] {
                   return 1
                } else {
                   return 0
                }
            }
            return 1
        }
    }

    if [string match $Ventana::Rizo::meta(content) "text/html"] {
        set esHtml 1
    } else {
        set esHtml 0
        if {$options(onlyHtml)==1} {
            return 0
        }
    }

    set Ventana::Rizo::curlReport(parar)    0
    set Ventana::Rizo::curlReport(nextFile) 0
    Ventana::FileDownloading $fichero $enlace $madre
    tkwait variable Ventana::Rizo::curlReport(nextFile)
    if {$errorCode!=""} {
        ::Herramientas::ErrorLog $fichero $enlace
        return 2
    }

    set root [file rootname $fichero]
    catch {file rename -force $fichero $root}
    if {$esHtml==1} {
        catch {file delete $root.orig}
    }

    if {$relocated(ok)==1} {
        return [expr 3 + $esHtml ]
    }
    return $esHtml
}

###############################################################################
# ChangePage
#    Changes a html page, so that there is consistency with the local
#    directories
#
# Parameters
#    pagina: filename of the page to change
#    hay_url: '1' if the file includes the site url, e.g., www.scriptics.com in
#             the links.
###############################################################################
proc ChangePage {pagina hay_url} {
    global siteUrl linkFile nLinks
    global directories

    set path_relativo [RelativePath $pagina]
    set script [open "$directories(base)/script.sed" w+]
    if {$hay_url==1} {
        puts $script "s#http://$siteUrl(www)##g"
        puts $script "s#ftp://$siteUrl(www)##g"
        puts $script {s#\(http:/\)\([^/]\)#/\2#g}
    }

    if [string match $siteUrl(dir) "/"] {
        set dir ""
    } else {
        set dir $siteUrl(dir)
    }
    for {set i 1} {$i<$nLinks} {incr i} {
        set linkToChange $linkFile($i)
        if [regsub -all {~|\?|\*} $linkFile($i) {} nuevo_enlace] {
            regsub {\*} $linkFile($i) {\\*} tmp
            set linkToChange $nuevo_enlace
            regsub -all {&} $nuevo_enlace {\\\&} nuevo_enlace
            puts $script "s#=\\\"*$tmp\\(\\\"\\|\\ \\)#=\"$nuevo_enlace\"#g"
        }
        if {![regexp {^/} $linkFile($i)]} {
            set tmp [TidyDir [TidyDir $dir]/[TidyDir $linkFile($i)]]
            set conAmper $tmp
            if {![string match $tmp $linkFile($i)]} {
                set tmp [TidyNames $tmp]
                regsub -all {&} $tmp {\\\&} tmp
                if [regexp {^\./} $linkFile($i)] {
                    puts $script "s#=\\\"*\\\$linkToChange\\(\\\"\\|\\ \\)#=\"$tmp\"#g"
                } else {
                    if {$linkFile($i)!="\\"} {
                        puts $script "s#=\\\"*$linkToChange\\(\\\"\\|\\ \\)#=\"$tmp\"#g"
                    }
                }
            }
            set linkFile($i) $conAmper
        }
    }
    puts $script "s#=\"/#=\"$path_relativo#g"
    puts $script "s#=/#=$path_relativo#g"

    close $script

    if {![file exists $pagina.orig]} {
        file copy $pagina $pagina.orig
        InvokeSed $pagina
    } else {
        file delete $directories(base)/script.sed
    }

    return
}

###############################################################################
# InvokeSed
#    Calls 'sed' to do the dirty job of changing the web pages.
#
# Parameters:
#    pagina: file containing the page to be changed.
#    expr:   (opcional), regular expression to be passed to 'sed'. If not given
#            'sed' will use the file 'script.sed'.
###############################################################################
proc InvokeSed {pagina {expr {}}} {
    global directories
    global labelTitles

    if {$expr!=""} {
        set file [open "$directories(base)/script.sed" w+]
        puts $file $expr
        close $file
    } else {
        if {![file exists [file join $directories(base) script.sed] ]} {
            return
        }
    }
    if [catch {exec sed -f "$directories(base)/script.sed"  $pagina > "$directories(base)/sed_temp"} error] {
        tk_messageBox -title $labelTitles(error) -message $error -type ok -icon error
    } else {
        file rename -force "$directories(base)/sed_temp" $pagina
    }
    file delete "$directories(base)/script.sed"

    return
}

###############################################################################
# ChangeLinkList
#    Changes or removes one link form the list that contains all the links of
#    one page
#
# Parameters:
#    linkList: list with the links
#    oldLink: link to change or remove
#    newLink: the new links, defaults to none to remove the old one.
#
# Returns
#    The new list
###############################################################################
proc ChangeLinkList {linkList oldLink {newLink {} } } {

    set index   [lsearch -exact $linkList $oldLink]
    if {$newLink!=""} {
        set linkList [lreplace $linkList $index $index $newLink]
    } else {
        set linkList [lreplace $linkList $index $index]
    }

    return $linkList
}

###############################################################################
# SaveSedExp
#    Saves one line of the sed script that will change the links to active
#    pages to the resulting page
#
# Parameter
#    exp: sed expresion to be saved
###############################################################################
proc SaveSedExp {exp} {
    global directories

    set handle [open [file join $directories(base) script.sed] a+]
    puts $handle $exp
    close $handle

    return
}

###############################################################################
# ControlDownloading
#    Send 'Downloading' all the links one by one.
#
# Parameters:
#    listaEnlaces: list with the links to download.
#    nivelActual: maximun level of recursion set by the user.
#    paginaMadre: html page from which the links come from.
###############################################################################
proc ControlDownloading {listaEnlaces nivelActual paginaMadre} {
    global siteUrl directories nLinks
    global options filesDone labelTitles labelMessages
    global relocated

    set n_oks [llength $listaEnlaces]
    if {$n_oks==0} return

    set cosa ""
    set invocarSed 0

    set j 0
    foreach enlace $listaEnlaces {
        update
        incr j
        if {$options(onlyHtml)==1} {
            if [regexp -nocase {(jpg$)|(jpeg$)|(gif$)|(gz$)|(tar$)|(zip$)|(exe$)|(ps$)|(pdf$)|(txt$)|(text$)|(avi$)|(ram$)|(wav$)} $enlace] {
                set isHtml($j) 0
                continue
            }
        }
        if {($options(filter)!="")&&([regexp -nocase "$options(filter)" $enlace])} {
            set isHtml($j) 0
            continue
        }
        if [regexp {/$} $enlace] {
            set tmp [TidyNames $enlace]
            set fichero_a_abrir [file join $tmp index.html]
            set isHtml($j) 1
            set listaEnlaces [ChangeLinkList $listaEnlaces $enlace \
                    [file join $enlace index.html]]
            regsub -all {&} $tmp {\\\&} tmpAmper
            SaveSedExp "s#\\($tmp\\)\\(\\ \\|\\\"\\)#${tmpAmper}index.html\\2#g"
            set invocarSed 1
        } else {
            set fichero_a_abrir [TidyNames $enlace]
        }
        set fichero [OpenFile $fichero_a_abrir]
        if {$fichero==1} {
            set isHtml($j) 0
            continue
        }
        if [regexp -nocase {(htm|html|/)$} $fichero_a_abrir] {
            set sufijo 1
        } else {
            set sufijo 0
        }

        set urlToDownload $siteUrl(www)$enlace
        set isHtml($j) [Download $urlToDownload $fichero $paginaMadre $sufijo]
        if {($isHtml($j)==1)&&($sufijo==0)} {
            set raiz_fichero [file rootname $fichero]
            catch {file rename $raiz_fichero $raiz_fichero.html}
            exec touch $raiz_fichero
            set listaEnlaces [ChangeLinkList $listaEnlaces \
                    $enlace $enlace.html]
            set enlace [file tail [TidyNames $enlace]]
            regsub -all {&} $enlace {\\\&} tmp
            SaveSedExp "s#\\($enlace\\)\\(\\ \\|\\\"\\)#$tmp.html\\2#g"
            set invocarSed 1
        }
        if {$isHtml($j)==3} {
            set tmp [TidyNames $enlace]
            regsub -all {&} $relocated(file) {\\\&} relocated(file)
            SaveSedExp "s#\\($tmp\\)\\(\\ \\|\\\"\\)#$relocated(file)\\2#g"
            set invocarSed 1
            set isHtml($j) 0
        }
        if {$isHtml($j)==4} {
            set isHtml($j) 1
            set invocarSed 1
            set tmp [TidyNames $enlace]
            regsub -all {&} $relocated(file) {\\\&} relocated(file)
            if {$sufijo==0} {
                if {[regexp -nocase {html?} $relocated(file)]} {
                    SaveSedExp "s#\\($tmp\\)\\(\\ \\|\\\"\\)#$relocated(file)\\2#g"
                    set listaEnlaces [ChangeLinkList $listaEnlaces \
                        $enlace $relocated(url)index.html]
                } else {
                    catch {file rename $relocated(file) $relocated(file).html}
                    exec touch $nuevoEnlace
                    SaveSedExp "s#\\($tmp\\)\\(\\ \\|\\\"\\)#$relocated(file).html\\2#g"
                    set listaEnlaces [ChangeLinkList $listaEnlaces \
                        $enlace $relocated(url).html]
                }
            } else {
                if [regexp {/$} $enlace] {
                    SaveSedExp "s#\\(${tmp}index.html\\)\\(\\ \\|\\\"\\)#$relocated(file)\\2#g"
                    set listaEnlaces [ChangeLinkList $listaEnlaces \
                        ${enlace}index.html $relocated(url)]
                } else {
                    SaveSedExp "s#\\($tmp\\)\\(\\ \\|\\\"\\)#$relocated(file)\\2#g"
                    set listaEnlaces [ChangeLinkList $listaEnlaces \
                        $enlace $relocated(url)]
                }
            }
        }
        if {$isHtml($j)==2} {
            if {[regexp {/$} $enlace]} {
                set listaEnlaces [ChangeLinkList $listaEnlaces [file join $enlace index.html] ]
            } else {
                set listaEnlaces [ChangeLinkList $listaEnlaces $enlace]
            }
            incr j -1
        }
        if {$options(stopFile)==1} {
            break
        }
        if {$options(pauseFile)==1} {
            set windowState [wm state .bajando]
            if {$windowState=="iconic"} {
                wm deiconify .bajando
            }
            wm withdraw .bajando
            tk_messageBox -title $labelTitles(paused) -icon info -type ok \
                    -message $labelMessages(paused)
            tkwait variable options(pauseFile)
            wm deiconify .bajando
            if {$windowState=="iconic"} {
                wm iconify .bajando
            }

        }
    }
    if {$options(pausePage)==1} {
        tk_messageBox -title $labelTitles(paused) -icon info -type ok \
                -message $labelMessages(paused)
        tkwait variable options(pausePage)
    }

    if {$invocarSed==1} {
        InvokeSed [TidyNames $paginaMadre]
    }

    if {$options(stopFile)==1} {
        return
    }

    set j 1
    foreach enlace $listaEnlaces {
        if {$isHtml($j)==1} {
            set pagina_a_tratar "$directories(base)$enlace"
            set pagina_a_tratar [TidyDir $pagina_a_tratar]
            set pagina_a_tratar [TidyNames $pagina_a_tratar]
            if {![info exists filesDone($pagina_a_tratar)]} {
                set tmp_url   $siteUrl(dir)
                set tmp_local $directories(local)
                set dir_enlace [TidyDir [file dirname $enlace]]
                if {$dir_enlace != "."} {
                    set siteUrl(dir) [TidyDir $dir_enlace]
                }
                if {(($nivelActual<$options(levels))||($options(levels)==-1))} {
                    Preprocessing $pagina_a_tratar
                    if {$nLinks!=1} {
                        Chosen {.} 1
                        set nextLevel [expr $nivelActual +1]
                        ControlDownloading [PrepareDownloading $nextLevel] $nextLevel \
                                 $pagina_a_tratar
                        if {($options(stopPage)==1)||($options(stopFile)==1)} {
                            break
                        }
                    }
                }
                set direcion(dir)      $tmp_url
                set directories(local) $tmp_local
            }
        }
        incr j
    }
    return
}

###############################################################################
# PrepareDownloading
#    Prepares the list of files to download
#
# Parameter
#    nivelActual: level of recursion
###############################################################################
proc PrepareDownloading {nivelActual } {
    global linkFile nLinks linkDescription linksOk
    global siteIndex siteMap options

    for {set i 1 ; set listaEnlaces ""} {$i<$nLinks} {incr i} {
        if {$linksOk($i)==1} {
            if {[lsearch $listaEnlaces $linkFile($i)] == -1} {
                lappend listaEnlaces $linkFile($i)
            }
        }
    }

    if {$options(map)==1} {
        for {set i 1} {$i<$nLinks} {incr i} {
            if {![info exists siteMap($linkFile($i))]} {
                set siteMap($siteIndex,0) $nivelActual
                set siteMap($siteIndex,1) $linkFile($i)
                set siteMap($siteIndex,2) $linkDescription($i)
                set siteMap($linkFile($i)) 1
                incr siteIndex
            }
        }
    }

    return $listaEnlaces
}

###############################################################################
# ReadSel
#    Reads the selection from the clipboard
#
# Returns
#    Whatever was in the clipboard
###############################################################################
proc ReadSel {} {
    if {[catch {selection get -selection CLIPBOARD} cosa]} {
        if {[catch {selection get} cosa]} return
    }
    return $cosa
}

###############################################################################
# EnterUrl
#    Creates a dialog box where you can write or paste the desired url.
###############################################################################
proc EnterUrl {} {
    global filesDone nLinks options urlHistory done
    global directories urlIndex coord getleftState
    global labelButtons labelTitles labelMessages

    if [winfo exists .introducir] {
        return
    }

    SetCoordinates

    set intro [toplevel .introducir]
    wm title $intro $labelTitles(enterUrl)
    wm resizable $intro 0 0
    wm geometry $intro +[expr $coord(x)+100]+[expr $coord(y)+50]

    set done                      0
    set options(stopPage)         0
    set options(stopFile)         0
    set options(pausePage)        0
    set options(pauseFile)        0
    set getleftState(filesChosen) 0
    set getleftState(errorFound)  0

    set marcoEx [frame $intro.marcoEx]
    set marco   [frame $marcoEx.marco -bd 2 -relief groove]

    for {set i 0; set histItems ""} {$i<10} {incr i} {
        if [info exists urlHistory($i)] {
            lappend histItems $urlHistory($i)
        } else {
            break
        }
    }
    cuadroCombinado $marco.pagina $histItems [ReadSel]

    set pagina $marco.pagina
    set botones  [frame $marcoEx.botones]
    set empezar  [button $intro.empezar  -text $labelButtons(start) \
            -width 8 -command {set done 1}]
    set cancelar [button $intro.cancelar -text $labelButtons(cancel) \
            -width 8 -command {set done 0}]

    bind $pagina.e <Return> "$empezar invoke"
    bind $pagina.e <Button-3> "
        $marco.pagina.e insert insert \[ReadSel\]
    "
    bind $empezar <Return> "$empezar invoke"

    pack $pagina -padx 20 -pady 20
    pack $marcoEx -ipadx 10 -ipady 5
    pack $marco -side bottom
    pack $cancelar -side right -padx 9 -pady 5
    pack $empezar -side right  -pady 5
    pack $botones -side right

    focus $pagina.e
    tkwait variable done

    if {$done==0} {
        destroy $intro
        return
    }

    set pagina_objetivo [$pagina.e get]
    destroy $intro

    if {![EnHistorial $pagina_objetivo]} {
        set urlHistory($urlIndex) $pagina_objetivo
        if {$urlIndex==9} {
            set urlIndex 0
        } else {
            incr urlIndex
        }
        SaveConfig
    }
    catch "unset filesDone"
    set getleftState(downloading) 1

    set nombre_fichero [InitializeDownload $pagina_objetivo]
    if {![string compare $nombre_fichero ""]} {
        set getleftState(downloading) 0
        return
    }
    if {($options(levels)==0)||(![regexp -nocase {html?$} $nombre_fichero])} {
        tk_messageBox -icon info -title $labelTitles(theEnd) -type ok \
                -message $labelMessages(theEnd)
        set getleftState(downloading) 0
        return
    }
    set pagina_a_tratar [TidyNames $nombre_fichero]
    Preprocessing $pagina_a_tratar
    if {$nLinks==1} {
        tk_messageBox -icon info -title $labelTitles(noLinks) -type ok \
                -message $labelMessages(noLinks)
        set getleftState(downloading) 0
        return
    }
    set chosen [ChooseDownloads]
    if {$chosen==1} {
        set getleftState(filesChosen) 1
        ControlDownloading [PrepareDownloading 0] 1 $pagina_a_tratar
        catch {wm withdraw .bajando}
        if {$options(map)==0} {
            tk_messageBox -icon info -title $labelTitles(theEnd) -type ok \
                    -message $labelMessages(theEnd)
            if {$getleftState(errorFound)==1} {
                ::Herramientas::ShowErrorLog
            }
        }
        set options(stopFile)  0
        set options(stopPage)  0
        set options(pauseFile) 0
        set options(pausePage) 0
        set getleftState(filesChosen) 0
    }
    catch "unset filesDone"
    set getleftState(downloading) 0

    return
}

###############################################################################
# GetSiteMap
#    Downloads the html pages in a site, to extract the site map from them
###############################################################################
proc GetSiteMap {} {
    global siteIndex siteMap options getleftState
    global siteUrl linksOk labelMessages labelTitles
    global linkFile linkDescription

    set options(map)       1
    set options(onlyHtml)  1
    set siteIndex           1
    catch {unset siteMap}

    EnterUrl

    set options(onlyHtml)  0
    set options(map)       0

    if {$siteIndex==1} {
        return
    }

    set done [ChooseDownloads 1]
    set tmp $options(levels)
    set $options(levels) 0

    if {$done==1} {
        for {set i 1} {$i<$siteIndex} {incr i} {
            if {$linksOk($i)==1} {
                set enlace $siteMap($i,1)
                if [regexp {/$} $enlace] {
                    set ficheroAbrir [file join [TidyNames $enlace] index.html]
                } else {
                    set ficheroAbrir [TidyNames $enlace]
                }
                set fichero [OpenFile $ficheroAbrir]
                if {$fichero==1} {
                    continue
                }
                set filename [file root $fichero]
                if {[file exists $filename]} {
                    continue
                } else {
                    set isHtml [regexp -nocase {html?} $filename]
                    Download $siteUrl(www)$enlace $fichero "-" $isHtml
               }
                if {$options(stopFile)==1} {
                    break
               }
            }
        }
        tk_messageBox -icon info -title $labelTitles(theEnd) -type ok \
                -message $labelMessages(theEnd)
    }
    unset siteMap
    unset linksOk
    unset linkFile
    unset linkDescription

    if {$getleftState(errorFound)==1} {
        ::Herramientas::ShowErrorLog
    }
    set options(levels) $tmp

    return
}

###############################################################################
# ReadConfig
#    Reads the configuration files
#
# Parameter:
#    Configuration file to read, if not present, the user configuration  file
#    will be read.
###############################################################################
proc ReadConfig { {confFile {} } } {
    global urlHistory urlIndex dirGetleft options
    global env tcl_platform


    if {$confFile!=""} {
        set ficherorc $confFile
    } else {
        if [regexp {Windows} $tcl_platform(os)] {
            set ficherorc [file join "$dirGetleft(main)" getleft.ini]
        } else {
            set ficherorc [file join $env(HOME) .Getleft]
        }
    }

    if [catch {open $ficherorc r} fichero] {
        exec touch $ficherorc
        set options(lang) en
        set urlIndex 0
        return 0
    }

    for {set enHistorial 0} {![eof $fichero]} {} {
        set linea [gets $fichero]
        if [regexp {(^\[historial\]$)|(^\[urlHistory\]$)} $linea] {
            gets $fichero
            for {set i 0} {$i<10} {incr i} {
                set urlHistory($i) [gets $fichero]
                if [string match $urlHistory($i) ""] {
                unset urlHistory($i)
                break
                }
            }
            set enHistorial [incr i -1]
        }

        if [regexp {(^\[urlIndex\]$)|(^\[hisIndex\]$)} $linea] {
            gets $fichero
            set urlIndex [gets $fichero]
        }

        if [regexp {^\[proxy\]$} $linea] {
            gets $fichero
            set options(dirProxy) [gets $fichero]
            set options(proxy) 1
        }

        if [regexp {^\[language\]$} $linea] {
            gets $fichero
            set options(lang) [gets $fichero]
        }
        if [regexp {^\[scripts\]$} $linea] {
            gets $fichero
            set dirGetleft(scripts) [gets $fichero]
        }
        if [regexp {^\[images\]$} $linea] {
            gets $fichero
            set dirGetleft(images) [gets $fichero]
        }
        if [regexp {^\[doc\]$} $linea] {
            gets $fichero
            set dirGetleft(doc) [gets $fichero]
        }
        if [regexp {^\[languages\]$} $linea] {
            gets $fichero
            set dirGetleft(languages) [gets $fichero]
        }
    }

    close $fichero

    if {![info exists options(lang)]} {
        set options(lang) en
    }
    if {![info exists urlIndex]} {
        set urlIndex 0
    }

    return $enHistorial
}

###############################################################################
# SaveConfig
#   Saves the configuration
###############################################################################
proc SaveConfig {} {
    global urlHistory urlIndex dirGetleft options
    global env tcl_platform
    global labelMessages labelTitles

    if [regexp {Windows} $tcl_platform(os)] {
        set ficherorc [file join "$dirGetleft(main)" getleft.ini]
    } else {
        set ficherorc [file join $env(HOME) .Getleft]
    }

    if [catch {open  $ficherorc w+} fichero] {
        tk_messageBox -title $labelTitles(error) -icon Error \
            -message $labelMessages(saveConfig)
        return
    }

    puts $fichero "\[urlHistory\]\n"
    for {set i 0} {$i<10} {incr i} {
        if {![catch {set urlHistory($i)} ]} {
            puts $fichero $urlHistory($i)
        }
    }

    puts $fichero "\n\[urlIndex\]\n"
    puts $fichero $urlIndex

    if [info exists options(dirProxy)] {
        puts $fichero "\n\[proxy\]\n"
        puts $fichero "$options(dirProxy)"
    }

    puts $fichero "\n\[language\]\n"
    puts $fichero "$options(lang)"

    close $fichero

    return
}

###############################################################################
# MostrarHistorial
#    Shows the url history
###############################################################################
proc MostrarHistorial {} {
    global urlHistory urlIndex

    for {set i 0} {$i<10} {incr i} {
        if {![catch {set urlHistory($i)} ]} {
            puts "$i: $urlHistory($i)"
        }
    }
    return
}

###############################################################################
# EnHistorial
#    Checks whether the given url is already in the url history
#
# Parameter:
#    pagina: given url
#
# Returns:
#    1 if it is.
#    0 if it is not
###############################################################################
proc EnHistorial {pagina} {
    global urlHistory

    for {set i 0} {$i<10} {incr i} {
        if [catch {regsub {(http://)||(ftp://)} $urlHistory($i) {} una}] {
            return 0
        }
        regsub {(http://)||(ftp://)} $pagina {} dos

        if [string match $una $dos] {
            return 1
        }
    }
    return 0
}

###############################################################################
# ChangeLanguage
#    Changes the default language for the program
#
# Parameters:
#    lang: language to be used
###############################################################################
proc ChangeLanguage {lang} {
    global options

    set options(lang) $lang
    MenusLabels $lang

    SaveConfig

    return
}

###############################################################################
# ReadSupportedLang
#    Reads the 'supported' file to know which languages this version supports
#    and puts then in supportedLang global variable
###############################################################################
proc ReadSupportedLang {} {
    global dirGetleft supportedLang
    global labelTitles

    if [catch {open [file join $dirGetleft(languages) supported] r} handle] {
        tk_messageBox -type ok -icon error -title Error -message $handle
        return
    }

    for {} {![eof $handle]} {} {
        set line [gets $handle]
        if [regexp {^#|^ |^$} $line] {
            continue
        }
        set language [lindex $line 1]
        set code     [lindex $line 0]

        set supportedLang($language) $code
    }
    return
}

###############################################################################
# ReadLangFile
#    Reads one of the files with translations
#
# Parameter
#    The two letters code that identifies one language, i.e., 'es', 'en'
#
# Returns
#    - '0' all went well
#    - '1' there was an error
###############################################################################
proc ReadLangFile {sufix} {
    global dirGetleft
    global labelMenus labelButtons  labelTitles labelMessages
    global labelDialogs labelMonths indexMenus

    set languageFile [file join $dirGetleft(languages) menus.$sufix]

    if [catch {open $languageFile r} handle] {
        tk_messageBox -icon error -type ok -title Error \
                -message "File not found"
        return 1
    }
    for {set active ""} {![eof $handle]} {} {
        set line [gets $handle]
        if [regexp {^#|^ |^$} $line] {
            continue
        }
        if {![string compare $line "End"]} {
            set active ""
            continue
        }
        set key   [lindex $line 0]
        set value [lindex $line 1]

        switch -exact -- $active {
            Menus {
                set indexMenus($key) [string first & $value]
                if {$indexMenus($key)!=-1} {
                    regsub {&} $value {} value
                }
                set labelMenus($key) $value
            }
            Buttons {
                set labelButtons($key) $value
            }
            Titles {
                set labelTitles($key) $value
            }
            Messages {
                set labelMessages($key) $value
            }
            Dialogs {
                set labelDialogs($key) $value
            }
            Months {
                set labelMonths($key) $value
            }
            "" {
                set active $line
            }
        }
    }
    return 0
}

###############################################################################
# MenusLabels
#    Puts the labels into the menus.
#
# Parameters:
#    labFile: file with the labels in the, hopefully, desired language
###############################################################################
proc MenusLabels {{labFile ""}} {
    global menus dirGetleft
    global labelButtons labelTitles labelMessages labelDialogs labelMonths
    global labelMenus indexMenus

    if {$labFile==""} {
        set labFile en
    }

    ReadLangFile $labFile

    .menus      entryconfigure 1 -label $labelMenus(file)           -underline $indexMenus(file)
    $menus(1)   entryconfigure 0 -label $labelMenus(enterUrl)       -underline $indexMenus(enterUrl)
    $menus(1)   entryconfigure 1 -label $labelMenus(siteMap)        -underline $indexMenus(siteMap)
    $menus(1)   entryconfigure 3 -label $labelMenus(stop)           -underline $indexMenus(stop)
    $menus(1,3) entryconfigure 0 -label $labelMenus(stopPage)       -underline $indexMenus(stopPage)
    $menus(1,3) entryconfigure 1 -label $labelMenus(stopFile)       -underline $indexMenus(stopFile)
    $menus(1,3) entryconfigure 2 -label $labelMenus(stopNow)        -underline $indexMenus(stopNow)
    $menus(1)   entryconfigure 4 -label $labelMenus(pause)          -underline $indexMenus(pause)
    $menus(1,4) entryconfigure 0 -label $labelMenus(stopPage)       -underline $indexMenus(stopPage)
    $menus(1,4) entryconfigure 1 -label $labelMenus(stopFile)       -underline $indexMenus(stopFile)
    $menus(1)   entryconfigure 6 -label $labelMenus(exit)           -underline $indexMenus(exit)

    .menus      entryconfigure 2 -label $labelMenus(options)        -underline $indexMenus(options)
    $menus(2)   entryconfigure 0 -label $labelMenus(uplinks)        -underline $indexMenus(uplinks)
    $menus(2,0) entryconfigure 0 -label $labelMenus(linksFollow)    -underline $indexMenus(linksFollow)
    $menus(2,0) entryconfigure 1 -label $labelMenus(linksIgnore)    -underline $indexMenus(linksIgnore)
    $menus(2)   entryconfigure 1 -label $labelMenus(levels)         -underline $indexMenus(levels)
    $menus(2,1) entryconfigure 0 -label $labelMenus(noLimit)        -underline $indexMenus(noLimit)
    $menus(2)   entryconfigure 2 -label $labelMenus(fileFilter)     -underline $indexMenus(fileFilter)
    $menus(2,2) entryconfigure 0 -label $labelMenus(onlyHtml)       -underline $indexMenus(onlyHtml)
    $menus(2,2) entryconfigure 1 -label $labelMenus(chooseFilter)   -underline $indexMenus(chooseFilter)
    $menus(2)   entryconfigure 3 -label $labelMenus(resume)         -underline $indexMenus(resume)
    $menus(2)   entryconfigure 4 -label $labelMenus(update)         -underline $indexMenus(update)
    $menus(2)   entryconfigure 5 -label $labelMenus(cgi)            -underline $indexMenus(cgi)
    $menus(2)   entryconfigure 6 -label $labelMenus(useProxy)       -underline $indexMenus(useProxy)
    $menus(2)   entryconfigure 7 -label $labelMenus(checkSize)      -underline $indexMenus(checkSize)

    .menus      entryconfigure 3 -label $labelMenus(tools)          -underline $indexMenus(tools)
    $menus(3)   entryconfigure 0 -label $labelMenus(purgeFiles)     -underline $indexMenus(purgeFiles)
    $menus(3)   entryconfigure 1 -label $labelMenus(restoreOrig)    -underline $indexMenus(restoreOrig)
    $menus(3)   entryconfigure 2 -label $labelMenus(configureProxy) -underline $indexMenus(configureProxy)
    $menus(3)   entryconfigure 3 -label $labelMenus(languages)      -underline $indexMenus(languages)
#    $menus(3)   entryconfigure 4 -label $labelMenus(resource)      -underline $indexMenus(resource)

    .menus      entryconfigure 4 -label $labelMenus(help)           -underline $indexMenus(help)
    $menus(4)   entryconfigure 0 -label $labelMenus(manual)         -underline $indexMenus(manual)
    $menus(4)   entryconfigure 2 -label $labelMenus(changes)        -underline $indexMenus(changes)
    $menus(4)   entryconfigure 3 -label $labelMenus(license)        -underline $indexMenus(license)
    $menus(4)   entryconfigure 5 -label $labelMenus(about)          -underline $indexMenus(about)

    return
}

###############################################################################
# ShowMenu1
#    Before posting the file menu, this procedure decides which entries
#    should be active, and which will have to wait
###############################################################################
proc ShowMenu1 {} {
    global getleftState

    if {$getleftState(downloading)==0} {
        .menus.fichero entryconfigure 0 -state normal
        .menus.fichero entryconfigure 1 -state normal
        .menus.fichero entryconfigure 3 -state disabled
        .menus.fichero entryconfigure 4 -state disabled
    } else {
        .menus.fichero entryconfigure 0 -state disabled
        .menus.fichero entryconfigure 1 -state disabled
        .menus.fichero entryconfigure 3 -state normal
        .menus.fichero entryconfigure 4 -state normal
    }
    return
}

###############################################################################
# CreateMenus
#   Creates the menus, without putting the labels
###############################################################################
proc CreateMenus {} {
    global options dirGetleft menus supportedLang

    menu .menus -relief flat

    set menus(1) [menu .menus.fichero      -tearoff 0 -postcommand ShowMenu1]
    .menus add cascade -menu $menus(1)
    set menus(2) [menu .menus.editar       -tearoff 0]
    .menus add cascade -menu $menus(2)
    set menus(3) [menu .menus.herramientas -tearoff 0]
    .menus add cascade -menu $menus(3)
    set menus(4) [menu .menus.help         -tearoff 0]
    .menus add cascade -menu $menus(4)

    $menus(1) add command -command EnterUrl
    $menus(1) add command -command GetSiteMap
    $menus(1) add separator
    $menus(1) add cascade -menu $menus(1).stop
    $menus(1) add cascade -menu $menus(1).pause
    $menus(1) add separator
    $menus(1) add command -command exit

    set menus(1,3) [menu $menus(1).stop -tearoff 0]
    $menus(1,3) add check -variable options(stopPage) -onvalue 1 -offvalue 0
    $menus(1,3) add check -variable options(stopFile) -onvalue 1 -offvalue 0
    $menus(1,3) add command -command ::Ventana::CancelDownloading

    set menus(1,4) [menu $menus(1).pause -tearoff 0]
    $menus(1,4) add check -variable options(pausePage) -onvalue 1 -offvalue 0
    $menus(1,4) add check -variable options(pauseFile) -onvalue 1 -offvalue 0

    $menus(2) add cascade -menu $menus(2).directorios
    $menus(2) add cascade -menu $menus(2).levels
    $menus(2) add cascade -menu $menus(2).filters
    $menus(2) add check   -variable options(resume)    -onvalue 1 -offvalue 0
    $menus(2) add check   -variable options(update)    -onvalue 1 -offvalue 0
    $menus(2) add check   -variable options(cgi)       -onvalue 1 -offvalue 0
    $menus(2) add check   -variable options(proxy)     -onvalue 1 -offvalue 0
    $menus(2) add check   -variable options(checkSize) -onvalue 1 -offvalue 0

    set options(cgi)       0
    set options(resume)    1
    set options(update)    0
    set options(checkSize) 1

    set menus(2,0) [menu $menus(2).directorios -tearoff 0]
    $menus(2,0) add radio -variable options(dir) -value 1
    $menus(2,0) add radio -variable options(dir) -value 0
    set options(dir) 1

    set menus(2,1) [menu $menus(2).levels -tearoff 0]
    $menus(2,1) add radio -variable options(levels) -value -1
    for {set i 0} {$i<6} {incr i} {
        $menus(2,1) add radio -label $i -variable options(levels) -value $i
    }

    set menus(2,2) [menu $menus(2).filters -tearoff 0]
    $menus(2,2) add check -variable options(onlyHtml) -onvalue 1 -offvalue 0
    $menus(2,2) add command -command Herramientas::FilterFiles

    set options(levels) -1

    $menus(3) add command -command Herramientas::PurgeFiles
    $menus(3) add command -command Herramientas::RestoreOriginals
    $menus(3) add command -command Herramientas::ConfProxy
    $menus(3) add cascade -menu $menus(3).idiomas
#    $menus(3) add command -command \
#           "source [file join "$dirGetleft(main)" Getleft.tcl]"

    set menus(3,0) [menu $menus(3).idiomas -tearoff 0]
    set i 0
    foreach lang [lsort [array names supportedLang]] {
        if [file exists [file join $dirGetleft(languages) menus.$supportedLang($lang)]] {
            $menus(3,0) add command -command "ChangeLanguage $supportedLang($lang)"
            $menus(3,0) entryconfigure $i -label $lang
            incr i
        }
    }

    $menus(4) add command -command Ayuda::Manual
    $menus(4) add separator
    $menus(4) add command -command Ayuda::Changes
    $menus(4) add command -command Ayuda::Licence
    $menus(4) add separator
    $menus(4) add command -command Ayuda::About

    MenusLabels $options(lang)

    return
}

###############################################################################
#				PROGRAMA PRINCIPAL
###############################################################################

set dirGetleft(main) [file dirname [info script]]
if [file exists /etc/getlet.conf] {
    ReadConfig /etc/getleft.conf
} elseif [file exists /usr/local/etc/getleft.conf] {
    ReadConfig /usr/local/etc/getleft.conf
} else {
    regexp {(?:.:)(.*)} "$dirGetleft(main)" nada dirGetleft(main)
    set dirGetleft(scripts)   [file join "$dirGetleft(main)" scripts]
    set dirGetleft(images)    [file join "$dirGetleft(main)" images]
    set dirGetleft(doc)       [file join "$dirGetleft(main)" doc]
    set dirGetleft(languages) [file join "$dirGetleft(main)" Languages]
}

source [file join "$dirGetleft(scripts)" Ccombinado.tcl]
source [file join "$dirGetleft(scripts)" Ventana.tcl]
source [file join "$dirGetleft(scripts)" Dialogos.tcl]
source [file join "$dirGetleft(scripts)" Herramientas.tcl]
source [file join "$dirGetleft(scripts)" Ayuda.tcl]


if {![winfo exists .menus]} {
    ReadSupportedLang
    ReadLangFile en

    wm title . "Getleft v 0.7.6"
    if [regexp {Windows} $tcl_platform(os)] {
        wm geometry . 300x20
    } else {
        wm geometry . 300x0
    }
    . config -menu .menus

    wm protocol . WM_DELETE_WINDOW exit

    ReadConfig
    set options(filter)            ""
    set options(map)               0
    set getleftState(downloading)  0

    CreateMenus
}

