/*
 *	$Source: /home/nlfm/Working/Frink/RCS/tcl.c,v $
 *	$Date: 1998/12/04 16:40:45 $
 *	$Revision: 1.2.1.36 $
 *
 *------------------------------------------------------------------------
 *   AUTHOR:  Lindsay Marshall <lindsay.marshall@newcastle.ac.uk>
 *------------------------------------------------------------------------
 *    Copyright 1994 The University of Newcastle upon Tyne (see COPYRIGHT)
 *========================================================================
 *
 */

#include <ctype.h>
#include <stdio.h>
#include <string.h>

#include "frink.h"

#ifdef __bsdi
#include <sys/types.h>
#include <sys/malloc.h>
#else
#include <malloc.h>
#endif

static int inClass = 0;
static Token lbraceToken = {LBRACE, (char *) 0, 0, (Token *) 0, (Token *) 0};
static Token rbraceToken = {RBRACE, (char *) 0, 0, (Token *) 0, (Token *) 0};
static Token xcontToken = {XCONT, (char *) 0, 0, (Token *) 0, (Token *) 0};
static Token ostartToken = {OSTART, (char *) 0, 0, (Token *) 0, (Token *) 0};
static Token startToken = {START, (char *) 0, 0, (Token *) 0,(Token *) 0};
static Token contToken = {CONT, (char *) 0, 0, (Token *) 0, (Token *) 0};
static Token econtToken = {ECONT, (char *) 0, 0, (Token *) 0, (Token *) 0};
static Token emToken = {EM, (char *) 0, 0, (Token *) 0, (Token *) 0};
static Token enToken = {EN, (char *) 0, 0, (Token *) 0, (Token *) 0};
static Token nospToken = {NOSP, (char *) 0, 0, (Token *) 0, (Token *) 0};
static Token *olsToken = &emToken;

static Token dqStart = {DQSTART, (char *) 0, 0, (Token *) 0, (Token *) 0};
static Token dqEnd = {DQEND, (char *) 0, 0, (Token *) 0, (Token *) 0};
static Token ifToken = {CONST, "if", 2, (Token *) 0, (Token *) 0};
static Token thenToken = {CONST, "then", 4, (Token *) 0, (Token *) 0};
static Token switchToken = {CONST, "switch", 6, (Token *) 0, (Token *) 0};
static Token procToken = {CONST, "proc", 4, (Token *) 0, (Token *) 0};
static Token elseToken = {CONST, "else", 4, (Token *) 0, (Token *) 0};
static Token elseifToken = {CONST, "elseif", 6, (Token *) 0, (Token *) 0};
static Token methodToken = {CONST, "method", 6, (Token *) 0, (Token *) 0};
static Token destToken = {CONST, "destructor", 10, (Token *) 0, (Token *) 0};
static Token consToken = {CONST, "constructor", 11, (Token *) 0, (Token *) 0};
static Token publicToken = {CONST, "public", 6, (Token *) 0, (Token *) 0};
static Token commonToken = {CONST, "common", 6, (Token *) 0, (Token *) 0};
static Token protToken = {CONST, "protected", 9, (Token *) 0, (Token *) 0};
static Token bindToken = {CONST, "bind", 4, (Token *) 0, (Token *) 0};
static Token catchToken = {CONST, "catch", 5, (Token *) 0, (Token *) 0};
static Token classToken = {CONST, "itcl_class", 10, (Token *) 0, (Token *) 0};
static Token forToken = {CONST, "for", 3, (Token *) 0, (Token *) 0};
static Token foreachToken = {CONST, "foreach", 7, (Token *) 0, (Token *) 0};
static Token loopToken = {CONST, "loop", 4, (Token *) 0, (Token *) 0};
static Token whileToken = {CONST, "while", 5, (Token *) 0, (Token *) 0};
static Token timeToken = {CONST, "time", 4, (Token *) 0, (Token *) 0};
static Token exprToken = {CONST, "expr", 4, (Token *) 0, (Token *) 0};
static Token semiToken = {SEMI, ";", 1, (Token *) 0, (Token *) 0};
static Token nspaceToken = {CONST, "namespace", 9, (Token *) 0, (Token *) 0};
static Token evalToken = {CONST, "eval", 4, (Token *) 0, (Token *) 0};
static Token codeToken = {CONST, "code", 4, (Token *) 0, (Token *) 0};
static Token interpToken = {CONST, "interp", 6, (Token *) 0, (Token *) 0};
static Token uplevelToken = {CONST, "uplevel", 7, (Token *) 0, (Token *) 0};

extern void setIndent();
extern void outdent();

static int tokEqual(Token *tp, char *val)
{
   return (tp != (Token *) 0 && tp->type == CONST && tp->text != (char *) 0 &&
	   strcmp(tp->text, val) == 0);
}

static int tokIsLevel(Token *tp)
{
    char *cp;
    if (tp == (Token *) 0 || tp->type != CONST || tp->text == (char *) 0)
    {
	return 0;
    }
    cp = tp->text;
    if (*cp == '#')
    {
	cp++;
    }
    while (*cp)
    {
	if (*cp < '0' || *cp > '9')
	{
            return 0;
	}
	cp++;
    }
    return 1;
}

static int oneLine(Token *seq, int semis)
{
    while (seq != (Token *) 0)
    {
	if (seq->type == SCOMMENT || seq->type == NL || (semis && seq->type == SEMI)) { return 0; }
	seq = seq->next;
    }
    return 1;
}

static int checkSpecial(char *val)
{
    char ch;
    if (*val == '{') return 0;
    while ((ch = *val++))
    {
	switch (ch)
	{
	case '$' :
	case '[' :
	    return 0;
	}
    }
    return 1;
}

static int single(Token * tp)
{
    if (tp != (Token *) 0 && tp->next == (Token *) 0)
    {
	switch (tp->type)
	{
	case CONST :
	    return checkSpecial(tp->text);
	case CONC :
	case CALL :
	case VAR :
	    return 1;
	default :
	    return 0;
	}
    }
    return 0;
}

static int constant(Token * tp)
{
    return (tp != (Token *) 0 && tp->next == (Token *) 0
	&& tp->type == CONST && !checkSpecial(tp->text));
}

/*
 * If there are no more tokens, print a useful message to the user and
 * exit.
 */
void failIfNullToken(Token *token, char *part, char *command, int ln)
{
    char msg[1024];

    if (token == (Token *) 0)
    {
        if (ln == 0)
	{
	    sprintf(msg, "Missing `%s' part in `%s'", part, command);
	}
	else
	{
	    sprintf(msg, "Missing `%s' part in `%s', starting line %d",
	      part, command, ln);
	}
	fail(token, msg);
    }
}

/*
 * If there are more tokens after this one, print a useful warning to
 * the user.
 */
void warnIfArgsAfter(Token *token, char *part, char *command)
{
    char msg[1024];

    if (token != 0 && token->next != 0)
    {
        if (token->next->type == SCOMMENT)
	{
	    output(token->next, 0);
	}
	else
	{
	    sprintf(msg, "Extra arguments after `%s' part in `%s', line %d",
	      part, command, token->lineNo);
	    warn(token, msg);
	    warn(token->next, "Extra Token is ");
	}
    }
}

void body(Token *bod, int addBraces)
{
    Input *idx;
    Token *bl;
    int lnr;
    extern void sprocess(Token *, int);

    switch (bod->type)
    {
    case CONST :
	if (embrace)
	{
	    output(&lbraceToken, 1);
	    setIndent();
	    output((oneliner) ? olsToken : &startToken, 0);
	}
	output(bod , 1);
	if (embrace)
	{
	    outdent();
	    if (oneliner) { output(olsToken, 0); }
	    output(&rbraceToken, oneliner);
	}
	break;

    case VAR :
    case CONC :
	if (addBraces && embrace)
	{
	    output(&lbraceToken, 1);
	    setIndent();
	    output((oneliner) ? olsToken : &startToken, 0);
	}
	output(bod , 1);
	if (addBraces && embrace)
	{
	    outdent();
	    if (oneliner) { output(olsToken, 0); }
	    output(&rbraceToken, oneliner);
	}
	break;

    case LIST :
	idx = tokenise(bod, 0);
	bl = accumulate(idx, 1);
	untokenise(idx);
	if (debrace && bl != (Token *) 0 && bl->next == 0
	  && bl->type == CONST && !checkSpecial(bl->text))
	{
	    output(bl, 1);
	    freeToken(bl);
	}
	else if (oneliner && oneLine(bl, 0))
	{
	    output(&lbraceToken, 1);
	    setIndent();
	    output(olsToken, 0);
	    lprocess(bl, 1);
	    outdent();
	    output(olsToken, 0);
	    output(&rbraceToken, 1);
	    
	}
	else
	{
	    output(&lbraceToken, 0);
	    setIndent();
	    lprocess(bl, 1);
	    outdent();
	    output(&rbraceToken, 0);
	}
	break;

    case STRING :
	output(&dqStart, 1);
	setIndent();
	if ((lnr = oneliner && oneLine(bod->sequence, 1)))
	{
	    output(olsToken, 0);
	}
	sprocess(bod->sequence, 1);
	if (lnr) { output(olsToken, 0); }
	outdent();
	output(&dqEnd, 0);
	bod->sequence = (Token *) 0;
	break;

    case SLIST :
	output(bod, 0);
	break;

    case CALL :
	lprocess(bod->sequence, 0);
	bod->sequence = (Token *) 0;
	break;

    case SCOMMENT:
        output(bod, 1);
        break;
    case SEMI:
    case NL:
	blank();
	break;

    default :
	fail(bod, "Block error");
    }
}

typedef enum flags_e {
    NOBRACE	= 001,
    ADDBRACES	= 002,
    SEMIS	= 004,
    PAREN	= 010
} PressFlags;

static void press(Token *v , PressFlags flags)
{
    Input *idx;
    Token *token, *lst = (Token *) 0;
    int parcnt = 0;
    
    switch (v->type)
    {
    case SLIST:
	output(v, 1);
	break;
    case LIST :
	idx = tokenise(v, flags & PAREN);
	for(;;)
	{
	    token = getToken(idx);
	    switch (token->type)
	    {
	    case ENDF :
		freeToken(token);
		goto done;
	    case NL :
	    case SEMI :
	    case SCOMMENT :
		if (flags & SEMIS)
		{
		    tokenPush(&lst, token);
		    break;
		}
	    case COMMENT :
	    case SP :
		freeToken(token);
	    	break;
	    case LPAREN:
		parcnt += 1;
		tokenPush(&lst, token);
	        break;
	    case RPAREN :
		if ((parcnt -= 1) < 0)
		{
		    warn(v, "Possible missing '('");
		}
		tokenPush(&lst, token);
	        break;
	    default :
		tokenPush(&lst, token);
	    }
	    idx->lineStart = 0;
	}
done :
	untokenise(idx);
	if (parcnt > 0)
	{
	    warn(v, "Possible missing ')'");
	}
	if (((flags & NOBRACE) && debrace && single(lst)) || constant(lst))
	{
	    output(lst, 1);
	}
	else
	{
	    output(&lbraceToken, 1);
	    token = lst;
	    while (token != (Token *) 0)
	    {
		switch (token->type)
		{
		case SEMI :
		    if (flags && SEMIS)
		    {
			output(&semiToken, 1);
		    }
		    else
		    {
			output(&startToken, 1);
		    }
		    break;
		default :
		    output(token, 1);
		}
		token = token->next;
	    }
	    output(&rbraceToken, 1);
	}
	freeToken(lst);
	break;
	
    case SCOMMENT:
        output(v, 1);
	break;
	
    case SEMI :
    case NL :
	if (flags & SEMIS) { output(&semiToken, 1); }
    	break;

    default :
	if (embrace && (flags & ADDBRACES)) { output(&lbraceToken, 1); }
	output(v , 1);
	if (embrace && (flags & ADDBRACES)) { output(&rbraceToken, 1); }
    }
}

void etcetera(Token *cmd, int v)
{
    while (cmd != (Token *) 0)
    {
        output(cmd, v);
	cmd = cmd->next;
    }
}

void catbin(Token *tp)
{
    Token *sp;
    int sem, oln;
    if (tp != (Token *) 0)
    {
	if (tp->type == LIST)
	{
	    if (tp->length == 0)
	    {
		output(tp, 1);
		return;
	    }
	}
	else if (tp->type == STRING)
	{
	    if ((sp = tp->sequence) != (Token *) 0)
	    {
		if (((sp->type == SPART && sp->length == 0) ||
		     sp->type == SP) && sp->next == (Token *) 0)
		{
		    output(tp, 1);
		    return;
		}
	    }
	    else
	    {
		output(tp, 1);
		return;
	    }
	}
	oln = oneliner;
	oneliner = 1;
	sem = embrace;
	embrace = 0;
	olsToken = &nospToken;
	body(tp, 0);
	olsToken = &emToken;
	embrace = sem;
	oneliner = oln;
    }
}

static void handlescom(Token *tp)
{
    if (tp != (Token *) 0)
    {
        output(tp, 1);
    }
}

void makeCall (Token *prc, Token *arg)
{
    int sol;
    output(&startToken, 0);
    if (tokEqual(prc, "}")) { warn(prc, "unmatched } found"); }
    output(prc, 1);
    if (xf)
    {
	while (arg != (Token *) 0)
	{
	    if (arg->type == CONST && arg->text[0] == '-' &&
		arg->next != (Token *) 0)
	    {
		output(&xcontToken, 1);
		output(arg, 1);
/*
  Be careful here - watch out for the case where -command is used as an
  access function rather than to set the command!!
*/
		if (strcmp(arg->text, "-command") == 0)
		{
		    arg = arg->next;
		    setIndent();
		    sol = oneliner;
		    oneliner = 1;
		    olsToken = &enToken;
		    body(arg, 0);
		    olsToken = &emToken;
		    oneliner = sol;
		    outdent();
		    arg = arg->next;
		    continue;
		}
		arg = arg->next;
	    }
	    output (arg, 1);
	    arg = arg->next;
	}
    }
    else
    {
	while (arg != (Token *) 0)
	{
	    output(arg, 1);
	    if (tokEqual(arg, "-command") && arg->next != (Token *) 0)
	    {
		arg = arg->next;
		setIndent();
		sol = oneliner;
		oneliner = 1;
		olsToken = &enToken;
		body(arg, 0);
		olsToken = &emToken;
		oneliner = sol;
		outdent();
	    }
	    arg = arg->next;
	}
    }
}

void doswitch(Token *cmd)
{
    Token *tp, *bod;
    Input *idx;
    int ln = cmd->lineNo;

    output(&startToken, 0);
    output(&switchToken, 0 );
    tp = cmd;
    while (tp != (Token *) 0 && tp->type == CONST && tp->text[0] == '-')
    {
	output(tp , 1 );
        if (tp->length == 2 && tp->text[1] == '-')
        {
            tp = tp->next;
            break;
        }
	tp = tp->next;
    }
    failIfNullToken(tp, "string", "switch", ln);
    output(tp, 1);
    tp = tp->next;
    failIfNullToken(tp, "pattern", "switch", ln);
    if (tp->next != (Token *) 0 && tp->next->type != SCOMMENT)
    { /* this the non-list format */
	if (switchIn) { setIndent(); }
	while(tp != (Token *) 0 && tp->type != SCOMMENT)
	{
	    output(&contToken, 1 );
	    output(tp, 1);
	    tp = tp->next;
	    failIfNullToken(tp, "body", "switch", ln);
	    if (tokEqual(tp, "-"))
	    {
		output(tp, 1);
	    }
	    else
	    {
		setIndent();
		body(tp, 0);
		outdent();
	    }
	    tp = tp->next;
	}
	handlescom(tp);
	if (switchIn) { outdent(); }
    }
    else
    {
	switch (tp->type)
	{
	case LIST :
	    idx = tokenise(tp, 0);
	    tp = bod = accumulate(idx , 0);
	    untokenise(idx );
	    output(&lbraceToken, 0);
	    if (switchIn) { setIndent(); }
	    while (tp != (Token *) 0)
	    {
		if (tp->type == COMMENT) {
		    output(tp, 1 );
		}
		else
		{
		    output(&ostartToken, 1);
		    output(tp, 1 );
		    tp = tp->next;
		    failIfNullToken(tp, "body", "switch", ln);
		    if (tokEqual(tp, "-"))
		    {
			output(tp, 1);
		    }
		    else
		    {
			setIndent();
			body(tp, 1);
			outdent();
		    }
		}
		tp = tp->next;
	    }
	    if (switchIn) { outdent(); }
	    output(&rbraceToken, 0);
	    freeToken(bod);
	    break;
	case STRING :
/*	    break; */
	default :
	    failIfNullToken((Token *) 0, "body", "switch", ln);
	}
    }
}

static Token *wrap(Token *tp)
{
    Token *t1;
    extern Token *newToken(TokenType);

    switch (tp->type)
    {
    case CALL:
    case VAR:
	t1 = newToken(CONC);
	t1->sequence = tp;
        return t1;
    default:
	break;
    }
    return tp;
}

static void doConc(Token *cmd)
{
    Token *t1, *t2, *t3;
    extern Token *newToken(TokenType);
    extern Token *createToken(TokenType t, char *text, Token *nxt);

    t1 = cmd->sequence;
    if ((t2 = t1->next) == (Token *) 0) return;
    if (t2->next != (Token *) 0)
    {
	return;
    }
    if (strcmp(t2->text, "=={}") == 0)
    {
	freeToken(t2);
	t1->next = (Token *) 0;
	t3 = newToken(CALL);
	t3->next = (Token *) 0;
	t3->sequence = createToken(CONST, "string",
		      createToken(CONST, "match",
		      createToken(LIST, "", wrap(t1))));
	cmd->sequence = t3;
    }
    else if (strcmp(t2->text, "!={}") == 0)
    {
	freeToken(t2);
	t1->next = (Token *) 0;
	t2 = newToken(CONC);
	t3 = newToken(CALL);
	t3->sequence = createToken(CONST, "string",
	      createToken(CONST, "match",
	      createToken(LIST, "", wrap(t1))));
	t2->sequence = createToken(CPART,"!", t3);
	cmd->sequence = t2;
    }
}

static void testOpt(Token *cmd)
{
    Token *t1, *t2, *t3, *t4;
    Input *idx;
    extern Token *newToken(TokenType);
    extern Token *createToken(TokenType t, char *text, Token *nxt);
    extern void dumpToken(Token*);

    dumpToken(cmd);
    switch (cmd->type)
    {
    case LIST :
        idx = tokenise(cmd, 0);
	t1 = getToken(idx);
	if (t1->type == ENDF) {
	     freeToken(t1);
	     untokenise(idx);
	     break;
	}
	t2 = getToken(idx);
	if (t2->type == ENDF) {
	    if (t1->type == CONC)
	    {
	    }
	    freeToken(t1); freeToken(t2);
	    untokenise(idx);
	    break;
	}
	t3 = getToken(idx);
	if (t3->type == ENDF) {
	    freeToken(t1); freeToken(t2); freeToken(t3);
	    untokenise(idx);
	    break;
	}
	if (t2->type != CONST || (strcmp(t2->text, "==") != 0 &&
				  strcmp(t2->text, "!=") != 0))
	{
	    freeToken(t1); freeToken(t2); freeToken(t3);
	    untokenise(idx);
	    break;
	}
	t4 = getToken(idx);
	if (t4->type != ENDF || (t1->type != LIST && t2->type != LIST)) {
	    freeToken(t1); freeToken(t2); freeToken(t3); freeToken(t4);
	    untokenise(idx);
	    break;
	}
	freeToken(t4);
	untokenise(idx);
	if (t1->type == LIST && t1->length == 0)
	{
	    
	}
	else if (t3->type == LIST && t3->length == 0)
	{
	    
	}
	break;
    case CONC : doConc(cmd); break;
    default : break;
    }
}

void doif(Token *cmd)
{
    Token *tp, *then;
    int efl = ADDBRACES, ln = cmd->lineNo;
    output(&startToken, 0 );
    if (eValue == 2) {
	/* generate switch statements instead of ifs... */
    }
    output(&ifToken, 0 );
    failIfNullToken(cmd, "condition", "if", ln);
    if (eValue == 1) { testOpt(cmd); }
    press(cmd, NOBRACE | ADDBRACES | PAREN);
    if (putThen) { output(&thenToken, 0); }
    then = cmd->next;
    if (tokEqual(then, "then")) { then = then->next; }
    failIfNullToken(then, "then", "if", ln);
    body(then, 0);
    tp = then->next;
    if (elseif) { efl |= NOBRACE; }
    while(tokEqual(tp, "elseif"))
    {
	output(&econtToken, 1);
	output(&elseifToken, 0);
	tp = tp->next;
	failIfNullToken(tp, "condition", "elseif", ln);
	if (eValue == 1) { testOpt(tp);	}
	press(tp, efl | PAREN);
	tp = tp->next;
	failIfNullToken(tp, "body", "elseif", ln);
	body(tp, 0);
	tp = tp->next;;
    }
    if (tokEqual(tp, "else"))
    {
	if (putElse) { output(&elseToken, 0 ); }
	tp = tp->next;
	failIfNullToken(tp, "body", "else", ln);
	body(tp, 0);
    }
    else if (tp != (Token *) 0)
    {
        if (tp->type != SCOMMENT)
	{
	    if (putElse) { output(&elseToken, 0 ); }
	    body(tp, 0);
	    warnIfArgsAfter(tp, "else", "if");
	}
	else
	{
	    output(tp, 0);
	}
    }
}

void checkNSBody(Token *cmd, int ln)
{
    if (tokEqual(cmd, "eval"))
    {
        output(&evalToken, 0);
	cmd = cmd->next;
	failIfNullToken(cmd, "name", "namespace", ln);
	output(cmd, 0);
	cmd = cmd->next;
	failIfNullToken(cmd, "script", "namespace", ln);
	if (cmd->next != (Token *) 0)
	{
	    etcetera(cmd, 0);
	}
	else
	{
	    catbin(cmd);
	}
    }
    else if (tokEqual(cmd, "code"))
    {
        output(&codeToken, 0);
	cmd = cmd->next;
	failIfNullToken(cmd, "script", "namespace", ln);
	catbin(cmd);
    }
    else
    {
	etcetera(cmd, 0);
    } 
}

void donamespace(Token *cmd)
{
    int ln = cmd->lineNo;
    output(&startToken, 0);
    output(&nspaceToken, 0);
    checkNSBody(cmd, ln);
}

void dointerp(Token *cmd)
{
    int ln = cmd->lineNo;
    output(&startToken, 0);
    output(&interpToken, 0);
    if (tokEqual(cmd, "eval"))
    {
        output(&evalToken, 0);
	cmd = cmd->next;
	failIfNullToken(cmd, "path", "interp", ln);
	output(cmd, 0);
	cmd = cmd->next;
	failIfNullToken(cmd, "script", "interp", ln);
	if (cmd->next != (Token *) 0)
	{
	    output(cmd, 0);
	}
	else
	{
	    catbin(cmd);
	}
    }
    else
    {
	etcetera(cmd, 0);
    } 
}

void checkNS(Token *cmd, int ln)
{
    if (tokEqual(cmd, "namespace"))
    {
	output(&nspaceToken, 0);
	checkNSBody(cmd->next, ln);
    }
    else
    {
        etcetera(cmd, 0);
    }
}

void douplevel(Token *cmd)
{
    int ln = cmd->lineNo;
    
    output(&startToken, 0);
    output(&uplevelToken, 0);
    if (tokIsLevel(cmd))
    {
	output(cmd, 0);
	cmd = cmd->next;
    }
    failIfNullToken(cmd, "command", "uplevel", ln);
    if (cmd->type == LIST && cmd->next == (Token *) 0)
    {
        catbin(cmd);
    }
    else
    {
        checkNS(cmd, ln);
    }
}

void doProc(Token *tag, Token *cmd)
{
    int ln = tag->lineNo;
  
    if (tag == &methodToken && !inClass)
    {
	makeCall(tag, cmd);
    }
    else
    {
	output(&startToken, 0);
	output(tag, 0);
	failIfNullToken(cmd, "name", tag->text, ln);
	output(cmd, 1);
	cmd = cmd->next;
	failIfNullToken(cmd, "args", tag->text, ln);
	press(cmd, NOBRACE | ADDBRACES);
	cmd = cmd->next;
	failIfNullToken(cmd, "body", tag->text, ln);
	body(cmd, 0);
	warnIfArgsAfter(cmd, "body", tag->text);
    }
}

void doproc(Token *cmd) { doProc(&procToken, cmd); }

void domethod(Token *cmd) { doProc(&methodToken ,cmd); }

void dodestructor(Token *cmd)
{
    if (!inClass)
    {
	makeCall(&destToken, cmd);
    }
    else
    {
	output(&startToken, 0);
	output(&destToken , 0);
        failIfNullToken(cmd, "body", "destructor", 0);
	body(cmd, 0);
        warnIfArgsAfter(cmd, "body", "destructor");
    }
}

void doconstructor(Token *cmd)
{
    if (!inClass)
    {
	makeCall(&consToken, cmd);
    }
    else
    {
	output(&startToken, 0);
	output(&consToken, 0);
        failIfNullToken(cmd, "args", "constructor", 0);
	press(cmd, NOBRACE | ADDBRACES);
        cmd = cmd->next;
        failIfNullToken(cmd, "body", "constructor", 0);
        body(cmd, 0);
        warnIfArgsAfter(cmd, "body", "constructor");
    }
}

void dobind(Token *cmd)
{
    Token *np;
    if (!doBind)
    {
	makeCall(&bindToken,cmd);
    }
    else
    {
	output(&startToken, 0);
	output(&bindToken, 0);
        failIfNullToken(cmd, "windowSpec", "bind", 0);
	output(cmd, 1);
	if ((np = cmd->next) != (Token *) 0)
	{
	    output(np, 1);
            if ((np->next) != (Token *) 0)
            {
                catbin(np->next);
                warnIfArgsAfter(np->next, "command", "bind");
            }
	}
    }
}

void docatch(Token *cmd)
{
    if (!doCatch)
    {
	makeCall(&catchToken,cmd);
    }
    else
    {
	output(&startToken, 0);
	output(&catchToken, 0);

        failIfNullToken(cmd, "script", "catch", 0);
	catbin(cmd);
        if ((cmd = cmd->next) != (Token *) 0)
        {
            output(cmd, 1);
            warnIfArgsAfter(cmd, "varName", "catch");
        }
    }
}

void doitcl_class(Token *cmd)
{
    output(&startToken, 0);
    output(&classToken, 0);
    failIfNullToken(cmd, "className", "itcl_class", 0);
    output(cmd, 1);
    inClass += 1;
    cmd = cmd->next;
    failIfNullToken(cmd, "body", "itcl_class", 0);
    body(cmd, 0);
    warnIfArgsAfter(cmd, "body", "itcl_class");
    inClass += -1;
}

void docvar(Token *cmd, Token *prt)
{
    if (!inClass)
    {
	makeCall(prt, cmd);
    }
    else
    {
	output(&startToken, 0);
	output(prt, 0);
        failIfNullToken(cmd, "varName", prt->text, 0);
	output(cmd, 1);
        if ((cmd = cmd->next) != (Token *) 0)
        {
            press(cmd, NOBRACE | ADDBRACES);
            warnIfArgsAfter(cmd, "init", prt->text);
        }
    }
}

void dopublic(Token *cmd)
{
    if (!inClass)
    {
	makeCall(&publicToken, cmd);
    }
    else
    {
	output(&startToken, 0);
	output(&publicToken, 0);
        failIfNullToken(cmd, "varName", "public", 0);
	output(cmd, 1);
	if ((cmd = cmd->next) != (Token *) 0)
	{
	    press(cmd, NOBRACE | ADDBRACES);
            if ((cmd = cmd->next) != (Token *) 0)
            {
                body(cmd, 0);
                warnIfArgsAfter(cmd, "config", "public");
            }
	}
    }
}

void doprotected(Token *cmd) { docvar(cmd, &protToken); }

void docommon(Token *cmd) { docvar(cmd, &commonToken); }

void doforeach(Token *cmd)
{
    output(&startToken, 0);
    output(&foreachToken, 0 );
    failIfNullToken(cmd, "varName", "foreach", 0);
    do
    {
	output(cmd, 0);
	cmd = cmd->next;
	failIfNullToken(cmd, "list", "foreach", 0);
	press(cmd, NOBRACE);
	cmd = cmd->next;
    }
    while (cmd != (Token *) 0 && cmd->next != (Token *) 0);
    failIfNullToken(cmd, "body", "foreach", 0);
    body(cmd, 0);
}

void dofor(Token *cmd )
{
   int ln = cmd->lineNo;
  
    output(&startToken, 0);
    output(&forToken, 0 );
    
    press(cmd, NOBRACE | ADDBRACES | SEMIS);
    cmd = cmd->next;
    failIfNullToken(cmd, "test", "for", ln);
    press(cmd, ADDBRACES | PAREN);
    cmd = cmd->next;
    failIfNullToken(cmd, "next", "for", ln);
    press(cmd, NOBRACE | ADDBRACES | SEMIS);
    cmd = cmd->next;
    failIfNullToken(cmd, "body", "for", ln);
    body(cmd, 0);
    warnIfArgsAfter(cmd, "body", "for");
}

void doloop(Token *cmd )
{
    Token *tp;
    if (!tclX)
    {
	makeCall(&loopToken, cmd);
    }
    else
    {
	output(&startToken, 0);
	output(&loopToken, 0 );				/* loop */
        failIfNullToken(cmd, "var", "loop", 0);
	press(cmd, NOBRACE | ADDBRACES);		/* var */
        tp = cmd->next;
        failIfNullToken(tp, "first", "loop", 0);
        press(tp, ADDBRACES);                           /* first */
        tp = tp->next;
        failIfNullToken(tp, "limit", "loop", 0);
        press(tp, ADDBRACES);                           /* limit */

	tp = tp->next;
        failIfNullToken(tp, "body", "loop", 0);

	if (tp->next != (Token *) 0)
	{
	    press(tp, NOBRACE | ADDBRACES);		/* incr */
            tp = tp->next;
	}
	body(tp, 0);			/* body */
        warnIfArgsAfter(tp, "body", "loop");
    }
}
 
void dowhile(Token *cmd)
{
    int ln = cmd->lineNo;
  
    output(&startToken, 0 );
    output(&whileToken, 0 );
    press(cmd, ADDBRACES | PAREN);
    cmd = cmd->next;
    failIfNullToken(cmd, "body", "while", ln);
    body(cmd, 0);
    warnIfArgsAfter(cmd, "body", "while");
}
 
 
void doexpr(Token *cmd)
{
    if (!doExpr)
    {
	makeCall(&catchToken,cmd);
    }
    else
    {
	output(&startToken, 0);
	output(&exprToken, 0);

        failIfNullToken(cmd, "expression", "expr", 0);
        if (cmd->next == (Token *) 0)
	{
	    press(cmd, ADDBRACES | PAREN);
	}
	else   
        {
	    etcetera(cmd, 1);
        }
    }
}
 
void dotime(Token *cmd)
{
    if (!doTime)
    {
	makeCall(&catchToken,cmd);
    }
    else
    {
	output(&startToken, 0 );
	output(&timeToken, 0 );

        failIfNullToken(cmd, "script", "time", 0);
	catbin(cmd);
	if (cmd->next != (Token *) 0)
	{
	    output(cmd = cmd->next, 1);
	    warnIfArgsAfter(cmd, "count", "time");
	}
    }
}

static struct OpTable
{
    char *procName;
    void (*func)(Token *);
} opTable[] =
{
    {"bind", dobind},				/* 0 */
    {"catch", docatch},				/* 1 */
    {"common", docommon},
    {"constructor", doconstructor},
    {"destructor", dodestructor},		/* 4 */
    {"expr", doexpr},				/* 5 */
    {"for", dofor},				/* 6 */
    {"foreach", doforeach},
    {"if", doif},				/* 8 */
    {"interp", dointerp},			
    {"itcl_class", doitcl_class},
    {"loop", doloop},				/* 11 */
    {"method", domethod},			/* 12 */
    {"namespace", donamespace},			/* 13 */
    {"proc", doproc},				/* 14 */
    {"protected", doprotected},
    {"public", dopublic},
    {"switch", doswitch},			/* 17 */
    {"time", dotime},				/* 18 */
    {"uplevel", douplevel}, 			/* 19 */
    {"while", dowhile},				/* 20 */
    {(char *) 0, 0}
};

static int starter[26] = {
    -1, 0, 1, 4, 5, 6, -1, -1, 8, -1, -1, 11, 12,
    13, -1, 14, -1, -1, 17, 18, 19, -1, 20, -1, -1, -1
};

int tclop(Token *hd, Token *line)
{
    struct OpTable *op = opTable;
    int indx;
    char ch = *hd->text, msg[80];

    if (!islower(ch) || (indx = starter[(int) ch - 'a']) < 0)
    {
	return 0;
    }
    op = &opTable[indx];
    while (op->procName != (char *) 0)
    {
	switch (strcmp(op->procName, hd->text))
	{
	case 0:
	    {
	    	if (line == 0)
		{
		    sprintf(msg, "%s used with no parameters!\n", hd->text);
		    warn(hd, msg);
		    output(&startToken, 0);
		    output(hd, 0);
		}
		else
		{
		    (*op->func)(line);
		}
		return 1;
	    }
	case 1 :
	    return 0;
	}
	op++;
    }
    return 0;
}
