# copyright (C) 1997-98 Jean-Luc Fontaine (mailto:jfontain@mygale.org)
# this program is free software: please refer to the BSD type license enclosed in this package

set rcsId {$Id: blt2d.tcl,v 1.13 1998/04/21 16:36:43 jfontain Exp $}

class blt2DViewer {

    # use pie colors for consistency
    set blt2DViewer::(colors) {#7FFFFF #7FFF7F #FF7F7F #FFFF7F #7F7FFF #FFBF00 #BFBFBF #FF7FFF #FFFFFF}

    proc blt2DViewer {this path} viewer {} {
        $path configure -plotbackground $widget::(default,ButtonBackgroundColor)
        $path yaxis configure -title {} -min 0 -tickfont $font::(smallNormal)
        $path legend configure -borderwidth 1 -font $font::(mediumNormal)

        # allow dropping of data cells
        set blt2DViewer::($this,drop) [new dropSite\
            -path $path -formats DATACELLS -command "viewer::view $this \$dragSite::data(DATACELLS)"\
        ]

        set blt2DViewer::($this,drag) [new dragSite -path $path -validcommand "blt2DViewer::validateDrag $this"]
        dragSite::provide $blt2DViewer::($this,drag) OBJECTS "blt2DViewer::dragData $this"                ;# provide list of objects
        dragSite::provide $blt2DViewer::($this,drag) DATACELLS "blt2DViewer::dragData $this"

        set blt2DViewer::($this,elements) {}                                             ;# initialize list of data line identifiers
        set blt2DViewer::($this,selector) [new objectSelection -selectcommand "blt2DViewer::setElementsState $this"]
        bind $path <ButtonRelease-1> "blt2DViewer::setSelection $this %x %y"
        bind $path <Control-ButtonRelease-1> "blt2DViewer::toggleSelection $this %x %y"
        bind $path <Shift-ButtonRelease-1> "blt2DViewer::extendSelection $this %x %y"

        set blt2DViewer::($this,colorIndex) 0
        set blt2DViewer::($this,path) $path
    }

    proc ~blt2DViewer {this} {
        delete $blt2DViewer::($this,drag) $blt2DViewer::($this,drop) $blt2DViewer::($this,selector)
        eval delete $blt2DViewer::($this,elements)                                                       ;# delete existing elements
    }

    proc supportedTypes {this} {
        return {integer real}
    }

    proc dragData {this format} {
        set selectedElements [objectSelection::selected $blt2DViewer::($this,selector)]
        switch $format {
            OBJECTS {
                if {[llength $selectedElements]>0} {
                    return $selectedElements                                            ;# return selected elements if there are any
                } elseif {[llength $blt2DViewer::($this,elements)]==0} {
                    return $this                                                   ;# return graph itself if it contains no elements
                } else {
                    return {}                                                                            ;# return nothing otherwise
                }
            }
            DATACELLS {
                return [cellsFromElements $this $selectedElements]
            }
        }
    }

    proc validateDrag {this x y} {
        if {\
            ([llength [objectSelection::selected $blt2DViewer::($this,selector)]]>0)||\
            ([llength $blt2DViewer::($this,elements)]==0)\
        } {
            return 1                                                              ;# allow drag of selected elements or empty viewer
        } else {
            return 0                                                                                              ;# nothing to drag
        }
    }

    proc monitorCell {this array row column} {
        viewer::registerTrace $this $array
        set cell ${array}($row,$column)
        if {[lsearch -exact [cellsFromElements $this $blt2DViewer::($this,elements)] $cell]>=0} return     ;# already charted, abort
        set element [newElement $this $blt2DViewer::($this,path) -label [viewer::label $array $row $column]]
        switched::configure $element -color [lindex $blt2DViewer::(colors) $blt2DViewer::($this,colorIndex)]
        # keep track of element existence
        switched::configure $element -deletecommand "blt2DViewer::deletedElement $this $array $element"
        # circle through colors
        set blt2DViewer::($this,colorIndex) [expr {($blt2DViewer::($this,colorIndex)+1)%[llength $blt2DViewer::(colors)]}]
        lappend blt2DViewer::($this,elements) $element                                                     ;# register new data line
        set blt2DViewer::($this,cell,$element) $cell
        objectSelection::add $blt2DViewer::($this,selector) $element
    }

    proc deletedElement {this array element} {
        viewer::unregisterTrace $this $array                                          ;# trace may no longer be needed on this array
        ldelete blt2DViewer::($this,elements) $element
        objectSelection::remove $blt2DViewer::($this,selector) $element
        unset blt2DViewer::($this,cell,$element)
        if {[llength $blt2DViewer::($this,elements)]==0} {
            delete $this                                                            ;# self destruct when there are no more elements
        }
    }

    proc update {this array args} {                                                               ;# update display using cells data
        updateTimeDisplay $this [set seconds [clock seconds]]
        foreach element $blt2DViewer::($this,elements) {
            set cell $blt2DViewer::($this,cell,$element)
            if {[string first $array $cell]<0} continue                                  ;# check that cell belongs to updated array
            set value {}                                                        ;# set default to void in case data no longer exists
            catch {set value [set $cell]}
            updateElement $this $element $seconds $value
        }
    }

    virtual proc newElement {this path args}                                       ;# let derived class create an element of its own

    virtual proc updateElement {this element seconds value}      ;# let derived class (such as graph, bar chart, ...) update element

    virtual proc updateTimeDisplay {this seconds} {}        ;# eventually let derived class (such as graph) update axis, for example

    proc cellsFromElements {this elements} {
        set cells {}
        foreach element $elements {
            lappend cells $blt2DViewer::($this,cell,$element)
        }
        return $cells
    }

    proc setElementsState {this elements select} {
        if {$select} {
            set action activate
        } else {
            set action deactivate
        }
        set path $blt2DViewer::($this,path)
        foreach element $elements {
            $path legend $action $element
        }
    }

    proc setSelection {this x y} {
        if {[string length [set element [$blt2DViewer::($this,path) legend get @$x,$y]]]>0} {                         ;# in a legend
            objectSelection::select $blt2DViewer::($this,selector) $element
        }
    }

    proc toggleSelection {this x y} {
        if {[string length [set element [$blt2DViewer::($this,path) legend get @$x,$y]]]>0} {                         ;# in a legend
            objectSelection::toggle $blt2DViewer::($this,selector) $element
        }
    }

    proc extendSelection {this x y} {
        if {[string length [set element [$blt2DViewer::($this,path) legend get @$x,$y]]]>0} {                         ;# in a legend
            objectSelection::extend $blt2DViewer::($this,selector) $element
        }
    }

}
