# copyright (C) 1997-98 Jean-Luc Fontaine (mailto:jfontain@mygale.org)
# this program is free software: please refer to the BSD type license enclosed in this package

set rcsId {$Id: canvhand.tcl,v 1.9 1998/02/20 21:39:32 jfontain Exp $}

class canvasHandles {     ;# embed widget in a frame with handles for resizing and moving inside a canvas acting as a window manager
    proc canvasHandles {this parentPath args} composite {[new frame $parentPath] $args} {
        if {[string compare [winfo class $parentPath] Canvas]!=0} {
            error "parent must be the manager canvas"
        }
        set path $widget::($this,path)
        bind $path <Configure> "canvasHandles::resize $this %w %h"                                           ;# monitor size changes
        bind $path <Motion> "canvasHandles::setCursor $this %x %y"
        bind $path <Enter> "canvasHandles::setCursor $this %x %y"     ;# when just entering window, no motion event is yet generated
        bind $path <Button1-Motion> "canvasHandles::buttonMotion $this %x %y"
        bind $path <ButtonPress-1> "canvasHandles::buttonPress $this %x %y"
        bind $path <ButtonRelease-1> "canvasHandles::buttonRelease $this"
        set canvasHandles::($this,item) [$parentPath create window 0 0 -window $path -anchor nw]
        set canvasHandles::($this,canvas) $parentPath
        set canvasHandles::($this,raised) 1
        composite::complete $this
    }

    proc ~canvasHandles {this} {
        $canvasHandles::($this,canvas) delete $canvasHandles::($this,item) outline       ;# delete canvas items (eventually outline)
        catch {delete $canvasHandles::($this,bindings)}                                                ;# eventually delete bindings
    }

    proc options {this} {
        return [list\
            [list\
                -background background Background $widget::(default,ButtonBackgroundColor) $widget::(default,ButtonBackgroundColor)\
            ]\
            [list -borderwidth borderWidth BorderWidth 3]\
            [list -handlesize handleSize HandleSize 7 7]\
            [list -path path Path {} {}]\
            [list -relief relief Relief ridge]\
        ]
    }

    proc set-handlesize {this value} {
        resize $this [winfo width $widget::($this,path)] [winfo height $widget::($this,path)]                 ;# recalculate handles
    }

    proc set-path {this value} {
        if {![winfo exists $value]} {
            error "invalid widget: \"$value\""
        }
        set path $widget::($this,path)
        catch {eval pack forget [pack slaves $path]}                                            ;# eventually forget existing widget
        catch {delete $canvasHandles::($this,bindings)}                                       ;# eventually delete existing bindings
        set canvasHandles::($this,bindings) [new bindings $value end]
        bindings::set $canvasHandles::($this,bindings) <Destroy> "delete $this"  ;# self destruct when managed widget gets destroyed
        raise $value $path
        pack $value -in $path -fill both -expand 1                                                        ;# expand as manager frame
    }

    foreach option {-background -relief -borderwidth} {
        proc set$option {this value} "\$widget::(\$this,path) configure $option \$value"
    }

    proc buttonMotion {this x y} {
        set canvasHandles::(motion) {}
        canvasHandles::updateOutline $this $x $y
    }

    proc buttonPress {this x y} {
        set canvasHandles::(xLast) $x; set canvasHandles::(yLast) $y
        canvasHandles::createOutline $this
    }

    proc buttonRelease {this} {
        if {[info exists canvasHandles::(motion)]} {                                                   ;# moving or resizing occured
            canvasHandles::updateGeometry $this
            raise $widget::($this,path)                                              ;# always place frame on top after acting on it
            set canvasHandles::($this,raised) 1
            unset canvasHandles::(motion)
        } else {         ;# no moving or resizing occured, toggle place in statcking order so that other managed widgets may be seen
            if {$canvasHandles::($this,raised)} {
                lower $widget::($this,path)
                set canvasHandles::($this,raised) 0
            } else {
                raise $widget::($this,path)
                set canvasHandles::($this,raised) 1
            }
        }
        catch {raise $composite::($this,-path) $widget::($this,path)}     ;# maintain managed widget (if it exists) just above frame
        canvasHandles::destroyOutline $this
        unset canvasHandles::(xLast) canvasHandles::(yLast) canvasHandles::(hidden)
    }

    proc resize {this width height} {
        # handle size should not be less than border width because of corners
        set size [maximum $composite::($this,-handlesize) $composite::($this,-borderwidth)]
        # recalculate handles limits (mid handles disappear when frame gets too small so that it stays movable)

        set halfHeight [expr {($height/2)}]
        set canvasHandles::($this,topHandleBottom) [minimum $size $halfHeight] ;# top corner handle bottom cannot exceed half height
        set canvasHandles::($this,bottomHandleTop) [expr {$height-$canvasHandles::($this,topHandleBottom)}]
        # mid handle top cannot be to close to top corner handle bottom
        set canvasHandles::($this,midHandleTop)\
            [maximum [expr {$halfHeight-$size}] [expr {$canvasHandles::($this,topHandleBottom)+$size}]]
        # mid handle bottom limit cannot be greater than bottom corner handle top
        set canvasHandles::($this,midHandleBottom)\
            [minimum [expr {$halfHeight+$size}] [expr {$canvasHandles::($this,bottomHandleTop)-$size}]]
        # note: mid handle top can be greater than mid handle bottom when handle disappears

        set halfWidth [expr {($width/2)}]
        set canvasHandles::($this,leftHandleRight) [minimum $size $halfWidth]   ;# left corner handle right cannot exceed half width
        set canvasHandles::($this,rightHandleLeft) [expr {$width-$canvasHandles::($this,leftHandleRight)}]
        # mid handle left cannot be less than left corner handle right
        set canvasHandles::($this,midHandleLeft)\
            [maximum [expr {$halfWidth-$size}] [expr {$canvasHandles::($this,leftHandleRight)+$size}]]
        # mid handle right limit cannot be greater than right corner handle left
        set canvasHandles::($this,midHandleRight)\
            [minimum [expr {$halfWidth+$size}] [expr {$canvasHandles::($this,rightHandleLeft)-$size}]]
        # note: mid handle left can be greater than mid handle right when handle disappears
    }

    proc setCursor {this x y} {
        if {[info exists canvasHandles::(motion)]} {
            return        ;# make sure not to change cursor while moving outline (may happen when pointer passes over manager frame)
        }
        set border $composite::($this,-borderwidth)
        set path $widget::($this,path)
        set cursor fleur                                                                        ;# use moving cursor outside borders
        set direction {}
        if {$x<$border} {
            set side left
            set direction w
        } elseif {$x>=([winfo width $path]-$border)} {
            set side right
            set direction e
        }
        if {[info exists side]} {                                                                            ;# in a vertical border
            if {$y<$canvasHandles::($this,topHandleBottom)} {
                set cursor top_${side}_corner
                append direction n
            } elseif {$y>$canvasHandles::($this,bottomHandleTop)} {
                set cursor bottom_${side}_corner
                append direction s
            } elseif {($y>$canvasHandles::($this,midHandleTop))&&($y<$canvasHandles::($this,midHandleBottom))} {
                set cursor ${side}_side
            } else {
                set cursor fleur
                set direction {}
            }
        } else {
            if {$y<$border} {
                set side top
                set direction n
            } elseif {$y>=([winfo height $path]-$border)} {
                set side bottom
                set direction s
            }
            if {[info exists side]} {                                                                     ;# in an horizontal border
                if {$x<$canvasHandles::($this,leftHandleRight)} {
                    set cursor ${side}_left_corner
                    append direction w
                } elseif {$x>$canvasHandles::($this,rightHandleLeft)} {
                    set cursor ${side}_right_corner
                    append direction e
                } elseif {($x>$canvasHandles::($this,midHandleLeft))&&($x<$canvasHandles::($this,midHandleRight))} {
                    set cursor ${side}_side
                } else {
                    set cursor fleur
                    set direction {}
                }
            }
        }
        if {[string compare $cursor [$widget::($this,path) cget -cursor]]!=0} {                    ;# update cursor only when needed
            $widget::($this,path) configure -cursor $cursor
            update idletasks                                                    ;# make cursor immediately visible for user feedback
        }
        set canvasHandles::($this,direction) $direction
    }

    proc updateOutline {this x y} {                                                     ;# coordinates are relative to manager frame
        if {$canvasHandles::(hidden)} {                                                        ;# make sure outline is fully visible
            positionOutlineInStackingOrder $this raise
        }
        set canvas $canvasHandles::($this,canvas)
        set coordinates [$canvas coords $canvasHandles::($this,item)]
        # make sure that pointer stays within canvas boundaries
        set xFrame [lindex $coordinates 0]
        set yFrame [lindex $coordinates 1]
        if {($xFrame+$x)<0} {
            set x [expr {-$xFrame}]                                                 ;# use expr to properly handle consecutive signs
        }
        if {($yFrame+$y)<0} {
            set y [expr {-$yFrame}]                                                 ;# use expr to properly handle consecutive signs
        }
        set width [winfo width $canvas]
        if {($xFrame+$x)>=$width} {
            set x [expr {$width-$xFrame-1}]
        }
        set height [winfo height $canvas]
        if {($yFrame+$y)>=$height} {
            set y [expr {$height-$yFrame-1}]
        }

        if {[string length $canvasHandles::($this,direction)]==0} {                                          ;# moving, not resizing
            $canvas move outline [expr {$x-$canvasHandles::(xLast)}] [expr {$y-$canvasHandles::(yLast)}]
            set canvasHandles::(xLast) $x
            set canvasHandles::(yLast) $y
            return
        }

        set width [winfo width $widget::($this,path)]
        set height [winfo height $widget::($this,path)]

        switch $canvasHandles::($this,direction) {                                                                       ;# resizing
            nw - wn {
                displayOutline $this [expr {$xFrame+$x}] [expr {$yFrame+$y}] [expr {$width-$x}] [expr {$height-$y}]
            }
            n {
                displayOutline $this $xFrame [expr {$yFrame+$y}] $width [expr {$height-$y}]
            }
            ne - en {
                displayOutline $this $xFrame [expr {$yFrame+$y}] $x [expr {$height-$y}]
            }
            e {
                displayOutline $this $xFrame $yFrame $x $height
            }
            se - es {
                displayOutline $this $xFrame $yFrame $x $y
            }
            s {
                displayOutline $this $xFrame $yFrame $width $y
            }
            sw - ws {
                displayOutline $this [expr {$xFrame+$x}] $yFrame [expr {$width-$x}] $y
            }
            w {
                displayOutline $this [expr {$xFrame+$x}] $yFrame [expr {$width-$x}] $height
            }
        }
    }

    proc createOutline {this} {
        # create outline borders (a single frame with no background cannot be used for it hides underlying windows)
        set canvas $canvasHandles::($this,canvas)
        foreach side {top bottom left right} {
            set frame $widget::([new frame $canvas -background black],path)                             ;# make sure frame is unique
            # items are static because there can be only 1 outline at a time
            set canvasHandles::($side,item) [$canvas create window 0 0 -window $frame -width 0 -height 0 -anchor nw -tags outline]
        }
        positionOutlineInStackingOrder $this lower                                    ;# hide outline for now and make it fit widget
        eval displayOutline $this [$canvas coords $canvasHandles::($this,item)]\
            [winfo width $widget::($this,path)] [winfo height $widget::($this,path)]
    }

    proc positionOutlineInStackingOrder {this order} {                                        ;# order must be either raise or lower
        set canvas $canvasHandles::($this,canvas)
        foreach side {top bottom left right} {
            $order [$canvas itemcget $canvasHandles::($side,item) -window]
        }
        set canvasHandles::(hidden) [string compare $order raise]
    }

    proc displayOutline {this x y width height} {                                              ;# coordinates are relative to canvas
        set minimum [expr {(2*$composite::($this,-borderwidth))+1}]                    ;# make sure managed widget is always visible
        set width [maximum $minimum $width]
        set height [maximum $minimum $height]
        set canvas $canvasHandles::($this,canvas)
        $canvas coords $canvasHandles::(top,item) $x $y
        $canvas coords $canvasHandles::(bottom,item) $x [expr {$y+$height-1}]
        $canvas coords $canvasHandles::(left,item) $x $y
        $canvas coords $canvasHandles::(right,item) [expr {$x+$width-1}] $y
        $canvas itemconfigure $canvasHandles::(top,item) -width $width
        $canvas itemconfigure $canvasHandles::(bottom,item) -width $width
        $canvas itemconfigure $canvasHandles::(left,item) -height $height
        $canvas itemconfigure $canvasHandles::(right,item) -height $height
    }

    proc destroyOutline {this} {
        set canvas $canvasHandles::($this,canvas)
        foreach side {top bottom left right} {
            destroy [$canvas itemcget $canvasHandles::($side,item) -window]                                    ;# destroy side frame
            unset canvasHandles::($side,item)
        }
        $canvas delete outline                                                                                  ;# delete side items
    }

    proc updateGeometry {this} {                    ;# update managed widget position and size according to outline current geometry
        set canvas $canvasHandles::($this,canvas)
        eval $canvas coords $canvasHandles::($this,item) [$canvas coords outline]
        $canvas itemconfigure $canvasHandles::($this,item) -width [$canvas itemcget $canvasHandles::(top,item) -width]\
            -height [$canvas itemcget $canvasHandles::(left,item) -height]
    }
}
