# copyright (C) 1997-98 Jean-Luc Fontaine (mailto:jfontain@mygale.org)
# this program is free software: please refer to the BSD type license enclosed in this package

set rcsId {$Id: datatab.tcl,v 1.15 1998/04/08 21:58:59 jfontain Exp $}

class dataTable {                       ;# module data view in the form of a table, that can be sorted using a column as a reference
    proc dataTable {this parentPath args} composite {[new scroll table $parentPath] $args} {
        set path $composite::($composite::($this,base),scrolled,path)
        $path tag configure title -font $font::(mediumBold) -anchor c
        $path configure -font $font::(mediumNormal) -rows 2 -state disabled -titlerows 1 -roworigin -1 -colstretchmode unset\
            -variable dataTable::${this}data
        $path tag configure title -font $font::(mediumBold) -anchor c
        # use selection background for highlighting
        $path tag configure highlight -font $font::(mediumNormal) -background [$path tag cget sel -background]

        bindtags $path "$path . all"           ;# remove all class bindings for we do not use any and they would cause interferences

        # allow dragging from data array
        set dataTable::($this,drag) [new dragSite -path $path -validcommand "dataTable::validateDrag $this"]
        dragSite::provide $dataTable::($this,drag) DATACELLS "dataTable::selectedDataCells $this"

        bind $path <ButtonRelease-1> "dataTable::columnSort $this %x %y %X %Y"

        set dataTable::($this,sortOrder) increasing
        set dataTable::($this,tablePath) $path
        composite::complete $this
    }

    proc ~dataTable {this} {
        variable ${this}data

        catch {unset ${this}data}                                                                      ;# eventually free table data
        delete $dataTable::($this,drag)
    }

    proc options {this} {
        return [list\
            [list -columns columns Columns 0 0]\
            [list -titlefont titleFont TitleFont $font::(mediumBold) $font::(mediumBold)]\
            [list -data data Data {} {}]\
        ]
    }

    proc set-columns {this value} {
        $dataTable::($this,tablePath) configure -cols $value
    }

    proc set-titlefont {this value} {
        $dataTable::($this,tablePath) tag configure title -font $value
    }

    proc set-data {this value} {                                                ;# value must be a fully qualified module data array
        variable ${this}data
        upvar $value data

        if {![array exists $value]} {
            error "\"$value\" argument is not an existing array"
        }
        set list {}
        for {set column 0} {$column<$data(columns)} {incr column} {
            lappend list -1,$column $data($column,label)
        }
        array set ${this}data $list                                                                              ;# set table titles
        $dataTable::($this,tablePath) configure -cols $column                                        ;# and fit to number of columns
        highlightColumn $this [lindex $data(sort) 0]
        set dataTable::($this,sortOrder) [lindex $data(sort) 1]
        trace variable ${value}(updates) w "dataTable::update $this"                                    ;# track module data updates
    }

    proc highlightColumn {this {column {}}} {                             ;# highlight column or return currently highlighted column
        set path $dataTable::($this,tablePath)
        set highlighted [lindex [$path tag col highlight] 0]
        if {[string length $column]==0} {                                     ;# return currently highlighted column if no arguments
            return $highlighted
        } else {
            if {[string length $highlighted]>0} {
                $path tag col {} $highlighted                                 ;# first eventually clear currently highlighted column
            }
            $path tag col highlight $column
        }
    }

    proc columnSort {this x y xRoot yRoot} {               ;# sort table rows using the column that the user selected as a reference
        set path $dataTable::($this,tablePath)
        # while dragging, mouse button may be released outside of table widget, so filter accordingly (use winfo instead of
        # comparing with window sizes as table may be clipped as container was scrolled or resized)
        if {[string compare [winfo containing $xRoot $yRoot] $path]!=0} return

        set number [expr {[$path cget -rows]-1}]                                                    ;# calculate number of data rows
        if {$number==0} return
        scan [$path index @$x,$y] %d,%d row column
        if {$row>=0} return                                                                                     ;# not the title row

        if {[string compare $column [highlightColumn $this]]==0} {                                           ;# sort the same column
            if {[string compare $dataTable::($this,sortOrder) increasing]==0} {                                ;# but toggle sorting
                set dataTable::($this,sortOrder) decreasing
            } else {
                set dataTable::($this,sortOrder) increasing
            }
        } else {                                                                ;# sort for the first time or for a different column
            set dataTable::($this,sortOrder) increasing
        }
        highlightColumn $this $column                                                                     ;# highlight sorted column
        update $this                                                                                     ;# update table immediately
    }

    proc update {this args} {                                   ;# update display using module data. ignore eventual trace arguments
        if {[string length $composite::($this,-data)]==0} return                                         ;# nothing to do if no data

        variable ${this}data
        upvar $composite::($this,-data) data

        set path $dataTable::($this,tablePath)

        set cursor [$path cget -cursor]                                                                               ;# save cursor
        $path configure -cursor watch                                                                  ;# show user that we are busy
        ::update idletasks

        set sortColumn [highlightColumn $this]                                                 ;# use highlighted column for sorting
        set lists {}
        foreach name [array names data *,0] {
            scan $name %u dataRow
            lappend lists [list $dataRow $data($dataRow,$sortColumn)]
        }
        # sort data rows according to highlighted column (column numbering is identical for table data and source data)
        set lists [lsort -$dataTable::($this,sortOrder) -$data($sortColumn,type) -index 1 $lists]

        set row 0
        foreach pair $lists {
            set dataRow [lindex $pair 0]
            set ${this}data($row,-1) $dataRow                                             ;# keep track of table / data rows mapping
            for {set column 0} {$column<$data(columns)} {incr column} {
                set ${this}data($row,$column) $data($dataRow,$column)
            }
            incr row
        }
        $path configure -rows [incr row]                                                ;# fit to data (take into account title row)
        highlightColumn $this $sortColumn                                      ;# refresh highlighted column in case rows were added

        $path configure -cursor $cursor                                                                            ;# restore cursor
        ::update idletasks
    }

### ultimately return list of selected cells ###
    proc selectedDataCells {this format} {
        variable ${this}data

        foreach {row column} [split [$dataTable::($this,tablePath) index @$dragSite::(x),$dragSite::(y)] ,] {}
        return $composite::($this,-data)([set ${this}data($row,-1)],$column)                ;# data cell format is array(row,column)
    }

    proc validateDrag {this x y} {
        scan [$dataTable::($this,tablePath) index @$x,$y] %d,%d row column
        if {($row<0)||($column<0)} {
            return 0                                                                      ;# drag can only originate from data cells
        }
        return 1
    }
}
