/*
 *  $Id: asciiexport.c 25063 2022-10-03 08:45:08Z yeti-dn $
 *  Copyright (C) 2003-2022 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/**
 * [FILE-MAGIC-USERGUIDE]
 * Text matrix of data values
 * .txt
 * Export
 **/

/**
 * [FILE-MAGIC-MISSING]
 * Export only.
 **/

#include "config.h"
#include <locale.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwyutils.h>
#include <libgwymodule/gwymodule-file.h>
#include <libprocess/datafield.h>
#include <app/gwyapp.h>

#include "err.h"

#define EXTENSION ".txt"

enum {
    PARAM_ADD_COMMENT,
    PARAM_DECIMAL_DOT,
    PARAM_CONCAT_ALL,
    PARAM_PRECISION,
};

typedef struct {
    gboolean needs_decimal_dot;
    guint decimal_dot_len;
    gchar *decimal_dot;
} DecimalDotInfo;

typedef struct {
    GwyParams *params;
    DecimalDotInfo decinfo;
} ModuleArgs;

static gboolean         module_register      (void);
static GwyParamDef*     define_module_params (void);
static gint             asciiexport_detect   (const GwyFileDetectInfo *fileinfo,
                                              gboolean only_name);
static gboolean         asciiexport_export   (GwyContainer *data,
                                              const gchar *filename,
                                              GwyRunType mode,
                                              GError **error);
static GwyDialogOutcome run_gui              (ModuleArgs *args);
static gboolean         export_one_channel   (GwyContainer *data,
                                              gint id,
                                              ModuleArgs *args,
                                              FILE *fh);
static void             fill_decimal_dot_info(DecimalDotInfo *info);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Exports data as simple ASCII matrix."),
    "Yeti <yeti@gwyddion.net>",
    "2.0",
    "David Nečas (Yeti)",
    "2004",
};

GWY_MODULE_QUERY2(module_info, asciiexport)

static gboolean
module_register(void)
{
    gwy_file_func_register("asciiexport",
                           N_("ASCII data matrix (.txt)"),
                           (GwyFileDetectFunc)&asciiexport_detect,
                           NULL,
                           NULL,
                           (GwyFileSaveFunc)&asciiexport_export);

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_file_func_current());
    gwy_param_def_add_boolean(paramdef, PARAM_ADD_COMMENT, "add-comment", _("Add _informational comment header"),
                              FALSE);
    gwy_param_def_add_boolean(paramdef, PARAM_DECIMAL_DOT, "decimal-dot", _("Use _dot as decimal separator"), TRUE);
    gwy_param_def_add_boolean(paramdef, PARAM_CONCAT_ALL, "concat-all", _("Conca_tenate exports of all images"), FALSE);
    gwy_param_def_add_int(paramdef, PARAM_PRECISION, "precision", _("_Precision"), 0, 16, 5);

    return paramdef;
}

static gint
asciiexport_detect(const GwyFileDetectInfo *fileinfo,
                   G_GNUC_UNUSED gboolean only_name)
{
    return g_str_has_suffix(fileinfo->name_lowercase, EXTENSION) ? 20 : 0;
}

static gboolean
asciiexport_export(GwyContainer *data,
                   const gchar *filename,
                   GwyRunType mode,
                   GError **error)
{
    ModuleArgs args;
    gint i, id, *ids;
    GwyDialogOutcome outcome;
    FILE *fh = NULL;
    gboolean ok = FALSE;

    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD_ID, &id, 0);
    if (id < 0) {
        err_NO_CHANNEL_EXPORT(error);
        return FALSE;
    }

    args.params = gwy_params_new_from_settings(define_module_params());
    fill_decimal_dot_info(&args.decinfo);
    if (!args.decinfo.needs_decimal_dot)
        gwy_params_set_boolean(args.params, PARAM_DECIMAL_DOT, TRUE);

    if (mode == GWY_RUN_INTERACTIVE) {
        outcome = run_gui(&args);
        gwy_params_save_to_settings(args.params);
        if (outcome == GWY_DIALOG_CANCEL) {
            err_CANCELLED(error);
            goto fail;
        }
    }

    if (!(fh = gwy_fopen(filename, "w"))) {
        err_OPEN_WRITE(error);
        goto fail;
    }

    if (gwy_params_get_boolean(args.params, PARAM_CONCAT_ALL)) {
        ids = gwy_app_data_browser_get_data_ids(data);
        for (i = 0; ids[i] >= 0; i++) {
            if (!export_one_channel(data, ids[i], &args, fh) || gwy_fprintf(fh, "\n") < 0) {
                err_WRITE(error);
                goto fail;
            }
        }
    }
    else {
        if (!export_one_channel(data, id, &args, fh)) {
            err_WRITE(error);
            goto fail;
        }
    }

    ok = TRUE;

fail:
    if (fh)
        fclose(fh);
    if (!ok)
        g_unlink(filename);

    g_object_unref(args.params);
    g_free(args.decinfo.decimal_dot);

    return ok;
}

static GwyDialogOutcome
run_gui(ModuleArgs *args)
{
    GwyDialog *dialog;
    GwyParamTable *table;

    dialog = GWY_DIALOG(gwy_dialog_new(_("Export Text")));
    gwy_dialog_add_buttons(dialog, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    table = gwy_param_table_new(args->params);
    gwy_param_table_append_header(table, -1, _("Options"));
    gwy_param_table_append_checkbox(table, PARAM_DECIMAL_DOT);
    gwy_param_table_set_sensitive(table, PARAM_DECIMAL_DOT, args->decinfo.needs_decimal_dot);
    gwy_param_table_append_checkbox(table, PARAM_ADD_COMMENT);
    gwy_param_table_append_checkbox(table, PARAM_CONCAT_ALL);
    gwy_param_table_append_slider(table, PARAM_PRECISION);
    gwy_param_table_slider_set_mapping(table, PARAM_PRECISION, GWY_SCALE_MAPPING_LINEAR);
    gwy_dialog_add_content(dialog, gwy_param_table_widget(table), FALSE, FALSE, 0);
    gwy_dialog_add_param_table(dialog, table);

    return gwy_dialog_run(dialog);
}

static inline gint
print_with_decimal_dot(FILE *fh,
                       gchar *formatted_number,
                       const gchar *decimal_dot,
                       guint decimal_dot_len)
{
    gchar *pos = strstr(formatted_number, decimal_dot);

    if (!pos)
        return fputs(formatted_number, fh);
    else {
        pos[0] = '.';
        if (decimal_dot_len == 1)
            return fputs(formatted_number, fh);
        else {
            gint l1, l2;

            pos[1] = '\0';
            if ((l1 = fputs(formatted_number, fh)) == EOF)
                return EOF;
            if ((l2 = fputs(pos + decimal_dot_len, fh)) == EOF)
                return EOF;
            return l1 + l2;
        }
    }
}

static gboolean
export_one_channel(GwyContainer *data, gint id, ModuleArgs *args, FILE *fh)
{
    GwyDataField *field = gwy_container_get_object(data, gwy_app_get_data_key_for_id(id));
    gboolean force_decimal_dot = (args->decinfo.needs_decimal_dot
                                  && gwy_params_get_boolean(args->params, PARAM_DECIMAL_DOT));
    gint precision = gwy_params_get_int(args->params, PARAM_PRECISION);
    gboolean add_comment = gwy_params_get_boolean(args->params, PARAM_ADD_COMMENT);
    gint xres, yres, i, ddlen = args->decinfo.decimal_dot_len;
    const gchar *ddot = args->decinfo.decimal_dot;
    const gdouble *d;
    gchar buf[40];

    g_return_val_if_fail(GWY_IS_DATA_FIELD(field), FALSE);
    xres = gwy_data_field_get_xres(field);
    yres = gwy_data_field_get_yres(field);
    d = gwy_data_field_get_data_const(field);

    if (add_comment) {
        gdouble xreal = gwy_data_field_get_xreal(field), yreal = gwy_data_field_get_yreal(field);
        GwySIUnit *units;
        GwySIValueFormat *vf = NULL;
        gchar *s;

        s = gwy_app_get_data_field_title(data, id);
        gwy_fprintf(fh, "# %s %s\n", _("Channel:"), s);
        g_free(s);

        vf = gwy_data_field_get_value_format_xy(field, GWY_SI_UNIT_FORMAT_VFMARKUP, vf);
        if (force_decimal_dot) {
            gwy_fprintf(fh, "# %s ", _("Width:"));
            g_snprintf(buf, sizeof(buf), "%.*f", vf->precision, xreal/vf->magnitude);
            print_with_decimal_dot(fh, buf, ddot, ddlen);
            gwy_fprintf(fh, " %s\n", vf->units);

            gwy_fprintf(fh, "# %s ", _("Height:"));
            g_snprintf(buf, sizeof(buf), "%.*f", vf->precision, yreal/vf->magnitude);
            print_with_decimal_dot(fh, buf, ddot, ddlen);
            gwy_fprintf(fh, " %s\n", vf->units);
        }
        else {
            gwy_fprintf(fh, "# %s %.*f %s\n", _("Width:"), vf->precision, xreal/vf->magnitude, vf->units);
            gwy_fprintf(fh, "# %s %.*f %s\n", _("Height:"), vf->precision, yreal/vf->magnitude, vf->units);
        }

        units = gwy_data_field_get_si_unit_z(field);
        s = gwy_si_unit_get_string(units, GWY_SI_UNIT_FORMAT_VFMARKUP);
        gwy_fprintf(fh, "# %s %s\n", _("Value units:"), s);
        g_free(s);

        gwy_si_unit_value_format_free(vf);
    }

    if (force_decimal_dot) {
        for (i = 0; i < xres*yres; i++) {
            g_snprintf(buf, sizeof(buf), "%.*g%c", precision, d[i], (i + 1) % xres ? '\t' : '\n');
            if (print_with_decimal_dot(fh, buf, ddot, ddlen) == EOF)
                return FALSE;
        }
    }
    else {
        for (i = 0; i < xres*yres; i++) {
            if (gwy_fprintf(fh, "%.*g%c", precision, d[i], (i + 1) % xres ? '\t' : '\n') < 2)
                return FALSE;
        }
    }

    return TRUE;
}

static void
fill_decimal_dot_info(DecimalDotInfo *info)
{
    struct lconv *locale_data;
    guint len;

    locale_data = localeconv();
    info->decimal_dot = g_strdup(locale_data->decimal_point);
    if (!info->decimal_dot || !(len = strlen(info->decimal_dot))) {
        g_warning("Cannot get decimal dot information from localeconv().");
        g_free(info->decimal_dot);
        info->decimal_dot = g_strdup(".");
        len = 1;
    }
    info->decimal_dot_len = len;
    info->needs_decimal_dot = !gwy_strequal(info->decimal_dot, ".");
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
