// Copyright (C) 2024 The Android Open Source Project
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

import {Trace} from './trace';
import {App} from './app';
import {RouteArgs} from './route_schema';

/**
 * This interface defines the shape of the plugins's class constructor (i.e. the
 * the constructor and all static members of the plugin's class.
 *
 * This class constructor is registered with the core.
 *
 * On trace load, the core will create a new class instance by calling new on
 * this constructor and then call its onTraceLoad() function.
 */
export interface PerfettoPluginStatic<T extends PerfettoPlugin> {
  readonly id: string;
  readonly description?: string;
  readonly dependencies?: ReadonlyArray<PerfettoPluginStatic<PerfettoPlugin>>;
  onActivate?(app: App, args: RouteArgs): void;
  new (trace: Trace): T;
}

/**
 * This interface defines the shape of a plugin's trace-scoped instance, which
 * is created from the class constructor above at trace load time.
 */
export interface PerfettoPlugin {
  onTraceLoad?(ctx: Trace, args?: {[key: string]: unknown}): Promise<void>;
}

export interface PluginManager {
  getPlugin<T extends PerfettoPlugin>(plugin: PerfettoPluginStatic<T>): T;
}
